/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-srt.h"
#include "ogmrip-backend.h"

#include "ogmjob-queue.h"
#include "ogmjob-enums.h"
#include "ogmjob-exec.h"

#include <string.h>
#include <unistd.h>

#include <glib/gstdio.h>

static gint ogmrip_srt_run (OGMJobSpawn *spawn);

G_DEFINE_TYPE (OGMRipSrt, ogmrip_srt, OGMRIP_TYPE_SUBP)

static void
ogmrip_srt_class_init (OGMRipSrtClass *klass)
{
  OGMJobSpawnClass *spawn_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  spawn_class->run = ogmrip_srt_run;
}

static void
ogmrip_srt_init (OGMRipSrt *srt)
{
}

static gint
ogmrip_srt_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gint result, fd;
  gchar **argv;

  GDir *dir;
  GPatternSpec *pspec;
  gchar *pattern, *str;
  const gchar *name;

  gchar *tmp_file, *srtx_file;

  result = OGMJOB_RESULT_ERROR;

  fd = g_file_open_tmp ("sub.XXXXXX", &tmp_file, NULL);
  if (fd < 0)
    return OGMJOB_RESULT_ERROR;
  g_unlink (tmp_file);
  close (fd);

  srtx_file = g_strconcat (tmp_file, ".srtx", NULL);

  argv = ogmrip_backend_vobsub_command (OGMRIP_SUBP (spawn), NULL, tmp_file);
  if (argv)
  {
    child = ogmjob_exec_newv (argv);
    ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_backend_vobsub_watch, spawn, TRUE, FALSE, FALSE);
    ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
    g_object_unref (child);

    result = OGMJOB_SPAWN_CLASS (ogmrip_srt_parent_class)->run (spawn);

    ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);
  }

  if (result == OGMJOB_RESULT_COMPLETED)
  {
    result = OGMJOB_RESULT_ERROR;

    argv = ogmrip_backend_pgm_command (OGMRIP_SUBP (spawn), tmp_file, NULL);
    if (argv)
    {
      child = ogmjob_exec_newv (argv);
      ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_backend_pgm_watch, spawn, TRUE, FALSE, FALSE);
      result = ogmjob_spawn_run (child, NULL);
      g_object_unref (child);
    }
  }

  if (result == OGMJOB_RESULT_COMPLETED)
  {
    result = OGMJOB_RESULT_ERROR;

    dir = g_dir_open (g_get_tmp_dir (), 0, NULL);
    if (dir)
    {
      gchar *basename;

      basename = g_path_get_basename (tmp_file);
      pattern = g_strconcat (basename, "*.pgm", NULL);
      pspec = g_pattern_spec_new (pattern);
      g_free (basename);
      g_free (pattern);

      while ((name = g_dir_read_name (dir)))
      {
        if (g_pattern_match (pspec, strlen (name), name, NULL))
        {
          str = g_build_filename (g_get_tmp_dir (), name, NULL);
          argv = ogmrip_backend_ocr_command (OGMRIP_SUBP (spawn), str, NULL);

          if (argv)
          {
            child = ogmjob_exec_newv (argv);
            ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_backend_ocr_watch, spawn, FALSE, TRUE, TRUE);
            result = ogmjob_spawn_run (child, NULL);
            g_object_unref (child);

            if (result != OGMJOB_RESULT_COMPLETED)
              break;
          }

          g_unlink (str);
          g_free (str);
        }
      }

      g_pattern_spec_free (pspec);
      g_dir_close (dir);
    }
  }

  if (result == OGMJOB_RESULT_COMPLETED)
  {
    result = OGMJOB_RESULT_ERROR;

    argv = ogmrip_backend_srt_command (OGMRIP_SUBP (spawn), srtx_file, NULL);
    if (argv)
    {
      child = ogmjob_exec_newv (argv);
      result = ogmjob_spawn_run (child, NULL);
      g_object_unref (child);
    }
  }

  if (g_file_test (srtx_file, G_FILE_TEST_EXISTS))
    g_unlink (srtx_file);
  g_free (srtx_file);

  srtx_file = g_strconcat (tmp_file, ".idx", NULL);
  if (g_file_test (srtx_file, G_FILE_TEST_EXISTS))
    g_unlink (srtx_file);
  g_free (srtx_file);

  srtx_file = g_strconcat (tmp_file, ".sub", NULL);
  if (g_file_test (srtx_file, G_FILE_TEST_EXISTS))
    g_unlink (srtx_file);
  g_free (srtx_file);

  dir = g_dir_open (g_get_tmp_dir (), 0, NULL);
  if (dir)
  {
    gchar *basename;

    basename = g_path_get_basename (tmp_file);
    pattern = g_strconcat (basename, "*.pgm*", NULL);
    pspec = g_pattern_spec_new (pattern);
    g_free (basename);
    g_free (pattern);

    while ((name = g_dir_read_name (dir)))
    {
      if (g_pattern_match (pspec, strlen (name), name, NULL))
      {
        str = g_build_filename (g_get_tmp_dir (), name, NULL);
        g_unlink (str);
        g_free (str);
      }
    }

    g_pattern_spec_free (pspec);
    g_dir_close (dir);
  }

  g_free (tmp_file);

  return result;
}

OGMJobSpawn *
ogmrip_srt_new (OGMDvdSubpStream *subp, const gchar *output)
{
  OGMJobSpawn *spawn;

  spawn = g_object_new (OGMRIP_TYPE_SRT, NULL);
  ogmrip_codec_set_output (OGMRIP_CODEC (spawn), output);
  ogmrip_subp_set_dvd_subp_stream (OGMRIP_SUBP (spawn), subp);

  return spawn;
}

