/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-lavc.h"
#include "ogmrip-backend.h"

#include "ogmjob-enums.h"
#include "ogmjob-exec.h"

#include <stdio.h>

#define OGMRIP_LAVC_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_LAVC, OGMRipLavcPriv))

struct _OGMRipLavcPriv
{
  guint header;
  guint cmp, precmp, subcmp;
  gint dia, predia;
};

static gint ogmrip_lavc_run (OGMJobSpawn *spawn);

G_DEFINE_TYPE (OGMRipLavc, ogmrip_lavc, OGMRIP_TYPE_VIDEO)

static void
ogmrip_lavc_class_init (OGMRipLavcClass *klass)
{
  OGMJobSpawnClass *spawn_class;

  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  spawn_class->run = ogmrip_lavc_run;

  g_type_class_add_private (klass, sizeof (OGMRipLavcPriv));
}

static void
ogmrip_lavc_init (OGMRipLavc *lavc)
{
  lavc->priv = OGMRIP_LAVC_GET_PRIVATE (lavc);
  lavc->priv->cmp = 2;
  lavc->priv->precmp = 2;
  lavc->priv->subcmp = 2;
  lavc->priv->dia = 2;
  lavc->priv->predia = 2;
}

static gint
ogmrip_lavc_run (OGMJobSpawn *spawn)
{
  OGMJobSpawn *child;
  gchar **argv;
  gint result;

  argv = ogmrip_backend_lavc_command (OGMRIP_VIDEO (spawn), NULL, NULL, NULL);
  if (!argv)
    return OGMJOB_RESULT_ERROR;

  child = ogmjob_exec_newv (argv);
  ogmjob_exec_add_watch_full (OGMJOB_EXEC (child), (OGMJobWatch) ogmrip_backend_lavc_watch, spawn, TRUE, FALSE, FALSE);
  ogmjob_container_add (OGMJOB_CONTAINER (spawn), child);
  g_object_unref (child);

  result = OGMJOB_SPAWN_CLASS (ogmrip_lavc_parent_class)->run (spawn);

  ogmjob_container_remove (OGMJOB_CONTAINER (spawn), child);

  return result;
}

OGMJobSpawn *
ogmrip_lavc_new (OGMDvdTitle *title, const gchar *output)
{
  OGMJobSpawn *spawn;

  spawn = g_object_new (OGMRIP_TYPE_LAVC, NULL);
  ogmrip_codec_set_output (OGMRIP_CODEC (spawn), output);
  ogmrip_codec_set_input (OGMRIP_CODEC (spawn), title);

  return spawn;
}

void
ogmrip_lavc_set_cmp (OGMRipLavc *lavc, guint cmp, guint precmp, guint subcmp)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));

  lavc->priv->cmp = MIN (cmp, 2000);
  lavc->priv->precmp = MIN (precmp, 2000);
  lavc->priv->subcmp = MIN (subcmp, 2000);
}

void
ogmrip_lavc_get_cmp (OGMRipLavc *lavc, guint *cmp, guint *precmp, guint *subcmp)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));
  g_return_if_fail (precmp != NULL);
  g_return_if_fail (subcmp != NULL);
  g_return_if_fail (cmp != NULL);

  *cmp = lavc->priv->cmp;
  *precmp = lavc->priv->precmp;
  *subcmp = lavc->priv->subcmp;
}

void
ogmrip_lavc_set_dia (OGMRipLavc *lavc, gint dia, gint predia)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));

  lavc->priv->dia = CLAMP (dia, -99, 6);
  lavc->priv->predia = CLAMP (predia, -99, 6);
}

void
ogmrip_lavc_get_dia (OGMRipLavc *lavc, gint *dia, gint *predia)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));
  g_return_if_fail (predia != NULL);
  g_return_if_fail (dia != NULL);

  *dia = lavc->priv->dia;
  *predia = lavc->priv->predia;
}

void
ogmrip_lavc_set_header (OGMRipLavc *lavc, guint header)
{
  g_return_if_fail (OGMRIP_IS_LAVC (lavc));

  lavc->priv->header = CLAMP (header, OGMRIP_LAVC_HEADER_AUTO, OGMRIP_LAVC_HEADER_COMBINE);
}

gint
ogmrip_lavc_get_header (OGMRipLavc *lavc)
{
  g_return_val_if_fail (OGMRIP_IS_LAVC (lavc), -1);

  return lavc->priv->header;
}

