/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "ogmrip-fs.h"

#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include <glib/gstdio.h>
#include <glib/gi18n-lib.h>

#ifdef HAVE_SYS_STATVFS_H
#include <sys/statvfs.h>
#endif

static gchar *ogmrip_tmp_dir = NULL;

GQuark
ogmrip_fs_error_quark (void)
{
  static GQuark quark = 0;

  if (quark == 0)
    quark = g_quark_from_static_string ("ogmrip-fs-error-quark");

  return quark;
}

G_CONST_RETURN gchar *
ogmrip_fs_get_tmp_dir (void)
{
  if (!ogmrip_tmp_dir)
    ogmrip_fs_set_tmp_dir (NULL);

  return ogmrip_tmp_dir;
}

void
ogmrip_fs_set_tmp_dir (const gchar *dir)
{
  if (ogmrip_tmp_dir)
    g_free (ogmrip_tmp_dir);

  if (!dir)
    dir = g_get_tmp_dir ();

  g_return_if_fail (g_file_test (dir, G_FILE_TEST_IS_DIR));

  ogmrip_tmp_dir = g_strdup (dir);
}

gboolean
ogmrip_fs_mkdir (const gchar *path, mode_t mode, GError **error)
{
  const gchar *sep;
  gchar *current;

  g_return_val_if_fail (path && *path, FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  for (sep = path; *sep; sep ++)
  {
    for (; *sep; sep ++)
      if (*sep == G_DIR_SEPARATOR)
        break;

    if (sep - path > 0)
    {
      current = g_strndup (path, sep - path);

      if (g_mkdir (current, mode) < 0)
      {
        g_set_error (error, OGMRIP_FS_ERROR, OGMRIP_FS_ERROR_INTERNAL, 
            _("Could not create directory %s: %s"), path, g_strerror (errno));
        g_free (current);
        return FALSE;
      }
      g_free (current);
    }
  }
  return TRUE;
}

gboolean
ogmrip_fs_rmdir (const gchar *path, gboolean recursive, GError **error)
{
  g_return_val_if_fail (path && *path, FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);
  g_return_val_if_fail (g_file_test (path, G_FILE_TEST_IS_DIR), FALSE);

  if (recursive)
  {
    GDir *dir;
    GError *tmp_error = NULL;

    gchar *filename;
    const gchar *name;

    dir = g_dir_open (path, 0, &tmp_error);
    if (!dir)
    {
      g_propagate_error (error, tmp_error);
      return FALSE;
    }

    while ((name = g_dir_read_name (dir)))
    {
      filename = g_build_filename (path, name, NULL);
      if (g_file_test (filename, G_FILE_TEST_IS_DIR))
      {
        if (!ogmrip_fs_rmdir (filename, TRUE, &tmp_error))
        {
          if (tmp_error)
            g_propagate_error (error, tmp_error);
          g_free (filename);
          return FALSE;
        }
      }
      else
      {
        if (g_unlink (filename) != 0)
        {
          g_free (filename);
          return FALSE;
        }
      }
      g_free (filename);
    }

    g_dir_close (dir);
  }

  if (g_rmdir (path) != 0)
    return FALSE;

  return TRUE;
}

gchar *
ogmrip_fs_mktemp (const gchar *tmpl, GError **error)
{
  gchar *filename;
  int fd;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

  filename = g_build_filename (ogmrip_fs_get_tmp_dir (), tmpl, NULL);
  fd = g_mkstemp (filename);
  if (fd < 0)
  {
    g_set_error (error, G_FILE_ERROR, g_file_error_from_errno (errno), 
        _("Failed to create file '%s': %s"), filename, g_strerror (errno));
    g_free (filename);
    return NULL;
  }

  return filename;
}

gchar *
ogmrip_fs_mkftemp (const gchar *tmpl, GError **error)
{
  GError *tmp_error = NULL;
  gchar *name;
  gint fd;

  fd = ogmrip_fs_open_tmp (tmpl, &name, &tmp_error);

  if (fd < 0)
  {
    g_propagate_error (error, tmp_error);
    return NULL;
  }

  close (fd);
  g_unlink (name);
  mkfifo (name, 0666);

  return name;
}

gchar *
ogmrip_fs_mkdtemp (const gchar *tmpl, GError **error)
{
  gchar *path;

  g_return_val_if_fail (tmpl && *tmpl, FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  path = g_build_filename (ogmrip_fs_get_tmp_dir (), tmpl, NULL);

  if (!mkdtemp (path))
  {
    g_set_error (error, OGMRIP_FS_ERROR, OGMRIP_FS_ERROR_INTERNAL, 
        _("Could not create directory %s: %s"), path, g_strerror (errno));
    g_free (path);
    return NULL;
  }

  return path;
}

gint
ogmrip_fs_open_tmp (const gchar *tmpl, gchar **name_used, GError **error)
{
  const gchar *tmpdir;
  gchar *fulltmpl;
  gint retval;

  g_return_val_if_fail (error == NULL || *error == NULL, -1);

  if (!tmpl)
    tmpl = ".XXXXXX";

  if (!g_str_has_suffix (tmpl, "XXXXXX"))
  {
    g_set_error (error, G_FILE_ERROR, G_FILE_ERROR_FAILED, 
        _("Template '%s' doesn't end with XXXXXX"), tmpl);
    return -1;
  }

  if ((strchr (tmpl, G_DIR_SEPARATOR)) != NULL
#ifdef G_OS_WIN32
      || (strchr (tmpl, '/') != NULL)
#endif
      )
  {
    g_set_error (error, G_FILE_ERROR, G_FILE_ERROR_FAILED, 
        _("Template '%s' invalid, should not contain a '/'"), tmpl);
    return -1;
  }

  tmpdir = ogmrip_fs_get_tmp_dir ();
  fulltmpl = g_build_filename (tmpdir, tmpl, NULL);

  retval = g_mkstemp (fulltmpl);
  if (retval < 0)
  {
    g_set_error (error, G_FILE_ERROR, g_file_error_from_errno (errno), 
        _("Failed to create file '%s': %s"), tmpl, g_strerror (errno));
    g_free (fulltmpl);
    return -1;
  }

  if (name_used)
    *name_used = fulltmpl;
  else
    g_free (fulltmpl);

  return retval;
}

gint64
ogmrip_fs_get_left_space (const gchar *filename)
{
  gint status;
  gchar *dirname;
#ifdef HAVE_STATVFS
  struct statvfs statfs_buf;
#else
  struct statfs statfs_buf;
#endif

  g_return_val_if_fail (filename && *filename, -1);

  if (g_file_test (filename, G_FILE_TEST_IS_DIR))
    dirname = g_strdup (filename);
  else
    dirname = g_path_get_dirname (filename);

  if (!g_file_test (dirname, G_FILE_TEST_EXISTS))
  {
    g_free (dirname);
    return -1;
  }

#ifdef HAVE_STATVFS
  status = statvfs (dirname, &statfs_buf);
#else
  status = statfs (dirname, &statfs_buf);
#endif

  g_free (dirname);

  if (status < 0)
    return -1;

#ifdef OGMRIP_DEBUG
  g_message ("Space left on device containing '%s': %" G_GINT64_FORMAT " bytes", 
      filename, (gint64) statfs_buf.f_bsize * (gint64) statfs_buf.f_bavail);
#endif

  return (gint64) statfs_buf.f_bsize * (gint64) statfs_buf.f_bavail;
}

gchar *
ogmrip_fs_get_mount_point (const gchar *filename)
{
  gchar *cwd, *dirname, *mp = NULL;
  struct stat cur_stat, last_stat;

  g_return_val_if_fail (filename && *filename, NULL);

  cwd = g_get_current_dir ();

  if (g_file_test (filename, G_FILE_TEST_IS_DIR))
    dirname = g_strdup (filename);
  else
    dirname = g_path_get_dirname (filename);

  if (!g_file_test (dirname, G_FILE_TEST_EXISTS))
    goto done;

  if (g_stat (dirname, &last_stat) < 0)
    goto done;

  if (g_chdir (dirname) < 0)
    goto done;

  for (;;)
  {
    if (g_stat ("..", &cur_stat) < 0)
      goto done;

    if (cur_stat.st_dev != last_stat.st_dev || cur_stat.st_ino == last_stat.st_ino)
      break;

    if (g_chdir ("..") < 0)
      goto done;

    last_stat = cur_stat;
  }

  mp = g_get_current_dir ();

done:
  g_chdir (cwd);
  g_free (cwd);
  g_free (dirname);

  return mp;
}

void
ogmrip_fs_unref (gchar *filename, gboolean do_unlink)
{
  if (filename)
  {
    if (do_unlink)
    {
      if (g_file_test (filename, G_FILE_TEST_IS_DIR))
        ogmrip_fs_rmdir (filename, TRUE, NULL);
      else if (g_file_test (filename, G_FILE_TEST_EXISTS))
        g_unlink (filename);
    }
    g_free (filename);
  }
}

gboolean
ogmrip_fs_rename (const gchar *old_name, const gchar *new_name, GError **error)
{
  g_return_val_if_fail (old_name != NULL, FALSE);
  g_return_val_if_fail (new_name != NULL, FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  if (strcmp (old_name, new_name) == 0)
    return TRUE;

  if (g_file_test (new_name, G_FILE_TEST_EXISTS))
  {
    if (!g_file_test (new_name, G_FILE_TEST_IS_REGULAR))
      return FALSE;
    if (g_unlink (new_name) < 0)
      return FALSE;
  }

  if (g_rename (old_name, new_name) < 0)
    return FALSE;

  return TRUE;
}

