/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-container.h"
#include "ogmrip-backend.h"
#include "ogmjob-enums.h"
#include "ogmjob-exec.h"

#include <unistd.h>
#include <glib/gstdio.h>

#define OGMRIP_CONTAINER_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_CONTAINER, OGMRipContainerPriv))

struct _OGMRipContainerPriv
{
  gchar *label;
  gchar *output;
  gchar *fourcc;

  guint tsize;
  guint tnumber;
  guint overhead;

  GList *subp;
  GList *audio;
  GList *chapters;
  OGMRipVideo *video;
};

typedef struct
{
  OGMRipCodec *codec;
  gint language;
  guint demuxer;
} OGMRipContainerChild;

enum 
{
  PROP_0,
  PROP_OUTPUT,
  PROP_LABEL,
  PROP_FOURCC,
  PROP_TSIZE,
  PROP_TNUMBER,
  PROP_OVERHEAD
};

static void ogmrip_container_finalize     (GObject      *gobject);
static void ogmrip_container_set_property (GObject      *gobject,
                                           guint        property_id,
                                           const GValue *value,
                                           GParamSpec   *pspec);
static void ogmrip_container_get_property (GObject      *gobject,
                                           guint        property_id,
                                           GValue       *value,
                                           GParamSpec   *pspec);

G_DEFINE_ABSTRACT_TYPE (OGMRipContainer, ogmrip_container, OGMJOB_TYPE_BIN)

static void
ogmrip_container_class_init (OGMRipContainerClass *klass)
{
  GObjectClass *gobject_class;
  OGMJobSpawnClass *spawn_class;

  gobject_class = G_OBJECT_CLASS (klass);
  spawn_class = OGMJOB_SPAWN_CLASS (klass);

  gobject_class->finalize = ogmrip_container_finalize;
  gobject_class->set_property = ogmrip_container_set_property;
  gobject_class->get_property = ogmrip_container_get_property;

  g_object_class_install_property (gobject_class, PROP_OUTPUT, 
        g_param_spec_string ("output", "Output property", "Set output file", 
           NULL, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_LABEL, 
        g_param_spec_string ("label", "Label property", "Set label", 
           NULL, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_FOURCC, 
        g_param_spec_string ("fourcc", "FourCC property", "Set fourcc", 
           NULL, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_TSIZE, 
        g_param_spec_uint ("target-size", "Target size property", "Set target size", 
           0, G_MAXUINT, 0, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_TNUMBER, 
        g_param_spec_uint ("target-number", "Target number property", "Set target number", 
           0, G_MAXUINT, 1, G_PARAM_READWRITE));

  g_object_class_install_property (gobject_class, PROP_OVERHEAD, 
        g_param_spec_uint ("overhead", "Overhead property", "Set overhead", 
           0, G_MAXUINT, 6, G_PARAM_CONSTRUCT_ONLY | G_PARAM_READWRITE));

  g_type_class_add_private (klass, sizeof (OGMRipContainerPriv));
}

static void
ogmrip_container_init (OGMRipContainer *container)
{
  container->priv = OGMRIP_CONTAINER_GET_PRIVATE (container);
  container->priv->overhead = 6;
  container->priv->tnumber = 1;
  container->priv->tsize = 0;
}

static void
ogmrip_stream_free (OGMRipContainerChild *child)
{
  if (child->codec)
    g_object_unref (child->codec);

  g_free (child);
}

static void
ogmrip_container_finalize (GObject *gobject)
{
  OGMRipContainer *container;

  container = OGMRIP_CONTAINER (gobject);

  if (container->priv->video)
    g_object_unref (container->priv->video);
  container->priv->video = NULL;

  g_list_foreach (container->priv->audio, (GFunc) ogmrip_stream_free, NULL);
  g_list_free (container->priv->audio);
  container->priv->audio = NULL;

  g_list_foreach (container->priv->chapters, (GFunc) ogmrip_stream_free, NULL);
  g_list_free (container->priv->chapters);
  container->priv->chapters = NULL;

  g_list_foreach (container->priv->subp, (GFunc) ogmrip_stream_free, NULL);
  g_list_free (container->priv->subp);
  container->priv->subp = NULL;

  g_free (container->priv->label);
  container->priv->label = NULL;

  g_free (container->priv->output);
  container->priv->output = NULL;

  g_free (container->priv->fourcc);
  container->priv->fourcc = NULL;

  G_OBJECT_CLASS (ogmrip_container_parent_class)->finalize (gobject);
}

static void
ogmrip_container_set_property (GObject *gobject, guint property_id, const GValue *value, GParamSpec *pspec)
{
  OGMRipContainer *container;

  container = OGMRIP_CONTAINER (gobject);

  switch (property_id) 
  {
    case PROP_OUTPUT:
      ogmrip_container_set_output (container, g_value_get_string (value));
      break;
    case PROP_LABEL: 
      ogmrip_container_set_label (container, g_value_get_string (value));
      break;
    case PROP_FOURCC: 
      ogmrip_container_set_fourcc (container, g_value_get_string (value));
      break;
    case PROP_TSIZE: 
      container->priv->tsize = g_value_get_uint (value);
      break;
    case PROP_TNUMBER: 
      container->priv->tnumber = g_value_get_uint (value);
      break;
    case PROP_OVERHEAD: 
      container->priv->overhead = g_value_get_uint (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static void
ogmrip_container_get_property (GObject *gobject, guint property_id, GValue *value, GParamSpec *pspec)
{
  OGMRipContainer *container;

  container = OGMRIP_CONTAINER (gobject);

  switch (property_id) 
  {
    case PROP_OUTPUT:
      g_value_set_static_string (value, container->priv->output);
      break;
    case PROP_LABEL:
      g_value_set_static_string (value, container->priv->label);
      break;
    case PROP_FOURCC:
      g_value_set_static_string (value, container->priv->fourcc);
      break;
    case PROP_TSIZE:
      g_value_set_uint (value, container->priv->tsize);
      break;
    case PROP_TNUMBER:
      g_value_set_uint (value, container->priv->tnumber);
      break;
    case PROP_OVERHEAD:
      g_value_set_uint (value, container->priv->overhead);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (gobject, property_id, pspec);
      break;
  }
}

static GList *
ogmrip_container_copy_list (GList *list)
{
  OGMRipContainerChild *child;
  GList *new_list = NULL;

  while (list)
  {
    child = list->data;
    new_list = g_list_append (new_list, child->codec);
    list = list->next;
  }

  return new_list;
}

static void
ogmrip_container_foreach (OGMRipContainer *container, GList *list, OGMRipContainerFunc func, gpointer data)
{
  OGMRipContainerChild *child;

  while (list)
  {
    child = (OGMRipContainerChild *) list->data;
    func (container, child->codec, child->demuxer, child->language, data);
    list = list->next;
  }
}

G_CONST_RETURN gchar *
ogmrip_container_get_output (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  return container->priv->output;
}

void
ogmrip_container_set_output (OGMRipContainer *container, const gchar *output)
{
  g_return_if_fail (OGMRIP_IS_CONTAINER (container));
  g_return_if_fail (output && *output);

  g_free (container->priv->output);
  container->priv->output = g_strdup (output);
}

G_CONST_RETURN gchar *
ogmrip_container_get_label (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  return container->priv->label;
}

void
ogmrip_container_set_label (OGMRipContainer *container, const gchar *label)
{
  g_return_if_fail (OGMRIP_IS_CONTAINER (container));

  g_free (container->priv->label);
  container->priv->label = label ? g_strdup (label) : NULL;
}

G_CONST_RETURN gchar *
ogmrip_container_get_fourcc (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  return container->priv->fourcc;
}

void
ogmrip_container_set_fourcc (OGMRipContainer *container, const gchar *fourcc)
{
  gchar *str;

  g_return_if_fail (OGMRIP_IS_CONTAINER (container));

  if (container->priv->fourcc)
    g_free (container->priv->fourcc);
  container->priv->fourcc = NULL;

  if (fourcc)
  {
    str = g_utf8_strup (fourcc, -1);
    container->priv->fourcc = g_strndup (str, 4);
    g_free (str);
  }
}

OGMRipVideo *
ogmrip_container_get_video (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  return container->priv->video;
}

void
ogmrip_container_set_video (OGMRipContainer *container, OGMRipVideo *video)
{
  g_return_if_fail (OGMRIP_IS_CONTAINER (container));
  g_return_if_fail (OGMRIP_IS_VIDEO (video));

  g_object_ref (video);
  if (container->priv->video)
    g_object_unref (container->priv->video);
  container->priv->video = video;
}

void
ogmrip_container_add_audio (OGMRipContainer *container, OGMRipAudio *audio, OGMRipAudioDemuxer demuxer, gint language)
{
  OGMRipContainerChild *child;

  g_return_if_fail (OGMRIP_IS_CONTAINER (container));
  g_return_if_fail (OGMRIP_IS_AUDIO (audio));

  child = g_new0 (OGMRipContainerChild, 1);

  g_object_ref (audio);
  child->codec = OGMRIP_CODEC (audio);

  child->language = language;
  child->demuxer = demuxer;

  container->priv->audio = g_list_append (container->priv->audio, child);
}

GList *
ogmrip_container_get_audio (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  return ogmrip_container_copy_list (container->priv->audio);
}

OGMRipAudio *
ogmrip_container_get_nth_audio (OGMRipContainer *container, gint n)
{
  OGMRipContainerChild *child;
  GList *link;

  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  if (n < 0)
    link = g_list_last (container->priv->audio);
  else
    link = g_list_nth (container->priv->audio, n);

  if (!link)
    return NULL;

  child = link->data;

  return OGMRIP_AUDIO (child->codec);
}

gint
ogmrip_container_get_n_audio (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), -1);

  return g_list_length (container->priv->audio);
}

void
ogmrip_container_foreach_audio (OGMRipContainer *container, OGMRipContainerFunc func, gpointer data)
{
  g_return_if_fail (OGMRIP_IS_CONTAINER (container));
  g_return_if_fail (func != NULL);

  ogmrip_container_foreach (container, container->priv->audio, func, data);
}

void
ogmrip_container_add_subp (OGMRipContainer *container, OGMRipSubp *subp, OGMRipSubpDemuxer demuxer, gint language)
{
  OGMRipContainerChild *child;

  g_return_if_fail (OGMRIP_IS_CONTAINER (container));
  g_return_if_fail (OGMRIP_IS_SUBP (subp));

  child = g_new0 (OGMRipContainerChild, 1);

  g_object_ref (subp);
  child->codec = OGMRIP_CODEC (subp);

  child->language = language;
  child->demuxer = demuxer;

  container->priv->subp = g_list_append (container->priv->subp, child);
}

GList *
ogmrip_container_get_subp (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  return ogmrip_container_copy_list (container->priv->subp);
}

OGMRipSubp *
ogmrip_container_get_nth_subp (OGMRipContainer *container, gint n)
{
  OGMRipContainerChild *child;
  GList *link;

  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  if (n < 0)
    link = g_list_last (container->priv->subp);
  else
    link = g_list_nth (container->priv->subp, n);

  if (!link)
    return NULL;

  child = link->data;

  return OGMRIP_SUBP (child->codec);
}

gint
ogmrip_container_get_n_subp (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), -1);

  return g_list_length (container->priv->subp);
}

void
ogmrip_container_foreach_subp (OGMRipContainer *container, OGMRipContainerFunc func, gpointer data)
{
  g_return_if_fail (OGMRIP_IS_CONTAINER (container));
  g_return_if_fail (func != NULL);

  ogmrip_container_foreach (container, container->priv->subp, func, data);
}

void
ogmrip_container_add_chapters (OGMRipContainer *container, OGMRipChapters *chapters)
{
  OGMRipContainerChild *child;

  g_return_if_fail (OGMRIP_IS_CONTAINER (container));
  g_return_if_fail (OGMRIP_IS_CHAPTERS (chapters));

  child = g_new0 (OGMRipContainerChild, 1);

  g_object_ref (chapters);
  child->codec = OGMRIP_CODEC (chapters);

  container->priv->chapters = g_list_append (container->priv->chapters, child);
}

GList *
ogmrip_container_get_chapters (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  return ogmrip_container_copy_list (container->priv->chapters);
}

OGMRipChapters *
ogmrip_container_get_nth_chapters (OGMRipContainer *container, gint n)
{
  OGMRipContainerChild *child;
  GList *link;

  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);

  if (n < 0)
    link = g_list_last (container->priv->chapters);
  else
    link = g_list_nth (container->priv->chapters, n);

  if (!link)
    return NULL;

  child = link->data;

  return OGMRIP_CHAPTERS (child->codec);
}

gint
ogmrip_container_get_n_chapters (OGMRipContainer *container)
{
  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), -1);

  return g_list_length (container->priv->chapters);
}

void
ogmrip_container_foreach_chapters (OGMRipContainer *container, OGMRipContainerFunc func, gpointer data)
{
  g_return_if_fail (OGMRIP_IS_CONTAINER (container));
  g_return_if_fail (func != NULL);

  ogmrip_container_foreach (container, container->priv->chapters, func, data);
}

void
ogmrip_container_set_split (OGMRipContainer *container, guint number, guint size)
{
   g_return_if_fail (OGMRIP_IS_CONTAINER (container));

   if (number > 0)
     container->priv->tnumber = number;

   if (size > 0)
     container->priv->tsize = size;
}

void
ogmrip_container_get_split (OGMRipContainer *container, guint *number, guint *size)
{
  g_return_if_fail (OGMRIP_IS_CONTAINER (container));

  if (number)
    *number = container->priv->tnumber;

  if (size)
    *size = container->priv->tsize;
}

static gint64
ogmrip_container_get_audio_overhead (OGMRipContainer *container, OGMRipContainerChild *child)
{
  glong length;
  gint samples_per_frame, sample_rate;
  guint numerator, denominator;

  length = ogmrip_codec_get_length (child->codec, NULL);
  sample_rate = ogmrip_audio_get_sample_rate (OGMRIP_AUDIO (child->codec));
  samples_per_frame = ogmrip_audio_get_samples_per_frame (OGMRIP_AUDIO (child->codec));
  ogmrip_codec_get_framerate (OGMRIP_CODEC (child->codec), &numerator, &denominator);

  return (gint64) length * sample_rate * denominator * container->priv->overhead / (gdouble) (samples_per_frame * numerator);
}

static gint64
ogmrip_container_get_subp_overhead (OGMRipContainer *container, OGMRipContainerChild *child)
{
  return 0;
}

gint64
ogmrip_container_get_overhead_size (OGMRipContainer *container)
{
  GList *link;
  gint64 overhead;
  guint numerator, denominator;
  glong length;

  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), -1);

  overhead = 0;

  if (container->priv->video)
  {
    length = ogmrip_codec_get_length (OGMRIP_CODEC (container->priv->video), NULL);
    ogmrip_codec_get_framerate (OGMRIP_CODEC (container->priv->video), &numerator, &denominator);
    overhead = (gint64) length * numerator * container->priv->overhead / denominator;
  }

  for (link = container->priv->audio; link; link = link->next)
    overhead += ogmrip_container_get_audio_overhead (container, link->data);

  for (link = container->priv->subp; link; link = link->next)
    overhead += ogmrip_container_get_subp_overhead (container, link->data);

  return overhead;
}

static gint64
ogmrip_container_get_size (OGMRipContainer *container, OGMRipContainerChild *child)
{
  struct stat buf;
  const gchar *filename;
  guint64 size = 0;

  filename = ogmrip_codec_get_output (child->codec);
  if (filename && g_file_test (filename, G_FILE_TEST_IS_REGULAR))
    if (g_stat (filename, &buf) == 0)
      size = (guint64) buf.st_size;

  return size;
}

gint64
ogmrip_container_get_nonvideo_size (OGMRipContainer *container)
{
  GList *link;
  gint64 nonvideo = 0;

  for (link = container->priv->audio; link; link = link->next)
    nonvideo += ogmrip_container_get_size (container, link->data);

  for (link = container->priv->subp; link; link = link->next)
    nonvideo += ogmrip_container_get_size (container, link->data);

  for (link = container->priv->chapters; link; link = link->next)
    nonvideo += ogmrip_container_get_size (container, link->data);

  return nonvideo;
}

