/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"

#include "ogmrip-backend.h"
#include "ogmrip-lavc.h"
#include "ogmrip-xvid.h"
#include "ogmrip-avi.h"
#include "ogmrip-ogg.h"

#ifdef HAVE_MKV_SUPPORT
#include "ogmrip-mkv.h"
#endif

#ifdef HAVE_LAVF_SUPPORT
#include "ogmrip-mp4.h"
#endif

#ifdef HAVE_AAC_SUPPORT
#include "ogmrip-aac.h"
#endif

#include <stdio.h>
#include <string.h>
#include <glib/gstdio.h>

#define OGMRIP_XVID_VERY_HIGH_OPTIONS "autoaspect:chroma_opt:vhq=4:bvhq=1:quant_type=mpeg"
#define OGMRIP_XVID_HIGH_OPTIONS      "autoaspect:chroma_opt:vhq=2:bvhq=1:quant_type=mpeg"
#define OGMRIP_XVID_FAST_OPTIONS      "autoaspect:vhq=0"

#define OGMRIP_LAVC_VERY_HIGH_OPTIONS "vcodec=mpeg4:autoaspect:mbd=2:vb_strategy=1:last_pred=3:mv0:preme=2:qns=2"
#define OGMRIP_LAVC_HIGH_OPTIONS      "vcodec=mpeg4:autoaspect:mbd=2:vb_strategy=1:last_pred=2:vqcomp=0.6"
#define OGMRIP_LAVC_FAST_OPTIONS      "vcodec=mpeg4:autoaspect:mbd=2"
/*
#define OGMRIP_X264_VERY_HIGH_OPTIONS "subq=6:b_pyramid:weight_b:8x8dct:frameref=5:me=3"
#define OGMRIP_X264_HIGH_OPTIONS      "subq=5:b_pyramid:weight_b:8x8dct:frameref=2"
*/
#define OGMRIP_X264_VERY_HIGH_OPTIONS "subq=6:b_pyramid:weight_b:frameref=5:me=3"
#define OGMRIP_X264_HIGH_OPTIONS      "subq=5:b_pyramid:weight_b:frameref=2"
#define OGMRIP_X264_FAST_OPTIONS      "subq=4:b_pyramid:weight_b"

#if MPLAYER_PRE >= 8
#define CROP_FRAMES 12
#else
#define CROP_FRAMES 30
#endif

static const gchar *deinterlacer[] = { "pp=lb", "pp=li", "pp=ci", "pp=md", "pp=fd", "kerndeint" };

static gint
ogmrip_backend_map_audio_id (OGMDvdAudioStream *astream)
{
  gint aid;

  aid = ogmdvd_audio_stream_get_nr (astream);

  switch (ogmdvd_audio_stream_get_format (astream))
  {
    case OGMDVD_AUDIO_FORMAT_MPEG1:
    case OGMDVD_AUDIO_FORMAT_MPEG2EXT:
      break;
    case OGMDVD_AUDIO_FORMAT_LPCM:
      aid += 160;
      break;
    case OGMDVD_AUDIO_FORMAT_DTS:
      aid += 136;
      break;
    default:
      aid += 128;
      break;
  }

  return aid;
}

gchar *
ogmrip_backend_get_output_fps (OGMRipCodec *codec, OGMDvdTitle *title)
{
  guint rate_numerator, rate_denominator;
  guint output_rate_numerator, output_rate_denominator;
  gint step;

  ogmdvd_title_get_framerate (title, &rate_numerator, &rate_denominator);
  ogmrip_codec_get_framerate (codec, &output_rate_numerator, &output_rate_denominator);
  step = ogmrip_codec_get_framestep (codec); 

  if (rate_numerator != output_rate_numerator || rate_denominator != output_rate_denominator * step)
    return g_strdup_printf ("%d/%d", output_rate_numerator, output_rate_denominator * step);

  return NULL;
}

gdouble
ogmrip_backend_wav_watch (OGMJobExec *exec, const gchar *buffer, OGMRipAudio *audio)
{
  gchar a[10], v[10], av[15], ct[15];
  gint frame, decoded;

  if (sscanf (buffer, "A:%s V:%s A-V:%s ct:%s %d/%d", a, v, av, ct, &decoded, &frame) == 6)
    return decoded / (gdouble) ogmrip_codec_get_length (OGMRIP_CODEC (audio), NULL);

  return -1.0;
}

gdouble
ogmrip_backend_acopy_watch (OGMJobExec *exec, const gchar *buffer, OGMRipAudio *audio)
{
  gint frames, progress;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
    return frames / (gdouble) ogmrip_codec_get_length (OGMRIP_CODEC (audio), NULL);

  return -1.0;
}

gdouble
ogmrip_backend_xvid_watch (OGMJobExec *exec, const gchar *buffer, OGMRipVideo *video)
{
  gint frames, progress;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
    return frames / (gdouble) ogmrip_codec_get_length (OGMRIP_CODEC (video), NULL);

  return -1.0;
}

gdouble
ogmrip_backend_lavc_watch (OGMJobExec *exec, const gchar *buffer, OGMRipVideo *video)
{
  gint frames, progress;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
    return frames / (gdouble) ogmrip_codec_get_length (OGMRIP_CODEC (video), NULL);

  return -1.0;
}

#ifdef HAVE_X264_SUPPORT
gdouble
ogmrip_backend_x264_watch (OGMJobExec *exec, const gchar *buffer, OGMRipVideo *video)
{
  gint frames, progress;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
    return frames / (gdouble) ogmrip_codec_get_length (OGMRIP_CODEC (video), NULL);

  return -1.0;
}
#endif /* HAVE_X264_SUPPORT */

#ifdef HAVE_THEORA_SUPPORT
gdouble
ogmrip_backend_yuv4mpeg_watch (OGMJobExec *exec, const gchar *buffer, OGMRipVideo *video)
{
  gchar v[10];
  gint frame, decoded;

  if (sscanf (buffer, "V:%s %d/%d", v, &frame, &decoded) == 3)
    return decoded / (gdouble) ogmrip_codec_get_length (OGMRIP_CODEC (video), NULL);

  return -1.0;
}
#endif /* HAVE_THEORA_SUPPORT */

typedef struct
{
  gpointer data;
  GSList *next;
  gint ref;
} GUList;

static GUList *
g_ulist_add (GUList *ulist, gpointer data)
{
  GUList *ulink;

  ulink = (GUList *) g_slist_find ((GSList *) ulist, data);
  if (ulink)
    ulink->ref ++;
  else
  {
    ulink = g_new0 (GUList, 1);
    ulink->data =data;
    ulink->ref = 1;

    ulink->next = (GSList *) ulist;
  }

  return ulink;
}

static gpointer
g_ulist_max_data (GUList *ulist)
{
  GUList *ulink, *umax;

  if (!ulist)
    return NULL;

  for (umax = ulink = ulist; ulink; ulink = (GUList *) ulink->next)
    if (ulink->ref > umax->ref)
      umax = ulink;

  return umax->data;
}

static void
g_ulist_free (GUList *ulist)
{
  g_slist_free ((GSList *) ulist);
}

gdouble
ogmrip_backend_crop_watch (OGMJobExec *exec, const gchar *buffer, OGMRipVideo *video)
{
  gchar *str;

  static guint frame = 0;
  static GUList *crop_x = NULL, *crop_y = NULL,
                *crop_w = NULL, *crop_h = NULL;

  str = strstr (buffer, "-vf crop=");
  if (str)
  {
    gint x, y, w, h;

    if (sscanf (str, "-vf crop=%d:%d:%d:%d", &w, &h, &x, &y) == 4)
    {
      crop_w = g_ulist_add (crop_w, GINT_TO_POINTER (w));
      crop_h = g_ulist_add (crop_h, GINT_TO_POINTER (h));
      crop_x = g_ulist_add (crop_x, GINT_TO_POINTER (x));
      crop_y = g_ulist_add (crop_y, GINT_TO_POINTER (y));
    }

    frame ++;
    if (frame == CROP_FRAMES - 2)
    {
      gpointer x, y, w, h;

      w = g_ulist_max_data (crop_w);
      g_ulist_free (crop_w);
      crop_w = NULL;

      h = g_ulist_max_data (crop_h);
      g_ulist_free (crop_h);
      crop_h = NULL;

      x = g_ulist_max_data (crop_x);
      g_ulist_free (crop_x);
      crop_x = NULL;

      y = g_ulist_max_data (crop_y);
      g_ulist_free (crop_y);
      crop_y = NULL;

      ogmrip_video_set_crop_size (video, 
          GPOINTER_TO_INT (x), GPOINTER_TO_INT (y), 
          GPOINTER_TO_INT (w), GPOINTER_TO_INT (h));

      frame = 0;

      return 1.0;
    }

    return frame / (gdouble) (CROP_FRAMES - 2);
  }

  return -1.0;
}

gdouble
ogmrip_backend_ogg_merge_watch (OGMJobExec *exec, const gchar *buffer, OGMRipContainer *ogg)
{
  gulong frames, total;
  guint percent, steps;

  ogmrip_container_get_split (ogg, &steps, NULL);
  steps = steps > 1 ? 2 : 1;

  if (sscanf (buffer, "progress: %ld/%ld frames (%d%%)", &frames, &total, &percent) == 3)
    return percent / (steps * 100.0);

  return -1.0;
}

gdouble
ogmrip_backend_ogg_split_watch (OGMJobExec *exec, const gchar *buffer, OGMRipContainer *ogg)
{
  gulong frames, total;
  guint percent;

  if (sscanf (buffer, "Processing bytes %ld/%ld (%d%%)", &frames, &total, &percent) == 3)
    return 0.5 + percent / 400.0;
  else if (sscanf (buffer, "Processing frame %ld/%ld (%d%%)", &frames, &total, &percent) == 3)
    return 0.5 + percent / 400.0;

  return -1.0;
}

#ifdef HAVE_MKV_SUPPORT
gdouble
ogmrip_backend_matroska_watch (OGMJobExec *exec, const gchar *buffer, OGMRipContainer *matroska)
{
  gulong frames, total;
  guint percent;

  if (sscanf (buffer, "progress: %ld/%ld frames (%u%%)", &frames, &total, &percent) == 3)
    return percent / 100.0;
  else if (sscanf (buffer, "progress: %u%%", &percent) == 1)
    return percent / 100.0;

  return -1.0;
}
#endif /* HAVE_MKV_SUPPORT */

gdouble
ogmrip_backend_avi_watch (OGMJobExec *exec, const gchar *buffer, OGMRipContainer *avi)
{
  gint frames, progress;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
    return progress / 100.;

  return -1.0;
}

#ifdef HAVE_LAVF_SUPPORT
gdouble
ogmrip_backend_lavf_watch (OGMJobExec *exec, const gchar *buffer, OGMRipContainer *lavf)
{
  gint frames, progress;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
    return progress / 100.;

  return -1.0;
}
#endif /* HAVE_LAVF_SUPPORT */

gdouble
ogmrip_backend_vobsub_watch (OGMJobExec *exec, const gchar *buffer, OGMRipSubp *subp)
{
  gint frames, progress;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
    return 0.98 * frames / (gdouble) ogmrip_codec_get_length (OGMRIP_CODEC (subp), NULL);

  return -1.0;
}

gdouble
ogmrip_backend_pgm_watch (OGMJobExec *exec, const gchar *buffer, OGMRipSubp *subp)
{
  guint files;

  if (sscanf (buffer, "%u files generated", &files) == 1)
  {
    g_object_set_data (G_OBJECT (subp), "__ogmrip_srt_index__", GUINT_TO_POINTER (0));
    g_object_set_data (G_OBJECT (subp), "__ogmrip_srt_files__", GUINT_TO_POINTER (files));
  }

  return -1.0;
}

gdouble
ogmrip_backend_ocr_watch (OGMJobExec *exec, const gchar *buffer, OGMRipSubp *subp)
{
  if (strncmp (buffer, "Elapsed time:", 13) == 0)
  {
    guint index, files;

    index = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (subp), "__ogmrip_srt_index__"));
    files = GPOINTER_TO_UINT (g_object_get_data (G_OBJECT (subp), "__ogmrip_srt_files__"));

    g_object_set_data (G_OBJECT (subp), "__ogmrip_srt_index__", GUINT_TO_POINTER (index + 1));

    return 0.98 + 0.02 * (index + 1) / (gdouble) files;
  }

  return -1.0;
}

gdouble
ogmrip_backend_dvdcpy_watch (OGMJobExec *exec, const gchar *buffer, OGMRipVideo *video)
{
  guint bytes, total, percent;

  if (sscanf (buffer, "%u/%u blocks written (%u%%)", &bytes, &total, &percent) == 3)
    return percent / 100.;

  return -1;
}

gchar **
ogmrip_backend_wav_command (OGMRipAudio *audio, gboolean header, const gchar *input, const gchar *output)
{
  OGMDvdTitle *title;
  OGMDvdAudioStream *astream;

  GPtrArray *argv;
  GString *options;

  const gchar *device;
  guint start, end;
  gint vid, aid;

#if MPLAYER_PRE >= 6
  gint srate;
#endif

  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (audio));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (audio));
  g_return_val_if_fail (title != NULL, NULL);

  astream = ogmrip_audio_get_dvd_audio_stream (audio);
  g_return_val_if_fail (astream != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mplayer"));
  g_ptr_array_add (argv, g_strdup ("-nolirc"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-noframedrop"));
  g_ptr_array_add (argv, g_strdup ("-mc"));
  g_ptr_array_add (argv, g_strdup ("0"));

  g_ptr_array_add (argv, g_strdup ("-vc"));
  g_ptr_array_add (argv, g_strdup ("null"));
  g_ptr_array_add (argv, g_strdup ("-vo"));
  g_ptr_array_add (argv, g_strdup ("null"));
  g_ptr_array_add (argv, g_strdup ("-ao"));

#if MPLAYER_PRE >= 8
  if (header)
    g_ptr_array_add (argv, g_strdup_printf ("pcm:fast:waveheader:file=%s", output));
  else
    g_ptr_array_add (argv, g_strdup_printf ("pcm:fast:nowaveheader:file=%s", output));
#elif MPLAYER_PRE >= 7
  if (header)
    g_ptr_array_add (argv, g_strdup_printf ("pcm:waveheader:file=%s", output));
  else
    g_ptr_array_add (argv, g_strdup_printf ("pcm:nowaveheader:file=%s", output));
#else /* MPLAYER_PRE < 7 */
  g_ptr_array_add (argv, g_strdup ("pcm"));
  if (header)
    g_ptr_array_add (argv, g_strdup ("-waveheader"));
  else
    g_ptr_array_add (argv, g_strdup ("-nowaveheader"));
  g_ptr_array_add (argv, g_strdup ("-aofile"));
  g_ptr_array_add (argv, g_strdup (output));
#endif /* MPLAYER_PRE */

  options = g_string_new (NULL);

  if (ogmrip_audio_get_normalize (audio))
  {
#if MPLAYER_PRE >= 8
    g_string_append (options, "volnorm=1");
#elif MPLAYER_PRE >= 6
    g_string_append (options, "volnorm");
#else /* MPLAYER_PRE */
    g_string_append (options, "list=volnorm");
#endif /* MPLAYER_PRE */
  }

#if MPLAYER_PRE >= 6
  srate = ogmrip_audio_get_sample_rate (audio);
  if (srate != 48000)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "lavcresample=%d", srate);
  }
#endif

  if (options->len == 0)
    g_string_free (options, TRUE);
  else
  {
#if MPLAYER_PRE >= 6
    g_ptr_array_add (argv, g_strdup ("-af"));
#else
    g_ptr_array_add (argv, g_strdup ("-aop"));
#endif
    g_ptr_array_add (argv, g_string_free (options, FALSE));
  }

  g_ptr_array_add (argv, g_strdup ("-channels"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_audio_get_channels (audio) + 1));

  ogmrip_codec_get_chapters (OGMRIP_CODEC (audio), &start, &end);
  g_ptr_array_add (argv, g_strdup ("-chapter"));
  g_ptr_array_add (argv, g_strdup_printf ("%d-%d", start + 1, end + 1));

  aid = ogmdvd_audio_stream_get_nr (astream);
  g_ptr_array_add (argv, g_strdup ("-aid"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_backend_map_audio_id (astream)));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

gchar **
ogmrip_backend_acopy_command (OGMRipAudio *audio, const gchar *input, const gchar *output)
{
  OGMDvdTitle *title;
  OGMDvdAudioStream *astream;

  GPtrArray *argv;
  const gchar *device;
  guint start, end;
  gint vid, aid;
  gchar *ofps;

  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (audio));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (audio));
  g_return_val_if_fail (title != NULL, NULL);

  astream = ogmrip_audio_get_dvd_audio_stream (audio);
  g_return_val_if_fail (astream != NULL, NULL);

  argv = g_ptr_array_new ();

  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-mc"));
  g_ptr_array_add (argv, g_strdup ("0"));

  g_ptr_array_add (argv, g_strdup ("-ovc"));
#if MPLAYER_PRE >= 8
  g_ptr_array_add (argv, g_strdup ("copy"));
  g_ptr_array_add (argv, g_strdup ("-of"));
  g_ptr_array_add (argv, g_strdup ("rawaudio"));
#else /* MPLAYER_PRE */
  g_ptr_array_add (argv, g_strdup ("frameno"));
#endif /* MPLAYER_PRE */

  g_ptr_array_add (argv, g_strdup ("-oac"));
  g_ptr_array_add (argv, g_strdup ("copy"));

  ofps = ogmrip_backend_get_output_fps (OGMRIP_CODEC (audio), title);
  if (ofps)
  {
    g_ptr_array_add (argv, g_strdup ("-ofps"));
    g_ptr_array_add (argv, ofps);
  }

  ogmrip_codec_get_chapters (OGMRIP_CODEC (audio), &start, &end);
  g_ptr_array_add (argv, g_strdup ("-chapter"));
  g_ptr_array_add (argv, g_strdup_printf ("%d-%d", start + 1, end + 1));

  aid = ogmdvd_audio_stream_get_nr (astream);
  g_ptr_array_add (argv, g_strdup ("-aid"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_backend_map_audio_id (astream)));

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

gchar **
ogmrip_backend_vorbis_command (OGMRipAudio *audio, const gchar *input, const gchar *output)
{
  GPtrArray *argv;
  gint quality;

  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (audio));
  g_return_val_if_fail (output != NULL, NULL);

  quality = ogmrip_audio_get_quality (audio);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("oggenc"));
  g_ptr_array_add (argv, g_strdup ("-r"));
  g_ptr_array_add (argv, g_strdup ("-R"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_audio_get_sample_rate (audio)));
  g_ptr_array_add (argv, g_strdup ("-q"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", quality));
  g_ptr_array_add (argv, g_strdup ("-C"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_audio_get_channels (audio) + 1));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

gchar **
ogmrip_backend_mp3_command (OGMRipAudio *audio, const gchar *input, const gchar *output)
{
  static gchar *presets[][2] =
  {
    { "96",       NULL },
    { "112",      NULL },
    { "128",      NULL },
    { "fast",    "medium" },
    { "medium",   NULL },
    { "fast",    "standard" },
    { "standard", NULL },
    { "fast",    "extreme" },
    { "extreme",  NULL },
    { "fast",    "insane" },
    { "insane",   NULL }
  };

  GPtrArray *argv;
  gint quality;

  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (audio));
  g_return_val_if_fail (output != NULL, NULL);

  quality = ogmrip_audio_get_quality (audio);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("lame"));
  g_ptr_array_add (argv, g_strdup ("--nohist"));
  g_ptr_array_add (argv, g_strdup ("-h"));
  g_ptr_array_add (argv, g_strdup ("-t"));
  g_ptr_array_add (argv, g_strdup ("-r"));
  g_ptr_array_add (argv, g_strdup ("-x"));
  g_ptr_array_add (argv, g_strdup ("-s"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_audio_get_sample_rate (audio) / 1000));
  g_ptr_array_add (argv, g_strdup ("--preset"));
  g_ptr_array_add (argv, g_strdup (presets[quality][0]));
  if (presets[quality][2])
    g_ptr_array_add (argv, g_strdup (presets[quality][1]));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

#ifdef HAVE_AAC_SUPPORT
gchar **
ogmrip_backend_aac_command (OGMRipAudio *audio, const gchar *input, const gchar *output)
{
  GPtrArray *argv;
  gint quality;

  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (audio));
  g_return_val_if_fail (output != NULL, NULL);

  quality = ogmrip_audio_get_quality (audio);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("faac"));
  g_ptr_array_add (argv, g_strdup ("-P"));
  g_ptr_array_add (argv, g_strdup ("-R"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_audio_get_sample_rate (audio)));
  g_ptr_array_add (argv, g_strdup ("-q"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", quality * 49 + 10));
  g_ptr_array_add (argv, g_strdup ("-C"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_audio_get_channels (audio) + 1));
  g_ptr_array_add (argv, g_strdup ("-X"));
/*
  g_ptr_array_add (argv, g_strdup ("-w"));
*/
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);

  return NULL;
}
#endif

static GPtrArray *
ogmrip_backend_video_command (OGMRipVideo *video, OGMDvdTitle *title)
{
  GPtrArray *argv;
  OGMRipScalerType scaler;
  OGMRipDeintType deint;
  OGMDvdAudioStream *astream;
  GString *options;

  guint crop_x, crop_y, crop_width, crop_height;
  guint scale_width, scale_height;
  gboolean crop, scale;
  gchar *ofps;

  argv = g_ptr_array_new ();

  scaler = ogmrip_video_get_scaler (video);
  deint = ogmrip_video_get_deinterlacer (video);

  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-noslices"));
  g_ptr_array_add (argv, g_strdup ("-zoom"));
  g_ptr_array_add (argv, g_strdup ("-mc"));
  g_ptr_array_add (argv, g_strdup ("0"));

  astream = ogmrip_video_get_ensure_sync (video);
  if (astream)
  {
    g_ptr_array_add (argv, g_strdup ("-oac"));
    g_ptr_array_add (argv, g_strdup ("pcm"));
    g_ptr_array_add (argv, g_strdup ("-srate"));
    g_ptr_array_add (argv, g_strdup ("4000"));
    g_ptr_array_add (argv, g_strdup ("-af"));
    g_ptr_array_add (argv, g_strdup ("channels=1,lavcresample=4000"));
    g_ptr_array_add (argv, g_strdup ("-aid"));
    g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_backend_map_audio_id (astream)));
/*
    g_ptr_array_add (argv, g_strdup ("-channels"));
    g_ptr_array_add (argv, g_strdup ("1"));
*/
  }
  else
    g_ptr_array_add (argv, g_strdup ("-nosound"));

  g_ptr_array_add (argv, g_strdup ("-sws"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", MAX (scaler, 0)));

  ogmrip_video_get_crop_size (video, &crop_x, &crop_y, &crop_width, &crop_height);
  crop = (crop_x > 0 || crop_y > 0 || crop_width > 0 || crop_height > 0);

  ogmrip_video_get_scale_size (video, &scale_width, &scale_height);
  scale = (scale_width > 0 || scale_height > 0);

  options = g_string_new (NULL);

  if (ogmrip_video_get_pullup (video))
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append (options, "pullup,softskip");
  }

  if (crop)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "crop=%u:%u:%u:%u", crop_width, crop_height, crop_x, crop_y);
  }

  if (deint != OGMRIP_DEINT_NONE)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "%s", deinterlacer[deint - 1]);
  }

  if (scale)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "scale=%u:%u", scale_width, scale_height);
  }

  if (ogmrip_video_get_denoise (video))
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append (options, "hqdn3d=2:1:2");
  }

  if (options->len > 0)
    g_string_append_c (options, ',');
  g_string_append (options, "harddup");

  if (options->len == 0)
    g_string_free (options, TRUE);
  else
  {
    g_ptr_array_add (argv, g_strdup ("-vf"));
    g_ptr_array_add (argv, g_string_free (options, FALSE));
  }

  ofps = ogmrip_backend_get_output_fps (OGMRIP_CODEC (video), title);
  if (ofps)
  {
    g_ptr_array_add (argv, g_strdup ("-ofps"));
    g_ptr_array_add (argv, ofps);
  }

  g_ptr_array_add (argv, g_strdup ("-dvdangle"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_video_get_angle (video)));

  return argv;
}

gchar **
ogmrip_backend_xvid_command (OGMRipVideo *video, const gchar *input, const gchar *output, const gchar *log)
{
  OGMDvdTitle *title;
  GPtrArray *argv;
  GString *options;

  const gchar *device;
  gint bitrate, vid, pass;
  guint start, end;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  pass = ogmrip_video_get_pass (video);
  if (pass > 0 && !log)
    log = ogmrip_video_get_log (video);
  g_return_val_if_fail (pass == 0 || log != NULL, NULL);

  argv = g_ptr_array_new ();
  argv = ogmrip_backend_video_command (video, title);

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("xvid"));

  switch (ogmrip_video_get_quality (video))
  {
    case OGMRIP_QUALITY_VERY_HIGH:
      options = g_string_new (OGMRIP_XVID_VERY_HIGH_OPTIONS);
      break;
    case OGMRIP_QUALITY_HIGH:
      options = g_string_new (OGMRIP_XVID_HIGH_OPTIONS);
      break;
    default:
      options = g_string_new (OGMRIP_XVID_FAST_OPTIONS);
      break;
  }

#if MPLAYER_PRE >= 6
  if (ogmrip_video_get_cartoon (video))
    g_string_append (options, ":cartoon");
#endif /* MPLAYER_PRE */

  if (ogmrip_video_get_qpel (video))
    g_string_append (options, ":qpel");
  if (ogmrip_video_get_turbo (video))
    g_string_append (options, ":turbo");
  if (ogmrip_video_get_trellis (video))
    g_string_append (options, ":trellis");
  if (ogmrip_video_get_grayscale (video))
    g_string_append (options, ":greyscale");

  if (ogmrip_xvid_get_gmc (OGMRIP_XVID (video)))
    g_string_append (options, ":gmc");

  g_string_append_printf (options, ":max_bframes=%d", ogmrip_video_get_max_b_frames (video));

  bitrate = ogmrip_video_get_bitrate (video);
  if (bitrate < 16001)
    g_string_append_printf (options, ":bitrate=%u", bitrate / 1000);
  else
    g_string_append_printf (options, ":bitrate=%u", bitrate);
  
  switch (pass)
  {
    case 1:
      g_string_append_printf (options, ":pass=%u", 1);
      g_ptr_array_add (argv, g_strdup ("-passlogfile"));
      g_ptr_array_add (argv, g_strdup (log));
      break;
    case 2:
      g_string_append_printf (options, ":pass=%u", 2);
      g_ptr_array_add (argv, g_strdup ("-passlogfile"));
      g_ptr_array_add (argv, g_strdup (log));
      break;
    default:
      break;
  }

  g_ptr_array_add (argv, g_strdup ("-xvidencopts"));
  g_ptr_array_add (argv, g_string_free (options, FALSE));

  ogmrip_codec_get_chapters (OGMRIP_CODEC (video), &start, &end);
  g_ptr_array_add (argv, g_strdup ("-chapter"));
  g_ptr_array_add (argv, g_strdup_printf ("%d-%d", start + 1, end + 1));

  g_ptr_array_add (argv, g_strdup ("-o"));
  if (pass == 1)
    g_ptr_array_add (argv, g_strdup ("/dev/null"));
  else
    g_ptr_array_add (argv, g_strdup (output));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

gchar **
ogmrip_backend_lavc_command (OGMRipVideo *video, const gchar *input, const gchar *output, const gchar *log)
{
  OGMDvdTitle *title;
  GPtrArray *argv;
  GString *options;

  const gchar *device;
  gint vid, pass, dia, predia, header;
  guint cmp, precmp, subcmp, start, end;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  pass = ogmrip_video_get_pass (video);
  if (pass > 0 && !log)
    log = ogmrip_video_get_log (video);
  g_return_val_if_fail (pass == 0 || log != NULL, NULL);

  argv = ogmrip_backend_video_command (video, title);

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("lavc"));

  switch (ogmrip_video_get_quality (video))
  {
    case OGMRIP_QUALITY_VERY_HIGH:
      options = g_string_new (OGMRIP_LAVC_VERY_HIGH_OPTIONS);
      break;
    case OGMRIP_QUALITY_HIGH:
      options = g_string_new (OGMRIP_LAVC_HIGH_OPTIONS);
      break;
    default:
      options = g_string_new (OGMRIP_LAVC_FAST_OPTIONS);
      break;
  }

#if MPLAYER_PRE >= 6
  if (ogmrip_video_get_turbo (video))
    g_string_append (options, ":turbo");
#endif /* MPLAYER_PRE */

  if (ogmrip_video_get_qpel (video))
    g_string_append (options, ":qpel");
  if (ogmrip_video_get_4mv (video))
    g_string_append (options, ":v4mv");
  if (ogmrip_video_get_trellis (video))
    g_string_append (options, ":trell:cbp");
  if (ogmrip_video_get_grayscale (video))
    g_string_append (options, ":gray");

  ogmrip_lavc_get_cmp (OGMRIP_LAVC (video), &cmp, &precmp, &subcmp);
  g_string_append_printf (options, ":precmp=%u:subcmp=%u:cmp=%u", precmp, subcmp, cmp);

  ogmrip_lavc_get_dia (OGMRIP_LAVC (video), &dia, &predia);
  g_string_append_printf (options, ":dia=%d:predia=%d", dia, predia);

  header = ogmrip_lavc_get_header (OGMRIP_LAVC (video));
  g_string_append_printf (options, ":vglobal=%d", header);

  g_string_append_printf (options, ":vmax_b_frames=%d", ogmrip_video_get_max_b_frames (video));
  g_string_append_printf (options, ":vbitrate=%u", ogmrip_video_get_bitrate (video));
  
  switch (pass)
  {
    case 1:
      g_string_append_printf (options, ":vpass=%u", 1);
      g_ptr_array_add (argv, g_strdup ("-passlogfile"));
      g_ptr_array_add (argv, g_strdup (log));
      break;
    case 2:
      g_string_append_printf (options, ":vpass=%u", 2);
      g_ptr_array_add (argv, g_strdup ("-passlogfile"));
      g_ptr_array_add (argv, g_strdup (log));
      break;
    default:
      break;
  }

  g_ptr_array_add (argv, g_strdup ("-lavcopts"));
  g_ptr_array_add (argv, g_string_free (options, FALSE));

  ogmrip_codec_get_chapters (OGMRIP_CODEC (video), &start, &end);
  g_ptr_array_add (argv, g_strdup ("-chapter"));
  g_ptr_array_add (argv, g_strdup_printf ("%d-%d", start + 1, end + 1));

  g_ptr_array_add (argv, g_strdup ("-o"));
  if (pass == 1)
    g_ptr_array_add (argv, g_strdup ("/dev/null"));
  else
    g_ptr_array_add (argv, g_strdup (output));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

#ifdef HAVE_X264_SUPPORT
gchar **
ogmrip_backend_x264_command (OGMRipVideo *video, const gchar *input, const gchar *output, const gchar *log)
{
  OGMDvdTitle *title;
  GPtrArray *argv;
  GString *options;

  const gchar *device;
  guint start, end;
  gint vid, pass;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  pass = ogmrip_video_get_pass (video);
  if (pass > 0 && !log)
    log = ogmrip_video_get_log (video);
  g_return_val_if_fail (pass == 0 || log != NULL, NULL);

  argv = ogmrip_backend_video_command (video, title);

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("x264"));

  switch (ogmrip_video_get_quality (video))
  {
    case OGMRIP_QUALITY_VERY_HIGH:
      options = g_string_new (OGMRIP_X264_VERY_HIGH_OPTIONS);
      break;
    case OGMRIP_QUALITY_HIGH:
      options = g_string_new (OGMRIP_X264_HIGH_OPTIONS);
      break;
    default:
      options = g_string_new (OGMRIP_X264_FAST_OPTIONS);
      break;
  }

#if MPLAYER_PRE >= 8
  if (ogmrip_video_get_turbo (video))
    g_string_append (options, ":turbo=1");
#endif /* MPLAYER_PRE */

  if (ogmrip_video_get_4mv (video))
    g_string_append (options, ":4x4mv");
  if (ogmrip_video_get_trellis (video))
    g_string_append (options, ":trellis=1");
  else
    g_string_append (options, ":trellis=0");

  g_string_append_printf (options, ":bframes=%d", ogmrip_video_get_max_b_frames (video));
  g_string_append_printf (options, ":bitrate=%u", ogmrip_video_get_bitrate (video) / 1000);
  
  switch (pass)
  {
    case 0:
      break;
    case 1:
      g_string_append_printf (options, ":pass=%u", 1);
      g_ptr_array_add (argv, g_strdup ("-passlogfile"));
      g_ptr_array_add (argv, g_strdup (log));
      break;
    case 2:
      g_string_append_printf (options, ":pass=%u", 2);
      g_ptr_array_add (argv, g_strdup ("-passlogfile"));
      g_ptr_array_add (argv, g_strdup (log));
      break;
    default:
      break;
  }

  g_ptr_array_add (argv, g_strdup ("-x264encopts"));
  g_ptr_array_add (argv, g_string_free (options, FALSE));

  ogmrip_codec_get_chapters (OGMRIP_CODEC (video), &start, &end);
  g_ptr_array_add (argv, g_strdup ("-chapter"));
  g_ptr_array_add (argv, g_strdup_printf ("%d-%d", start + 1, end + 1));

  g_ptr_array_add (argv, g_strdup ("-o"));
  if (pass == 1)
    g_ptr_array_add (argv, g_strdup ("/dev/null"));
  else
    g_ptr_array_add (argv, g_strdup (output));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}
#endif /* HAVE_X264_SUPPORT */

#ifdef HAVE_THEORA_SUPPORT
gchar **
ogmrip_backend_yuv4mpeg_command (OGMRipVideo *video, const gchar *input, const gchar *output, const gchar *log)
{
  OGMDvdTitle *title;
  OGMRipScalerType scaler;
  OGMRipDeintType deint;
  GString *options;
  GPtrArray *argv;

  const gchar *device;
  gboolean crop, scale;
  guint crop_x, crop_y, crop_width, crop_height;
  guint scale_width, scale_height;
  guint start, end;
  gint vid;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  argv = g_ptr_array_new ();

  g_ptr_array_add (argv, g_strdup ("mplayer"));
  g_ptr_array_add (argv, g_strdup ("-nolirc"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-noframedrop"));
  g_ptr_array_add (argv, g_strdup ("-nosound"));
  g_ptr_array_add (argv, g_strdup ("-noslices"));
  g_ptr_array_add (argv, g_strdup ("-zoom"));
  g_ptr_array_add (argv, g_strdup ("-mc"));
  g_ptr_array_add (argv, g_strdup ("0"));

  scaler = ogmrip_video_get_scaler (video);
  g_ptr_array_add (argv, g_strdup ("-sws"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", MAX (scaler, 0)));

  ogmrip_video_get_crop_size (video, &crop_x, &crop_y, &crop_width, &crop_height);
  crop = (crop_x > 0 || crop_y > 0 || crop_width > 0 || crop_height > 0);

  ogmrip_video_get_scale_size (video, &scale_width, &scale_height);
  scale = (scale_width > 0 || scale_height);

  options = g_string_new (NULL);

  if (ogmrip_video_get_pullup (video))
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append (options, "pullup,softskip");
  }

  if (crop)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "crop=%u:%u:%u:%u", crop_width, crop_height, crop_x, crop_y);
  }

  deint = ogmrip_video_get_deinterlacer (video);
  if (deint != OGMRIP_DEINT_NONE)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "%s", deinterlacer[deint - 1]);
  }

  if (scale)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "scale=%u:%u", scale_width, scale_height);
  }

  if (ogmrip_video_get_denoise (video))
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append (options, "hqdn3d=2:1:2");
  }

  if (options->len > 0)
    g_string_append_c (options, ',');
  g_string_append (options, "harddup");

  if (options->len == 0)
    g_string_free (options, TRUE);
  else
  {
    g_ptr_array_add (argv, g_strdup ("-vf"));
    g_ptr_array_add (argv, g_string_free (options, FALSE));
  }
/*
  ofps = ogmrip_backend_get_output_fps (OGMRIP_CODEC (video), title);
  if (ofps)
  {
    g_ptr_array_add (argv, g_strdup ("-ofps"));
    g_ptr_array_add (argv, ofps);
  }
*/
  g_ptr_array_add (argv, g_strdup ("-dvdangle"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_video_get_angle (video)));

  g_ptr_array_add (argv, g_strdup ("-vo"));
#if MPLAYER_PRE >= 6
  g_ptr_array_add (argv, g_strdup_printf ("yuv4mpeg:file=%s", output));
#else /* MPLAYER_PRE */
  g_ptr_array_add (argv, g_strdup ("yuv4mpeg"));
#endif /* MPLAYER_PRE */

  ogmrip_codec_get_chapters (OGMRIP_CODEC (video), &start, &end);
  g_ptr_array_add (argv, g_strdup ("-chapter"));
  g_ptr_array_add (argv, g_strdup_printf ("%d-%d", start + 1, end + 1));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

gchar **
ogmrip_backend_theora_command (OGMRipVideo *video, const gchar *input, const gchar *output, const gchar *log)
{
  GPtrArray *argv;
  gint bitrate;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  argv = g_ptr_array_new ();

  g_ptr_array_add (argv, g_strdup ("theoraenc"));

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  bitrate = ogmrip_video_get_bitrate (video) / 1000;

  g_ptr_array_add (argv, g_strdup ("-V"));
  g_ptr_array_add (argv, g_strdup_printf ("%u", CLAMP (bitrate, 45, 2000)));

  g_ptr_array_add (argv, g_strdup (input));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}
#endif /* HAVE_THEORA_SUPPORT */

gchar **
ogmrip_backend_crop_command (OGMRipVideo *video, const gchar *input, const gchar *output)
{
  OGMDvdTime time_;
  OGMDvdTitle *title;
  GPtrArray *argv;

  const gchar *device;
  gint vid;

#if MPLAYER_PRE >= 8
  gint sstep;
#endif

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (video));
  g_return_val_if_fail (title != NULL, NULL);

  argv = g_ptr_array_new ();

#if MPLAYER_PRE >= 8
  g_ptr_array_add (argv, g_strdup ("mplayer"));
  g_ptr_array_add (argv, g_strdup ("-nolirc"));
  g_ptr_array_add (argv, g_strdup ("-vo"));
  g_ptr_array_add (argv, g_strdup ("null"));
#else
  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("lavc"));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup ("/dev/null"));
#endif

  g_ptr_array_add (argv, g_strdup ("-nosound"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));

  g_ptr_array_add (argv, g_strdup ("-vf"));
  g_ptr_array_add (argv, g_strdup ("cropdetect"));

  ogmrip_codec_get_length (OGMRIP_CODEC (video), &time_);

#if MPLAYER_PRE >= 8
  sstep = (time_.hour * 3600 + time_.min * 60 + time_.sec) / (3 * CROP_FRAMES) + 1;
  g_ptr_array_add (argv, g_strdup ("-sstep"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", sstep));
#else
  g_ptr_array_add (argv, g_strdup ("-ss"));
  g_ptr_array_add (argv, g_strdup_printf ("%02u:%02u:%02u",
        time_.hour / 4, time_.min / 4 + (time_.hour % 4) * 15, time_.sec / 4 + (time_.min % 4) * 15));
#endif

  g_ptr_array_add (argv, g_strdup ("-frames"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", CROP_FRAMES));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static void
ogmrip_backend_merge_foreach_audio (OGMRipContainer *ogg, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  if (language > -1)
  {
    g_ptr_array_add (argv, g_strdup ("-c"));
    g_ptr_array_add (argv, g_strdup_printf ("LANGUAGE=%s", 
          g_strdup (ogmdvd_get_language_label (language))));
  }

  g_ptr_array_add (argv, g_strdup ("--novideo"));
  g_ptr_array_add (argv, g_strdup ("--notext"));

  input = ogmrip_codec_get_output (codec);
  g_ptr_array_add (argv, g_strdup (input));
}

static void
ogmrip_backend_merge_foreach_subp (OGMRipContainer *ogg, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  if (demuxer != OGMRIP_SUBP_DEMUXER_VOBSUB)
  {
    if (language > -1)
    {
      g_ptr_array_add (argv, g_strdup ("-c"));
      g_ptr_array_add (argv, g_strdup_printf ("LANGUAGE=%s",
            g_strdup (ogmdvd_get_language_label (language))));
    }

    g_ptr_array_add (argv, g_strdup ("--novideo"));
    g_ptr_array_add (argv, g_strdup ("--noaudio"));

    input = ogmrip_codec_get_output (codec);
    g_ptr_array_add (argv, g_strdup (input));
  }
}

static void
ogmrip_backend_merge_foreach_chapters (OGMRipContainer *ogg, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  if (language > -1)
  {
    g_ptr_array_add (argv, g_strdup ("-c"));
    g_ptr_array_add (argv, g_strdup_printf ("LANGUAGE=%s",
          g_strdup (ogmdvd_get_language_label (language))));
  }

  g_ptr_array_add (argv, g_strdup ("--novideo"));
  g_ptr_array_add (argv, g_strdup ("--noaudio"));

  input = ogmrip_codec_get_output (codec);
  g_ptr_array_add (argv, g_strdup (input));
}

gchar **
ogmrip_backend_ogg_merge_command (OGMRipContainer *ogg, const gchar *output)
{
  GPtrArray *argv;
  OGMRipVideo *video;
  const gchar *label, *fourcc, *filename;

  g_return_val_if_fail (OGMRIP_IS_OGG (ogg), NULL);

  if (!output)
    output = ogmrip_container_get_output (ogg);
  g_return_val_if_fail (output != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("ogmmerge"));

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  fourcc = ogmrip_container_get_fourcc (ogg);
  if (fourcc)
  {
    g_ptr_array_add (argv, g_strdup ("--fourcc"));
    g_ptr_array_add (argv, g_strdup (fourcc));
  }

  label = ogmrip_container_get_label (ogg);
  if (label)
  {
    g_ptr_array_add (argv, g_strdup ("-c"));
    g_ptr_array_add (argv, g_strdup_printf ("TITLE=%s", label));
  }

  video = ogmrip_container_get_video (ogg);
  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));

  g_ptr_array_add (argv, g_strdup ("--noaudio"));
  g_ptr_array_add (argv, g_strdup (filename));

  ogmrip_container_foreach_audio (ogg, 
      (OGMRipContainerFunc) ogmrip_backend_merge_foreach_audio, argv);
  ogmrip_container_foreach_subp (ogg, 
      (OGMRipContainerFunc) ogmrip_backend_merge_foreach_subp, argv);
  ogmrip_container_foreach_chapters (ogg, 
      (OGMRipContainerFunc) ogmrip_backend_merge_foreach_chapters, argv);

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

gchar **
ogmrip_backend_ogg_split_command (OGMRipContainer *ogg, const gchar *input)
{
  GPtrArray *argv;
  const gchar *output;
  guint tsize;

  g_return_val_if_fail (OGMRIP_IS_CONTAINER (ogg), NULL);
  g_return_val_if_fail (input && *input, NULL);

  output = ogmrip_container_get_output (ogg);
  g_return_val_if_fail (output && *output, NULL);

  ogmrip_container_get_split (OGMRIP_CONTAINER (ogg), NULL, &tsize);
  g_return_val_if_fail (tsize > 0, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("ogmsplit"));
  g_ptr_array_add (argv, g_strdup ("--frontend"));
  g_ptr_array_add (argv, g_strdup ("-s"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", tsize));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

#ifdef HAVE_MKV_SUPPORT
static void
ogmrip_backend_matroska_foreach_audio (OGMRipContainer *matroska, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  if (language > -1)
  {
    const gchar *iso639_2;

    iso639_2 = ogmdvd_get_language_iso639_2 (language);
    if (iso639_2)
    {
      g_ptr_array_add (argv, g_strdup ("--language"));
      g_ptr_array_add (argv, g_strconcat ("0:", iso639_2, NULL));
#if MPLAYER_PRE < 8
      g_ptr_array_add (argv, g_strdup ("--language"));
      g_ptr_array_add (argv, g_strconcat ("1:", iso639_2, NULL));
#endif
    }
  }

  g_ptr_array_add (argv, g_strdup ("-D"));
  g_ptr_array_add (argv, g_strdup ("-S"));

  input = ogmrip_codec_get_output (codec);
  g_ptr_array_add (argv, g_strdup (input));
}

static void
ogmrip_backend_matroska_foreach_chapters (OGMRipContainer *matroska, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  if (language > -1)
  {
    const gchar *iso639_2;

    iso639_2 = ogmdvd_get_language_iso639_2 (language);
    if (iso639_2)
    {
      g_ptr_array_add (argv, g_strdup ("--chapter-language"));
      g_ptr_array_add (argv, g_strdup (iso639_2));
    }
  }

  g_ptr_array_add (argv, g_strdup ("--chapter-charset"));
  g_ptr_array_add (argv, g_strdup ("UTF-8"));
  g_ptr_array_add (argv, g_strdup ("--chapters"));

  input = ogmrip_codec_get_output (codec);
  g_ptr_array_add (argv, g_strdup (input));
}

static void
ogmrip_backend_matroska_foreach_subp (OGMRipContainer *matroska, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  if (language > -1)
  {
    const gchar *iso639_2;

    iso639_2 = ogmdvd_get_language_iso639_2 (language);
    if (iso639_2)
    {
      g_ptr_array_add (argv, g_strdup ("--language"));
      g_ptr_array_add (argv, g_strconcat ("0:", iso639_2, NULL));
    }
  }

  g_ptr_array_add (argv, g_strdup ("--sub-charset"));
  g_ptr_array_add (argv, g_strdup ("0:UTF-8"));
  g_ptr_array_add (argv, g_strdup ("-s"));
  g_ptr_array_add (argv, g_strdup ("0"));
  g_ptr_array_add (argv, g_strdup ("-D"));
  g_ptr_array_add (argv, g_strdup ("-A"));

  input = ogmrip_codec_get_output (codec);
  if (demuxer == OGMRIP_SUBP_DEMUXER_VOBSUB && !g_str_has_suffix (input, ".idx"))
    g_ptr_array_add (argv, g_strconcat (input, ".idx", NULL));
  else
    g_ptr_array_add (argv, g_strdup (input));
}

gchar **
ogmrip_backend_matroska_command (OGMRipContainer *matroska)
{
  GPtrArray *argv;
  OGMRipVideo *video;
  const gchar *output, *label, *filename, *fourcc;
  guint tsize, tnumber;

  g_return_val_if_fail (OGMRIP_IS_MATROSKA (matroska), NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mkvmerge"));

  output = ogmrip_container_get_output (matroska);
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  fourcc = ogmrip_container_get_fourcc (matroska);
  if (fourcc)
  {
    g_ptr_array_add (argv, g_strdup ("--fourcc"));
    g_ptr_array_add (argv, g_strconcat ("0:", fourcc, NULL));
  }

#ifdef HAVE_X264_SUPPORT
  /*
   * Option to merge h264 streams
   */
  g_ptr_array_add (argv, g_strdup ("--engage"));
  g_ptr_array_add (argv, g_strdup ("allow_avc_in_vfw_mode"));
#endif /* HAVE_X264_SUPPORT */

  g_ptr_array_add (argv, g_strdup ("--command-line-charset"));
  g_ptr_array_add (argv, g_strdup ("UTF-8"));

  video = ogmrip_container_get_video (matroska);
  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));

  g_ptr_array_add (argv, g_strdup ("-d"));
  g_ptr_array_add (argv, g_strdup ("0"));
  g_ptr_array_add (argv, g_strdup ("-A"));
  g_ptr_array_add (argv, g_strdup ("-S"));
  g_ptr_array_add (argv, g_strdup (filename));

  ogmrip_container_foreach_audio (matroska, 
      (OGMRipContainerFunc) ogmrip_backend_matroska_foreach_audio, argv);
  ogmrip_container_foreach_subp (matroska, 
      (OGMRipContainerFunc) ogmrip_backend_matroska_foreach_subp, argv);
  ogmrip_container_foreach_chapters (matroska, 
      (OGMRipContainerFunc) ogmrip_backend_matroska_foreach_chapters, argv);

  label = ogmrip_container_get_label (matroska);
  if (label)
  {
    g_ptr_array_add (argv, g_strdup ("--title"));
    g_ptr_array_add (argv, g_strdup_printf ("%s", label));
  }

  ogmrip_container_get_split (matroska, &tnumber, &tsize);
  if (tnumber > 1)
  {
    g_ptr_array_add (argv, g_strdup ("--split"));
    g_ptr_array_add (argv, g_strdup_printf ("%dM", tsize));
  }

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}
#endif /* HAVE_MKV_SUPPORT */

static void
ogmrip_backend_mplayer_foreach_audio (OGMRipContainer *mplayer, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  const gchar *input;

  input = ogmrip_codec_get_output (codec);

#if MPLAYER_PRE >= 8
  g_ptr_array_add (argv, g_strdup ("-audiofile"));
  g_ptr_array_add (argv, g_strdup (input));

  if (demuxer != OGMRIP_AUDIO_DEMUXER_AUTO)
  {
    g_ptr_array_add (argv, g_strdup ("-audio-demuxer"));
    g_ptr_array_add (argv, g_strdup ("rawaudio"));
    g_ptr_array_add (argv, g_strdup ("-rawaudio"));
    g_ptr_array_add (argv, g_strdup_printf ("format=0x%x", demuxer));
  }
#else
  if (demuxer == OGMRIP_AUDIO_DEMUXER_AUTO)
  {
    g_ptr_array_add (argv, g_strdup ("-audiofile"));
    g_ptr_array_add (argv, g_strdup (input));
  }
  else
  {
    gchar *dirname, *new_name;

    dirname = g_path_get_dirname (input);
    new_name = g_build_filename (dirname, "frameno.avi", NULL);
    g_rename (input, new_name);
    strcpy ((gchar *) input, new_name);
  }
#endif

#ifdef HAVE_AAC_SUPPORT
  /*
   * TODO
   * Temporary workaround for AAC in AVI/MP4 support
   */
  if (OGMRIP_IS_AAC (codec))
  {
    g_ptr_array_add (argv, g_strdup ("-audio-demuxer"));
    g_ptr_array_add (argv, g_strdup ("aac"));
    g_ptr_array_add (argv, g_strdup ("-fafmttag"));
    g_ptr_array_add (argv, g_strdup ("0x706D"));
  }
#endif
}

static GPtrArray *
ogmrip_backend_mencoder_command (OGMRipContainer *container)
{
  GPtrArray *argv;
  const gchar *fourcc;

#if MPLAYER_PRE >= 8
  const gchar *label;
#endif

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));

  g_ptr_array_add (argv, g_strdup ("-nosound"));
  g_ptr_array_add (argv, g_strdup ("-noskip"));
  g_ptr_array_add (argv, g_strdup ("-mc"));
  g_ptr_array_add (argv, g_strdup ("0"));

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("copy"));
  g_ptr_array_add (argv, g_strdup ("-oac"));
  g_ptr_array_add (argv, g_strdup ("copy"));

  fourcc = ogmrip_container_get_fourcc (container);
  if (fourcc)
  {
    g_ptr_array_add (argv, g_strdup ("-ffourcc"));
    g_ptr_array_add (argv, g_strdup (fourcc));
  }

#if MPLAYER_PRE >= 8
  label = ogmrip_container_get_label (container);
  if (label)
  {
    g_ptr_array_add (argv, g_strdup ("-info"));
    g_ptr_array_add (argv, g_strdup_printf ("name=%s", label));
  }
#endif

  return argv;
}

gchar **
ogmrip_backend_avi_command (OGMRipContainer *avi)
{
  GPtrArray *argv;
  OGMRipVideo *video;
  const gchar *output, *filename;

  g_return_val_if_fail (OGMRIP_IS_AVI (avi), NULL);

  argv = ogmrip_backend_mencoder_command (avi);

  output = ogmrip_container_get_output (avi);
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  ogmrip_container_foreach_audio (avi, 
      (OGMRipContainerFunc) ogmrip_backend_mplayer_foreach_audio, argv);

  video = ogmrip_container_get_video (avi);
  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_ptr_array_add (argv, g_strdup (filename));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

#ifdef HAVE_LAVF_SUPPORT
gchar **
ogmrip_backend_mp4_command (OGMRipContainer *mp4)
{
  GPtrArray *argv;
  OGMRipVideo *video;
  const gchar *output, *filename;

  g_return_val_if_fail (OGMRIP_IS_MP4 (mp4), NULL);

  argv = ogmrip_backend_mencoder_command (mp4);

  g_ptr_array_add (argv, g_strdup ("-of"));
  g_ptr_array_add (argv, g_strdup ("lavf"));
  g_ptr_array_add (argv, g_strdup ("-lavfopts"));
  g_ptr_array_add (argv, g_strdup ("format=mp4:i_certify_that_my_video_stream_does_not_use_b_frames"));

  output = ogmrip_container_get_output (mp4);
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  ogmrip_container_foreach_audio (mp4, 
      (OGMRipContainerFunc) ogmrip_backend_mplayer_foreach_audio, argv);

  video = ogmrip_container_get_video (mp4);
  filename = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_ptr_array_add (argv, g_strdup (filename));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}
#endif /* HAVE_LAVF_SUPPORT */

gchar **
ogmrip_backend_vobsub_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  OGMDvdTitle *title;
  OGMDvdSubpStream *sstream;

  GPtrArray *argv;
  const gchar *device;
  guint start, end;
  gint vid, sid;
  gchar *ofps;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (subp));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (subp));
  g_return_val_if_fail (title != NULL, NULL);

  sstream = ogmrip_subp_get_dvd_subp_stream (OGMRIP_SUBP (subp));
  g_return_val_if_fail (sstream != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-nosound"));

#if MPLAYER_PRE >= 8
  g_ptr_array_add (argv, g_strdup ("-of"));
  g_ptr_array_add (argv, g_strdup ("rawaudio"));
#endif

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("copy"));

  ofps = ogmrip_backend_get_output_fps (OGMRIP_CODEC (subp), title);
  if (ofps)
  {
    g_ptr_array_add (argv, g_strdup ("-ofps"));
    g_ptr_array_add (argv, ofps);
  }

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup ("/dev/null"));

  sid = ogmdvd_subp_stream_get_nr (sstream);
  g_ptr_array_add (argv, g_strdup ("-vobsubout"));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, g_strdup ("-vobsuboutindex"));
  g_ptr_array_add (argv, g_strdup ("0"));
  g_ptr_array_add (argv, g_strdup ("-sid"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", sid));

  if (ogmrip_subp_get_forced_only (subp))
    g_ptr_array_add (argv, g_strdup ("-forcedsubsonly"));

  ogmrip_codec_get_chapters (OGMRIP_CODEC (subp), &start, &end);
  g_ptr_array_add (argv, g_strdup ("-chapter"));
  g_ptr_array_add (argv, g_strdup_printf ("%d-%d", start + 1, end + 1));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_MAJOR > 0
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_MAJOR */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_MAJOR */

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

#ifdef HAVE_SRT_SUPPORT
gchar **
ogmrip_backend_pgm_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("subp2pgm"));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

gchar **
ogmrip_backend_ocr_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("gocr"));
  g_ptr_array_add (argv, g_strdup ("-v"));
  g_ptr_array_add (argv, g_strdup ("1"));
  g_ptr_array_add (argv, g_strdup ("-f"));
  g_ptr_array_add (argv, g_strdup ("UTF8"));
  g_ptr_array_add (argv, g_strdup ("-m"));
  g_ptr_array_add (argv, g_strdup ("4"));
  g_ptr_array_add (argv, g_strdup ("-m"));
  g_ptr_array_add (argv, g_strdup ("64"));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strconcat (input, ".txt", NULL));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

gchar **
ogmrip_backend_srt_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  GPtrArray *argv;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (subp));
  g_return_val_if_fail (output != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("srttool"));
  g_ptr_array_add (argv, g_strdup ("-s"));
  g_ptr_array_add (argv, g_strdup ("-w"));
  g_ptr_array_add (argv, g_strdup ("-i"));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}
#endif /* HAVE_SRT_SUPPORT */

gchar **
ogmrip_backend_dvdcpy_command (OGMRipDvdCpy *dvdcpy, const gchar *input, const gchar *output)
{
  OGMDvdTitle *title;
  GPtrArray *argv;

  const gchar *device;
  gint vid;

  g_return_val_if_fail (OGMRIP_IS_DVDCPY (dvdcpy), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (dvdcpy));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (dvdcpy));
  g_return_val_if_fail (title != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("dvdcpy"));
  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, g_strdup ("-m"));

  vid = ogmdvd_title_get_nr (title);
  g_ptr_array_add (argv, g_strdup ("-t"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup (device));

  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

static G_CONST_RETURN gchar *
g_filename_get_extension (const gchar *filename)
{
  gchar *dot;

  dot = strrchr (filename, '.');
  if (dot && ++dot)
    return dot;

  return NULL;
}

static gchar *
g_filename_set_extension (const gchar *filename, const gchar *extension)
{
  gchar *dot;

  dot = strrchr (filename, '.');
  if (!dot)
    return g_strconcat (filename, extension, NULL);
  {
    if (dot[1] == '\0')
      return g_strconcat (filename, ".", extension, NULL);
    else
    {
      if (strcmp (dot + 1, extension) == 0)
        return g_strdup (filename);
      else
      {
        gchar *name;

        name = g_new0 (gchar, dot - filename + 5);
        strncpy (name, filename, dot - filename + 1);
        strcat (name, extension);

        return name;
      }
    }
  }
}

gchar **
ogmrip_backend_copy_command (OGMRipContainer *container, const gchar *input, const gchar *output)
{
  GPtrArray *argv;
  gchar *real_output;

  g_return_val_if_fail (OGMRIP_IS_CONTAINER (container), NULL);
  g_return_val_if_fail (input != NULL, NULL);

  if (output)
    real_output = g_strdup (output);
  else
  {
    const gchar *ext;

    output = ogmrip_container_get_output (container);
    g_return_val_if_fail (output != NULL, NULL);

    ext = g_filename_get_extension (input);

    if (ext && strcmp (ext, "idx") == 0)
      real_output = g_filename_set_extension (output, ext);
    else
      real_output = g_filename_set_extension (output, "sub");
  }

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("cp"));
  g_ptr_array_add (argv, g_strdup ("-f"));
  g_ptr_array_add (argv, g_strdup (input));
  g_ptr_array_add (argv, real_output);
  g_ptr_array_add (argv, NULL);

  return (gchar **) g_ptr_array_free (argv, FALSE);
}

