/* OGMJob - A library to spawn processes
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmjob-spawn.h"
#include "ogmjob-marshal.h"
#include "ogmjob-enums.h"

#define OGMJOB_SPAWN_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMJOB_TYPE_SPAWN, OGMJobSpawnPriv))

struct _OGMJobSpawnPriv
{
  gint result;
  gint priority;
  gboolean async;
  GError *error;
  OGMJobSpawn *parent;
};

enum
{
  RUN,
  CANCEL,
  PROGRESS,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

GQuark
ogmjob_spawn_error_quark (void)
{
  static GQuark quark = 0;

  if (quark == 0)
    quark = g_quark_from_static_string ("ogmjob-spawn-error-quark");

  return quark;
}

G_DEFINE_ABSTRACT_TYPE (OGMJobSpawn, ogmjob_spawn, G_TYPE_OBJECT)

static void
ogmjob_spawn_class_init (OGMJobSpawnClass *klass)
{
  signals[RUN] = g_signal_new ("run", 
      G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_LAST, 
      G_STRUCT_OFFSET (OGMJobSpawnClass, run), 
      NULL, NULL, ogmjob_cclosure_marshal_INT__VOID, 
      G_TYPE_INT, 0);

  signals[CANCEL] = g_signal_new ("cancel", 
      G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_FIRST, 
      G_STRUCT_OFFSET (OGMJobSpawnClass, cancel), 
      NULL, NULL, g_cclosure_marshal_VOID__VOID, 
      G_TYPE_NONE, 0);

  signals[PROGRESS] = g_signal_new ("progress", 
      G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_FIRST, 
      G_STRUCT_OFFSET (OGMJobSpawnClass, progress), 
      NULL, NULL, g_cclosure_marshal_VOID__DOUBLE, 
      G_TYPE_NONE, 1, G_TYPE_DOUBLE);

  g_type_class_add_private (klass, sizeof (OGMJobSpawnPriv));
}

static void
ogmjob_spawn_init (OGMJobSpawn *spawn)
{
  spawn->priv = OGMJOB_SPAWN_GET_PRIVATE (spawn);
  spawn->priv->priority = G_MININT;
}

gint
ogmjob_spawn_run (OGMJobSpawn *spawn, GError **error)
{
  g_return_val_if_fail (OGMJOB_IS_SPAWN (spawn), OGMJOB_RESULT_ERROR);
  g_return_val_if_fail (error == NULL || *error == NULL, OGMJOB_RESULT_ERROR);

  spawn->priv->result = OGMJOB_RESULT_ERROR;
  spawn->priv->error = NULL;

  g_signal_emit (spawn, signals[RUN], 0, &spawn->priv->result);

  if (spawn->priv->result == OGMJOB_RESULT_ERROR && spawn->priv->error)
    g_propagate_error (error, spawn->priv->error);

  return spawn->priv->result;
}

void
ogmjob_spawn_cancel (OGMJobSpawn *spawn)
{
  g_return_if_fail (OGMJOB_IS_SPAWN (spawn));

  if (spawn->priv->result != OGMJOB_RESULT_CANCEL)
  {
#ifdef OGMJOB_DEBUG
    g_printerr ("Canceling %s\n", G_OBJECT_TYPE_NAME (spawn));
#endif

    spawn->priv->result = OGMJOB_RESULT_CANCEL;

    g_signal_emit (spawn, signals[CANCEL], 0);
  }
}

void
ogmjob_spawn_set_async (OGMJobSpawn *spawn, gboolean async)
{
  g_return_if_fail (OGMJOB_IS_SPAWN (spawn));

  spawn->priv->async = async;
}

gboolean
ogmjob_spawn_get_async (OGMJobSpawn *spawn)
{
  g_return_val_if_fail (OGMJOB_IS_SPAWN (spawn), FALSE);

  return spawn->priv->async;
}

OGMJobSpawn *
ogmjob_spawn_get_parent (OGMJobSpawn *spawn)
{
  g_return_val_if_fail (OGMJOB_IS_SPAWN (spawn), NULL);

  return spawn->priv->parent;
}

void
ogmjob_spawn_set_parent (OGMJobSpawn *spawn, OGMJobSpawn *parent)
{
  g_return_if_fail (OGMJOB_IS_SPAWN (spawn));
  g_return_if_fail (parent == NULL || OGMJOB_IS_SPAWN (parent));

  spawn->priv->parent = parent;
}

void
ogmjob_spawn_propagate_error (OGMJobSpawn *spawn, GError *error)
{
  g_return_if_fail (OGMJOB_IS_SPAWN (spawn));

  if (error)
    g_propagate_error (&spawn->priv->error, error);
}

void
ogmjob_spawn_set_priority (OGMJobSpawn *spawn, gint priority)
{
  g_return_if_fail (OGMJOB_IS_SPAWN (spawn));

  spawn->priv->priority = CLAMP (priority, -20, 19);
}

gint
ogmjob_spawn_get_priority (OGMJobSpawn *spawn)
{
  g_return_val_if_fail (OGMJOB_IS_SPAWN (spawn), G_MININT);

  if (spawn->priv->priority != G_MININT)
    return spawn->priv->priority;

  if (spawn->priv->parent)
    return ogmjob_spawn_get_priority (spawn->priv->parent);

  return 0;
}

