/* OGMDvd - A wrapper library around libdvdread
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n-lib.h>

#include <gtk/gtkliststore.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcelllayout.h>

#include "ogmdvd-title-chooser.h"

#define OGMDVD_TITLE_CHOOSER_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMDVD_TYPE_TITLE_CHOOSER, OGMDvdTitleChooserPriv))

struct _OGMDvdTitleChooserPriv
{
  OGMDvdDisc *disc;
};

static void ogmdvd_title_chooser_finalize (GObject *object);

G_DEFINE_TYPE (OGMDvdTitleChooser, ogmdvd_title_chooser, GTK_TYPE_COMBO_BOX);

static void
ogmdvd_title_chooser_class_init (OGMDvdTitleChooserClass *klass)
{
  GObjectClass *object_class;

  object_class = (GObjectClass *) klass;

  object_class->finalize = ogmdvd_title_chooser_finalize;

  g_type_class_add_private (klass, sizeof (OGMDvdTitleChooserPriv));
}

static void
ogmdvd_title_chooser_init (OGMDvdTitleChooser *chooser)
{
  GtkCellRenderer *cell;
  GtkListStore *store;

  chooser->priv = OGMDVD_TITLE_CHOOSER_GET_PRIVATE (chooser);

  store = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (chooser), GTK_TREE_MODEL (store));

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (chooser), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (chooser), cell, "text", 0, NULL);
}

static void
ogmdvd_title_chooser_finalize (GObject *object)
{
  OGMDvdTitleChooser *chooser;

  chooser = OGMDVD_TITLE_CHOOSER (object);

  if (chooser->priv->disc)
    ogmdvd_disc_unref (chooser->priv->disc);

  (*G_OBJECT_CLASS (ogmdvd_title_chooser_parent_class)->finalize) (object);
}

GtkWidget *
ogmdvd_title_chooser_new (void)
{
  return g_object_new (OGMDVD_TYPE_TITLE_CHOOSER, NULL);
}

void
ogmdvd_title_chooser_set_disc (OGMDvdTitleChooser *chooser, OGMDvdDisc *disc)
{
  g_return_if_fail (OGMDVD_IS_TITLE_CHOOSER (chooser));

  if (chooser->priv->disc != disc)
  {
    GtkTreeModel *model;
    OGMDvdTitle *title;

    gint vid, nvid, lvid, format, aspect;
    glong length, longest;
    gchar *str;

    if (disc)
      ogmdvd_disc_ref (disc);
    if (chooser->priv->disc)
      ogmdvd_disc_unref (chooser->priv->disc);
    chooser->priv->disc = disc;

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (chooser));
    gtk_list_store_clear (GTK_LIST_STORE (model));

    if (!disc)
      gtk_combo_box_set_active (GTK_COMBO_BOX (chooser), -1);
    else
    {
      nvid = ogmdvd_disc_get_n_titles (disc);
      for (vid = lvid = 0, longest = 0; vid < nvid; vid++)
      {
        title = ogmdvd_disc_get_nth_title (disc, vid);

        format = ogmdvd_title_get_video_format (title);
        aspect = ogmdvd_title_get_display_aspect (title);
        length = ogmdvd_title_get_length (title, NULL);

        if (length > longest)
        {
          longest = length;
          lvid = vid;
        }

        str = g_strdup_printf ("%s %02d (%s, %s)", _("Title"), vid + 1, 
            ogmdvd_get_video_format_label (format), 
            ogmdvd_get_display_aspect_label (aspect)); 
        gtk_combo_box_append_text (GTK_COMBO_BOX (chooser), str);
        g_free (str);

        ogmdvd_title_unref (title);
      }

      if (nvid > 0)
        gtk_combo_box_set_active (GTK_COMBO_BOX (chooser), lvid);
    }
  }
}

OGMDvdDisc *
ogmdvd_title_chooser_get_disc (OGMDvdTitleChooser *chooser)
{
  g_return_val_if_fail (OGMDVD_IS_TITLE_CHOOSER (chooser), NULL);

  return chooser->priv->disc;
}

OGMDvdTitle *
ogmdvd_title_chooser_get_active (OGMDvdTitleChooser *chooser)
{
  gint nr;

  g_return_val_if_fail (OGMDVD_IS_TITLE_CHOOSER (chooser), NULL);

  if (!chooser->priv->disc)
    return NULL;

  nr = gtk_combo_box_get_active (GTK_COMBO_BOX (chooser));
  if (nr < 0)
    return NULL;

  return ogmdvd_disc_get_nth_title (chooser->priv->disc, nr);
}

