/* OGMDvd - A wrapper library around libdvdread
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "config.h"

#include <string.h>
#include <glib/gi18n.h>

#include <gtk/gtkmenu.h>
#include <gtk/gtkcombobox.h>
#include <gtk/gtkliststore.h>
#include <gtk/gtkcelllayout.h>
#include <gtk/gtkcellrenderertext.h>

#include "ogmdvd-drive-chooser.h"
#include "ogmdvd.h"

#define OGMDVD_DRIVE_CHOOSER_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMDVD_TYPE_DRIVE_CHOOSER, OGMDvdDriveChooserPriv))

enum
{
  DEVICE_CHANGED,
  LAST_SIGNAL
};

struct OGMDvdDriveChooserPriv
{
  GList *drives;
};

static void ogmdvd_drive_chooser_init     (OGMDvdDriveChooser *chooser);
static void ogmdvd_drive_chooser_finalize (GObject          *object);

static int signals[LAST_SIGNAL] = { 0 };

G_DEFINE_TYPE (OGMDvdDriveChooser, ogmdvd_drive_chooser, GTK_TYPE_COMBO_BOX);

static NautilusBurnDrive *
get_drive (OGMDvdDriveChooser *chooser, GtkTreeIter *iter)
{
  NautilusBurnDrive *drive = NULL;
  GtkTreeModel *model;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (chooser));
  if (iter)
    gtk_tree_model_get (model, iter, 1, &drive, -1);
  else
  {
    GtkTreeIter it;

    if (gtk_tree_model_get_iter_first (model, &it))
      gtk_tree_model_get (model, &it, 1, &drive, -1);
  }

  return drive;
}

static void
ogmdvd_drive_chooser_class_init (OGMDvdDriveChooserClass *klass)
{
  GObjectClass *object_class;
  GtkWidgetClass *widget_class;

  object_class = (GObjectClass *) klass;
  widget_class = (GtkWidgetClass *) klass;

  object_class->finalize = ogmdvd_drive_chooser_finalize;

  signals[DEVICE_CHANGED] = g_signal_new ("device-changed", G_TYPE_FROM_CLASS (object_class), 
      G_SIGNAL_RUN_LAST, G_STRUCT_OFFSET (OGMDvdDriveChooserClass, device_changed), 
      NULL, NULL, g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);

  g_type_class_add_private (klass, sizeof (OGMDvdDriveChooserPriv));
}

static void
combo_device_changed (OGMDvdDriveChooser *chooser)
{
  NautilusBurnDrive *drive;

  drive = get_drive (chooser, NULL);
  if (drive)
    g_signal_emit (G_OBJECT (chooser), signals[DEVICE_CHANGED], 0, drive->device);
  else
    g_signal_emit (G_OBJECT (chooser), signals[DEVICE_CHANGED], 0, NULL);
}

static void
ogmdvd_drive_chooser_media_changed (OGMDvdDriveChooser *chooser)
{
  GtkTreeModel *model;

  if (gtk_combo_box_get_active (GTK_COMBO_BOX (chooser)) == -1)
    gtk_combo_box_set_active (GTK_COMBO_BOX (chooser), 0);

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (chooser));
  gtk_widget_set_sensitive (GTK_WIDGET (chooser), 
      gtk_tree_model_iter_n_children (model, NULL) > 0);
}

static void
ogmdvd_drive_chooser_media_removed (OGMDvdDriveChooser *chooser, NautilusBurnDrive *drive1)
{
  NautilusBurnDrive *drive2;
  GtkTreeModel *model;
  GtkTreeIter iter;

  model = gtk_combo_box_get_model (GTK_COMBO_BOX (chooser));
  if (gtk_tree_model_get_iter_first (model, &iter))
  {
    do
    {
      drive2 = get_drive (chooser, &iter);
      if (drive1 == drive2)
      {
        gtk_list_store_remove (GTK_LIST_STORE (model), &iter);
        break;
      }
    }
    while (gtk_tree_model_iter_next (model, &iter));
  }

  ogmdvd_drive_chooser_media_changed (chooser);
}

static void
ogmdvd_drive_chooser_media_added (OGMDvdDriveChooser *chooser, NautilusBurnDrive *drive)
{
  OGMDvdDisc *disc;

  disc = ogmdvd_disc_open (drive->device, NULL);
  if (!disc)
    ogmdvd_drive_chooser_media_removed (chooser, drive);
  else
  {
    GtkTreeIter iter;
    GtkTreeModel *model;
    gchar *title, *text;

    title = ogmdvd_disc_get_label (disc);
    text = g_strdup_printf ("<b>%s</b>\n%s", title ? title : _("Unknown"),
        drive->display_name ? drive->display_name : _("Unknown Drive"));
    g_free (title);

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (chooser));
    gtk_list_store_append (GTK_LIST_STORE (model), &iter);
    gtk_list_store_set (GTK_LIST_STORE (model), &iter, 0, text, 1, drive, -1);
    g_free (text);

    ogmdvd_disc_unref (disc);

    ogmdvd_drive_chooser_media_changed (chooser);
  }
}

static void
ogmdvd_drive_chooser_fill (OGMDvdDriveChooser *chooser)
{
  GList *drive;

  chooser->priv->drives = nautilus_burn_drive_get_list ();

  for (drive = chooser->priv->drives; drive; drive = drive->next)
  {
    if (NAUTILUS_BURN_DRIVE (drive->data)->type & NAUTILUS_BURN_DRIVE_TYPE_DVD_DRIVE)
    {
      nautilus_burn_drive_set_monitor_enabled (NAUTILUS_BURN_DRIVE (drive->data), TRUE);

      g_signal_connect_swapped (drive->data, "media-added", 
          G_CALLBACK (ogmdvd_drive_chooser_media_added), chooser);
      g_signal_connect_swapped (drive->data, "media-removed", 
          G_CALLBACK (ogmdvd_drive_chooser_media_removed), chooser);

      ogmdvd_drive_chooser_media_added (chooser, NAUTILUS_BURN_DRIVE (drive->data));
    }
  }
}

static void
ogmdvd_drive_chooser_init (OGMDvdDriveChooser *chooser)
{
  GtkCellRenderer *cell;
  GtkListStore *store;

  chooser->priv = OGMDVD_DRIVE_CHOOSER_GET_PRIVATE (chooser);

  store = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_POINTER);
  gtk_combo_box_set_model (GTK_COMBO_BOX (chooser), GTK_TREE_MODEL (store));

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (chooser), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (chooser), cell, "markup", 0, NULL);

  ogmdvd_drive_chooser_fill (chooser);

  g_signal_connect_swapped (G_OBJECT (chooser), "changed", G_CALLBACK (combo_device_changed), chooser);
}

static void
ogmdvd_drive_chooser_finalize (GObject *object)
{
  OGMDvdDriveChooser *chooser;

  g_return_if_fail (object != NULL);
  g_return_if_fail (OGMDVD_IS_DRIVE_CHOOSER (object));

  chooser = OGMDVD_DRIVE_CHOOSER (object);

  if (chooser->priv->drives)
  {
    g_list_foreach (chooser->priv->drives, (GFunc) g_object_unref, NULL);
    g_list_free (chooser->priv->drives);
    chooser->priv->drives = NULL;
  }

  if (G_OBJECT_CLASS (ogmdvd_drive_chooser_parent_class)->finalize != NULL)
    (*G_OBJECT_CLASS (ogmdvd_drive_chooser_parent_class)->finalize) (object);
}

GtkWidget *
ogmdvd_drive_chooser_new (void)
{
  GtkWidget *widget;

  widget = g_object_new (OGMDVD_TYPE_DRIVE_CHOOSER, NULL);

  return widget;
}

NautilusBurnDrive *
ogmdvd_drive_chooser_get_drive (OGMDvdDriveChooser *chooser)
{
  NautilusBurnDrive *drive;

  g_return_val_if_fail (chooser != NULL, NULL);
  g_return_val_if_fail (OGMDVD_IS_DRIVE_CHOOSER (chooser), NULL);

  drive = get_drive (chooser, NULL);
  if (drive)
    g_object_ref (drive);

  return drive;
}

