/* OGMDvd - A wrapper library around libdvdread
 * Copyright (C) 2004-2006 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <glib/gi18n-lib.h>

#include <gtk/gtkliststore.h>
#include <gtk/gtkcellrenderertext.h>
#include <gtk/gtkcelllayout.h>

#include "ogmdvd-audio-chooser.h"

#define OGMDVD_AUDIO_CHOOSER_GET_PRIVATE(o) \
    (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMDVD_TYPE_AUDIO_CHOOSER, OGMDvdAudioChooserPriv))

struct _OGMDvdAudioChooserPriv
{
  OGMDvdTitle *title;
};

static void ogmdvd_audio_chooser_finalize (GObject *object);

G_DEFINE_TYPE (OGMDvdAudioChooser, ogmdvd_audio_chooser, GTK_TYPE_COMBO_BOX);

static void
ogmdvd_audio_chooser_class_init (OGMDvdAudioChooserClass *klass)
{
  GObjectClass *object_class;

  object_class = (GObjectClass *) klass;

  object_class->finalize = ogmdvd_audio_chooser_finalize;

  g_type_class_add_private (klass, sizeof (OGMDvdAudioChooserPriv));
}

static void
ogmdvd_audio_chooser_init (OGMDvdAudioChooser *chooser)
{
  GtkCellRenderer *cell;
  GtkListStore *store;

  chooser->priv = OGMDVD_AUDIO_CHOOSER_GET_PRIVATE (chooser);

  store = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_combo_box_set_model (GTK_COMBO_BOX (chooser), GTK_TREE_MODEL (store));

  cell = gtk_cell_renderer_text_new ();
  gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (chooser), cell, TRUE);
  gtk_cell_layout_set_attributes (GTK_CELL_LAYOUT (chooser), cell, "text", 0, NULL);
}

static void
ogmdvd_audio_chooser_finalize (GObject *object)
{
  OGMDvdAudioChooser *chooser;

  chooser = OGMDVD_AUDIO_CHOOSER (object);

  if (chooser->priv->title)
    ogmdvd_title_unref (chooser->priv->title);

  (*G_OBJECT_CLASS (ogmdvd_audio_chooser_parent_class)->finalize) (object);
}

GtkWidget *
ogmdvd_audio_chooser_new (void)
{
  return g_object_new (OGMDVD_TYPE_AUDIO_CHOOSER, NULL);
}

void
ogmdvd_audio_chooser_set_title (OGMDvdAudioChooser *chooser, OGMDvdTitle *title, gint pref)
{
  g_return_if_fail (OGMDVD_IS_AUDIO_CHOOSER (chooser));
  g_return_if_fail (title != NULL);

  if (chooser->priv->title != title)
  {
    GtkTreeModel *model;
    OGMDvdAudioStream *astream;
    gint channels, format, lang, content;
    gint aid, naid, selected;
    gchar *str;

    ogmdvd_title_ref (title);
    if (chooser->priv->title)
      ogmdvd_title_unref (chooser->priv->title);
    chooser->priv->title = title;

    model = gtk_combo_box_get_model (GTK_COMBO_BOX (chooser));
    gtk_list_store_clear (GTK_LIST_STORE (model));

    naid = ogmdvd_title_get_n_audio_streams (title);
    for (aid = selected = 0; aid < naid; aid++)
    {
      astream = ogmdvd_title_get_nth_audio_stream (title, aid);

      channels = ogmdvd_audio_stream_get_channels (astream);
      content = ogmdvd_audio_stream_get_content (astream);
      format = ogmdvd_audio_stream_get_format (astream);
      lang = ogmdvd_audio_stream_get_language (astream);

      if (!selected && lang == pref)
        selected = aid;

      if (content > 0)
        str = g_strdup_printf ("%s %02d: %s (%s, %s, %s)", _("Track"), aid + 1, 
            ogmdvd_get_audio_content_label (content), ogmdvd_get_language_label (lang), 
            ogmdvd_get_audio_format_label (format), ogmdvd_get_audio_channels_label (channels));
      else
        str = g_strdup_printf ("%s %02d: (%s, %s, %s)", _("Track"), aid + 1, 
            ogmdvd_get_language_label (lang), ogmdvd_get_audio_format_label (format), 
            ogmdvd_get_audio_channels_label (channels));

      gtk_combo_box_append_text (GTK_COMBO_BOX (chooser), str);
      g_free (str);

      ogmdvd_audio_stream_unref (astream);
    }

    gtk_widget_set_sensitive (GTK_WIDGET (chooser), naid > 0);
    if (naid > 0)
      gtk_combo_box_set_active (GTK_COMBO_BOX (chooser), selected);
  }
}

OGMDvdTitle *
ogmdvd_audio_chooser_get_video_stream (OGMDvdAudioChooser *chooser)
{
  g_return_val_if_fail (OGMDVD_IS_AUDIO_CHOOSER (chooser), NULL);

  return chooser->priv->title;
}

OGMDvdAudioStream *
ogmdvd_audio_chooser_get_active (OGMDvdAudioChooser *chooser)
{
  gint aid;

  g_return_val_if_fail (OGMDVD_IS_AUDIO_CHOOSER (chooser), NULL);

  if (!chooser->priv->title)
    return NULL;

  aid = gtk_combo_box_get_active (GTK_COMBO_BOX (chooser));
  if (aid < 0)
    return NULL;

  return ogmdvd_title_get_nth_audio_stream (chooser->priv->title, aid); 
}

