/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-
 *
 * nautilus-burn-drive.c: easy to use cd burner software
 *
 * Copyright (C) 2002-2004 Red Hat, Inc.
 * Copyright (C) 2005 William Jon McCann <mccann@jhu.edu>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * Authors: Alexander Larsson <alexl@redhat.com>
 *          Bastien Nocera <hadess@hadess.net>
 *          William Jon McCann <mccann@jhu.edu>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#if defined(USE_HAL4) || defined(USE_HAL5)

#include <string.h>
#include <libhal.h>
#include <stdio.h>
#include <glib.h>

#include "nautilus-burn-drive.h"
#include "nautilus-burn-drive-common.h"

#ifdef USE_HAL5
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>
#endif

static LibHalContext *
get_hal_context (void)
{
  static LibHalContext *ctx = NULL;

#ifdef USE_HAL5
  DBusError error;
  DBusConnection *dbus_conn;

  if (ctx == NULL)
  {
    ctx = libhal_ctx_new ();
    if (ctx == NULL)
      g_warning ("Could not create a HAL context");
    else
    {
      dbus_error_init (&error);
      dbus_conn = dbus_bus_get (DBUS_BUS_SYSTEM, &error);

      if (dbus_error_is_set (&error))
      {
        g_warning ("Could not connect to system bus: %s", error.message);
        dbus_error_free (&error);
        return NULL;
      }

      libhal_ctx_set_dbus_connection (ctx, dbus_conn);

      if (!libhal_ctx_init (ctx, &error))
      {
        g_warning ("Could not initalize the HAL context: %s", error.message);

        if (dbus_error_is_set (&error))
          dbus_error_free (&error);

        libhal_ctx_free (ctx);
        ctx = NULL;
      }
    }
  }
#else /* USE_HAL5 */
  LibHalFunctions hal_functions = {
    NULL,                       /* mainloop integration */
    NULL,                       /* device_added */
    NULL,                       /* device_removed */
    NULL,                       /* device_new_capability */
    NULL,                       /* device_lost_capability */
    NULL,                       /* property_modified */
    NULL,                       /* device_condition */
  };

  if (ctx == NULL)
    ctx = hal_initialize (&hal_functions, FALSE);
#endif /* USE_HAL5 */

  return ctx;
}

#ifdef USE_HAL5

#define LIBHAL_PROP_EXTRACT_BEGIN if (FALSE)
#define LIBHAL_PROP_EXTRACT_INT(_property_, _where_) \
  else if (strcmp (key, _property_) == 0 && type == LIBHAL_PROPERTY_TYPE_INT32) \
    _where_ = libhal_psi_get_int (&it)
#define LIBHAL_PROP_EXTRACT_INT_DIV(_property_, _where_, _div_) \
  else if (strcmp (key, _property_) == 0 && type == LIBHAL_PROPERTY_TYPE_INT32) \
    _where_ = libhal_psi_get_int (&it) / _div_
#define LIBHAL_PROP_EXTRACT_STRING(_property_, _where_) \
  else if (strcmp (key, _property_) == 0 && type == LIBHAL_PROPERTY_TYPE_STRING) \
    _where_ = (libhal_psi_get_string (&it) != NULL && strlen (libhal_psi_get_string (&it)) > 0) ? strdup (libhal_psi_get_string (&it)) : NULL
#define LIBHAL_PROP_EXTRACT_BOOL(_property_, _where_) \
  else if (strcmp (key, _property_) == 0 && type == LIBHAL_PROPERTY_TYPE_BOOLEAN) \
    _where_ = libhal_psi_get_bool (&it)
#define LIBHAL_PROP_EXTRACT_BOOL_BITFIELD(_property_, _where_, _field_) \
  else if (strcmp (key, _property_) == 0 && type == LIBHAL_PROPERTY_TYPE_BOOLEAN) \
    _where_ |= libhal_psi_get_bool (&it) ? _field_ : 0
#define LIBHAL_PROP_EXTRACT_STRLIST(_property_, _where_) \
  else if (strcmp (key, _property_) == 0 && type == LIBHAL_PROPERTY_TYPE_STRLIST) \
    _where_ = my_strvdup (libhal_psi_get_strlist (&it))
#define LIBHAL_PROP_EXTRACT_END ;

static NautilusBurnDrive *
hal_drive_from_udi (LibHalContext * ctx, const char *udi)
{
  LibHalPropertySet *pset;
  LibHalPropertySetIterator it;
  DBusError error;
  NautilusBurnDrive *drive;

  LIBHAL_CHECK_LIBHALCONTEXT (ctx, FALSE);

  dbus_error_init (&error);

  if ((pset = libhal_device_get_all_properties (ctx, udi, &error)) == NULL)
  {
    if (dbus_error_is_set (&error))
    {
      g_warning ("Could not get all properties: %s", error.message);
      dbus_error_free (&error);
    }

    return NULL;
  }

  drive = nautilus_burn_drive_new ();
  drive->type = NAUTILUS_BURN_DRIVE_TYPE_CD_DRIVE;

  for (libhal_psi_init (&it, pset); libhal_psi_has_more (&it); libhal_psi_next (&it))
  {
    int type;
    char *key;

    type = libhal_psi_get_type (&it);
    key = libhal_psi_get_key (&it);

    LIBHAL_PROP_EXTRACT_BEGIN;

    LIBHAL_PROP_EXTRACT_STRING ("block.device", drive->device);
    LIBHAL_PROP_EXTRACT_STRING ("storage.model", drive->display_name);

    LIBHAL_PROP_EXTRACT_INT_DIV ("storage.cdrom.read_speed", drive->max_speed_read, CD_ROM_SPEED);
    LIBHAL_PROP_EXTRACT_INT_DIV ("storage.cdrom.write_speed", drive->max_speed_write, CD_ROM_SPEED);

    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.cdr", drive->type, NAUTILUS_BURN_DRIVE_TYPE_CD_RECORDER);
    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.cdrw", drive->type, NAUTILUS_BURN_DRIVE_TYPE_CDRW_RECORDER);
    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.dvd", drive->type, NAUTILUS_BURN_DRIVE_TYPE_DVD_DRIVE);
    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.dvdplusr", drive->type, NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_R_RECORDER);
    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.dvdplusrw", drive->type, NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_RW_RECORDER);
    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.dvdplusrdl", drive->type, NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_R_DL_RECORDER);

    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.dvdr", drive->type, NAUTILUS_BURN_DRIVE_TYPE_DVD_RW_RECORDER);
    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.dvdrw", drive->type, NAUTILUS_BURN_DRIVE_TYPE_DVD_RW_RECORDER);
    LIBHAL_PROP_EXTRACT_BOOL_BITFIELD ("storage.cdrom.dvdram", drive->type, NAUTILUS_BURN_DRIVE_TYPE_DVD_RAM_RECORDER);

    LIBHAL_PROP_EXTRACT_END;
  }

  libhal_free_property_set (pset);

  drive->udi = g_strdup (udi);
  drive->cdrecord_id = g_strdup (drive->device);

  if (!drive->display_name)
    drive->display_name = g_strdup_printf ("Unnamed Drive (%s)", drive->device);

  return drive;
}

static dbus_bool_t
hal_mainloop_integration (LibHalContext *ctx,
            DBusError     *error)
{
  DBusConnection *dbus_connection;

  dbus_connection = dbus_bus_get (DBUS_BUS_SYSTEM, error);

  if (dbus_error_is_set (error))
    return FALSE;

  dbus_connection_setup_with_g_main (dbus_connection, NULL);

  libhal_ctx_set_dbus_connection (ctx, dbus_connection);

  return TRUE;
}

void
nautilus_burn_drive_hal_init (NautilusBurnDrive *drive)
{
  LibHalContext *ctx;
  DBusError      error;

  if (! (ctx = libhal_ctx_new ())) 
  {
    g_warning ("failed to initialize HAL!");
    return;
  }

  dbus_error_init (&error);
  if (! hal_mainloop_integration (ctx, &error)) 
  {
    g_warning ("hal_initialize failed: %s", error.message);
    dbus_error_free (&error);
    return;
  }

  if (! libhal_ctx_init (ctx, &error)) 
  {
    g_warning ("hal_initialize failed: %s", error.message);
    dbus_error_free (&error);
    libhal_ctx_free (ctx);
    return;
  }

  drive->ctx = ctx;
}

void
nautilus_burn_drive_hal_shutdown (NautilusBurnDrive *drive)
{
  DBusError error;

  dbus_error_init (&error);

  if (! libhal_ctx_shutdown (drive->ctx, &error)) 
  {
    g_warning ("hal_shutdown failed: %s\n", error.message);
    dbus_error_free (&error);
    return;
  }

  if (! libhal_ctx_free (drive->ctx))
    g_warning ("hal_shutdown failed - unable to free hal context\n");
}

static gboolean
hal_udi_is_our_drive (LibHalContext *ctx, const char *udi, NautilusBurnDrive *drive)
{
  char    *device = NULL;
  gboolean res    = FALSE;

  if (drive == NULL || drive->device == NULL )
    return res;

  if (! libhal_device_property_exists (ctx, udi, "block.device", NULL))
    return res;

  device = libhal_device_get_property_string (ctx, udi, "block.device", NULL);
  if (device && strcmp (device, drive->device) == 0)
    res = TRUE;

  if (device)
    libhal_free_string (device);

  return res;
}

static void
hal_device_added (LibHalContext *ctx, const char *udi)
{
  NautilusBurnDrive *drive;

  drive = libhal_ctx_get_user_data (ctx);

  g_return_if_fail (drive != NULL);
  g_return_if_fail (udi != NULL);

  if (hal_udi_is_our_drive (ctx, udi, drive)) 
  {
    g_free (drive->monitor_udi);
    drive->monitor_udi = g_strdup (udi);

    g_signal_emit_by_name (drive, "media-added");
  }
}

static void
hal_device_removed (LibHalContext *ctx, const char *udi)
{
  NautilusBurnDrive *drive;

  drive = libhal_ctx_get_user_data (ctx);

  g_return_if_fail (drive != NULL);
  g_return_if_fail (udi != NULL);

  if (drive->monitor_udi && strcmp (drive->monitor_udi, udi) == 0) 
  {
    g_free (drive->monitor_udi);
    drive->monitor_udi = NULL;

    g_signal_emit_by_name (drive, "media-removed");
  }
}

void
nautilus_burn_drive_hal_set_monitor (NautilusBurnDrive *drive, gboolean enabled)
{
  g_free (drive->monitor_udi);
  drive->monitor_udi = NULL;

  if (enabled) 
  {
    char    **device_names;
    int       num_devices = -1;
    DBusError error;

    libhal_ctx_set_user_data (drive->ctx, drive);
    libhal_ctx_set_device_added (drive->ctx, hal_device_added);
    libhal_ctx_set_device_removed (drive->ctx, hal_device_removed);

    dbus_error_init (&error);
    device_names = libhal_manager_find_device_string_match (drive->ctx,
        "info.parent", drive->udi, &num_devices, &error);

    if (dbus_error_is_set (&error)) 
    {
      g_warning ("%s\n", error.message);
      dbus_error_free (&error);
    }

    if (num_devices > 0)
      drive->monitor_udi = g_strdup (device_names [0]);

    libhal_free_string_array (device_names);
  } 
  else 
  {
    libhal_ctx_set_user_data (drive->ctx, NULL);
    libhal_ctx_set_device_added (drive->ctx, NULL);
    libhal_ctx_set_device_removed (drive->ctx, NULL);
  }

/* FIXME: check for devices already in the drive and emit signals for them? */
}

NautilusBurnMediaType
nautilus_burn_drive_hal_get_media_type_full (NautilusBurnDrive *drive)
{
	if (drive->udi)
  {
		LibHalContext        *ctx;
		char                **device_names;
		int                   num_devices = -1;
		NautilusBurnMediaType type;
		char                 *hal_type;
		DBusError             error;
		
		ctx = drive->ctx;

		dbus_error_init (&error);
		if (ctx != NULL) 
    {
			device_names = libhal_manager_find_device_string_match (ctx, 
          "info.parent", drive->udi, &num_devices, &error);

			if (dbus_error_is_set (&error)) 
      {
				g_warning ("%s\n", error.message);
				dbus_error_free (&error);
				libhal_free_string_array (device_names);

				return NAUTILUS_BURN_MEDIA_TYPE_ERROR;
			}

			if (num_devices <= 0) 
      {
				g_warning ("No HAL devices found for UDI '%s'", drive->udi);
				libhal_free_string_array (device_names);

				return NAUTILUS_BURN_MEDIA_TYPE_ERROR;
			}

			/* just look at the first child */
			if (libhal_device_get_property_bool (ctx, 
            device_names [0], "volume.is_mounted", NULL))
				type = NAUTILUS_BURN_MEDIA_TYPE_BUSY;
			else 
      {
				type = NAUTILUS_BURN_MEDIA_TYPE_BUSY;
				hal_type = libhal_device_get_property_string (ctx, 
            device_names [0], "volume.disc.type", NULL);

				if (hal_type == NULL || strcmp (hal_type, "unknown") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN;
				else if (strcmp (hal_type, "cd_rom") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_CD;
				else if (strcmp (hal_type, "cd_r") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_CDR;
				else if (strcmp (hal_type, "cd_rw") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_CDRW;
				else if (strcmp (hal_type, "dvd_rom") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_DVD;
				else if (strcmp (hal_type, "dvd_r") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_DVDR;
				else if (strcmp (hal_type, "dvd_ram") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_DVD_RAM;
				else if (strcmp (hal_type, "dvd_rw") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_DVDRW;
				else if (strcmp (hal_type, "dvd_plus_rw") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_RW;
				else if (strcmp (hal_type, "dvd_plus_r") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_R;
				else if (strcmp (hal_type, "dvd_plus_r_dl") == 0)
					type = NAUTILUS_BURN_MEDIA_TYPE_DVD_PLUS_R_DL;
				else
					type = NAUTILUS_BURN_MEDIA_TYPE_UNKNOWN;
				
				if (hal_type != NULL)
					libhal_free_string (hal_type);
			}

			libhal_free_string_array (device_names);

			return type;
		}
	}

	return nautilus_burn_drive_get_media_type_from_path_full (drive->device);
}

#else

#define GET_BOOL_PROP(x) (hal_device_property_exists (ctx, udi, x) && hal_device_get_property_bool (ctx, udi, x))

static NautilusBurnDrive *
hal_drive_from_udi (LibHalContext * ctx, const char *udi)
{
  NautilusBurnDrive *drive;
  char *string;
  gboolean is_cdr;

  is_cdr = GET_BOOL_PROP ("storage.cdrom.cdr");

  drive = nautilus_burn_drive_new ();
  drive->type = NAUTILUS_BURN_DRIVE_TYPE_CD_DRIVE;
  if (is_cdr != FALSE)
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CD_RECORDER;
  if (GET_BOOL_PROP ("storage.cdrom.cdrw"))
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_CDRW_RECORDER;
  if (GET_BOOL_PROP ("storage.cdrom.dvd"))
  {
    drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_DRIVE;

    if (GET_BOOL_PROP ("storage.cdrom.dvdram"))
      drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_RAM_RECORDER;
    if (GET_BOOL_PROP ("storage.cdrom.dvdr"))
      drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_RW_RECORDER;
    if (GET_BOOL_PROP ("storage.cdrom.dvd"))
      drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_DRIVE;
    if (GET_BOOL_PROP ("storage.cdrom.dvdplusr"))
      drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_R_RECORDER;
    if (GET_BOOL_PROP ("storage.cdrom.dvdplusrw"))
      drive->type |= NAUTILUS_BURN_DRIVE_TYPE_DVD_PLUS_RW_RECORDER;
  }

  drive->udi = g_strdup (udi);
  drive->device = hal_device_get_property_string (ctx, udi, "block.device");
  drive->cdrecord_id = g_strdup (drive->device);

  string = hal_device_get_property_string (ctx, udi, "storage.model");
  if (string != NULL)
    drive->display_name = string;
  else
    drive->display_name = g_strdup_printf ("Unnamed Drive (%s)", drive->device);

  drive->max_speed_read = hal_device_get_property_int (ctx, udi, "storage.cdrom.read_speed") / CD_ROM_SPEED;

  if (hal_device_property_exists (ctx, udi, "storage.cdrom.write_speed"))
    drive->max_speed_write = hal_device_get_property_int (ctx, udi, "storage.cdrom.write_speed") / CD_ROM_SPEED;

  return drive;
}

#endif /* USE_HAL5 */

GList *
nautilus_burn_drive_hal_scan (void)
{
  GList *drives = NULL;
  LibHalContext *ctx;
  int i, num_devices;
  char **device_names;

  ctx = get_hal_context ();

  if (ctx == NULL)
  {
    return NULL;
  }

#ifdef USE_HAL5
  device_names = libhal_find_device_by_capability (ctx, "storage.cdrom", &num_devices, NULL);
#else
  device_names = hal_find_device_by_capability (ctx, "storage.cdrom", &num_devices);
#endif

  if (device_names == NULL)
    return NULL;

  for (i = 0; i < num_devices; i++)
  {
    NautilusBurnDrive *drive;

    drive = hal_drive_from_udi (ctx, device_names[i]);

    nautilus_burn_drive_add_whitelist (drive);

    drives = g_list_prepend (drives, drive);
  }

#ifdef USE_HAL5
  libhal_free_string_array (device_names);
#else
  hal_free_string_array (device_names);
#endif

  drives = g_list_reverse (drives);

  return drives;
}

#endif /* USE_HAL */

