/* upsdrvctl.c - UPS driver controller

   Copyright (C) 2001  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include "config.h"
#include "shared.h"
#include "proto.h"
#include "version.h"
#include "common.h"
#include "upsconf.h"

typedef struct {
	char	*upsname;
	char	*driver;
	char	*port;
	int	sdorder;
	int	maxstartdelay;
	void	*next;
}	ups_t;

	ups_t	*upstable = NULL;

	int	execwait, verbose = 0, maxsdorder = 0, testmode = 0;

	/* timer - keeps us from getting stuck if a driver hangs */
	int	maxstartdelay = 45;

void do_upsconf_args(char *upsname, char *var, char *val)
{
	ups_t	*tmp, *last;

	/* only one global argument right now */
	if (!upsname) {
		if (!strcmp(var, "maxstartdelay"))
			maxstartdelay = atoi(val);

		return;
	}

	last = tmp = upstable;

	while (tmp) {
		last = tmp;

		if (!strcmp(tmp->upsname, upsname)) {
			if (!strcmp(var, "driver")) 
				tmp->driver = xstrdup(val);
			if (!strcmp(var, "port")) 
				tmp->port = xstrdup(val);
			if (!strcmp(var, "maxstartdelay"))
				tmp->maxstartdelay = atoi(val);

			if (!strcmp(var, "sdorder")) {
				tmp->sdorder = atoi(val);

				if (tmp->sdorder > maxsdorder)
					maxsdorder = tmp->sdorder;
			}

			return;
		}

		tmp = tmp->next;
	}

	tmp = xmalloc(sizeof(ups_t));
	tmp->upsname = xstrdup(upsname);
	tmp->driver = NULL;
	tmp->port = NULL;
	tmp->next = NULL;
	tmp->sdorder = 0;
	tmp->maxstartdelay = -1;	/* use global value by default */

	if (!strcmp(var, "driver"))
		tmp->driver = xstrdup(val);
	if (!strcmp(var, "port"))
		tmp->port = xstrdup(val);

	if (last)
		last->next = tmp;
	else
		upstable = tmp;
}

int read_pid_file(char *upsname, char *driver, char *port)
{
	char	pidfn[SMALLBUF], buf[SMALLBUF];
	int	ret, pid;
	struct	stat	fs;
	FILE	*pidf;

	snprintf(pidfn, sizeof(pidfn), "%s/%s-%s.pid", ALTPIDPATH,
		driver, xbasename(port));
	ret = stat(pidfn, &fs);

	if (ret != 0) {
		upslog(LOG_ERR, "Can't open %s", pidfn);
		return -1;
	}

	pidf = fopen(pidfn, "r");

	if (!pidf) {
		upslog(LOG_ERR, "Can't open %s", pidfn);
		return -1;
	}

	fgets(buf, sizeof(buf), pidf);
	fclose(pidf);

	buf[strlen(buf)-1] = '\0';

	pid = strtol(buf, (char **)NULL, 10);

	return pid;
}

/* handle sending the signal */
void send_term(char *upsname, char *driver, char *port)
{
	char	pidfn[SMALLBUF], buf[SMALLBUF];
	int	ret, pid;
	struct	stat	fs;
	FILE	*pidf;

	printf("Stopping UPS: %s\n", upsname);

	snprintf(pidfn, sizeof(pidfn), "%s/%s-%s.pid", ALTPIDPATH,
		driver, xbasename(port));
	ret = stat(pidfn, &fs);

	if (ret != 0) {
		upslog(LOG_ERR, "Can't open %s", pidfn);
		return;
	}

	pidf = fopen(pidfn, "r");

	if (!pidf) {
		upslog(LOG_ERR, "Can't open %s", pidfn);
		return;
	}

	fgets(buf, sizeof(buf), pidf);
	buf[strlen(buf)-1] = '\0';

	pid = strtol(buf, (char **)NULL, 10);

	if (pid < 2) {
		upslogx(LOG_NOTICE, "Ignoring invalid pid %d in %s",
			pid, pidfn);
		return;
	}

	if (verbose)
		printf("Sending signal: kill -TERM %d\n", pid);

	if (testmode)
		return;

	ret = kill(pid, SIGTERM);

	if (ret < 0) {
		upslog(LOG_ERR, "kill pid %d failed", pid);
		return;
	}
}

/* stop user-selected driver */
void stop_one_driver(char *upsname)
{
	ups_t	*tmp = upstable;

	if (!tmp)
		fatalx("Error: no UPS definitions found in ups.conf!\n");

	while (tmp) {
		if (!strcmp(tmp->upsname, upsname)) {
			send_term(tmp->upsname, tmp->driver, tmp->port);
			return;
		}

		tmp = tmp->next;
	}

	fatalx("UPS %s not found in ups.conf", upsname);
}

/* walk ups table, but stop drivers instead */
void stop_all_drivers(void)
{
	ups_t	*tmp = upstable;

	if (!tmp)
		fatalx("Error: no UPS definitions found in ups.conf!\n");

	while (tmp) {
		send_term(tmp->upsname, tmp->driver, tmp->port);

		tmp = tmp->next;
	}

	exit(0);
}

/* check user-selected driver */
void check_one_driver(char *upsname)
{
	ups_t	*tmp = upstable;
	int	pid, stat_error = 0;

	if (!tmp)
		fatalx("Error: no UPS definitions found in ups.conf!\n");

	while (tmp) {
		if (!strcmp(tmp->upsname, upsname)) {
			pid = read_pid_file(tmp->upsname, tmp->driver, 
				tmp->port);

			if (pid < 2)
				stat_error = 1;

			if (verbose) {
				if (pid > 1)
					printf("UPS %s: driver %s (PID %d)\n", 
						tmp->upsname, tmp->driver, pid);
				else
					printf("UPS %s: driver %s not running "
						"or invalid PID\n",
						tmp->upsname, tmp->driver);
			}

			exit(stat_error);
		}

		tmp = tmp->next;
	}

	fatalx("UPS %s not found in ups.conf", upsname);
}

/* walk ups table and make sure they're still running */
void check_all_drivers(void)
{
	ups_t	*tmp = upstable;
	int	pid, stat_error = 0;

	if (!tmp)
		fatalx("Error: no UPS definitions found in ups.conf!\n");

	while (tmp) {
		pid = read_pid_file(tmp->upsname, tmp->driver, tmp->port);

		if (pid < 2)
			stat_error = 1;

		if (verbose) {
			if (pid > 1)
				printf("UPS %s: driver %s (PID %d)\n", 
					tmp->upsname, tmp->driver, pid);
			else
				printf("UPS %s: driver %s not running or "
					"invalid PID\n",
					tmp->upsname, tmp->driver);
		}

		tmp = tmp->next;
	}

	exit(stat_error);
}

/* reap exiting children */
void sigchld(int sig)
{
	execwait = 0;
}

/* break free of drivers that don't start up quickly */
void sigalrm(int sig)
{
	upslogx(LOG_NOTICE, "Startup timer elapsed, continuing...");
	execwait = 0;
}

void forkexec(char *prog, ups_t *ups)
{
	int	ret;
	char	*argv[4];

	/* TODO: consider redoing this with wait() depending on portability */

	/* parent spins until the child exits */
	execwait = 1;

	ret = fork();

	if (ret < 0)
		fatal("fork");

	/* parent */
	if (ret != 0) {

		/* catching this will unset execwait */
		signal(SIGCHLD, sigchld);

		/* give us a way out if the child gets stuck */
		signal(SIGALRM, sigalrm);

		if (ups->maxstartdelay != -1)
			alarm(ups->maxstartdelay);
		else
			alarm(maxstartdelay);

		/* spin until the child is done */
		while (execwait)
			usleep(250000);

		alarm(0);
		signal(SIGALRM, SIG_IGN);

		return;
	}

	if (verbose)
		printf("exec: %s -a %s\n", prog, ups->upsname);

	argv[0] = xstrdup(prog);
	argv[1] = "-a";
	argv[2] = ups->upsname;
	argv[3] = NULL;

	/* child */

	if (testmode)
		exit(0);

	ret = execve(prog, argv, NULL);

	/* should not be reached */
	fatal("execve");
}

void start_driver(ups_t *ups)
{
	char	dfn[SMALLBUF];
	int	ret;
	struct	stat	fs;

	snprintf(dfn, sizeof(dfn), "%s/%s", DRVPATH, ups->driver);
	ret = stat(dfn, &fs);

	if (ret < 0)
		fatal("Can't start %s", dfn);

	forkexec(dfn, ups);
}

/* start user-selected driver */
void start_one_driver(char *upsname)
{
	ups_t	*tmp = upstable;

	if (!tmp)
		fatalx("Error: no UPS definitions found in ups.conf!\n");

	while (tmp) {
		if (!strcmp(tmp->upsname, upsname)) {
			start_driver(tmp);
			return;
		}

		tmp = tmp->next;
	}

	fatalx("UPS %s not found in ups.conf", upsname);
}

/* walk ups table and invoke drivers */
void start_all_drivers(void)
{
	ups_t	*tmp = upstable;

	if (!tmp)
		fatalx("Error: no UPS definitions found in ups.conf!\n");

	while (tmp) {
		start_driver(tmp);
		tmp = tmp->next;
	}
}

void help(const char *progname)
{
	printf("Starts and stops UPS drivers via ups.conf.\n\n");
	printf("usage: %s [-h] [-t] [-v] (start | stop | shutdown [<ups>])\n\n", progname);

	printf("  -h			display this help\n");
	printf("  -v			enable verbose messages\n");
	printf("  -t			testing mode - prints actions without doing them\n");
	printf("  start			start all UPS drivers in ups.conf\n");
	printf("  start	<ups>		only start driver for UPS <ups>\n");
	printf("  stop			stop all UPS drivers in ups.conf\n");
	printf("  stop <ups>		only stop driver for UPS <ups>\n");
	printf("  status		check all driver PIDs, returns 0 on success\n");
	printf("  status <ups>		check just UPS <ups>\n");
	printf("  shutdown		shutdown all UPS drivers in ups.conf\n");
	printf("  shutdown <ups>	only shutdown UPS <ups>\n");
	exit(1);
}

void shutdown_driver(ups_t *ups)
{
	int	ret;
	char	*argv[7], dfn[SMALLBUF];

	/* parent spins until the child exits */
	execwait = 1;

	ret = fork();

	if (ret < 0)
		fatal("fork");

	/* parent */
	if (ret != 0) {

		/* wait for child to return */
		signal(SIGCHLD, sigchld);

		/* give us a way out if the child gets stuck */
		signal(SIGALRM, sigalrm);

		if (ups->maxstartdelay != -1)
			alarm(ups->maxstartdelay);
		else
			alarm(maxstartdelay);

		while (execwait)
			usleep(250000);

		alarm(0);
		signal(SIGALRM, SIG_IGN);

		return;
	}

	/* child */

	snprintf(dfn, sizeof(dfn), "%s/%s", DRVPATH, ups->driver);

	if (verbose)
		printf("exec: %s -a %s -d 0 -k\n", dfn, ups->upsname);

	argv[0] = dfn;
	argv[1] = "-a";
	argv[2] = xstrdup(ups->upsname);
	argv[3] = "-d";
	argv[4] = "0";
	argv[5] = "-k";
	argv[6] = NULL;

	if (testmode)
		exit(0);

	ret = execve(dfn, argv, NULL);

	/* should not be reached */
	fatal("execve");
}

void shutdown_one_driver(char *upsname)
{
	ups_t	*tmp = upstable;

	if (!tmp)
		fatalx("Error: no UPS definitions found in ups.conf!\n");

	while (tmp) {
		if (!strcmp(tmp->upsname, upsname)) {
			shutdown_driver(tmp);
			return;
		}

		tmp = tmp->next;
	}

	fatalx("UPS %s not found in ups.conf", upsname);
}

/* walk UPS table and shut down all UPSes according to sdorder */
void shutdown_all_drivers(void)
{
	ups_t	*tmp;
	int	i;

	if (!upstable)
		fatalx("Error: no UPS definitions found in ups.conf");

	for (i = 0; i <= maxsdorder; i++) {
		tmp = upstable;

		while (tmp) {
			if (tmp->sdorder == i)
				shutdown_driver(tmp);
			
			tmp = tmp->next;
		}
	}

	exit(0);
}

int main(int argc, char **argv)
{
	int	i;
	char	*prog;

	printf("Network UPS Tools - UPS driver controller %s\n",
		UPS_VERSION);

	prog = argv[0];
	while ((i = getopt(argc, argv, "+htv")) != EOF) {
		switch(i) {
			case 'v':
				verbose++;
				break;

			case 't':
				testmode = 1;

				/* force verbose mode while testing */
				if (verbose == 0)
					verbose = 1;

				break;

			case 'h':
			default:
				help(prog);
				break;
		}
	}

	argc -= optind;
        argv += optind;

	if (argc < 1)
		help(prog);

	if (testmode)
		printf("*** Testing mode: not calling exec/kill\n");

	if (!strcmp(argv[0], "start")) {
		read_upsconf(1);

		if (argc == 1)
			start_all_drivers();
		else
			start_one_driver(argv[1]);

		exit(0);
	}

	if (!strcmp(argv[0], "stop")) {
		read_upsconf(1);

		if (argc == 1)
			stop_all_drivers();
		else
			stop_one_driver(argv[1]);

		exit(0);
	}

	if (!strcmp(argv[0], "status")) {
		read_upsconf(1);

		if (argc == 1)
			check_all_drivers();
		else
			check_one_driver(argv[1]);
	}

	if (!strcmp(argv[0], "shutdown")) {
		read_upsconf(1);

		if (argc == 1)
			shutdown_all_drivers();
		else
			shutdown_one_driver(argv[1]);

		exit(0);
	}

	fatalx("Error: unrecognized command [%s]\n", argv[0]);

	/* NOTREACHED */
	return 0;
}
