/*  snmp.c - monitor SNMP UPS (RFC 1628 compliant) for NUT
 *
 *  Based on NetSNMP API (Simple Network Management Protocol V1-2)
 *  
 *  Copyright (C) 2002 Arnaud Quette <arnaud.quette@free.fr>
 *  some parts are Copyright (C) :
 *                Russell Kroll <rkroll@exploits.org>
 *                Hans Ekkehard Plesser <hans.plesser@itf.nlh.no>
 *
 *  Sponsored by MGE UPS SYSTEMS <http://www.mgeups.com>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

/* NUT includes and global data */
#include "main.h"
#include "snmp-ups.h"

/* UCD SNMP includes and global data */
#include <arpa/inet.h> /* for ucd-snmp include bug */
#include <ucd-snmp/ucd-snmp-config.h>
#include <ucd-snmp/ucd-snmp-includes.h>
#include <ucd-snmp/system.h>

struct snmp_session session, *ss;
struct snmp_pdu *pdu;
struct snmp_pdu *response;

oid anOID[MAX_OID_LEN];
size_t anOID_len = MAX_OID_LEN;
struct variable_list *vars;
int status;
int count=1;

/* functions definition */
void ups_snmpwalk(int startindex, int mode);

/* function implementation */
void upsdrv_initinfo(void)
{
   upsdebugx(1, "SNMP UPS driver : entering upsdrv_initinfo()\n");

  /* initialize all other INFO_ fields from list */
  ups_snmpwalk(0, 0);

  /* upsh.instcmd = instcmd; */
}

void upsdrv_updateinfo(void)
{
  upsdebugx(1,"SNMP UPS driver : entering upsdrv_updateinfo()\n");
  
  /* update all dynamic info fields */
  ups_snmpwalk(3, 1);

  writeinfo();	
}

void upsdrv_shutdown(void)
{
  /*
   * Clean up:
   *  1) free the response.
   *  2) close the session.
   */
  if (response)
    snmp_free_pdu(response);

  snmp_close(ss);
  SOCK_CLEANUP;
 
  /* replace with a proper shutdown function */
  fatalx("shutdown not supported");

}

void instcmd (int auxcmd, int dlen, char *data)
{
  switch (auxcmd) {
    /* case CMD_BTEST0:	// stop battery test
    *	upssend("???");
    *	break;
    * case CMD_BTEST1:	// start battery test
      *	upssend("???");
    *	break;
    */
  default:
    upslogx(LOG_INFO, "instcmd: unknown type 0x%04x\n", auxcmd);
  }
}

void upsdrv_help(void)
{
  upsdebugx(1, "entering upsdrv_help");
}

/* list flags and values that you want to receive via -x */
void upsdrv_makevartable(void)
{
  upsdebugx(1, "entering upsdrv_makevartable()");

  addvar(VAR_VALUE, "community", "Set community name (default=public)");
  addvar(VAR_VALUE, "snmp_version", "Set SNMP version (default=v1, allowed v2c)");
}

void upsdrv_banner(void)
{
  printf("Network UPS Tools - SNMP UPS driver 0.01 (%s)\n\n", UPS_VERSION);
  printf("Warning: This is an experimental driver.\n");
  printf("Some features may not function correctly.\n\n");
}

void upsdrv_initups(void)
{
  char *community = NULL; 
  char *snmp_version = NULL;
  char *host_name = (char *)malloc(strlen(device_path) + 1);

  /* Initialize the SNMP library */
  init_snmp("nut-snmp");
  
  upsdebugx(1, "SNMP UPS driver : entering upsdrv_initups()\n");

  /* Initialize session */
  snmp_sess_init( &session );     /* set up defaults */
  strcpy(host_name, device_path);
  session.peername = host_name;   /* hostname        */

  /* set the SNMP version number */
  snmp_version = getval("snmp_version");
  if(snmp_version == NULL)
    session.version = SNMP_VERSION_1;
  else {
    if(!strncmp(snmp_version, "v2c", 3))
      session.version = SNMP_VERSION_2c;
    else
      session.version = SNMP_VERSION_1;
  }
  upsdebugx(1, "SNMP UPS driver : snmp_version is set to %s\n", 
	    getval("snmp_version"));
   
  /* set the SNMP community name (for authentication) */
  community = getval("community");
  if (community != NULL)
    session.community = community;
  else
    session.community = "public";

  session.community_len = strlen(session.community);
  
  upsdebugx(1, "SNMP UPS driver : community is set to %s\n", session.community);


/*    snmp_synch_setup(&session); */

  /* the upsh handlers can't be done here, as they get initialized
   * shortly after upsdrv_initups returns to main.
   */

  /*
   * Open the session
   */
  SOCK_STARTUP;
  ss = snmp_open(&session);                     /* establish the session */
  
  if (!ss) {
    snmp_perror("ack");
    snmp_log(LOG_ERR, "SNMP UPS driver : something horrible happened!!!\n");
    exit(2);
  }

  /* Get upsMIB .1.3.6.1.2.1.33.1.1.2 (UPS Model) 
   * node to see if there's an UPS SNMP part.    */
  pdu = snmp_pdu_create(SNMP_MSG_GET);
  read_objid(OID_MODEL_NAME, anOID, &anOID_len);  
  snmp_add_null_var(pdu, anOID, anOID_len);

  /* Send the Request out */
  status = snmp_synch_response(ss, pdu, &response);

  /* Process the response */
  if ((status == STAT_SUCCESS) && (response->errstat == SNMP_ERR_NOERROR))
    printf("SNMP UPS driver : Detected %s on host %s\n", 
	   response->variables->val.string, device_path);
  else
    fatalx("SNMP UPS driver : UPS MIB (%s) wasn't found on %s.\n", 
	   OID_UPS_MIB, device_path);   

  /* Clean up: free the response */
  if (response)
    snmp_free_pdu(response);

}

/* tell main how many items you need */
int upsdrv_infomax(void)
{
  return SNMP_INFOMAX;
}

/* specific processing for INFO_STATUS */
void read_status(char *OID, char *buf)
{
  int status_value;
  char *cur_status = getdata (INFO_STATUS);

  status_value = atoi(buf);

  if (!strcmp(OID, OID_POWER_STATUS)) {
    switch (status_value) {
    case PWR_OTHER : /*  INFO_STATUS/? */
      break;
    case PWR_NONE : /* none -> INFO_STATUS/OFF */
      sprintf(buf, "OFF ");
      break;
    case PWR_NORMAL : /* normal -> INFO_STATUS/OL */
      sprintf(buf, "OL ");
      break;
    case PWR_BYPASS : /* bypass -> INFO_STATUS/? */
      break;
    case PWR_BATTERY : /* battery -> INFO_STATUS/OB */
      sprintf(buf, "OB ");
      break;
    case PWR_BOOSTER : /* booster -> INFO_STATUS/BOOST */
      sprintf(buf, "BOOST ");
      break;
    case PWR_REDUCER : /* reducer -> INFO_STATUS/TRIM */
      sprintf(buf, "TRIM ");
      break;
    }
  } else {
    switch (status_value) {
    case BATT_UNKNOWN : /* unknown -> INFO_STATUS/? */
      break;
    case BATT_NORMAL : /* batteryNormal -> INFO_STATUS/? */
      sprintf(buf, "%s", cur_status);
      break;
    case BATT_LOW : /* batteryLow -> INFO_STATUS/LB */
      sprintf(buf, "%s%s", cur_status, "LB ");
      break;
    case BATT_DEPLETED : /* batteryDepleted -> INFO_STATUS/? */
      break;
    }
  }
}

/* walk through available OIDs and set INFO_* */
/* starting in list from startindex (0 when init, 2 when update */
/* mode : 0 = addinfo(), 1 = setinfo() */ 
void ups_snmpwalk(int startindex, int mode) {

  int value ;
  char *sp = NULL;
  snmp_info_item *item;

  /* loop on data matrix */
  for ( item = snmp_info ; item->type != INFO_UNUSED ; item++ ) {
    if ( item->ok ) {
      
      /* Create and send request for this item */
      pdu = snmp_pdu_create(SNMP_MSG_GET);
      read_objid(item->cmd, anOID, &anOID_len);  
      snmp_add_null_var(pdu, anOID, anOID_len);
      status = snmp_synch_response(ss, pdu, &response);
      
      /* Process the response */
      if ((status == STAT_SUCCESS) && (response->errstat == SNMP_ERR_NOERROR)) {

	if (response->variables->type == ASN_OCTET_STR) {
          sp = (char *)malloc(1 + response->variables->val_len);
          memcpy(sp, response->variables->val.string, response->variables->val_len);
          sp[response->variables->val_len] = '\0';

/*  	  sprint_variable (buf, response->variables->name,  */
/*  			   response->variables->name_length,  */
/*  			   response->variables); */
/*  	  upsdebugx(1, "SNMP UPS driver : (ASN_OCTET_STR) %s\n", buf); */
        }
        else if (response->variables->type == ASN_INTEGER) {
          sp = (char *)malloc(16);
	  value = *response->variables->val.integer * item->length;
	  sprintf(sp, "%d", value);

/*  	  sprint_variable (buf, response->variables->name,  */
/*  			   response->variables->name_length,  */
/*  			   response->variables); */
/*  	  upsdebugx(1, "SNMP UPS driver : (ASN_INTEGER) %s\n", buf); */
	}
/*  	else */
/*  	  upsdebugx(1, "SNMP UPS driver : (ASN_? (%d))\n",  */
/*  		    response->variables->type); */

	/* specific processing for INFO_STATUS*/
	if (item->type == INFO_STATUS) {
	  read_status(item->cmd, sp);
	}

	/* add or update INFO_ field */
	if (getdata (item->type) != NULL)
	  setinfo(item->type, sp);
	else
	  addinfo(item->type, sp, item->flags, 
		  response->variables->val_len);

	free(sp);
      } 
      else { /* an error occured */
	if (mode == 0) { /* this works only for MFR and MODEL */
	  if ( item->dfl != "" )
	    addinfo(item->type, item->dfl, item->flags, 
		    response->variables->val_len);
	} else { 
	  item->ok = FALSE;
	  upslogx(LOG_ERR, "SNMP UPS driver : missing data");
	}
      }

      /* Clean up: free the response */
      if (response)
	snmp_free_pdu(response);
      
    } /* if ok */
  }
}
