/* sms.c - driver for SMS UPS hardware

   Copyright (C) 2001  Marcio Gomes  <tecnica@microlink.com.br>

   based on fentonups.c:

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   2001/05/17 - Version 0.1 - Initial release
   2001/06/01 - Version 0.2 - Add Battery Informations in driver
   2001/06/04 - Version 0.3 - Updated Battery Volts range, to reflect a correct
                              percent ( % )
   2002/12/02 - Version 0.4 - Update driver to new-model, based on Fentonups 
                              driver version 0.90  

   Microlink ISP/Pop-Rio contributed with MANAGER III 1300, MANAGER III 650 UPS
   for my tests ( http://www.microlink.com.br and http://www.pop-rio.com.br )

*/

#include "main.h"
#include "sms.h"

#define ENDCHAR 13	/* replies end with CR */

static	int	cap_upstemp = 0;
static	float	lowvolt = 0, voltrange;
static	int	lownorm, highnorm, poll_failures = 0;

void guessmodel(const char *raw)
{
	char	mch, mstr[256];

	mch = raw[18];

/* MPG  
        printf ("0         1         2        3         \n");
        printf ("012345678901234567890123567890123456789\n");
        printf ("%s\n", raw);
  MPG  END */

	strlcpy(mstr, &raw[18], sizeof(mstr));
	mstr[10] = '\0';	/* 10 chars max, per the protocol */

	/* trim whitespace */
	rtrim(mstr, ' ');

     addinfo (INFO_MODEL, "", 0, 0);
     setinfo(INFO_MODEL, "SMS XX  %s", mstr);
     cap_upstemp = 1;

}

void getbaseinfo(void)
{
	char	temp[256], model[32], *raw;
	int	modelnum, i;

	/* dummy read attempt to sync - throw it out */
	upssend("I\r");
	upsrecv(temp, sizeof(temp), ENDCHAR, "");

	/* now retrieve information and parse */
	upssend("I\r");
	upsrecv(temp, sizeof(temp), ENDCHAR, "");
	raw = xstrdup(temp);

	if (temp[0] != '#')
		fatalx("Bad UPS info start character [%s]", temp);

	temp[11] = 0;
	temp[27] = 0;

	/* manufacturer */
	rtrim(&temp[1], ' ');
	addinfo(INFO_MFR, &temp[1], 0, 0);

	/* L660A = PowerPal (L) @ 660 VA, American (A) version (115V) */
        /*  SMS LTDA         1300 VA   VER 1.0 */

	/* grab full model string */
	rtrim(&temp[18], ' ');
	snprintf(model, sizeof(model), "%s", &temp[18]);
/*        printf("Model ->%s<-\n",model);  */
	modelnum = -1;

	/* figure out official model name and voltage info from table */
	for (i = 0; modeltab[i].mtext != NULL; i++) {
		if (!strcmp(modeltab[i].mtext, model)) {
			modelnum = i;
			lowvolt = modeltab[i].lowvolt;
			voltrange = modeltab[i].voltrange;
			cap_upstemp = modeltab[i].has_temp;
			break;
		}
	}

	/* table lookup fails -> guess */
	if (modelnum == -1)
		guessmodel (raw);
	else {
		setinfo(INFO_MODEL, "%s", modeltab[modelnum].desc);
		snprintf(temp, sizeof(temp), "%i", modeltab[modelnum].lowxfer);
		addinfo(INFO_LOWXFER, temp, 0, 0);

		snprintf(temp, sizeof(temp), "%i", modeltab[modelnum].highxfer);
		addinfo(INFO_HIGHXFER, temp, 0, 0);

		lownorm = modeltab[modelnum].lownorm;
		highnorm = modeltab[modelnum].highnorm;
	}

	if (cap_upstemp == 1)
		addinfo(INFO_UPSTEMP, "", 0, 0);

	/* now add instant command support info */
        addinfo(INFO_INSTCMD, "", 0, CMD_BTEST0);
        addinfo(INFO_INSTCMD, "", 0, CMD_BTEST1);

	printf("Detected %s on %s\n", getdata(INFO_MODEL), device_path);
	free(raw);

	/* paranoia - cancel any shutdown that might already be running */
	upssend("C\r");
}

void instcmd(int auxcmd, int dlen, char *data)
{
	/* TODO: reply to upsd? */

	switch (auxcmd) {
		case CMD_BTEST0:	/* stop battery test */
			upssend("CT\r");
			break;
		case CMD_BTEST1:	/* start battery test */
			upssend("T\r");
			break;
		default:
			upslogx(LOG_INFO, "instcmd: unknown type 0x%04x", auxcmd);
	}
}

void upsdrv_initinfo(void)
{
	addinfo(INFO_MODEL, "", 0, 0);
	addinfo(INFO_OUTVOLT, "", 0, 0);
	addinfo(INFO_UTILITY, "", 0, 0);
	addinfo(INFO_BATTVOLT, "", 0, 0);
	addinfo(INFO_BATTPCT, "", 0, 0);
	addinfo(INFO_STATUS, "", 0, 0);
	addinfo(INFO_ACFREQ, "", 0, 0);
	addinfo(INFO_LOADPCT, "", 0, 0);

	getbaseinfo();

	upsh.instcmd = instcmd;
}

void pollfail(char *why)
{
	poll_failures++;

	/* don't spew into the syslog forever */
	if (poll_failures < 3)
		upslogx(LOG_ERR, why);

	return;
}

void upsdrv_updateinfo(void)
{
	char	temp[256], utility[16], loadpct[16], acfreq[16], battvolt[16],
		upstemp[16], stat[16], outvolt[16];
	int	util, ret;
	double	bvoltp;

	upssend("Q1\r");

	ret = upsrecv (temp, sizeof(temp), ENDCHAR, "");

	/* sanity checks for poll data */
	if (strlen(temp) < 46) {
		pollfail("Poll failed: short read from UPS");
		return;
	}

	if (strlen(temp) > 46) {
		pollfail("Poll failed: oversized read from UPS");
		return;
	}

	if (temp[0] != '(') {
		pollfail("Poll failed: invalid start character");
		return;
	}

	if (poll_failures > 0)
		upslogx(LOG_NOTICE, "UPS poll succeeded");

	poll_failures = 0;		

	/* (MMM.M NNN.N PPP.P QQQ RR.R S.SS TT.T  b7b6b5b4b3b2b1b0<cr>
	 *
	 * MMM.M : input voltage (utility)
	 * NNN.N : fault voltage (ignored)
	 * PPP.P : output voltage
	 */

	sscanf(temp, "%*c%s %*s %s %s %s %s %s %s", utility, outvolt, loadpct,
		acfreq, battvolt, upstemp, stat);

	setinfo(INFO_OUTVOLT, "%s", outvolt);
	setinfo(INFO_UTILITY, "%s", utility);
	setinfo(INFO_BATTVOLT, "%s", battvolt);

	bvoltp = ((atof(battvolt) - lowvolt) / voltrange) * 100.0;

	if (bvoltp > 100.0)
		bvoltp = 100.0;

	setinfo(INFO_BATTPCT, "%02.1f", bvoltp);

	status_init();

	util = atoi(utility);

	if (stat[0] == '0') {
		status_set("OL");		/* on line */

		/* only allow these when OL since they're bogus when OB */
		if (stat[2] == '1') {		/* boost or trim in effect */
			if (util < lownorm)
				status_set("BOOST");

			if (util > highnorm)
				status_set("TRIM");
		}

	} else {
		status_set("OB");		/* on battery */
	}

	if (stat[1] == '1')
		status_set("LB");		/* low battery */

	status_commit();

	if (cap_upstemp == 1)
		setinfo(INFO_UPSTEMP, "%s", upstemp);

	setinfo(INFO_ACFREQ, "%s", acfreq);
	setinfo(INFO_LOADPCT, "%s", loadpct);

	writeinfo();
}

/* power down the attached load immediately */
void upsdrv_shutdown(void)
{
	char	temp[256], stat[32];

	/* basic idea: find out line status and send appropriate command */

	upssend("Q1\r");
	upsrecv (temp, sizeof(temp), ENDCHAR, "");
	sscanf (temp, "%*s %*s %*s %*s %*s %*s %*s %s", stat);

	/* on battery: send S01<cr>, ups will return by itself on utility */
	/* on line: send S01R0003<cr>, ups will cycle and return soon */

	upssend("S01");

	if (stat[0] == '0') {			/* on line */
		printf("On line, sending shutdown+return command...\n");
		upssend("R0003");
	} else
		printf("On battery, sending normal shutdown command...\n");

	upssendchar(13);	/* end sequence */
}

void upsdrv_help(void)
{
}

void upsdrv_makevartable(void)
{
}

void upsdrv_banner(void)
{
	printf("Network UPS Tools - SMS UPS driver 0.40 (%s)\n", UPS_VERSION);
        printf("by Marcio Gomes at Microlink - tecnica@microlink.com.br\n");
}

void upsdrv_initups(void)
{
	open_serial(device_path, B2400);
}

/* tell main how many entries we need */
int upsdrv_infomax(void)
{
	return 16;
}

