/* engetron.c - model specific routines for Engetron Jr model

   Copyright (C) 2000  Adriano Roberto de Lima <arlima@bigfoot.com>
                       Marcio Argollo F. de Menezes <marcio@if.uff.br>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <math.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

#define INFOMAX  16
#define UPSDELAY 5
#define MAXTRIES 10

int	upsfd;
int	shmok = 1;
char	statefn[512], *modelname, *va;
itype	*info;
extern	int	flag_timeoutfailure;
float	lowvolt = 0, highvolt = 0, voltrange = 0, 
  vn,   /* nominal tension - model dependent */ 
  il;   /* maximum load current - model dependent */
int	infomax = 16;



void 
usage (char *prog)
{
	printf ("usage: %s [-h] [-k] <device>\n", prog);
	printf ("Example: %s /dev/ttyS1\n", prog);
	exit (1);
}

void 
help (char *prog)
{
	printf ("usage: %s [-h] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-h       - display this help\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
	exit (1);
}

void 
initinfo (void)
{
  int	i;
  
  info = create_info (INFOMAX, shmok);
  
  /* clear out everything first */
  for (i = 0; i < INFOMAX; i++)
    info[i].type = INFO_UNUSED;
  
  /* number of variables - max possible */
  info[0].type = INFO_MEMBERS;
  snprintf (info[0].value, sizeof(info[0].value), "%i", INFOMAX);
  
  /* now set up room for all future variables that are supported */
  info[1].type = INFO_MFR;
  info[2].type = INFO_MODEL;
  info[3].type = INFO_UTILITY;
  info[4].type = INFO_BATTPCT;
  info[5].type = INFO_STATUS;
  info[6].type = INFO_ACFREQ;
  info[7].type = INFO_LOADPCT;
  info[8].type = INFO_UPSTEMP;
}


void 
setup_serial(void)
{	
  struct termios tio;
  
  tio.c_iflag = IXON | IXOFF;
  tio.c_oflag = 0;
  tio.c_cflag = (B9600 | CS8 | CREAD | HUPCL | CLOCAL);
  tio.c_lflag = 0;
  tio.c_cc[VMIN] = 1;
  tio.c_cc[VTIME] = 0;
  
  if (tcsetattr (upsfd, TCSANOW, &tio) == -1) {
    perror ("tcsetattr");
    exit (1);
  }
}

char 
readchar (void)
{
  char	c;
  int	ret;
  
  ret = read (upsfd, &c, 1);
  
  if (ret > 0)
    return (c & 0x7f);
  else
    return 0;
}

int 
xsend (char *str)
{
  int	ret;
  
  while (*str) {
    ret = write (upsfd, str++, 1);
    if (ret == -1)
      return (-1);
  }
  
  return 0;
}

int 
xrecv (char *buf, int max_count)
{
  char	ch, *p;
  int counter;

  p = buf;
  *p = '\0';
  counter=0;

  while ((ch = readchar ())) {
    if (ch == '\n') {
      counter++;
      if(counter==max_count){
	*p = '\0';
	return 0;
      }
    }
    
    *p++ = ch;
  }
  
  *p = '\0';
  return 0;
}

void 
ups_ident(void)
{
  char	
    buf[512], 
    *ptr, 
    *com;
  int	
    tries = 0;

  tcflush(upsfd,TCIOFLUSH);	
  printf ("Identifying UPS: ");
  fflush (stdout);
  
  for (;;) {
    tries++;
    if (tries > MAXTRIES) {
      printf ("\nFailed - giving up...\n");
      exit (1);
    }
    xsend ("Identidade\r");
    printf (".");
    fflush (stdout);
    sleep (UPSDELAY);
    printf (".");
    fflush (stdout);
    xrecv (buf,9);
    printf (".");
    fflush (stdout);
    if ((buf[0] != '\0') && (buf[0] != 'I'))
      break;
  }
  tcflush(upsfd,TCIOFLUSH);	
  
  printf (" done\n");
  
  modelname = va = NULL;
  
  ptr = strdup(buf);
  com = strchr (ptr, 'N');
  ptr = com;

  com = strchr (ptr, 'V') - 1;
  if (com)
    *com = '\0';
  
  modelname = strdup (ptr);
  
  ptr=com+1;
  
  com = strchr (ptr, '\n');
  if (com)
    *com = '\0';

  va = strdup (ptr);
  
  snprintf (info[1].value, sizeof(info[1].value), "ENGETRON");
  snprintf (info[2].value, sizeof(info[2].value), "%s", modelname);
  
  ptr=com+1;
  com = strchr (ptr, ':');
  ptr = com + 1;
  com = strchr (ptr, ':');
  ptr = com + 1;
  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf (ptr, "%f %*s %f", &lowvolt, &highvolt);
  voltrange = highvolt - lowvolt;
}

void 
ups_update (void)
{
  char	
    *ptr, 
    *com,
    onoff,
    *num,
    vr[128],
    buf[300];
  
  int
    mask,
    onbattery,
    status;
  
  float
    bvoltp,
    info_eng[24];
    
  int i;

  vn = 12.0;
  il = 2.8;

  xsend ("Status\r");     
  sleep (UPSDELAY);
  xrecv (buf,7);
  
  if ((strlen(buf) < 271) || (buf[0] != 'S'))
    return;
  
  ptr = strdup(buf);
  com = strchr(ptr, ':');
  ptr = com + 1;
  
  com = strchr(ptr, 'V') - 1;
  if (com)
    *com = '\0';
  
  num = strdup(ptr);
  
  ptr=com+1;
  
  sscanf (ptr, "%*s %s", vr);

  for (i = 1; i < 16; i++) {
    com = strchr (ptr, ':');
    ptr = com + 1;
    sscanf(ptr, "%f", &info_eng[i]);
  }
  
  com = strchr (ptr, ':');
  ptr = com + 1;
  if(info_eng[8]!=0.0){
    sscanf(ptr, "%f %*s %f", &info_eng[16], &info_eng[17]);
  }
  
  com = strchr (ptr, ':');
  ptr = com + 1;
  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf(ptr, "%f", &info_eng[18]);

  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf(ptr, "%x", &status);

  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf(ptr, "%f %f %f", &info_eng[19], &info_eng[20], &info_eng[21]);

  com = strchr (ptr, ':');
  ptr = com + 1;
  sscanf(ptr, "%f %c", &info_eng[22], &onoff);


  snprintf (info[3].value, sizeof(info[3].value), "%3.1f",
	    info_eng[2]);

 
  
  strcpy (info[5].value, "");
  
  mask = 1;
  onbattery=0;
  for(i=0;i<16;i++){
    if((status&mask)!=0){
      switch (i) {
      case 0: strcat (info[5].value, "RB ");
	break;
      case 2: strcat (info[5].value, "OVER ");
	break;
      case 10: strcat (info[5].value, "LB ");
	break;
      case 11: strcat (info[5].value, "TRIM ");
	break;
      case 12:
	onbattery=1;
	strcat (info[5].value, "OB ");
	break;
      default:
	break;
      }
    }
    mask <<= 1;
  }
  
  if(onoff=='D')
    strcat (info[5].value, "OFF ");
  else{
    if(onbattery==0)
      strcat (info[5].value, "OL ");
  }
  
  /* lose trailing space if present */
  if (info[5].value[strlen(info[5].value)-1] == ' ')
    info[5].value[strlen(info[5].value)-1] = 0;
  

  /*********************** modificar tudo ******************/
  if(onbattery){
    bvoltp = ((6060 / vn) * info_eng[11])/10 - 567; 
  } 
  else {

    /*    bvoltp = (80.0 * (info_eng[11] - vn) / (1.15 * vn) + 20.0 * (il - info_eng[12]) / (0.6 * il));*/

    bvoltp = (87.0 * info_eng[11])  / vn - (2.0 * info_eng[12]) / il;
  }
  if (bvoltp < 0.0) {
    bvoltp = 0;
  }
  else {
    if (bvoltp > 100.0) {
      bvoltp = 100.0;
    }
  }
  
  snprintf (info[4].value, sizeof(info[4].value), "%3.1f", bvoltp); 

  /*********************** ate aqui ******************/
 
  snprintf (info[6].value, sizeof(info[6].value), "%2.2f", info_eng[5]);
  snprintf (info[7].value, sizeof(info[7].value), "%3.1f", 100.0*info_eng[8]/info_eng[1]);
  snprintf (info[8].value, sizeof(info[8].value), "%3.1f", info_eng[18]);
  
  writeinfo(info);
}



int 
main (int argc, char **argv)
{
  char	
    *prog, 
    *portname;
  int
    i;

  printf ("Network UPS Tools - Engetron Jr. UPS driver 0.10 (%s)\n", UPS_VERSION);
  openlog ("engetron", LOG_PID, LOG_FACILITY);

  prog = argv[0];
  while ((i = getopt(argc, argv, "+hk:")) != EOF) {
    switch (i) {
    case 'h':
      help(prog);
      break;
    default:
      usage(prog);
      break;
    }
  }
  
  argc -= optind;
  argv += optind;

  if (argc < 1) {
	printf ("Error: no device specified!\n");
	usage (prog);
  }
  
  droproot();
  
  portname = NULL;
  for (i = strlen(argv[0]); i >= 0; i--)
    if (argv[0][i] == '/') {
      portname = &argv[0][i+1];
      break;
    }
  
  if (portname == NULL) {
    printf ("Unable to abbreviate %s\n", argv[0]);
    exit (1);
  }

  snprintf (statefn, sizeof(statefn), "%s/engetron-%s", STATEPATH,
	    portname);

  upsfd = open(argv[0], O_RDWR | O_NDELAY);
  if (upsfd == -1) {
    perror ("open");
    exit (1);
  }
  
  setup_serial();
  initinfo();
  ups_ident();
  createmsgq();	/* try to create IPC message queue */
  
  printf ("Detected %s %s on %s\n", info[1].value, info[2].value, argv[0]);
  
  background();
  
  for (;;)
    ups_update();

  return 0;
}
