/* apcsmart.c - model specific routines for APC smart protocol units

   Copyright (C) 1999  Russell Kroll <rkroll@exploits.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.              
*/

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/termios.h>

#include "config.h"
#include "proto.h"
#include "apcsmart.h"
#include "shared.h"
#include "version.h"
#include "upscommon.h"
#include "common.h"

	int	shmok = 1;
	char	statefn[256];
	itype	*info;
extern	int	flag_timeoutfailure;
	int	cap_measureupsii = 0;
	int	infomax = 128;
	int	upslevel = 0;

	int	sddelay = 60;	/* wait 60 seconds for shutdown */
	int	shutdowncmdset = 0; /* The shutdown command set to use */
extern	struct	ups_handler	upsh;

	/* alternate cable handling support */
	char	*cable = NULL;
#define ALT_CABLE_1 "940-0095B"

extern	int	upsfd;

/* try to find AMBTEMP, CONTACTS and AMBHUMID */
void test_measureupsii(void)
{
	char	temp[256];

	upssendchar (REQ_AMBTEMP);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if (!strcmp(temp, "NA"))	/* not supported */
		return;

	/* we got something, so it must be supported */

	cap_measureupsii = 1;		/* log this fact */

	addinfo (INFO_AMBTEMP, temp, 0, 0);

	/* Now probe the contacts */

	upssendchar (REQ_CONTACTS);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	addinfo (INFO_CONTACTS, temp, 0, 0);

	/* now try for ambient humidity support */

	upssendchar (REQ_AMBHUMID);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if (!strcmp(temp, "NA"))	/* not supported */
		return;

	addinfo (INFO_AMBHUMID, temp, 0, 0);
}

void addcap (char cmdchar, char *value)
{
	switch (cmdchar) {
		case 'u': addenum (INFO_HIGHXFER, value);
			  setflag (INFO_HIGHXFER, FLAG_RW | FLAG_ENUM);
			  break;
		case 'l': addenum (INFO_LOWXFER, value);
			  setflag (INFO_LOWXFER, FLAG_RW | FLAG_ENUM);
			  break;
		case 'e': break;	/* TODO: return threshold	*/
		case 'o': break;	/* output voltages, unused now	*/
		case 's': addenum (INFO_LINESENS, value);
			  setflag (INFO_LINESENS, FLAG_RW | FLAG_ENUM);
			  break;
		case 'q': break;	/* TODO: low batt warning	*/
		case 'p': break;	/* TODO: shutdown grace delay	*/
		case 'k': break;	/* TODO: alarm delay		*/
		case 'r': addenum (INFO_WAKEDELAY, value);
			  setflag (INFO_WAKEDELAY, FLAG_RW | FLAG_ENUM);
			  break;
		case 'E': break;	/* TODO: selftest intervals	*/
		case 0x06: break;	/* TODO: find out what ^F is	*/
		case 0x0c: break;	/* TODO: front panel language	*/
		case 'w': break;	/* TODO: runtime conservation	*/
		default:
			printf ("Unrecognized cmdchar %c, value %s\n",
			        cmdchar, value);
	}		
}	

void do_capabilities (void)
{
	char	upsloc, temp[512], *ptr, cmd, loc, *entptr, etmp[16], *endtemp;
	int	nument, entlen, i, matrix, recvret;

	/* just in case ... */
	usleep (50000);

	/* get location byte for later comparisons */
	upssendchar ('V');
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	upsloc = temp[strlen(temp) - 1];

	/* add base entries for these variables */

	/* create them as normal variables for now - enum and rw flags    *
	 * get added later if the model actually supports such operations */

	addinfo (INFO_LOWXFER, "", 0, 0);
	addinfo (INFO_HIGHXFER, "", 0, 0);
	addinfo (INFO_WAKEDELAY, "", 0, 0);
	addinfo (INFO_LINESENS, "", 0, 0);
	installinfo (INFO_LOWXFER, REQ_LOWXFER, ENDCHAR, IGNCHARS);
	installinfo (INFO_HIGHXFER, REQ_HIGHXFER, ENDCHAR, IGNCHARS);
	installinfo (INFO_WAKEDELAY, REQ_WAKEDELAY, ENDCHAR, IGNCHARS);
	installinfo (INFO_LINESENS, REQ_LINESENS, ENDCHAR, IGNCHARS);

	/* battery change date */
	addinfo (INFO_BATTDATE, "", FLAG_RW | FLAG_STRING, 8);
	installinfo (INFO_BATTDATE, REQ_BATTDATE, ENDCHAR, IGNCHARS);

	/* get capability string */
	upssendchar (REQ_CAPABILITIES);	/* ^Z */
	strcpy (temp, "NA");
	recvret = upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if ((recvret == -1) || (!strcmp(temp, "NA"))) {
		/* Early Smart-UPS, not as smart as later ones */
		return;
	}

	/* recv always puts a \0 at the end, so this is safe */
	/* however it assumes a zero byte cannot be embedded */
	endtemp = &temp[0] + strlen(temp);

	if (temp[0] != '#') {
		printf ("Unrecognized capability start char %c\n", temp[0]);
		printf ("Please report this error [%s]\n", temp);
		return;
	}

	if (temp[1] == '#') {		/* Matrix-UPS */
		matrix = 1;
		ptr = &temp[0];
	}
	else {
		ptr = &temp[1];
		matrix = 0;
	}

	/* command char, location, # of entries, entry length */

	while (ptr[0] != '\0') {
		if (matrix)
			ptr += 2;	/* jump over repeating ## */

		/* check for idiocy */
		if (ptr >= endtemp) {
			printf ("Capability string has overflowed\n");
			printf ("Please report this error\n");
			syslog (LOG_ERR, "ERROR: capability overflow!\n");
			exit(1);
		}

		cmd = ptr[0];
		loc = ptr[1];
		nument = ptr[2] - 48;
		entlen = ptr[3] - 48;
		entptr = &ptr[4];

		for (i = 0; i < nument; i++) {
			snprintf (etmp, entlen + 1, "%s", entptr);
			if ((loc == upsloc) || (loc == '4'))
				addcap (cmd, etmp);
			entptr += entlen;
		}
		ptr = entptr;
	}
}

void do_status(void)
{
	char	temp[256], itemp[256];
	int	tval;

	/* parse the status */
	upssendchar (REQ_STATUS);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	/* APC SU600 models return this when the FPTEST is still active */
	if (!strncmp(temp, "BC", 2))
		return;			/* bogus, we don't want to parse it */

	tval = strtol (temp, 0, 16);

	strcpy (itemp, "");
	if (tval & 1)
		strcat (itemp, "CAL ");		/* calibration */
	if (tval & 2)
		strcat (itemp, "TRIM ");	/* SmartTrim */
	if (tval & 4)
		strcat (itemp, "BOOST ");	/* SmartBoost */
	if (tval & 8)
		strcat (itemp, "OL ");		/* on line */
	if (tval & 16)
		strcat (itemp, "OB ");		/* on battery */
	if (tval & 32)
		strcat (itemp, "OVER ");	/* overload */
	if (tval & 64)
		strcat (itemp, "LB ");		/* low battery */
	if (tval & 128)
		strcat (itemp, "RB ");		/* replace batt */

	if (!strcmp(temp, "00"))
		strcpy (itemp, "OFF");

	/* lose trailing space if present */
	if (itemp[strlen(itemp)-1] == ' ')
		itemp[strlen(itemp)-1] = 0;

	setinfo (INFO_STATUS, itemp);
}

void getbaseinfo (char *port)
{
	char	temp[256];
	int	silentfail = 0, oldmatrix = 0;
	int	recvret = 0;

	upslevel = 1;

	/* really old models ignore REQ_MODEL, so find them first */
	upssendchar (REQ_MODEL);
	strcpy (temp, "NA");

	/* disable timeout complaints temporarily */
	flag_timeoutfailure = -1;
	recvret = upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	flag_timeoutfailure = 0;

	if ((recvret == -1) || (!strcmp(temp, "NA"))) {
		/* force the model name */
		addinfo (INFO_MODEL, "Smart-UPS", 0, 0);
		silentfail = 1;
	}
	else
		addinfo (INFO_MODEL, temp, 0, 0);

	/* see if this might be an old Matrix-UPS instead */
	upssendchar ('/');
	flag_timeoutfailure = -1;
	strcpy (temp, "NA");
	recvret = upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	flag_timeoutfailure = 0;

	/* it returned a useful number, so it must be a Matrix-UPS */
	if ((recvret != -1) && (strcmp(temp, "NA")) != 0) {
		oldmatrix = 1;	/* doesn't do 'b' */
		setinfo (INFO_MODEL, "Matrix-UPS");
	}

	addinfo (INFO_SERIAL, "", 0, 0);
	installinfo (INFO_SERIAL, REQ_SERIAL, ENDCHAR, IGNCHARS);

	addinfo (INFO_STATUS, "", 0, 0);
	do_status();

	/* upslevel 1: Smart-UPS v/s, older Back-UPS Pros */
	upssendchar (REQ_UTILITY);
	strcpy (temp, "NA");
	recvret = upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	if ((!strcmp(temp, "NA")) || (recvret == -1)) {
		upslevel = 1;
		return;
	}

	/* upslevel 2: at least a modern Smart-UPS / Back-UPS Pro */
	upslevel = 2;

	/* add other things to monitor and prime them with values */

	addinfo (INFO_UTILITY, "", 0, 0);
	installinfo (INFO_UTILITY, REQ_UTILITY, ENDCHAR, IGNCHARS);

	addinfo (INFO_BATTPCT, "", 0, 0);
	installinfo (INFO_BATTPCT, REQ_BATTPCT, ENDCHAR, IGNCHARS);

	addinfo (INFO_ACFREQ, "", 0, 0);
	installinfo (INFO_ACFREQ, REQ_ACFREQ, ENDCHAR, IGNCHARS);

	addinfo (INFO_LOADPCT, "", 0, 0);
	installinfo (INFO_LOADPCT, REQ_LOADPCT, ENDCHAR, IGNCHARS);

	addinfo (INFO_BATTVOLT, "", 0, 0);
	installinfo (INFO_BATTVOLT, REQ_BATTVOLT, ENDCHAR, IGNCHARS);

	addinfo (INFO_OUTVOLT, "", 0, 0);
	installinfo (INFO_OUTVOLT, REQ_OUTVOLT, ENDCHAR, IGNCHARS);

	/* upslevel 3: at least a Smart-UPS (beyond the Back-UPS Pro) */

	/* test for UPSTEMP support */
	upssendchar (REQ_UPSTEMP);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	if (strcmp(temp, "NA") != 0) {
		upslevel = 3;
		addinfo (INFO_UPSTEMP, temp, 0, 0);
	}

	addinfo (INFO_UPSIDENT, "", FLAG_RW | FLAG_STRING, 8);
	installinfo (INFO_UPSIDENT, REQ_UPSIDENT, ENDCHAR, IGNCHARS);

	/* now add the instant commands */
	addinfo (INFO_INSTCMD, "", 0, CMD_FPTEST);
	addinfo (INFO_INSTCMD, "", 0, CMD_BTEST1);
	addinfo (INFO_INSTCMD, "", 0, CMD_OFF);
	addinfo (INFO_INSTCMD, "", 0, CMD_ON);
	addinfo (INFO_INSTCMD, "", 0, CMD_CAL0);
	addinfo (INFO_INSTCMD, "", 0, CMD_CAL1);

	do_capabilities();

	/* that's all for 'silent failure' units */
	if (silentfail)
		return;

	/* see if we have a Measure-UPS II connected */
	test_measureupsii();
}

/* normal idle loop - keep up with the current state of the UPS */
void updateinfo (void)
{
	char	temp[256];

	/* try to wake up a dead ups once in awhile */
	if (flag_timeoutfailure == 1) {
		upssendchar (GO_SMART);
		upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	}

	do_status();

	/* upslevel 1 only has a status readback */
	if (upslevel < 2) {
		writeinfo(info);
		return;
	}

	installinfo (INFO_UTILITY, REQ_UTILITY, ENDCHAR, IGNCHARS);
	installinfo (INFO_BATTPCT, REQ_BATTPCT, ENDCHAR, IGNCHARS);
	installinfo (INFO_ACFREQ, REQ_ACFREQ, ENDCHAR, IGNCHARS);
	installinfo (INFO_LOADPCT, REQ_LOADPCT, ENDCHAR, IGNCHARS);
	installinfo (INFO_BATTVOLT, REQ_BATTVOLT, ENDCHAR, IGNCHARS);
	installinfo (INFO_OUTVOLT, REQ_OUTVOLT, ENDCHAR, IGNCHARS);

	/* need at least 3 for ups temperature and Measure-UPS II */
	if (upslevel < 3) {
		writeinfo (info);
		return;
	}

	installinfo (INFO_UPSTEMP, REQ_UPSTEMP, ENDCHAR, IGNCHARS);

	if (cap_measureupsii) {
		installinfo (INFO_AMBHUMID, REQ_AMBHUMID, ENDCHAR, IGNCHARS);
		installinfo (INFO_AMBTEMP, REQ_AMBTEMP, ENDCHAR, IGNCHARS);
		installinfo (INFO_CONTACTS, REQ_CONTACTS, ENDCHAR, IGNCHARS);
	}

	writeinfo(info);
}

/* TODO: roll this into upscommon ala bestups */
void sendstring(char * string, int len, int delay)
{
	int	i;

	for (i=0; (i<len); i++) {
	        upssendchar(string[i]);
		usleep(delay);
	}
}

/* get the UPS talking to us in smart mode */
int smartmode (void)
{
	int	tries;
	char	temp[256];

	tries = 0;
	for (;;) {
		tries++;

		upssendchar (GO_SMART);
		upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if (!strcmp(temp, "SM"))
			return 1;	/* success */

		if (tries > 5)
			return 0;	/* failure */

		sleep (1);	/* wait before trying again */
	}

	return 0;	/* notreached */
}

/* 940-0095B support: set DTR, lower RTS */
void init_serial(void)
{
	int	dtr_bit = TIOCM_DTR;
	int	rts_bit = TIOCM_RTS;

	ioctl(upsfd, TIOCMBIS, &dtr_bit);
	ioctl(upsfd, TIOCMBIC, &rts_bit);
}

/* power down the attached load immediately */
void forceshutdown()
{
	char	temp[32];
	int	tval;

	syslog (LOG_INFO, "Initiating UPS shutdown\n");
	printf ("Initiating forced UPS shutdown!\n");

	/* check the line status */

	upssendchar (REQ_STATUS);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	tval = strtol (temp, 0, 16);

	switch (shutdowncmdset) {
		case 3:		/* shutdown with grace period */
			printf ("Sending delayed power off command to UPS\n");
			upssendchar (REQ_SHUTDOWN);
			usleep (1500000);
			upssendchar (REQ_SHUTDOWN);
			break;

		case 2:		/* instant shutdown */
			printf ("Sending power off command to UPS\n");
			upssendchar (REQ_OFF);
			usleep (1500000);
			upssendchar (REQ_OFF);
			break;

		case 1:

	/* Send a combined set of shutdown commands which can work better */
	/* if the UPS gets power during shutdown process */
	/* Specifically it sends both the soft shutdown 'S' */
	/* and the powerdown after grace period - '@000' commands */
			printf ("UPS - currently %s - sending shutdown/powerdown\n",
			       (tval & 8) ? "on-line" : "on battery");
			sendstring ("S@000", 4, 50000);
			break;

		default:

		/* @000 - shutdown after 'p' grace period             */
		/*      - returns after 000 minutes (i.e. right away) */

		/* S    - shutdown after 'p' grace period, only on battery */
		/*        returns after 'e' charge % plus 'r' seconds      */

			if (tval & 8) {                 /* on line */
				printf ("On line, sending shutdown+return command...\n");
				sendstring ("@000", 4, 50000);
			}
			else {
				printf ("On battery, sending normal shutdown command...\n");
				upssendchar ('S');
			}
	}
	if (sddelay > 0) {
		sleep (sddelay);
		printf ("Hmm, did the shutdown fail?  Oh well...\n");
		exit (1);
	}
	exit (0);
}

/* set a value in the hardware using the <cmdchar> '-' (repeat) approach */
void hw_set (char cmdchar, char *newval)
{
	char	temp[256], orig[256];
	int	tries;

	upssendchar (cmdchar);
	upsrecv (orig, sizeof(orig), ENDCHAR, IGNCHARS);

	/* don't change it if already set - save wear and tear on the eeprom */
	if (!strcmp(orig, newval))
		return;

	tries = 0;
	while (tries < 6) {
		upssendchar (NEXT_VAL);
		upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

		/* TODO: test for "OK", bail out on "NO", etc */
		upssendchar (cmdchar);
		upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

		if (!strcmp(temp, newval)) 		/* got it */
			return;

		if (!strcmp(temp, orig)) {		/* wrapped around */
			syslog (LOG_ERR, "setvar: variable %c wrapped!\n",
			        cmdchar);
			return;
		}

		tries++;
	}

	syslog (LOG_ERR, "setvar: gave up after 6 tries\n");
}

/* set string values in the UPS EEPROM (battery date, local identity) */
void setstring (char *data, char reqchar)
{
	char	temp[256];
	int	i;

	upssendchar (reqchar);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	upssendchar (NEXT_VAL);
	usleep (50000);

	for (i = 0; i < strlen(data); i++) {
		upssendchar (data[i]);
		usleep (50000);
	}

	/* pad to 8 chars with CRs */
	for (i = strlen(data); i < 8; i++) {
		upssendchar (13);
		usleep (50000);
	}

	usleep (50000);
	upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
	usleep (50000);

	/* TODO: check for "OK" or "NO" or similar */
}

void setvar (int auxcmd, int dlen, char *data)
{
	/* TODO: look this up from a table? */
	switch (auxcmd) {
		case INFO_WAKEDELAY:
			hw_set (REQ_WAKEDELAY, data);
			installinfo (INFO_WAKEDELAY, REQ_WAKEDELAY, ENDCHAR, IGNCHARS);
			break;
		case INFO_LOWXFER:
			hw_set (REQ_LOWXFER, data);
			installinfo (INFO_LOWXFER, REQ_LOWXFER, ENDCHAR, IGNCHARS);
			break;
		case INFO_HIGHXFER:
			hw_set (REQ_HIGHXFER, data);
			installinfo (INFO_HIGHXFER, REQ_HIGHXFER, ENDCHAR, IGNCHARS);
			break;
		case INFO_LINESENS:
			hw_set (REQ_LINESENS, data);
			installinfo (INFO_LINESENS, REQ_LINESENS, ENDCHAR, IGNCHARS);
			break;
		case INFO_UPSIDENT:
			setstring (data, REQ_UPSIDENT);
			installinfo (INFO_UPSIDENT, REQ_UPSIDENT, ENDCHAR, IGNCHARS);
			break;
		case INFO_BATTDATE:
			setstring (data, REQ_BATTDATE);
			installinfo (INFO_BATTDATE, REQ_BATTDATE, ENDCHAR, IGNCHARS);
			break;
		default:
			syslog (LOG_ERR, "setvar: unknown type 0x%04x\n",
			        auxcmd);
	}
}

/* check for calibration status and either start or stop */
void do_cal (int start)
{
	char	temp[256];
	int	tval, ret;

	upssendchar (REQ_STATUS);
	ret = upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);

	if (!ret)		/* didn't get status, bail out */
		return;

	tval = strtol (temp, 0, 16);

	if (tval & 1) {		/* calibration currently happening */
		if (start == 0)	{		/* stop requested */
			syslog (LOG_INFO, "Stopping runtime calibration");
			upssendchar (REQ_CAL);
			return;
		}

		/* requested start while calibration still running */
		syslog (LOG_INFO, "Runtime calibration already in progress");
		return;
	}

	/* calibration not happening */

	if (start == 0) {		/* stop requested */
		syslog (LOG_INFO, "Runtime calibration not occurring");
		return;
	}

	syslog (LOG_INFO, "Starting runtime calibration");
	upssendchar (REQ_CAL);
}

/* handle the CMD_OFF with some paranoia */
void do_off (void)
{
	static	time_t lastcmd = 0;
	time_t	now, elapsed;

#ifdef CONFIRM_DANGEROUS_COMMANDS
	time (&now);
	elapsed = now - lastcmd;

	/* reset the timer every call - this means if you call it too      *
	 * early, then you have to wait MINCMDTIME again before sending #2 */
	lastcmd = now;

	if ((elapsed < MINCMDTIME) || (elapsed > MAXCMDTIME)) {

		/* FUTURE: tell the user (via upsd) to try it again */
		/* msgreply (UPSMSG_REPAGAIN); */
		return;
	}
#endif

	syslog (LOG_INFO, "Sending powerdown command to UPS\n");
	upssendchar (REQ_OFF);
	usleep (1500000);
	upssendchar (REQ_OFF);
}

void instcmd (int auxcmd, int dlen, char *data)
{
	char	temp[256];

	/* TODO: reply to upsd? */

	switch (auxcmd) {
		case CMD_FPTEST:	/* front panel test */
			upssendchar (REQ_FPTEST);
			upsrecv (temp, sizeof(temp), ENDCHAR, IGNCHARS);
			break;
		case CMD_BTEST1:	/* start battery test */
			upssendchar (REQ_BTEST);
			break;
		case CMD_OFF:		/* power off the load */
			do_off();
			break;
		case CMD_ON:		/* power on the load */
			upssendchar (REQ_ON);
			usleep (500000);
			upssendchar (REQ_ON);
			break;
		case CMD_CAL0:		/* stop runtime calibration */
			do_cal(0);
			break;
		case CMD_CAL1:		/* start runtime calibration */
			do_cal(1);
			break;
		default:
			syslog (LOG_INFO, "instcmd: unknown type 0x%04x\n",
			        auxcmd);
	}
}

/* install pointers to functions for msg handlers called from msgparse */
void setuphandlers(void)
{
	upsh.setvar = setvar;
	upsh.instcmd = instcmd;
}

void usage (char *prog)
{
	printf ("usage: %s [-h] [-c <cable>] [-d <num>] [-s <num>] [-k] <device>\n", prog);
	printf ("Example: %s /dev/ttyS0\n", prog);
	exit (1);
}

void help (char *prog)
{
	printf ("usage: %s [-h] [-c <cable>] [-d <num>] [-s <num>] [-k] <device>\n", prog);
	printf ("\n");
	printf ("-c       - specify '" ALT_CABLE_1 "' if using this cable\n");
	printf ("-d <num> - wait <num> seconds after sending shutdown command\n");
	printf ("-h       - display this help\n");
	printf ("-k       - force shutdown\n");
	printf ("-s <num> - use shutdown command set <num> for forced shutdown\n");
	printf ("           0: soft shutdown or powerdown, depending on battery status\n");
	printf ("           1: soft shutdown followed by powerdown\n");
	printf ("           2: instant power off\n");
	printf ("           3: power off with grace period\n");
	printf ("           Modes 0-1 will make the UPS come back when power returns\n");
	printf ("           Modes 2-3 will make the UPS stay turned off when power returns\n");
	printf ("<device> - /dev entry corresponding to UPS port\n");
	exit (1);
}

int main (int argc, char **argv)
{
	char	*portname, *prog;
	int	shutdown = 0;
	int	i;

	printf ("Network UPS Tools - APC Smart protocol driver 0.53 (%s)\n", UPS_VERSION);
	openlog ("apcsmart", LOG_PID, LOG_FACILITY);

	prog = argv[0];

	if (argc == 1)
		usage (prog);

	while ((i = getopt(argc, argv, "+hd:ks:c:")) != EOF) {
		switch (i) {
			case 'd':
				sddelay = atoi(optarg);
				break;
			case 'k':
				shutdown = 1;
				break;
			case 'h':
				help(prog);
				break;
			case 's':
				shutdowncmdset = atoi(optarg);
				break;
			case 'c':
				cable = strdup(optarg);
				break;
			default:
				usage(prog);
				break;
		}
	}

	argc -= optind;
	argv += optind;

	droproot();

	if (argc < 1) {
		printf ("Error: no device specified!\n");
		usage (prog);
	}

	portname = NULL;
	for (i = strlen(argv[0]); i >= 0; i--)
		if (argv[0][i] == '/') {
			portname = &argv[0][i+1];
			break;
		}

	if (portname == NULL) {
		printf ("Unable to abbreviate %s\n", argv[0]);
		exit (1);
	}

	snprintf (statefn, sizeof(statefn), "%s/apcsmart-%s", STATEPATH,
	          portname);
	open_serial (argv[0], B2400);

	if (cable != 0 && strcmp(cable,ALT_CABLE_1) == 0)
		init_serial();

	/* do this first - ALWAYS try a shutdown even if detection failed */
	if (shutdown) {
		forceshutdown();
		/* NOT REACHED */
	}

	if (smartmode() == 0) {
		printf ("Unable to detect an APC Smart protocol UPS on port %s\n", argv[0]);
		printf ("Check the cabling, port name or model name and try again\n");
		exit (1);
	}

	info = create_info (infomax, shmok);

	/* manufacturer ID - hardcoded in this particular module */
	addinfo (INFO_MFR, "APC", 0, 0);

	createmsgq();	/* try to create IPC message queue */

	setuphandlers();

	/* see what's out there */
	getbaseinfo(argv[0]);

	printf ("Detected %s [%s] on %s (level %i)\n", getdata(INFO_MODEL), 
		getdata(INFO_SERIAL), argv[0], upslevel);

	if (cap_measureupsii)
		printf ("Measure-UPS II detected!\n");

	background();
	addcleanup();

	for (;;) {
		updateinfo();

		/* wait up to 2 seconds for a message from the upsd */

		if (getupsmsg(2))	/* TODO: remove debug scaffolding */
			syslog (LOG_INFO, "Received a message from upsd\n");
	}
}
