/*
 * Copyright (c) 1996 University College London
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *      This product includes software developed by the Computer Science
 *      Department at University College London
 * 4. Neither the name of the University nor of the Department may be used
 *    to endorse or promote products derived from this software without
 *    specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE UNIVERSITY AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE UNIVERSITY OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
/*
   File: ui_fns.c
   Author: mhandley
   Date: Mon Jul  5 1994
   Purpose: Functions for converting between internal format and Tk text format
 */
#include "prototypes.h"
#include <stdio.h>
#include "tcl.h"
#include "tk.h"
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include <time.h>

extern int checksum_active;	/* used to store checksum activation status */
extern int minimise_updates;
extern page *p;
extern user_data me;
extern typeface tf;
extern Tcl_Interp *interp;
extern int tx_fd;
extern queue sendq;
extern int max_block_len;
extern p_list plist;
block *cur_block = NULL;
block *active_block = NULL;
block *clipboard_block = NULL;	/* pointer to block currently 'on the clipboard' */
int clipboard_counter = 1;	/* used to count the copies being made of a block, so that
				   subsequent pastings can be placed in descending positions */
int clipboard_x;		/* x-ord of original block, used when placing pastings. */
int clipboard_y;		/* y-ord of original block, used when placing pastings. */
line *cur_line;
u_int8 default_status = NORMAL;
u_int8 cur_col;
extern const int COLOUR_CHANGE;
extern int s_message_type;
extern max_block_length;	/* maximum number of lines in a block, as defined in main.c */
int i;				/* quick fix in for loops - jim */
#define MAX_DELETED_BLOCKS 50
block *last_deleted_block[MAX_DELETED_BLOCKS];	/* pointers to last ?x? deleted blocks */
int no_of_del_blocks = 0;

extern int Set_Key(const char *key);

int ui_new_block(ClientData dummy, Tcl_Interp * interp, int argc, char **argv)
{
    block *prev, *next, *new_blk;
    line *new_ln;
    struct timeval t;
    char lineid[ID_LEN];

    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);

    for (i = 0; i < argc; i++) {
	debug("Argv[%d]=%s ", i, argv[i]);
    }
    debug("\n");
    if (p->first_block == NULL) {
	/*this is the first block in the page */
	prev = NULL;
	next = NULL;
    } else {
	/*there's at least one block there */
	prev = p->last_block;
	next = NULL;
    }
    get_cur_time(&t);
    tf.colindex = (u_int8) atoi(argv[4]);
    tf.fontindex = (u_int8) atoi(argv[5]);
    new_blk = init_block(argv[1], default_status, atoi(argv[2]), atoi(argv[3]), &t, prev, next, &me, &me, &tf);
    generate_id(lineid);

    new_ln = init_line(lineid, 0, &t, NULL, "NULL", NULL, "NULL");
    new_blk->first_line = new_ln;

    cur_block = new_blk;
    cur_line = new_ln;
    cur_block->no_of_lines = 1;

    time_copy(&t, &(cur_line->last_mod));
    time_copy(&t, &(cur_block->last_mod));
    time_copy(&t, &(p->last_mod));
    user_copy(&me, &(cur_block->modifier));
    user_copy(&me, &(cur_line->modifier));

    if (p->first_block == NULL) {
	p->first_block = new_blk;
	p->last_block = new_blk;
    } else {
	p->last_block = new_blk;
    }

    new_ln->block = new_blk;

    /* now we transmit the new block */
    queue_packet_for_sending(tx_fd, new_blk, new_blk->blockid, &sendq, BLOCKMSG);
    add_to_recent_list(new_blk->blockid, &(new_blk->last_mod), BLOCKMSG);

    /* JIM CHANGE - SEND OUR NEW LINE AS WELL - THIS AVOIDS NULL POINTERS IN RECEIVING HOSTS.. */
    queue_packet_for_sending(tx_fd, new_ln, new_ln->lineid, &sendq, LINEMSG);
    add_to_recent_list(new_ln->lineid, &(new_ln->last_mod), LINEMSG);

    if (new_blk == NULL) {
	debug("no new block\n");
    }
    if (new_blk->first_line == NULL) {
	debug("new block - no first line\n");
    } else {
	debug("new block - first line exists");
    }

    debug("\nBlock list:\n");
    print_blocks(p);
    return TCL_OK;
}

int ui_new_line(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    line *new_ln;
    struct timeval t;
    char lineid[ID_LEN];

    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);

    debug("new line:");
    for (i = 0; i < argc; i++)
	debug("Argv[%d]=%s ", i, argv[i]);
    debug("\n");

    /* new check to make sure that we don't exceed the max block length */
    /* this is handled now by insert_cr in nt.tcl */
    /*
       if(atoi(argv[2])>=max_block_len)
       {
       debug("line number %d , maxblock length %d\n",atoi(argv[2]),max_block_len);
       return TCL_OK;
       } */

    get_cur_time(&t);
    cur_block = find_block_by_id(p, argv[1]);
    cur_line = find_line(cur_block, atoi(argv[2]));
    if (cur_line != NULL) {
	char *id;
	if (cur_line->next_line == NULL)
#ifdef OLDWAY
	    id = "OK";
#endif
	id = "NULL";
	else
	id = cur_line->next_line->lineid;
	generate_id(lineid);
	new_ln = init_line(lineid, atoi(argv[2]), &t,
			   cur_line, cur_line->lineid,
			   cur_line->next_line, id);
    } else {
#ifdef ABORT
	abort();
#endif
	/*Shouldn't ever need this... */
	generate_id(lineid);
	new_ln = init_line(lineid, atoi(argv[2]), &t, NULL, "OK", NULL, "OK");
    }
    cur_block->no_of_lines = count_lines(cur_block);
    cur_line = new_ln;
    new_ln->block = cur_block;
    renumber_block(cur_block);
    user_copy(&me, &(cur_line->modifier));
    /*queue_packet_for_sending(tx_fd, new_ln, new_ln->lineid, &sendq, LINEMSG);  */

    add_to_recent_list(new_ln->lineid, &t, LINEMSG);

    debug("after adding new line, no_of_lines == %d\n", cur_block->no_of_lines);

    return TCL_OK;
}

int ui_set_line(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    struct timeval t;
    int y;

    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);

    for (i = 0; i < argc; i++)
	debug("Argv[%d]=%s ", i, argv[i]);

    debug("\n");
    if (argv[3] == NULL)
	return 0;
    y = atoi(argv[2]);

    cur_block = find_block_by_id(p, argv[1]);
    if (cur_block == NULL) {
	debug("No current block %s - not sending change\n", argv[1]);
	print_blocks(p);
#ifdef ABORT
	debug("deliberate core dump...\n");
	abort();
#endif
    }
    cur_line = find_line(cur_block, y);
    if (cur_line == NULL) {
	debug("No current line %d in %s\n", y, argv[1]);
	return TCL_OK;
    } else {
	debug("setting line %d in %s\n", y, argv[1]);
    }
    get_cur_time(&t);
    time_copy(&t, &(cur_line->last_mod));
    time_copy(&t, &(cur_block->last_mod));
    time_copy(&t, &(p->last_mod));
    user_copy(&me, &(cur_block->modifier));
    user_copy(&me, &(cur_line->modifier));
    cur_line->no_of_chars = strlen(argv[3]);
    strncpy(cur_line->line_data, argv[3], MAX_LINE_LEN);



    queue_packet_for_sending(tx_fd, cur_line, cur_line->lineid, &sendq, LINEMSG);
    add_to_recent_list(cur_line->lineid, &t, LINEMSG);

    return TCL_OK;
}

int ui_delete_line(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    int y;
    struct timeval t;


    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);

    y = atoi(argv[2]);
    cur_block = find_block_by_id(p, argv[1]);
    cur_line = find_line(cur_block, y);
    if (cur_line == NULL) {
	debug("No current line\n");
	return TCL_OK;
    }
    debug("delete line: %s", cur_line->line_data);
    for (i = 0; i < argc; i++)
	debug("Argv[%d]=%s ", i, argv[i]);
    debug("\n");

    get_cur_time(&t);		/* get current time */
    cur_line->linenum = LINE_DELETED;	/* set linenumber to 'deleted' */

    time_copy(&t, &(cur_line->last_mod));	/* set line, block and page last modified times. */
    time_copy(&t, &(cur_block->last_mod));
    time_copy(&t, &(p->last_mod));

    user_copy(&me, &(cur_block->modifier));	/* set our name in the current line and */
    user_copy(&me, &(cur_line->modifier));	/* block modifiers */




    queue_packet_for_sending(tx_fd, cur_line, cur_line->lineid, &sendq, LINEMSG);
    /* send a deleted line transmission.. */
    add_to_recent_list(cur_line->lineid, &t, LINEMSG);	/* & add to recent list */


    delete_line(cur_line, cur_block);	/* delete the line */
    renumber_block(cur_block);	/* renumber the lines in the current block */
    /* to sort it out */

    if (cur_line->linenum != LINE_DELETED) {
	debug("Deleted line number has been changed\n");	/* line number has been changed of current line */
    } else {
	debug("deleted line still numbered ok\n");
    }

    add_to_recent_list(cur_line->lineid, &t, LINEMSG);	/* add the deleted line to the recent list. */

    if (cur_line->linenum != LINE_DELETED) {
	debug("2Deleted line number has been changed\n");	/* line number has been changed of current line */
    } else {
	debug("2deleted line still numbered ok\n");
    }
    return TCL_OK;		/* return TCL control */
}

int ui_delete_block(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    block *b;
    struct timeval t;

    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);

    b = find_block_by_id(p, argv[1]);

    if (b != NULL) {
	if ((b->status == NORMAL) || (b->status == LOCKED)) {
	    b->status = DELETED;
	    get_cur_time(&t);
	    time_copy(&t, &(b->last_mod));
	    time_copy(&t, &(p->last_mod));

	    add_to_recent_list(b->blockid, &t, BLOCKMSG);
	    queue_packet_for_sending(tx_fd, b, b->blockid,
				     &sendq, BLOCKMSG);

	    /* set the last deleted block as this block - for undelete function */
	    if (no_of_del_blocks >= MAX_DELETED_BLOCKS) {
		/* shift the other pointers */
		/* 0 = 1, 18 = 19, etc */
		for (i = 0; i < 19; i++) {
		    last_deleted_block[i] = last_deleted_block[i + 1];
		}		/*end for */

	    }
	    last_deleted_block[no_of_del_blocks] = b;	/* add to end of list */
	    no_of_del_blocks++;	/* increase size */

	}
	/* endif block hasn't been already deleted */
	else {
	    debug("trying to delete an already deleted block\n");
	}
    } else {
	debug("trying to delete a non-existent block\n");
    }

    debug("got here\n");
    if (b->first_line == NULL) {
	debug("delete block - first line gone missing\n");
    } else {
	debug("delete block - first line ok");
    }

    return TCL_OK;
}

int ui_undelete_block(dummy, interp, argc, argv)	/* jim's undelete block function */
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{

    struct timeval t;
    block *b;			/* for our new block */
    block *c;			/* for the old block    */

    line *l;
    line *x;			/* for line copying */
    line *y;
    line *z;
    char lineid[ID_LEN];

    char command[80];		/* for tcl command lines. */

    UNUSED(dummy);
    UNUSED(argc);

    for (i = 0; i < argc; i++)
	debug("Argv[%d]=%s ", i, argv[i]);
    debug("\n");

    if (no_of_del_blocks > 0) {	/* if we have an undeleted block */
	debug("here now\n");
	c = last_deleted_block[no_of_del_blocks - 1];

	/* now fix up array of deleted blocks. */
	no_of_del_blocks--;
	get_cur_time(&t);
	/* initialize block b ,then copy deleted block c's lines into b.. */
	b = init_block(argv[1],
		       NORMAL,
		       c->xpos,
		       c->ypos,
		       &t,
		       p->last_block,
		       NULL,
		       &(c->originator),
		       &(c->modifier),
		       &(c->face)
	    );

	/* transmit block as undeleted to the other sites, so that they remove */
	/* it from their deleted histories. */
	c->xpos = UNDELETED;	/* store undeleted state of block. */
	get_cur_time(&t);
	time_copy(&t, &(c->last_mod));
	time_copy(&t, &(p->last_mod));

	add_to_recent_list(c->blockid, &t, BLOCKMSG);
	queue_packet_for_sending(tx_fd, c, c->blockid,
				 &sendq, BLOCKMSG);


	if (p->first_block == NULL)	/* this shouldn't happen but just in case */
	    p->first_block = b;

	p->last_block = b;

	sprintf(command, "set_block_posn %s %d %d",
		b->blockid, b->xpos, b->ypos);
	Tcl_Eval(interp, command);
	sprintf(command, "set_block_colour %s %d",
		b->blockid, b->face.colindex);
	Tcl_Eval(interp, command);
	sprintf(command, "set_block_font %s %d",
		b->blockid, b->face.fontindex);
	Tcl_Eval(interp, command);

	/* transmit our new block - before we do the lines so that the */
	/* receivers don't have to put anything on their missing lists. */
	/* we'll put everything on our recent list anyway.. */
	debug("block transmission.\n");

	queue_packet_for_sending(tx_fd, b, b->blockid, &sendq, BLOCKMSG);
	add_to_recent_list(b->blockid, &t, BLOCKMSG);

	/* now have to copy line data to new lines, so that we can retransmit them as new lines.. */

	x = c->first_line;	/* start at first line */

	y = NULL;		/* where we store our new lines. */

	z = NULL;		/* used to store first line. */

	while (x != NULL) {	/* copy all lines.. */


	    if (x->linenum == LINE_DELETED) {
		/* don't copy a deleted line.. */
	    } else {

		/* create a copy line. */
		generate_id(lineid);	/* get new line id */

		l = init_line(lineid, x->linenum, &t, y, "NULL", NULL, "NULL");

		if (y != NULL) {
		    strncpy(l->previd, y->lineid, ID_LEN);	/* amend pointers to/from a previous line */
		    strncpy(y->nextid, l->lineid, ID_LEN);
		    y->next_line = l;	/* amend y->next pointer */
		    l->prev_line = y;
		} else {
		    debug("undelete - no previous line..\n");
		}

		l->block = b;	/* set this line's block. */

		user_copy(&(x->modifier), &(l->modifier));	/* copy modifier */

		strncpy(l->line_data, x->line_data, MAX_LINE_LEN);	/* copy string text.. */
		l->no_of_chars = strlen(l->line_data);	/* set string length */

		if (z == NULL) {
		    z = l;	/* store first line pointer */
		    b->first_line = l;	/* set block's first line.              */
		}
		b->no_of_lines = count_lines(b);	/* set the number of lines */

		debug("Line transmitting \n");

		/* transmit line L */
		queue_packet_for_sending(tx_fd, l, l->lineid, &sendq, LINEMSG);
		add_to_recent_list(l->lineid, &t, LINEMSG);

		y = l;		/* keep 'back' pointer */

	    }			/* end if this line is not deleted */

	    x = x->next_line;	/* now 'do' next line. */
	}


	if (b->first_line == NULL)
	    debug("undelete check 1 - first line null\n");
	else
	    debug("undelete check 1 - first line ok\n");

	p->last_block = b;	/* page's pointer to this block as it's last block. */

	time_copy(&t, &(b->last_mod));	/* set line, block and page last modified times. */
	time_copy(&t, &(b->last_mod));
	time_copy(&t, &(p->last_mod));

	user_copy(&me, &(b->modifier));		/* set our name in the current block modifier */
	/* will this need extending for each line in block? */

	renumber_block(b);	/* renumber all of the lines in block b. */
	b->new_data=1;
	redisplay_block(b);	/* re display the block */


	if (b->first_line == NULL)
	    debug("undelete check 2 - first line null\n");
	else
	    debug("undelete check 2 - first line ok\n");


    }
    /* end if we have a block to un delete */
    else {
	/* we haven't got a block to un delete */
	/* do nothing - perhaps should put a message up in tcl to tell users */
	/*                                          there's nothing to un delete. */
    }


    return TCL_OK;

}				/* end un_delete_block() */

int ui_copy_block(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    block *b;
    struct timeval t;


    block *c;			/* for the old block    */

    line *l;
    line *x;			/* for line copying */
    line *y;
    line *z;
    /*char lineid[ID_LEN];
       char blockid[ID_LEN];

       char command[80];    */

    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    /* block id in argv[1] */
    c = find_block_by_id(p, argv[1]);

    /* old version of the copy block routine - flawed in that if user made changes to 
       original block, they were copied to new block. */
    /*clipboard_block = b; */

    clipboard_counter = 1;
    clipboard_x = c->xpos;
    clipboard_y = c->ypos;

    /* need to make our own copy of the block here, otherwise we will not be
       able to stop the user changing the original from which all copies are made */

    /* first we need to check that we don't already have a clipboard block. If we do
       then we need to free up the memory used by it.. */

    if (clipboard_block != NULL) {
	line *next_line;
	line *this_line;
	/* for each line in the block */
	if (clipboard_block->first_line != NULL) {
	    next_line = clipboard_block->first_line->next_line;

	    while (next_line != NULL) {
		this_line = next_line;
		next_line = next_line->next_line;
		free(this_line);
	    }			/*end while */

	}			/* end if */
	free(clipboard_block);
    }				/*end if we had to destroy a block */
    /* now create a copy of the block in a similar fashion to 'undelete block' does */
    /* copy and paste the block as in undelete block */
    /* now fix up array of deleted blocks. */
    get_cur_time(&t);
    /* initialize block b ,then copy deleted block c's lines into b.. */
    b = init_block("copy",
		   NORMAL,
		   clipboard_x + (1 * clipboard_counter),	/* use clip_counter to position subsequent */
		   clipboard_y + (1 * clipboard_counter),	/*      pastings in a descending formation */
		   &t,
		   p->last_block,
		   NULL,
		   &(c->originator),
		   &(c->modifier),
		   &(c->face)
	);


    /* now have to copy line data to new lines, so that we can retransmit them as new lines.. */

    x = c->first_line;		/* start at first line */

    y = NULL;			/* where we store our new lines. */

    z = NULL;			/* used to store first line. */

    while (x != NULL) {		/* copy all lines.. */
	if (x->linenum == LINE_DELETED) {
	    /* don't copy deleted lines */
	} else {
	    /* create a copy line. */

	    l = init_line("copy", x->linenum, &t, y, "NULL", NULL, "NULL");

	    if (y != NULL) {
		strncpy(l->previd, y->lineid, ID_LEN);	/* amend pointers to/from a previous line */
		strncpy(y->nextid, l->lineid, ID_LEN);
		y->next_line = l;	/* amend y->next pointer */
		l->prev_line = y;
	    } else {
		debug("paste_block - no previous line..\n");
	    }

	    l->block = b;	/* set this line's block. */

	    user_copy(&(x->modifier), &(l->modifier));	/* copy modifier */

	    strncpy(l->line_data, x->line_data, MAX_LINE_LEN);	/* copy string text.. */
	    l->no_of_chars = strlen(l->line_data);	/* set string length */

	    if (z == NULL) {
		z = l;		/* store first line pointer */
		b->first_line = l;	/* set block's first line.              */
	    }
	    b->no_of_lines = count_lines(b);	/* set the number of lines */

	    y = l;		/* keep 'back' pointer */

	}			/* end if not a deleted line */

	x = x->next_line;	/* now 'do' next line. */
    }


    if (b->first_line == NULL)
	debug("copy check 1 - first line null\n");
    else
	debug("copy check 1 - first line ok\n");

    p->last_block = b;		/* page's pointer to this block as it's last block. */

    time_copy(&t, &(b->last_mod));	/* set line, block and page last modified times. */

    user_copy(&me, &(b->modifier));	/* set our name in the current block modifier */
    /* will this need extending for each line in block? */

    renumber_block(b);		/* renumber all of the lines in block b. */

    clipboard_block = b;


/* ** end of copy the block ** */



    return 0;
}


int ui_paste_block()
{
    /* paste the last cut/copied block */

    if (clipboard_block == NULL) {
	/*nothing to paste */
    } else {
	/* copy and paste the block as in undelete block */
	struct timeval t;

	block *b;		/* for our new block */
	block *c;		/* for the old block    */

	line *l;
	line *x;		/* for line copying */
	line *y;
	line *z;
	char lineid[ID_LEN];
	char blockid[ID_LEN];

	char command[80];	/* for tcl command lines. */

	c = clipboard_block;	/* pasting the block currently on the clipboard */

	generate_id(blockid);
	/* now fix up array of deleted blocks. */
	get_cur_time(&t);
	/* initialize block b ,then copy deleted block c's lines into b.. */
	b = init_block( /*new id */ blockid,
		       NORMAL,
		       clipboard_x + (1 * clipboard_counter),	/* use clip_counter to position subsequent */
		       clipboard_y + (1 * clipboard_counter),	/*      pastings in a descending formation */
		       &t,
		       p->last_block,
		       NULL,
		       &(c->originator),
		       &(c->modifier),
		       &(c->face)
	    );

	clipboard_counter++;	/* increment clipboard_counter for next paste */



	if (p->first_block == NULL)	/* this shouldn't happen but just in case */
	    p->first_block = b;

	p->last_block = b;

	sprintf(command, "set_block_posn %s %d %d",
		b->blockid, b->xpos, b->ypos);
	Tcl_Eval(interp, command);
	sprintf(command, "set_block_colour %s %d",
		b->blockid, b->face.colindex);
	Tcl_Eval(interp, command);
	sprintf(command, "set_block_font %s %d",
		b->blockid, b->face.fontindex);
	Tcl_Eval(interp, command);
	/* transmit our new block - before we do the lines so that the */
	/* receivers don't have to put anything on their missing lists. */
	/* we'll put everything on our recent list anyway.. */
	debug("block transmission.\n");

	queue_packet_for_sending(tx_fd, b, b->blockid, &sendq, BLOCKMSG);
	add_to_recent_list(b->blockid, &t, BLOCKMSG);

	/* now have to copy line data to new lines, so that we can retransmit them as new lines.. */

	x = c->first_line;	/* start at first line */

	y = NULL;		/* where we store our new lines. */

	z = NULL;		/* used to store first line. */

	while (x != NULL) {	/* copy all lines.. */
	    /* create a copy line. */

	    generate_id(lineid);	/* get new line id */

	    l = init_line(lineid, x->linenum, &t, y, "NULL", NULL, "NULL");

	    if (y != NULL) {
		strncpy(l->previd, y->lineid, ID_LEN);	/* amend pointers to/from a previous line */
		strncpy(y->nextid, l->lineid, ID_LEN);
		y->next_line = l;	/* amend y->next pointer */
		l->prev_line = y;
	    } else {
		debug("paste_block - no previous line..\n");
	    }

	    l->block = b;	/* set this line's block. */

	    user_copy(&(x->modifier), &(l->modifier));	/* copy modifier */

	    strncpy(l->line_data, x->line_data, MAX_LINE_LEN);	/* copy string text.. */
	    l->no_of_chars = strlen(l->line_data);	/* set string length */

	    if (z == NULL) {
		z = l;		/* store first line pointer */
		b->first_line = l;	/* set block's first line.              */
	    }
	    b->no_of_lines = count_lines(b);	/* set the number of lines */

	    debug("Line transmitting \n");

	    /* transmit line L */
	    queue_packet_for_sending(tx_fd, l, l->lineid, &sendq, LINEMSG);
	    add_to_recent_list(l->lineid, &t, LINEMSG);

	    y = l;		/* keep 'back' pointer */


	    x = x->next_line;	/* now 'do' next line. */
	}


	if (b->first_line == NULL)
	    debug("undelete check 1 - first line null\n");
	else
	    debug("undelete check 1 - first line ok\n");

	p->last_block = b;	/* page's pointer to this block as it's last block. */

	time_copy(&t, &(b->last_mod));	/* set line, block and page last modified times. */
	time_copy(&t, &(b->last_mod));
	time_copy(&t, &(p->last_mod));

	user_copy(&me, &(b->modifier));		/* set our name in the current block modifier */
	/* will this need extending for each line in block? */

	renumber_block(b);	/* renumber all of the lines in block b. */
	b->new_data=1;
	redisplay_block(b);	/* re display the block */


/* ** end of paste the block ** */

    }
    return 0;
}


int ui_find_line_len(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    static char lenstr[7];
    int y;
    u_int8 len;
    UNUSED(dummy);
    UNUSED(argc);
    debug("find line len:");
    for (i = 0; i < argc; i++)
	debug("Argv[%d]=%s ", i, argv[i]);
    debug("\n");

    y = atoi(argv[2]);
    len = find_line_len(cur_block, y);
    debug("Line len: %d\n", len);
    sprintf(lenstr, "%d", len);
    interp->result = lenstr;
    return TCL_OK;
}

int ui_find_block_len(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    static char lenstr[7];
    int len;
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    UNUSED(argv);
    debug("find block len");
    for (i = 0; i < argc; i++)
	debug("Argv[%d]=%s ", i, argv[i]);
    debug("\n");
    len = find_block_len(cur_block);
    debug("Block len: %d\n", len);
    sprintf(lenstr, "%d", len);
    interp->result = lenstr;
    return TCL_OK;
}

int ui_findblock(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    /*WARNING:  the return value is static!  you better not still be using
       a previous returned value when you call this!!! */
    static char blockid[ID_LEN];
    u_int8 x;
    u_int16 y;
    block *b;

    UNUSED(dummy);
    UNUSED(argc);
    debug("block id");
    for (i = 0; i < argc; i++)
	debug("Argv[%d]=%s ", i, argv[i]);
    debug("\n");
    y = (u_int8) atoi(argv[2]);
    x = (u_int16) atoi(argv[1]);
    b = findblock(blockid, p, x, y);
    if (b != NULL)
	cur_block = b;
    interp->result = blockid;
    return TCL_OK;
}

int ui_load_text_file(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    FILE *file;
    line *new_ln, *l;
    char str[MAX_LINE_LEN];
    struct timeval t;
    int y = 0;
    int no_of_lines = 0;
    int first = TRUE;
    char lineid[ID_LEN];
    UNUSED(dummy);
    UNUSED(argc);
    file = fopen(argv[1], "r");
    if (file == NULL) {
	Tcl_VarEval(interp, "errorpopup \"Load Failed\" \"Check you have permission to load this file\"", NULL);
	return TCL_OK;
    }
    while (!feof(file)) {
	fgets(str, MAX_LINE_LEN, file);
	no_of_lines++;
    }
    fclose(file);


    if (no_of_lines > max_block_len) {
	Tcl_VarEval(interp, "errorpopup \"Load Truncated\" \"The file you were trying to load is too long and has been truncated\"", NULL);
    }
    file = fopen(argv[1], "r");
    get_cur_time(&t);
    l = cur_block->first_line;
    while (!feof(file)) {
	strcpy(str, " ");	/* empty string first */
	fgets(str, MAX_LINE_LEN, file);
	if (str == NULL)
	    return TCL_OK;
	if (str[strlen(str) - 1] == '\n')
	    str[strlen(str) - 1] = '\0';
	if ((first != TRUE) || (l == NULL)) {
	    generate_id(lineid);
#ifdef OLDWAY
	    new_ln = init_line(lineid, LOCAL_LINE, &t, l, "OK", NULL, "OK");
#endif
	    new_ln = init_line(lineid, LOCAL_LINE, &t, l, l->lineid, NULL, "NULL");
	} else {
	    new_ln = l;
	    time_copy(&t, &(l->last_mod));
	}
	if (first == TRUE) {
	    /*first line */
	    cur_block->first_line = new_ln;
	    cur_block->no_of_lines = 1;
	    first = FALSE;
	} else {
	    cur_block->no_of_lines++;
	    debug("~ current number of lines..%d \n", cur_block->no_of_lines);
	    debug("~ current string %s \n", str);
	}

	strncpy(new_ln->line_data, str, MAX_LINE_LEN);

	new_ln->no_of_chars = strlen(str);
	new_ln->block = cur_block;
	user_copy(&me, &(new_ln->modifier));



	queue_packet_for_sending(tx_fd, new_ln, new_ln->lineid, &sendq, LINEMSG);
	add_to_recent_list(new_ln->lineid, &t, LINEMSG);
	l = new_ln;
	y++;
	if (y > max_block_len)
	    break;
    }

    renumber_block(cur_block);	/*renumber the block correctly */
	cur_block->new_data=1;
    redisplay_block(cur_block);
    return TCL_OK;
}

int ui_save_text_file(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    FILE *file;
    block *b;
    int i, j, max_len;
    char str[MAX_LINE_LEN];
    line *l[100];
    int xpos[100];
    int no_of_lines = 0;
    int max_line_num = 0;
    UNUSED(dummy);
    UNUSED(argc);
    if (p == NULL)
	return TCL_OK;
    if (p->first_block == NULL)
	return TCL_OK;
    b = p->first_block;
    file = fopen(argv[2], "w");
    if (file == NULL) {
	Tcl_VarEval(interp, "errorpopup \"Save Failed\" \"Check you have permission to save this to this file\"", NULL);
	return TCL_OK;
    }
    if (strcmp(argv[1], "TRUE") == 0) {		/* SAVE participants if required to do so */
	save_participants(file, &plist);
    }
    while (b != NULL) {		/* visit each block in the page - get ourselves a maximum line number */
	if (b->status != DELETED)
	    if ((b->no_of_lines) + (b->ypos) > max_line_num)
		max_line_num = (b->no_of_lines) + (b->ypos);
	b = b->next_block;

    }				/* end while */

    for (i = 0; i < (max_line_num + 1); i++) {	/* going down the page  - max+1 ->jim */
	max_len = 0;
	for (j = 0; j < MAX_LINE_LEN; j++)	/* clear the string - for this line? */
	    str[j] = ' ';
	b = p->first_block;	/* point to first block */

	while (b != NULL) {	/* for each block in the page */
	    if (b->status != DELETED)
		if (b->ypos == i) {	/* if this block is not deleted and starts at this y co-ord */
		    l[no_of_lines] = b->first_line;	/* ?????? */
		    xpos[no_of_lines++] = b->xpos;	/* ?????? */
		}
	    b = b->next_block;	/* check next block */
	}			/*end while */

	for (j = 0; j < no_of_lines; j++) {	/* for each line */
	    if ((l[j] == NULL) && (no_of_lines > 0)) {
		l[j] = l[no_of_lines - 1];
		xpos[j] = xpos[no_of_lines - 1];
		no_of_lines--;
		if (l[j] == NULL)
		    continue;
	    }			/*end if */
	    if (xpos[j] + l[j]->no_of_chars > max_len)
		max_len = xpos[j] + l[j]->no_of_chars;
	    memcpy(&(str[xpos[j]]), l[j]->line_data, l[j]->no_of_chars);
	    l[j] = l[j]->next_line;
	}			/* end for */
	str[max_len] = '\0';
	fprintf(file, "%s\n", str);
    }				/* end for */
    fclose(file);

    return TCL_OK;
}

int ui_save_struct_file(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    FILE *file;
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    file = fopen(argv[1], "w");
    if (file == NULL)
	return TCL_OK;
    save_struct_file(file, p);
    fclose(file);
    return TCL_OK;
}

int ui_load_struct_file(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    FILE *file;
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    file = fopen(argv[1], "r");
    if (file == NULL)
	return TCL_OK;
    load_struct_file(file);
    fclose(file);
    return TCL_OK;
}

int ui_generate_id(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    char id[ID_LEN];
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    UNUSED(argv);
    generate_id(id);
    strncpy(interp->result, id, ID_LEN);
    return TCL_OK;
}

int ui_set_block_colour(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    struct timeval t;
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    cur_block = find_block_by_id(p, argv[1]);
    if (cur_block != NULL) {
	cur_block->face.colindex = (u_int8) atoi(argv[2]);
	get_cur_time(&t);
	time_copy(&t, &(cur_block->last_mod));
	time_copy(&t, &(p->last_mod));
	queue_packet_for_sending(tx_fd, cur_block, cur_block->blockid,
				 &sendq, BLOCKMSG);
	add_to_recent_list(cur_block->blockid, &t, BLOCKMSG);
    }
    return TCL_OK;
}

int ui_set_block_font(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    struct timeval t;
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    cur_block = find_block_by_id(p, argv[1]);
    if (cur_block != NULL) {
	cur_block->face.fontindex = (u_int8) atoi(argv[2]);
	get_cur_time(&t);
	time_copy(&t, &(cur_block->last_mod));
	time_copy(&t, &(p->last_mod));
	queue_packet_for_sending(tx_fd, cur_block, cur_block->blockid,
				 &sendq, BLOCKMSG);
	add_to_recent_list(cur_block->blockid, &t, BLOCKMSG);
    }
    return TCL_OK;
}

int ui_set_block_posn(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    struct timeval t;
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    cur_block = find_block_by_id(p, argv[1]);
    if (cur_block != NULL) {
	cur_block->xpos = (u_int8) atoi(argv[2]);
	cur_block->ypos = (u_int8) atoi(argv[3]);
	get_cur_time(&t);
	time_copy(&t, &(cur_block->last_mod));
	time_copy(&t, &(p->last_mod));
	queue_packet_for_sending(tx_fd, cur_block, cur_block->blockid,
				 &sendq, BLOCKMSG);
	add_to_recent_list(cur_block->blockid, &t, BLOCKMSG);
    }
    return TCL_OK;
}

int ui_get_max_block_size(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    UNUSED(argv);
    sprintf(interp->result, "%d", max_block_len);
    return TCL_OK;
}				/*end get max block size */

int ui_get_block_length(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    block *b;
    UNUSED(dummy);
    UNUSED(argc);
    b = find_block_by_id(p, argv[1]);
    sprintf(interp->result, "%d", b->no_of_lines);
    return TCL_OK;
}				/*end get block size */

int ui_get_creator(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    block *b;
    struct in_addr addr;
    UNUSED(dummy);
    UNUSED(argc);
    b = find_block_by_id(p, argv[1]);
	/* make sure the block is up to date */
    if (b == NULL) return TCL_OK;
	redisplay_block (b);

    addr.s_addr = htonl(b->originator.hostaddr.s_addr);
    sprintf(interp->result, "%s@%s", b->originator.username,
	    inet_ntoa(addr));
    return TCL_OK;
}

int ui_get_modifier(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    block *b;
    struct in_addr addr;
    UNUSED(dummy);
    UNUSED(argc);
    b = find_block_by_id(p, argv[1]);
    if (b == NULL)
	return TCL_OK;
    addr.s_addr = htonl(b->modifier.hostaddr.s_addr);
    sprintf(interp->result, "%s@%s", b->modifier.username,
	    inet_ntoa(addr));
    return TCL_OK;
}


int ui_get_modification_time(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    block *b;
    struct tm *time;
    UNUSED(dummy);
    UNUSED(argc);
    b = find_block_by_id(p, argv[1]);
    if (b == NULL)
	return TCL_OK;
    time = localtime((time_t *) & (b->last_mod.tv_sec));
    if (time->tm_min < 10)
	sprintf(interp->result, "%2d:0%d", time->tm_hour, time->tm_min);
    else
	sprintf(interp->result, "%2d:%2d", time->tm_hour, time->tm_min);

    return TCL_OK;
}

/*is it locked for others */
int ui_get_lock_status(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    block *b;
    UNUSED(dummy);
    UNUSED(argc);
    b = find_block_by_id(p, argv[1]);
    if (b == NULL)
	return TCL_OK;
    if (b->status == LOCKED)
	sprintf(interp->result, "LOCKED");
    else
	sprintf(interp->result, "NORMAL");
    return TCL_OK;
}

/*is it locked for me */
int ui_get_my_lock_status(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    block *b;
    UNUSED(dummy);
    UNUSED(argc);
    b = find_block_by_id(p, argv[1]);
    if (b == NULL)
	return TCL_OK;
    if ((b->status == LOCKED) && (user_cmp(&me, &(b->modifier)) != TRUE))
	sprintf(interp->result, "LOCKED");
    else
	sprintf(interp->result, "NORMAL");
    return TCL_OK;
}

int ui_set_lock_mode(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    if (strcmp(argv[1], "LOCKED") == 0)
	default_status = LOCKED;
    else
	default_status = NORMAL;

    return TCL_OK;
}

int ui_set_my_colour(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    cur_col = atoi(argv[1]);
    debug("setting colour\n");

    s_message_type = COLOUR_CHANGE;
    send_session_message();
    add_receiver_to_participants(&me, "", cur_col, NULL);
    return TCL_OK;
}

int ui_keep_userlist(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    if (strcmp(argv[1], "TRUE") == 0)
	keep_userlist(TRUE);
    else
	keep_userlist(FALSE);
    return TCL_OK;
}

int ui_send_shared_pointer(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    static pointer p;
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    p.x = atoi(argv[1]);
    p.y = atoi(argv[2]);

    p.col = atoi(argv[3]);

    generate_pointer_id(p.id, argv[4]);

    strncpy(p.username, me.username, MAX_USER_NAME);
    strncpy(p.style, argv[4], STYLE_LEN);
    queue_packet_for_sending(tx_fd, &p, p.id,
			     &sendq, POINTERMSG);
    return TCL_OK;
}

int ui_quit()
{
    send_leave_message();
    debug("Now exiting NTE\n");
    exit(2);
    return TCL_OK;
}

int ui_get_participant_by_number(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;			/* Argument strings. */
{
    participant *pt;
    UNUSED(dummy);
    UNUSED(argc);
    pt = get_participant_by_number(atoi(argv[1]), &plist);
    /* jim fix - replace final variable substitution %u by %lu */
	if (pt != NULL) {
		sprintf(interp->result, "{%s} {%s} {%s} %d %lu %i", pt->text_name,
		 pt->nt_version, pt->os_version, (int) pt->col, pt->time, pt->packets);
	}
	return TCL_OK;
}

int uigetustimeofday()
{
    struct timeval tp;
    gettimeofday(&tp, NULL);
    sprintf(interp->result, "%ld.%ld", (unsigned long) tp.tv_sec, (unsigned long) tp.tv_usec);
    return TCL_OK;
}

int ui_set_encryption_key(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;
{
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    Set_Key(argv[1]);
    return TCL_OK;
}

int ui_reset_participants(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;
{
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);
    UNUSED(argv);

    reset_participants();

    return TCL_OK;
}

int ui_set_checksum_status(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;
{
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);


    if (atoi(argv[1]) == 1) {
	checksum_active = 1;	/* user has asked to activate the checksum 
				   protocols */
    } else {
	if (atoi(argv[1]) == 0) {
	    checksum_active = 0;	/* user has asked to de-activate the checksum
					   protocols */
	}
    }

    return TCL_OK;

}

int ui_active_block(dummy, interp, argc, argv)
ClientData dummy;		/* Not used. */
Tcl_Interp *interp;		/* Current interpreter. */
int argc;			/* Number of arguments. */
char **argv;
{
	block *b;
    UNUSED(dummy);
    UNUSED(interp);
    UNUSED(argc);

    if (strcmp (argv[1],"1")==0) {
	    minimise_updates = 1;	
    } else if (strcmp (argv[1],"0")==0) {
	    minimise_updates = 0;	
	} else {
        b = find_block_by_id(p, argv[1]);
		active_block = b;
	}

    return TCL_OK;
}
