#include <stdio.h>
#include <glib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>

#include "apassign.h"
#include "nls.h"

extern void gui_err(char *);
extern char *db_name;

enum {
	NSC_VIEWER,
	NSC_INDEXDB,
	NSC_LAST
};

struct rc_terms {
	char *name;
	int id;
};

static struct rc_terms cfgrc[] = {
	{"Viewer:", NSC_VIEWER},
	{"IndexDB:", NSC_INDEXDB}
};

char *get_1qstr(str)
char *str;
{
	static char *p = NULL;
	char *pom1 = NULL , *pom2 = NULL;
	int found;

	if (str) p = str;

	for ( ; p && *p && (!pom1 || !pom2) ; p++)
	{
		found = FALSE;

		if (*p == '\"')
		{
			if ((p == str) || (*(p-1) != '\\'))
				found = TRUE;
		}

		if (!pom1 && found)
			pom1 = p+1;
		else if (!pom2 && found)
			pom2 = p;
	}
	if (pom1 && pom2)
	{
		*pom2 = '\0';
		return pom1;
	}
	else
	{
		p = NULL;
		return NULL;
	}
}

char *escape_str(str , unsafe)
char *str;
char *unsafe;
{
	char sbuf[2048];
	char *p,*r;

	for (p = str, r = sbuf ; *p ; p++)
	{
		if (strchr(unsafe, *p))
		{
			*r = '\\';
			r++;
			*r = *p;
			r++;
		}
		else
		{
			*r = *p;
			r++;
		}
	}
	*r = '\0';
	return g_strdup(sbuf);
}

void load_rc()
{
	char pom[2048];
	FILE *f;
	char *line;
	int found = FALSE;
	int i;

	sprintf(pom, "%s/.nscacherc" , g_get_home_dir());

	f = fopen(pom , "r");

	if (!f) return;

	while((line = fgets(pom, sizeof(pom) , f)))
	{
		if (*line == '#') continue;
		if (!strcspn(line , " \t\r\n")) continue;

		found = FALSE;
		for (i = 0 ; i < NSC_LAST ; i++)
		{
			if (!strncasecmp(cfgrc[i].name, line , strlen(cfgrc[i].name)))
			{
				found = TRUE;
				break;
			}
		}
		if (!found)
			fprintf(stderr, gettext("Unable to parse: %s\n"), line);
		else
		{
			line += strlen(cfgrc[i].name);
			line += strspn(line , "\t ");
			switch(cfgrc[i].id)
			{
				case NSC_VIEWER:
				{
					apassign *ad = g_malloc(sizeof(apassign));

					ad->mimetype = g_strdup(get_1qstr(line));
					ad->fapplication = g_strdup(get_1qstr(NULL));
					ad->uapplication = g_strdup(get_1qstr(NULL));
					if (!*ad->fapplication)
					{
						g_free(ad->fapplication);
						ad->fapplication = NULL;
					}
					if (!*ad->uapplication)
					{
						g_free(ad->uapplication);
						ad->uapplication = NULL;
					}

					apassign_data = g_slist_append(apassign_data, ad);
				}
				break;
				case NSC_INDEXDB:
				{
					int l;

					l = strcspn(line , "\r\n");
					if (l) db_name = g_strndup(line , l);
				}
				break;
				default:
				break;
			}
		}
	}
	fclose(f);
}

void save_rc()
{
	char pom[2048];
	GSList *ptr;
	FILE *f;

	sprintf(pom , "%s/.nscacherc" , g_get_home_dir());

	f = fopen(pom , "w+");

	if (!f)
	{
		gui_err(gettext("Unable to open ~/.nscacherc file"));
		return;
	}

	fprintf(f, gettext("# This .nscacherc file was generated by nscache\n# You may edit it if you're careful!\n\n"));


	if (db_name)
		fprintf(f , "%s %s\n\n", cfgrc[NSC_INDEXDB].name , db_name);

	ptr = apassign_data;
	while(ptr)
	{
		apassign *ad = (apassign *)ptr->data;
		char *mt,*fv,*uv;

		mt = escape_str(ad->mimetype , "\\\"");
		fv = escape_str(ad->fapplication , "\\\"");
		uv = escape_str(ad->uapplication , "\\\"");
		fprintf(f , "%s \"%s\" \"%s\" \"%s\"\n" , cfgrc[NSC_VIEWER].name, mt, fv, uv);
		g_free(mt);
		g_free(fv);
		g_free(uv);
		ptr = ptr->next;
	}

	fclose(f);
	
}

static void usage(prgname)
char *prgname;
{
	printf(gettext("Usage: %s [OPTION]\n"), prgname);
	printf(gettext("  -f,  --db_file=FILENAME  set cache file to FILENAME\n"));
	printf(gettext("  -v,  --version           display version information and exit\n"));
	printf(gettext("  -h,  --help              display this help and exit\n"));
	exit(0);
}

int cmd_setup(argc,argv)
int argc;
char **argv;
{
	int c;
	int index;
	struct option options[] =
	{
		{"db_file", 1, NULL, 'f'},
		{"help", 0, NULL, 'h'},
		{"version", 0, NULL, 'v'}
	};

	while(1)
	{
		c = getopt_long(argc, argv, "vhf:", options, &index);
		if (c == -1) break;
		if (!c) c = options[index].val;
		switch (c)
		{
			case 'v':
				printf("%s-%s\n", PACKAGE, VERSION);
				exit(0);
			break;
			case 'h':
				usage(argv[0]);
			break;
			case 'f':
				db_name = g_strdup(optarg);
			break;
			default:
				return -1;
		}

	}

	if (optind < argc)
	{
		printf(gettext("Illegal argument: %s\n"), argv[optind]);
		return -1;
	}

	return 0;
}

