/*
 * Copyright (c) 2004, 2005, 2006 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/mman.h>

#include <netinet/in.h>
#include <arpa/inet.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <netdb.h>
#include <openssl/ssl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h> 

#include "../libmy/str.h"
#include "../libmy/file.h"
#ifdef __linux__
#include "../libbsd/strlcpy.h"
#include "../libbsd/strlcat.h"
#endif
#include "config.h"
#include "proto.h"
#include "extern.h"

/*
 * global vars local
 */
static int		ppid;
static const char	*configfile = "/var/nostromo/conf/nhttpd.conf";

/*
 * global vars extern
 */
volatile sig_atomic_t	quit = 0;
int			debug = 0;
int			mimes_size;
int			fdbsdauth[2];
char			*mimes;
char			logn[1024];

const char *month[12] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct",
	"Nov", "Dec"
};

const char *day[7] = {
	"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

const char *http_s_200		= "200 OK";
const char *http_s_206		= "206 Partial Content";
const char *http_s_301		= "301 Moved Permanently";
const char *http_s_302		= "302 Found";
const char *http_s_304		= "304 Not Modified";
const char *http_s_400		= "400 Bad Request";
const char *http_s_401		= "401 Unauthorized";
const char *http_s_403		= "403 Forbidden";
const char *http_s_404		= "404 Not Found";
const char *http_s_413		= "413 Request Entity Too Large";
const char *http_s_500		= "500 Internal Server Error";
const char *http_s_501		= "501 Not Implemented";
const char *http_s_503		= "503 Service Unavailable";
const char *http_fn_dat		= "Date:";
const char *http_fn_srv		= "Server:";
const char *http_fn_lmd		= "Last-Modified:";
const char *http_fn_clt		= "Content-Length:";
const char *http_fn_ims		= "If-Modified-Since:";
const char *http_fn_ref		= "Referer:";
const char *http_fn_agt		= "User-Agent:";
const char *http_fn_con		= "Connection:";
const char *http_fn_alv		= "Keep-Alive:";
const char *http_fn_cok		= "Cookie:";
const char *http_fn_teg		= "Transfer-Encoding:";
const char *http_fn_cte		= "Content-Type:";
const char *http_fn_loc		= "Location:";
const char *http_fn_hos		= "Host:";
const char *http_fn_aut		= "Authorization:";
const char *http_fn_auw		= "WWW-Authenticate:";
const char *http_fn_ran		= "Range:";
const char *http_fn_rac		= "Content-Range:";
const char *http_fn_aen		= "Accept-Encoding:";
const char *http_fv_srv		= "nostromo 1.8.8";
const char *http_fv_pro		= "HTTP/1.1";
const char *http_fv_cgi		= "CGI/1.1";
const char *http_fv_teg		= "chunked";
const char *http_fv_lch		= "0\r\n\r\n";
const char *http_fv_con_alv	= "Keep-Alive";
const char *http_fv_con_cls	= "close";

char	http_fv_alv[128];
char	http_fv_cte[128];
char	http_url[128];
char	http_urls[128];
char	http_sig[128];
char	http_path[128];

struct cfg		config;
struct connection	c[CON];

/*
 * usage()
 *	print usage message
 * Return:
 *	none
 */
void
usage(int mode)
{
	extern char	*__progname;

	if (mode != 2) {
		fprintf(stderr, "usage: %s ", __progname);
		fprintf(stderr, "[-dhrv46] [-c configfile]\n");
	}

	if (mode == 1) {
		fprintf(stderr, "\noptions:\n");
		fprintf(stderr, "  -d\t\t: Debug mode.\n");
		fprintf(stderr, "  -h\t\t: This help.\n");
		fprintf(stderr,
		    "  -r\t\t: nhttpd will chroot to serverroot.\n");
		fprintf(stderr, "  -v\t\t: Shows version.\n");
		fprintf(stderr, "  -4\t\t: Enable IPv4 and IPv6.\n");
		fprintf(stderr, "  -6\t\t: Enable IPv6 only.\n");
		fprintf(stderr,
		    "  -c configfile\t: Use an alternate configfile.\n");
	}

	if (mode == 2)
		fprintf(stderr, "%s\n", http_fv_srv);

	exit(1);
}

/*
 * sig_handler()
 *	signal handler
 * Return:
 *	none
 */
void
sig_handler(const int sig)
{
	switch (sig) {
	case SIGCHLD: {
		int	status;

		while (waitpid(-1, &status, WNOHANG) > 0);

		break;
	}
	case SIGINT:
	case SIGTERM:
		quit = 1;
		break;
	case SIGPIPE:
		/* ignore */
		break;
	case SIGHUP:
		/* ignore */
		break;
	case SIGQUIT:
		/* ignore */
		break;
	case SIGALRM:
		/* ignore */
		break;
	default:
		flogd(config.logserver, "[info] unhandled signal %i, ignored\n",
		    sig);
		break;
	}
}

/*
 * send_full()
 *	send a fully generated header/body block
 * Return:
 *	none
 */
void
send_full(const int sfd, const int hr)
{
	int	s, len;
	char	*offset;

	s = 0;

	len = strlen(c[sfd].pfdh[hr]) - c[sfd].wsnd;

	if (len > 0) {
		offset = c[sfd].pfdh[hr] + c[sfd].wsnd;

		if (c[sfd].x_ssl)
			s = sys_write_ssl(c[sfd].ssl, offset, len);
		else
			s = sys_write_a(sfd, offset, len);
		if (s != -1)
			c[sfd].wsnd += s;
	}

	if (len == 0 || s < 0) {
		c[sfd].pfdo--;
		c[sfd].wred = 0;
 		c[sfd].wsnd = 0;
		c[sfd].pfdn[hr] = 0;
		c[sfd].x_chk[hr] = 0;
		c[sfd].x_ful[hr] = 0;
		free(c[sfd].pfdh[hr]);
	}
}

/*
 * send_chunk()
 *	send a file in chunks
 * Return:
 *	none
 */
void
send_chunk(const int sfd, const int hr)
{
	int	k, h, s, len;
	char	*offset;

	k = h = s = 0;

	len = c[sfd].wred - c[sfd].wsnd;

	/* read block from file */
	if (len == 0) {
		if (c[sfd].pfds[hr] == 0)
			h = strlen(c[sfd].pfdh[hr]);

		k = http_chunk_ovr(BS);

		c[sfd].wred = sys_read(c[sfd].pfdn[hr], c[sfd].wbuf,
		    (BS - h) - k);

		/* chunk block */
		if (c[sfd].wred > 0) {
			offset = http_chunk(c[sfd].wbuf, c[sfd].wred);
			c[sfd].wred += http_chunk_ovr(c[sfd].wred);

			if (c[sfd].pfds[hr] == 0) {
				/* attach http header */
				h = strlcpy(c[sfd].wbuf, c[sfd].pfdh[hr], BS);
				c[sfd].pfds[hr] = 1;
			}

			memcpy(c[sfd].wbuf + h, offset, c[sfd].wred);
			free(offset);
			c[sfd].wred += h;
		}

		c[sfd].wsnd = 0;
		len = c[sfd].wred;
	}

	offset = c[sfd].wbuf + c[sfd].wsnd;

	/* send a chunk */
	if (c[sfd].wred > 0) {
		if (c[sfd].x_ssl)
			s = sys_write_ssl(c[sfd].ssl, offset, len);
		else
			s = sys_write_a(sfd, offset, len);
		if (s != 1)
			c[sfd].wsnd += s;
	}

	/* send last chunk, cleanup */
	if (c[sfd].wred < 1 || s < 0) {
		if (c[sfd].x_ssl)
			s = sys_write_ssl(c[sfd].ssl, http_fv_lch,
			    strlen(http_fv_lch));
		else
			s = sys_write_a(sfd, http_fv_lch,
			    strlen(http_fv_lch));

		close(c[sfd].pfdn[hr]);
		c[sfd].pfdo--;
		c[sfd].wred = 0;
	 	c[sfd].wsnd = 0;
		c[sfd].pfdn[hr] = 0;
		c[sfd].pfds[hr] = 0;
		c[sfd].x_chk[hr] = 0;
		free(c[sfd].pfdh[hr]);
	}
}

/*
 * send_file()
 *	send a file in blocks, the size is delivered in the header
 * Return:
 *	none
 */
void
send_file(const int sfd, const int hr)
{
	int	h, s, len;
	char	*offset;

	h = s = 0;

	len = c[sfd].wred - c[sfd].wsnd;

	/* read block from file */
	if (len == 0) {
		if (c[sfd].pfds[hr] == 0) {
			/* attach http header */
			h = strlcpy(c[sfd].wbuf, c[sfd].pfdh[hr], BS);
			c[sfd].wred = sys_read(c[sfd].pfdn[hr],
			    c[sfd].wbuf + h, BS - h);
			c[sfd].wred += h;
		} else
			c[sfd].wred = sys_read(c[sfd].pfdn[hr],
			    c[sfd].wbuf, BS);

		c[sfd].wsnd = 0;
		len = c[sfd].wred;
	}

	offset = c[sfd].wbuf + c[sfd].wsnd;

	/* send a block */
	if (c[sfd].wred > 0) {
		if (c[sfd].x_ssl)
			s = sys_write_ssl(c[sfd].ssl, offset, len);
		else
			s = sys_write_a(sfd, offset, len);
		if (s != -1) {
			c[sfd].pfds[hr] += h ? s - h : s;
			c[sfd].wsnd += s;
		}
	}

	/* file sent, write access log, cleanup */
	if (c[sfd].wred < 1 || s < 0 || c[sfd].pfds[hr] == 0) {
		http_alog(sfd, hr);
		close(c[sfd].pfdn[hr]);
		c[sfd].pfdo--;
		c[sfd].wred = 0;
		c[sfd].wsnd = 0;
		c[sfd].pfdn[hr] = 0;
		c[sfd].pfds[hr] = 0;
		free(c[sfd].pfdh[hr]);
		free(c[sfd].plreq[hr]);
		free(c[sfd].plref[hr]);
		free(c[sfd].plagt[hr]);
		free(c[sfd].pllog[hr]);
	}
}

/*
 * main()
 *	nostromo webserver
 * Return:
 *	0 = terminated successfull, 1 = terminated with error
 */
int
main(int argc, char *argv[])
{
	int			sdlis1, sdlis2, sdlis3, sdlis4;
	int			sdmax, sdnew, sdnow, sdlim;
	int			i, j, k, r, s, ch, rd, rt, wr, post, size;
	int			fd, erro, pid, opt_c, opt_r, opt_4, opt_6;
	char			in[HDN * HDS + 1], tmp[HDS + 1];
	char			header[HDN][HDS + 1];
	char			*b, *h, *offset, *body = NULL;
	extern char		**environ;
	struct passwd		*pwd = NULL;
	struct group		*grp = NULL;
	struct stat		st;
	struct hostent		*host;
	struct sockaddr_in	sa, ca;
	struct sockaddr_in6	sa6, ca6;
	struct in6_addr		in6addr_any = IN6ADDR_ANY_INIT;
	struct timeval		tv;
	socklen_t		ssize;
	fd_set			master_r, master_w, set_r, set_w;
	time_t			tnow;
	SSL_CTX			*ssl_ctx = NULL;
	FILE			*file;

	i = r = s = rd = sdmax = sdnow = 0;
	sdlis1 = sdlis2 = sdlis3 = sdlis4 = -1;

	/*
	 * command line arguments
	 */
	opt_4 = 1;
	opt_6 = opt_c = opt_r = 0;
	while ((ch = getopt(argc, argv, "dhrv46c:")) != -1) {
		switch (ch) {
		case 'd':
			debug = 1;
			break;
		case 'h':
			usage(1);
			break;
		case 'r':
			opt_r = 1;
			break;
		case 'v':
			usage(2);
			break;
		case '4':
			opt_6 = 1;
			break;
		case '6':
			opt_4 = 0;
			opt_6 = 1;
			break;
		case 'c':
			strlcpy(config.file, optarg, sizeof(config.file));
			opt_c = 1;
			break;
		default:
			usage(0);
			/* NOTREACHED */
		}
	}
	/* if no custom configuration file is given take the default */
	if (!opt_c)
		strlcpy(config.file, configfile, sizeof(config.file));
	/* check if configuration file exists */
	if ((fd = open(config.file, O_RDONLY)) == -1)
		err(1, config.file);
	else
		close(fd);

	/*
	 * parse configuration file
	 */

	/* mandatory configuration parameters */
	if (fparse(config.servername, "servername", config.file,
	    sizeof(config.servername)) == -1) {
		fprintf(stderr, "Missing <servername> in config file.\n");
		r = -1;
	} else {
		/* get servers port */
		if (strcuts(config.serverportc, config.servername, ':', '\0',
		    sizeof(config.serverportc)) == -1) {
			/* default port */
			config.serverport = PRT;
			snprintf(config.serverportc, sizeof(config.serverportc),
			    "%d", PRT);
		} else
			/* custom port */
			config.serverport = atoi(config.serverportc);

		/* remove port from servername */
		strcuts(config.servername, config.servername, '\0', ':',
		    sizeof(config.servername));
	}
	if (fparse(config.serveradmin, "serveradmin", config.file,
	    sizeof(config.serveradmin)) == -1) {
		fprintf(stderr, "Missing <serveradmin> in config file.\n");
		r = -1;
	}
	if (fparse(config.serverroot, "serverroot", config.file,
	    sizeof(config.serverroot)) == -1) {
		fprintf(stderr, "Missing <serverroot> in config file.\n");
		r = -1;
	}
	if (fparse(config.servermimes, "servermimes", config.file,
	    sizeof(config.servermimes)) == -1) {
		fprintf(stderr, "Missing <servermimes> in config file.\n");
		r = -1;
	}
	if (fparse(config.logserver, "logserver", config.file,
	    sizeof(config.logserver)) == -1) {
		fprintf(stderr, "Missing <logserver> in config file.\n");
		r = -1;
	}
	if (fparse(config.logpid, "logpid", config.file,
	    sizeof(config.logpid)) == -1) {
		fprintf(stderr, "Missing <logpid> in config file.\n");
		r = -1;
	}
	if (fparse(config.logaccess, "logaccess", config.file,
	    sizeof(config.logaccess)) == -1) {
		fprintf(stderr, "Missing <logaccess> in config file.\n");
		r = -1;
	} else
		strlcpy(logn, config.logaccess, sizeof(logn));
	if (fparse(config.docroot, "docroot", config.file,
	    sizeof(config.docroot)) == -1) {
		fprintf(stderr, "Missing <docroot> in config file.\n");
		r = -1;
	}
	if (fparse(config.docindex, "docindex", config.file,
	    sizeof(config.docindex)) == -1) {
		fprintf(stderr, "Missing <docindex> in config file.\n");
		r = -1;
	}

	/* optional configuration parameters */
	if (fparse(config.user, "user", config.file, sizeof(config.user))
	    == -1)
		strlcpy(config.user, "0", sizeof(config.user));
	if (fparse(config.htaccess, "htaccess", config.file,
	    sizeof(config.htaccess)) == -1)
		strlcpy(config.htaccess, "0", sizeof(config.htaccess));
	if (fparse(config.htpasswd, "htpasswd", config.file,
	    sizeof(config.htpasswd)) == -1)
		strlcpy(config.htpasswd, "0", sizeof(config.htpasswd));
#ifdef __OpenBSD__
	else {
		if (strcmp(config.htpasswd, "+bsdauth") == 0)
			config.bsdauth = 1;
		else if (strcmp(config.htpasswd, "+bsdauthnossl") == 0)
			config.bsdauth = 2;
		else
			config.bsdauth = 0;
	}
#else
	config.bsdauth = 0;
#endif
	if (fparse(config.sslportc, "sslport", config.file,
	    sizeof(config.sslportc)) == -1)
		config.ssl = 0;
	else {
		config.ssl = 1;
		config.sslport = atoi(config.sslportc);
		if (fparse(config.sslcert, "sslcert", config.file,
		    sizeof(config.sslcert))  == -1) {
			fprintf(stderr, "Missing <sslcert> in config file.\n");
			r = -1;
		}
		if (fparse(config.sslcertkey, "sslcertkey", config.file,
		    sizeof(config.sslcertkey)) == -1) {
			fprintf(stderr, "Missing <sslcertkey> in config file."
			    "\n");
			r = -1;
		}
	}
	if (fparse(config.c401, "custom_401", config.file, sizeof(config.user))
	    == -1)
		strlcpy(config.c401, "0", sizeof(config.c401));
	if (fparse(config.c403, "custom_403", config.file, sizeof(config.user))
	    == -1)
		strlcpy(config.c403, "0", sizeof(config.c403));
	if (fparse(config.c404, "custom_404", config.file, sizeof(config.user))
	    == -1)
		strlcpy(config.c404, "0", sizeof(config.c404));
	if (fparse(config.homedirs, "homedirs", config.file,
	    sizeof(config.homedirs)) == -1)
		strlcpy(config.homedirs, "0", sizeof(config.homedirs));

	/* we need all mandatory parameters otherwise no start */
	if (r == -1)
		exit(1);

	/*
	 * change to serverroot directory
	 */
	if (chdir(config.serverroot) == -1)
		err(1, "chdir");

	/*
	 * get UID and GID
	 */
	if (config.user[0] != '0') {
		if ((pwd = getpwnam(config.user)) == NULL)
			errx(1, "%s: no such user.", config.user);
	}

	/*
	 * map mime types file
	 */
	stat(config.servermimes, &st);
	mimes_size = st.st_size;
	if ((fd = open(config.servermimes, O_RDONLY, 0)) == -1)
		err(1, config.servermimes);
	else {
		if ((mimes = mmap(NULL, mimes_size, PROT_READ,
		    MAP_FILE|MAP_SHARED, fd, 0)) == MAP_FAILED)
			err(1, "mmap");
		close(fd);
	}

	/*
	 * assemble some strings
	 */

	/* servers full url */
	if (config.serverport == PRT)
		snprintf(http_url, sizeof(http_url), "%s", config.servername);
	else
		snprintf(http_url, sizeof(http_url), "%s:%d",
		    config.servername, config.serverport);
	if (config.ssl && config.sslport == PRTS)
		snprintf(http_urls, sizeof(http_url), "%s", config.servername);
	if (config.ssl && config.sslport != PRTS)
		snprintf(http_urls, sizeof(http_url), "%s:%d",
		    config.servername, config.sslport);
	/* servers connection timeout */
	snprintf(http_fv_alv, sizeof(http_fv_alv), "timeout=%d, max=%d", TON,
	    TON);
	/* servers default content type */
	sys_mime(http_fv_cte, sizeof(http_fv_cte), mimes, mimes_size, "html");

	/*
	 * clear all environment variables and save PATH
	 */
	strlcpy(http_path, getenv("PATH"), sizeof(http_path));
	memset(environ, 0, sizeof(environ));

	/*
	 * get servers IP adress(es)
	 */
	if (opt_4) {
		if ((host = gethostbyname2(config.servername, AF_INET)) ==
		    NULL) {
			herror(config.servername);
			exit(1);
		}
		inet_ntop(AF_INET, host->h_addr, config.serverip4,
		    sizeof(config.serverip4));
	}
	if (opt_6) {
		if ((host = gethostbyname2(config.servername, AF_INET6)) ==
		    NULL) {
			herror(config.servername);
			exit(1);
		}
		inet_ntop(AF_INET6, host->h_addr, config.serverip6,
		    sizeof(config.serverip6));
	}

	/*
	 * SSL
	 */
	if (config.ssl) {
		/* initialize SSL library */
		SSL_load_error_strings();
		SSL_library_init();

		/* create SSL context */
		if ((ssl_ctx = SSL_CTX_new(SSLv23_server_method())) == NULL)
			err(1, "SSL_CTX_new");

		/* load certificate and private key and check them */
		if (SSL_CTX_use_certificate_file(ssl_ctx, config.sslcert,
		    SSL_FILETYPE_PEM) != 1)
			err(1, "SSL_CTX_use_certificate_file");
		if (SSL_CTX_use_PrivateKey_file(ssl_ctx, config.sslcertkey,
		    SSL_FILETYPE_PEM) != 1)
			err(1, "SSL_CTX_use_PrivateKey_file");
		if (SSL_CTX_check_private_key(ssl_ctx) != 1)
			err(1, "SSL_CTX_check_private_key");
	}

	/*
	 * initialize select sets
	 */
	FD_ZERO(&master_r);
	FD_ZERO(&master_w);
	FD_ZERO(&set_r);
	FD_ZERO(&set_w);

	/*
	 * IPv4
	 */
	if (opt_4) {
		/* socket */
		if ((sdlis1 = socket(AF_INET, SOCK_STREAM, 0)) == -1)
			err(1, "socket");

		/* set socket non-blocking */
		if (fcntl(sdlis1, F_SETFL, fcntl(sdlis1, F_GETFL) |
		    O_NONBLOCK) == -1)
			err(1, "fcntl");
		/* allow rapid reuse of the address */
		size = 1;
		if (setsockopt(sdlis1, SOL_SOCKET, SO_REUSEADDR, (char *) &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");

		/* bind */
		sa.sin_family = AF_INET;
		sa.sin_port = htons(config.serverport);
		sa.sin_addr.s_addr = INADDR_ANY;
		memset(&sa.sin_zero, 0, sizeof(sa.sin_zero));
		if (bind(sdlis1, (struct sockaddr *)&sa, sizeof(sa)) == -1)
			err(1, "bind");

		/* listen */
		if (listen(sdlis1, CON) == -1)
			err(1, "listen");

		/* add listener socket to master set */
		FD_SET(sdlis1, &master_r);

		/* keep track of the biggest socket descriptor */
		sdmax = sdlis1;
	}

	/*
	 * IPv4 SSL
	 */
	if (opt_4 && config.ssl) {
		/* socket */
		if ((sdlis2 = socket(AF_INET, SOCK_STREAM, 0)) == -1)
			err(1, "socket");

		/* set socket non-blocking */
		if (fcntl(sdlis2, F_SETFL, fcntl(sdlis2, F_GETFL) |
		    O_NONBLOCK) == -1)
			err(1, "fcntl");
		/* allow rapid reuse of the address */
		size = 1;
		if (setsockopt(sdlis2, SOL_SOCKET, SO_REUSEADDR, (char *) &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");

		/* bind */
		sa.sin_family = AF_INET;
		sa.sin_port = htons(config.sslport);
		sa.sin_addr.s_addr = INADDR_ANY;
		memset(sa.sin_zero, 0, sizeof(sa.sin_zero));
		if (bind(sdlis2, (struct sockaddr *)&sa, sizeof(sa)) == -1)
			err(1, "bind");

		/* listen */
		if (listen(sdlis2, CON) == -1)
			err(1, "listen");

		/* add ssl listener socket to master set */
		FD_SET(sdlis2, &master_r);

		/* keep track of the biggest socket descriptor */
		sdmax = sdlis2;
	}

	/*
	 * IPv6
	 */
	if (opt_6) {
		/* socket */
		if ((sdlis3 = socket(AF_INET6, SOCK_STREAM, 0)) == -1)
			err(1, "socket");

		/* set socket non-blocking */
		if (fcntl(sdlis3, F_SETFL, fcntl(sdlis3, F_GETFL) |
		    O_NONBLOCK) == -1)
			err(1, "fcntl");
		/* allow rapid reuse of the address */
		size = 1;
		if (setsockopt(sdlis3, SOL_SOCKET, SO_REUSEADDR, (char *) &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");

		/* bind */
		#ifndef __linux__
		sa6.sin6_len = sizeof(sa6);
		#endif
		sa6.sin6_family = AF_INET6;
		sa6.sin6_port = htons(config.serverport);
		sa6.sin6_flowinfo = 0;
		sa6.sin6_scope_id = 0;
		memcpy(sa6.sin6_addr.s6_addr, &in6addr_any,
		    sizeof(in6addr_any));
		if (bind(sdlis3, (struct sockaddr *)&sa6, sizeof(sa6)) == -1)
			err(1, "bind");

		/* listen */
		if (listen(sdlis3, CON) == -1)
			err(1, "listen");

		/* add ipv6 listener to master set */
		FD_SET(sdlis3, &master_r);

		/* keep track of the biggest socket descriptor */
		sdmax = sdlis3;
	}

	/*
	 * IPv6 SSL
	 */
	if (opt_6 && config.ssl) {
		/* socket */
		if ((sdlis4 = socket(AF_INET6, SOCK_STREAM, 0)) == -1)
			err(1, "socket");

		/* set socket non-blocking */
		if (fcntl(sdlis4, F_SETFL, fcntl(sdlis4, F_GETFL) |
		    O_NONBLOCK) == -1)
			err(1, "fcntl");
		/* allow rapid reuse of the address */
		size = 1;
		if (setsockopt(sdlis4, SOL_SOCKET, SO_REUSEADDR, (char *) &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");

		/* bind */
		#ifndef __linux__
		sa6.sin6_len = sizeof(sa6);
		#endif
		sa6.sin6_family = AF_INET6;
		sa6.sin6_port = htons(config.sslport);
		sa6.sin6_flowinfo = 0;
		sa6.sin6_scope_id = 0;
		memcpy(sa6.sin6_addr.s6_addr, &in6addr_any,
		    sizeof(in6addr_any));
		if (bind(sdlis4, (struct sockaddr *)&sa6, sizeof(sa6)) == -1)
			err(1, "bind");

		/* listen */
		if (listen(sdlis4, CON) == -1)
			err(1, "listen");

		/* add ipv6 listener to master set */
		FD_SET(sdlis4, &master_r);

		/* keep track of the biggest socket descriptor */
		sdmax = sdlis4;
	}

	/*
	 * calculate maximal concurrent connections
	 */
	sdlim = CON + sdmax;

	/*
	 * signal handler
	 */
	signal(SIGCHLD, sig_handler);
	signal(SIGTERM, sig_handler);
	signal(SIGINT, sig_handler);
	signal(SIGPIPE, sig_handler);
	signal(SIGHUP, sig_handler);
	signal(SIGQUIT, sig_handler);
	signal(SIGALRM, sig_handler);

	/*
	 * daemonize
	 */
	ppid = fork();
	if (ppid == -1)
		err(1, "fork");
	if (ppid > 0)
		exit(0);
	setsid();
	ppid = fork();
	if (ppid == -1)
		err(1, "fork");
	if (ppid > 0)
		exit(0);
	ppid = getpid();
	umask(022);
	close(0);
	close(1);
	close(2);
	if (open("/dev/null", O_RDWR) == -1) {
		flogd(config.logserver, "[error] open: %s\n", strerror(errno));
		exit(1);
	}
	dup(0);
	dup(0);

        /*
         * basic authentication via bsd authentication framework
         */
        if (config.bsdauth) {
		pipe(fdbsdauth);
		pid = fork();
		if (pid == -1)
			err(1, "fork");
		/* bsd auth daemon */
		if (pid == 0) {
			/* child closes all fds except pipe to parent */
			sys_close_except(fdbsdauth[1]);
			grp = getgrnam("auth");
			setgid(grp->gr_gid);
			if (config.user[0] != '0')
				setuid(pwd->pw_uid);
			sys_daemon_bsdauth();
			exit(0);
		}
		close(fdbsdauth[1]);
	}

	/*
	 * chroot to serverroot directory
	 */
	if (opt_r) {
		if (chroot(config.serverroot) == -1) {
			flogd(config.logserver, "[error] chroot: %s\n",
			    strerror(errno));
			exit(1);
		}
		strcuti(config.file, config.file, strlen(config.serverroot),
		    strlen(config.file), sizeof(config.file));
	}

	/*
	 * set UID and GID
	 */
	if (config.user[0] != '0') {
		if (setgid(pwd->pw_gid) == -1) {
			flogd(config.logserver, "[error] setgid: %s\n",
			    strerror(errno));
			exit(1);
		}
		if (setuid(pwd->pw_uid) == -1) {
			flogd(config.logserver, "[error] setuid: %s\n",
			    strerror(errno));
			exit(1);
		}
	}
	/* dont run as root */
	if (getuid() == 0) {
		flogd(config.logserver,
		    "[error] we dont run as root, choose another user\n");
		exit(1);
	}

	/*
	 * PID file creation
	 */
	if ((file = fopen(config.logpid, "w")) != NULL) {
		fclose(file);
		flog(config.logpid, "%d\n", ppid);
	} else {
		flogd(config.logserver, "[error] fopen: %s: %s\n",
		    config.logpid, strerror(errno));
		exit(1);
	}

	/*
	 * nostromo ready
	 */
	flogd(config.logserver, "[info] nhttpd started\n");
	if (config.ssl)
		flogd(config.logserver, "[info] SSL enabled on port %d\n",
		    config.sslport);
	if (opt_r)
		flogd(config.logserver, "[info] chroot to %s\n",
		    config.serverroot);
	if (opt_4 && opt_6)
		flogd(config.logserver, "[info] IPv4 and IPv6 enabled\n");
	if (!opt_4 && opt_6)
		flogd(config.logserver, "[info] IPv6 only enabled\n");
	if (debug)
		flogd(config.logserver, "[info] debug mode enabled\n");

	/*
	 * main loop
	 */
	while (!quit) {
		/* copy master set */
		set_r = master_r;
		set_w = master_w;

		/* select is waiting for some action */
		tv.tv_sec = TON;
		tv.tv_usec = 0;
		if (select(sdmax + 1, &set_r, &set_w, NULL, &tv) == -1) {
			if (errno == EINTR) {
				usleep(1);
				continue;
			}
			/* EFAULT, EBADF, and EINVAL are not acceptable */
			flogd(config.logserver, "[fatal] select: %s\n",
			    strerror(errno));
			break;
		}

		/* check all existing connections */
		for (sdnow = 3; sdnow <= sdmax; sdnow++) {
			/*
			 * Unused connection
			 */
			if (c[sdnow].to == 0 && sdnow != sdlis1 &&
			    sdnow != sdlis2 && sdnow != sdlis3 &&
			    sdnow != sdlis4)
				continue;

			/* check connection read and write state */
			rd = FD_ISSET(sdnow, &set_r);
			wr = FD_ISSET(sdnow, &set_w);

			/*
			 * existing connection idles, check for timeout
			 */
			if (rd == 0 && wr == 0) {
				if (sdnow != sdlis1 && sdnow != sdlis2 &&
				    sdnow != sdlis3 && sdnow != sdlis4 &&
				    c[sdnow].pfdo == 0 &&
				    (time(&tnow) - c[sdnow].to) > TON) {
					sys_log(config.logserver, debug,
					"main: socket closed by us: timeout\n");
					goto quit;
				}
				continue;
			}

			/*
			 * new connection
			 */
			if (rd > 0 && (sdnow == sdlis1 || sdnow == sdlis2 ||
			    sdnow == sdlis3 || sdnow == sdlis4)) {
				/* accept */
				if (sdnow == sdlis3 || sdnow == sdlis4) {
					ssize = sizeof(ca6);
					if ((sdnew = accept(sdnow,
					    (struct sockaddr *)&ca6, &ssize)) ==
					    -1)
						continue;
				} else {
					ssize = sizeof(ca);
					if ((sdnew = accept(sdnow,
					    (struct sockaddr *)&ca, &ssize)) ==
					    -1)
						continue;
				}

				/* set socket options */
				if (fcntl(sdnew, F_SETFL,
				    fcntl(sdnew, F_GETFL) | O_NONBLOCK) == -1)
					continue;
				if (SBS > 0) {
					size = SBS;
					if (setsockopt(sdnew, SOL_SOCKET,
					    SO_SNDBUF, &size, sizeof(size)) ==
					    -1)
						continue;
				}

				/* initialize connection structure */
				c[sdnew].to = time(&tnow);
				c[sdnew].rbuf = malloc((HDN * HDS) + 1);
				c[sdnew].wbuf = malloc(BS);
				c[sdnew].roff = 0;
				c[sdnew].wred = 0;
				c[sdnew].wsnd = 0;
				c[sdnew].pfdo = 0;
				c[sdnew].x_ssl = 0;
				c[sdnew].x_ip6 = 0;
				c[sdnew].x_sta = 0;

				/* SSL ready */
				if (sdnow == sdlis2 || sdnow == sdlis4) {
					c[sdnew].ssl = SSL_new(ssl_ctx);
					SSL_set_fd(c[sdnew].ssl, sdnew);
					c[sdnew].x_ssl = 1;
				}

				/* save ip and port */
				if (sdnow == sdlis3 || sdnow == sdlis4) {
					inet_ntop(AF_INET6, &ca6.sin6_addr,
					    c[sdnew].ip, sizeof(c[sdnew].ip));
					snprintf(c[sdnew].pt,
					    sizeof(c[sdnew].pt), "%d",
					    ca6.sin6_port);
					c[sdnew].x_ip6 = 1;
				} else {
					strlcpy(c[sdnew].ip,
					    inet_ntoa(ca.sin_addr),
					    sizeof(c[sdnew].ip));
					snprintf(c[sdnew].pt,
					    sizeof(c[sdnew].pt), "%d",
					    ca.sin_port);
				}

				/* check connection limit */
				if (sdnew > sdlim) {
					h = http_head(http_s_503, "-",
					    c[sdnew].ip, 0);
					b = http_body(http_s_503, "", h, 0);
					c[sdnew].pfdo++;
					c[sdnew].pfdn[i] = 1;
					c[sdnew].pfdh[i] = strdup(b);
					c[sdnew].x_ful[i] = 1;
					c[sdnew].x_chk[i] = 0;
					c[sdnew].x_sta = 0;
					free(h);
					free(b);
					FD_SET(sdnew, &master_w);
				}

				/* add new connection to read fdset */
				FD_SET(sdnew, &master_r);

				/* set highest socket */
				if (sdnew > sdmax)
					sdmax = sdnew;

				continue;
			}

			/*
			 * SSL handshake
			 */
			if (c[sdnow].x_ssl == 1) {
				r = SSL_accept(c[sdnow].ssl);
				if (r == 1) {
					c[sdnow].x_ssl = 2;
					continue;
				}
				erro = SSL_get_error(c[sdnow].ssl, r);
				if (erro == SSL_ERROR_WANT_READ)
					continue;
				if (erro == SSL_ERROR_WANT_WRITE)
					continue;
				/* SSL handshake error */
				goto quit;
			}

			/*
			 * active connection wants partial file send
			 */
			if (c[sdnow].pfdo > 0) {
				for (i = 0; i < HDN; i++) {
					if (c[sdnow].pfdn[i] == 0)
						continue;

					/* send plain block */
				 	if (c[sdnow].x_chk[i] == 0 &&
					    c[sdnow].x_ful[i] == 0)
						send_file(sdnow, i);
					/* send chunked block */
					if (c[sdnow].x_chk[i] == 1 &&
					    c[sdnow].x_ful[i] == 0)
						send_chunk(sdnow, i);
					/* send full response block */
					if (c[sdnow].x_ful[i] == 1 &&
					    c[sdnow].x_chk[i] == 0)
						send_full(sdnow, i);

					break;
				}
				/* all partial files sent */
				if (c[sdnow].pfdo == 0) {
					FD_CLR(sdnow, &master_w);
					if (c[sdnow].x_sta == 0)
						goto quit;
				}
				continue;
			}

			/*
			 * active connection sends request
			 */
			memset(tmp, 0, sizeof(tmp));

			/* size check for data */
 			if (c[sdnow].roff > (sizeof(in) - 1) - BS) {
				h = http_head(http_s_413, "-", c[sdnow].ip, 0);
				b = http_body(http_s_413, "", h, 0);
				c[sdnow].pfdo++;
				c[sdnow].pfdn[i] = 1;
				c[sdnow].pfdh[i] = strdup(b);
				c[sdnow].x_ful[i] = 1;
				c[sdnow].x_chk[i] = 0;
       				c[sdnow].x_sta = 0;
				free(h);
				free(b);
				continue;
			}

			/* receive data */
			offset = c[sdnow].rbuf + c[sdnow].roff;

			if (c[sdnow].x_ssl)
				r = sys_read_ssl(c[sdnow].ssl, offset, BS);
			else
				r = sys_read_a(sdnow, offset, BS);
			if (r < 1)
				goto quit;

			c[sdnow].roff += r;
			offset = c[sdnow].rbuf + c[sdnow].roff;
			*offset = '\0';

			/* set timestamp for connection timeout */
			c[sdnow].to = time(&tnow);

			/* we need a full header sequence */
			size = http_header_comp(c[sdnow].rbuf, c[sdnow].roff);
			if (!size)
				continue;

			/* we got a full header sequence */
			memcpy(in, c[sdnow].rbuf, c[sdnow].roff);
			rt = c[sdnow].roff;
			c[sdnow].roff = 0;

			/* size check for header count */
			if (size > HDN) {
				h = http_head(http_s_413, "-", c[sdnow].ip, 0);
				b = http_body(http_s_413, "", h, 0);
				c[sdnow].pfdo++;
				c[sdnow].pfdn[i] = 1;
				c[sdnow].pfdh[i] = strdup(b);
				c[sdnow].x_ful[i] = 1;
				c[sdnow].x_chk[i] = 0;
       				c[sdnow].x_sta = 0;
				free(h);
				free(b);
				continue;
			}

			/* post */
			if (!strncasecmp("POST ", in, 5))
				post = 1;
			else
				post = 0;

			/* separate every header */
			for (i = 0, j = 0, k = 0; i < rt; i++) {
				/* size check for header */
				if (j == sizeof(header[k]) - 1) {
					h = http_head(http_s_413, "-",
					    c[sdnow].ip, 0);
					b = http_body(http_s_413, "", h, 0);
					c[sdnow].pfdo++;
					c[sdnow].pfdn[i] = 1;
					c[sdnow].pfdh[i] = strdup(b);
					c[sdnow].x_ful[i] = 1;
					c[sdnow].x_chk[i] = 0;
       					c[sdnow].x_sta = 0;
					free(h);
					free(b);
					continue;
				}
				if (in[i] == '\n' && in[i + 2] == '\n') {
					tmp[j] = in[i];
					i = i + 2;
					strlcpy(header[k], tmp,
					    sizeof(header[k]));
					k++;
					j = 0;
					memset(tmp, 0, sizeof(tmp));
				} else {
					tmp[j] = in[i];
					j++;
				}
				/* we got the post header */
				if (post == 1 && k == 1)
					break;
			}

			/* for post get initial body content */
			if (post) {
				body = in + (strlen(header[0]) + 2);
				size = rt - (strlen(header[0]) + 2);
			}

			/* and process every single header */
			for (i = 0; i < k; i++) {
				s = 0;
				r = 1;
				s = http_verify(header[i], c[sdnow].ip, sdnow,
				    i);
				if (s == 1)
					r = http_proc(header[i], body, i, size,
					    sdnow);
				if (r == 0)
					goto quit;
				/* on error abort processing */
				if (r == -1) {
					strcutl(tmp, header[i], 1, sizeof(tmp));
					h = http_head(http_s_500, tmp,
					    c[sdnow].ip, 0);
					b = http_body(http_s_500, "", h, 0);
					c[sdnow].pfdo++;
					c[sdnow].pfdn[i] = 1;
					c[sdnow].pfdh[i] = strdup(b);
					c[sdnow].x_ful[i] = 1;
					c[sdnow].x_chk[i] = 0;
       					c[sdnow].x_sta = 0;
					free(h);
					free(b);
				}
			}

			/* connection has open files, add to write fdset */
			if (c[sdnow].pfdo > 0)
				FD_SET(sdnow, &master_w);

			/* check keep-alive */
			if (r != 1) {
				quit:
				if (c[sdnow].x_ssl)
					SSL_free(c[sdnow].ssl);
				c[sdnow].to = 0;
				c[sdnow].x_ip6 = 0;
				c[sdnow].x_ssl = 0;
				free(c[sdnow].rbuf);
				free(c[sdnow].wbuf);
				close(sdnow);
				FD_CLR(sdnow, &master_r);
				FD_CLR(sdnow, &master_w);
			}
		}
	}

	/*
	 * nostromo shutdown
	 */
	munmap(mimes, mimes_size);
	flogd(config.logserver, "[info] nhttpd stopped\n");
	kill(0, SIGTERM);	/* terminate all childs */

	return (0);
}
