/*
 * Copyright (c) 2004, 2005 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <dirent.h>
#include <errno.h>
#include <fcntl.h>
#include <openssl/ssl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "../libmy/str.h"
#include "../libmy/file.h"
#ifdef __linux__
#include <crypt.h>
#include "../libbsd/strlcpy.h"
#endif
#include "proto.h"
#include "extern.h"

/*
 * sys_mime()
 * 	searches mime type in our memory mapped mime list
 * Return:
 *	0 = mime not found, 1 = mime found
 */
int
sys_mime(char *dst, const int dsize, const char *mimes, const int msize,
    const char *type)
{
	int	i, j, k;
	char	*x, l[2][1024];
	
	for (i = 0, j = 0, k = 0; i <= msize; i++) {
		/* no control characters found, read */
		if (mimes[i] != ' ' && mimes[i] != '\t' && mimes[i] != '\n') {
			l[j][k] = mimes[i];
			k++;
		}
		/* second field starts */
		if (mimes[i] == ' ' || mimes[i] == '\t') {
			l[j][k] = '\0';
			j = 1;
			k = 0;
		}
		/* end of line */
		if (mimes[i] == '\n') {
			l[j][k] = '\0';
			for (x = strtok(l[1], ","); x; x = strtok(NULL, ",")) {
				if (strcasecmp(x, type) == 0) {
					strlcpy(dst, l[0], dsize);
					return 1;
				}
			}
			j = 0;
			k = 0;
		}
	}

	return 0;
}

/*
 * sys_access_auth()
 *	basic user authentication
 * Return:
 *	0 = authentication failed, 1 = authentication successfull, -1 = error
 */
int
sys_access_auth(const char *clientuser, const char *clientpw)
{
	int		i, r, fd, found;
	char		ch[1], *password;
	char		serveruserpw[1024], serveruser[1024], serverpw[1024];
	char		salt[2];

	i = r = found = 0;

	memset(serveruserpw, 0, sizeof(serveruserpw));

	/* open htpasswd */
	if ((fd = open(config.htpasswd, O_RDONLY)) == -1)
		return -1;

	/* search user in htpasswd */
	while (read(fd, ch, 1)) {
		if (ch[0] != '\n') {
			serveruserpw[i] = ch[0];
			i++;
		} else {
			strcuts(serveruser, serveruserpw, '\0', ':',
			    sizeof(serveruser));
			strcuts(serverpw, serveruserpw, ':', '\0',
			    sizeof(serverpw));
			if (strcmp(serveruser, clientuser) == 0) {
				found = 1;
				break;
			}
			memset(serveruserpw, 0, sizeof(serveruserpw));
			i = 0;
		}
	}

	/* user not found in htpasswd */
	if (found == 0) {
		close(fd);
		return r;
	}

	/* get DES password salt from htpasswd */
	salt[0] = serverpw[0];
	salt[1] = serverpw[1];

	/* generate DES encrypted password */
	password = crypt(clientpw, salt);

	/* compare DES encrypted passwords */
	if (strcmp(password, serverpw) == 0)
		r = 1;

	/* close htpasswd */
	close(fd);

	return r;
}

/*
 * sys_read()
 *	synchronous read
 * Return:
 *	<bytes> = success, 0 = end-of-file, -1 = error
 */
int
sys_read(const int sfd, char *buf, const int len)
{
	int		r;
	fd_set		seto_r;
	struct timeval	tv;

	FD_ZERO(&seto_r);
	FD_SET(sfd, &seto_r);
	while (1) {
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		r = select(sfd + 1, &seto_r, NULL, NULL, &tv);
		if (r == -1) {
			if (errno == EINTR)
				continue;
			return -1;
		}
		if (r == 0)
			return -1;
		r = read(sfd, buf, len);
		if (r == -1) {
			if (errno == EINTR)
				continue;
			return -1;
		}
		if (r == 0)
			break;
		if (r > 0)
			break;
	}

	return r;
}

/*
 * sys_write()
 *	synchronous write
 * Return:
 *	<bytes> = success, -1 = error
 */
int
sys_write(const int sfd, const char *buf, const int len)
{
	int		r, sent;
	fd_set		seto_w;
	struct timeval	tv;

	sent = 0;

	FD_ZERO(&seto_w);
	FD_SET(sfd, &seto_w);
	while (1) {
		tv.tv_sec = 1;
		tv.tv_usec = 0;
		r = select(sfd + 1, NULL, &seto_w, NULL, &tv);
		if (r == -1) {
			if (errno == EINTR)
				continue;
			return -1;
		}
		if (r == 0)
			return -1;
		r = write(sfd, buf + sent, len - sent);
		if (r == -1) {
			if (errno == EINTR)
				continue;
			return -1;
		}
		if (r > 0)
			sent += r;

		if (sent == len)
			break;
	}

	return sent;
}

/*
 * sys_close_except()
 *	close all file descriptors except fd and stdin, stdout, stderr
 * Return:
 *	0 = success, -1 = error
 */
int
sys_close_except(const int fd)
{
	int		i;
	struct rlimit	rlp;

	/* get maximal file limit */
	if (getrlimit(RLIMIT_NOFILE, &rlp) == -1)
		return -1;

	/* close all fds higher than me */
	for (i = fd + 1; i < rlp.rlim_cur; i++)
		close(i);

	/* close all fds lower than me, except stdin, stdout, stderr */
	for (i = fd - 1; i > 2; i--)
		close(i);

	return 0;
}

/*
 * sys_date()
 *	converts struct tm to a CLF conform date string
 * Return:
 *	pointer to date string = sucessfull, NULL = error
 */
char *
sys_date(struct tm *t)
{
	int		diff;
	char		*d, sign;
	static char	date[64];

	diff = t->tm_gmtoff / 3600;

	if (diff < 0) {
		diff = -diff;
		sign = '-';
	} else {
		sign = '+';
	}

	snprintf(date, sizeof(date), "[%02d/%s/%d:%02d:%02d:%02d %c%02d00]",
	    t->tm_mday, month[t->tm_mon], t->tm_year + 1900, t->tm_hour,
	    t->tm_min, t->tm_sec, sign, diff);

	d = date;

	return d;
}

/*
 * sys_benv()
 *	build an environment string and return its address so we can pass it
 *	to an environment pointer array
 * Return:
 *	pointer to environment string = successfull, NULL = error
 */
char *
sys_benv(const char *str, ...)
{
	char	buf[1024];
	char	*p;
	va_list	ap;

	va_start(ap, str);
	vsnprintf(buf, sizeof(buf), str, ap);
	va_end(ap);

	p = strdup(buf);

	return p;
}
