/*
 * Copyright (c) 2004, 2005 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <sys/mman.h>

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <openssl/ssl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h> 

#include "../libmy/str.h"
#include "../libmy/file.h"
#ifdef __linux__
#include "../libbsd/strlcpy.h"
#include "../libbsd/strlcat.h"
#endif
#include "config.h"
#include "proto.h"
#include "extern.h"

/*
 * Global vars local
 */
static int		ppid;
static const char	*configfile = "/var/nostromo/conf/nhttpd.conf";

/*
 * Global vars extern
 */
int	mimes_size;
char	*mimes;

const char *month[12] = {
	"Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct",
	"Nov", "Dec"
};

const char *day[7] = {
	"Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

const char *http_s_200		= "200 OK";
const char *http_s_206		= "206 Partial Content";
const char *http_s_301		= "301 Moved Permanently";
const char *http_s_302		= "302 Found";
const char *http_s_304		= "304 Not Modified";
const char *http_s_401		= "401 Unauthorized";
const char *http_s_403		= "403 Forbidden";
const char *http_s_404		= "404 Not Found";
const char *http_s_413		= "413 Request Entity Too Large";
const char *http_s_500		= "500 Internal Server Error";
const char *http_s_501		= "501 Not Implemented";
const char *http_s_503		= "503 Service Unavailable";
const char *http_fn_dat		= "Date:";
const char *http_fn_srv		= "Server:";
const char *http_fn_lmd		= "Last-Modified:";
const char *http_fn_clt		= "Content-Length:";
const char *http_fn_ims		= "If-Modified-Since:";
const char *http_fn_ref		= "Referer:";
const char *http_fn_agt		= "User-Agent:";
const char *http_fn_con		= "Connection:";
const char *http_fn_alv		= "Keep-Alive:";
const char *http_fn_cok		= "Cookie:";
const char *http_fn_teg		= "Transfer-Encoding:";
const char *http_fn_cte		= "Content-Type:";
const char *http_fn_loc		= "Location:";
const char *http_fn_hos		= "Host:";
const char *http_fn_aut		= "Authorization:";
const char *http_fn_auw		= "WWW-Authenticate:";
const char *http_fn_ran		= "Range:";
const char *http_fn_rac		= "Content-Range:";
const char *http_fv_srv		= "Nostromo/1.7.4 (Unix)";
const char *http_fv_pro		= "HTTP/1.1";
const char *http_fv_cgi		= "CGI/1.1";
const char *http_fv_teg		= "chunked";
const char *http_fv_lch		= "0\r\n\r\n";
const char *http_fv_con_alv	= "Keep-Alive";
const char *http_fv_con_cls	= "close";

char	http_fv_alv[128];
char	http_fv_cte[128];
char	http_url[128];
char	http_sig[128];
char	http_path[128];

struct cfg		config;
struct connection	c[CON];

/*
 * usage()
 *	print usage message
 * Return:
 *	none
 */
void
usage(int mode)
{
	extern char	*__progname;

	if (mode != 2) {
		fprintf(stderr, "usage: %s ", __progname);
		fprintf(stderr, "[-hrv46] [-c configfile]\n");
	}
	if (mode == 1) {
		fprintf(stderr, "\noptions:\n");
		fprintf(stderr, "  -h\t\t: This help.\n");
		fprintf(stderr, "  -v\t\t: Shows version.\n");
		fprintf(stderr,
		    "  -r\t\t: nhttpd will chroot to serverroot.\n");
		fprintf(stderr, "  -4\t\t: Enable IPv4 and IPv6.\n");
		fprintf(stderr, "  -6\t\t: Enable IPv6 only.\n");
		fprintf(stderr,
		    "  -c configfile\t: Use an alternate configfile.\n");
	}
	if (mode == 2)
		fprintf(stderr, "%s\n", http_fv_srv);

	exit(1);
}

/*
 * sig_handler()
 *	signal handler
 * Return:
 *	none
 */
void
sig_handler(const int sig)
{
	switch (sig) {
	case SIGCHLD: {
		int	status;

		while (waitpid(-1, &status, WNOHANG) > 0);

		break;
	}
	case SIGINT:
	case SIGTERM:
		/* terminate nhttpd gracefully */
		munmap(mimes, mimes_size);
		flogd(config.logserver, "[info] nhttpd stopped\n");
		exit(0);
		break;
	case SIGPIPE:
		/* ignore */
		break;
	case SIGHUP:
		/* ignore */
		break;
	case SIGQUIT:
		/* ignore */
		break;
	case SIGALRM:
		/* ignore */
		break;
	default:
		flogd(config.logserver, "[info] unhandled signal %i, ignored\n",
		    sig);
		break;
	}
}

/*
 * send_chunk()
 *	send a file in chunks
 * Return:
 *	none
 */
void
send_chunk(const int sfd, const int hr)
{
	int		r, s;
	char		*b;
	static char	fbuf[BS];

	r = s = 0;

	/* we need to initialize buf to do nice chunks */
	memset(fbuf, 0, sizeof(fbuf));

	/* read block from file */
	r = read(c[sfd].pfdn[hr], fbuf, BS);

	/* send a chunk */
	if (r > 0) {
		b = http_chunk(fbuf);
		if (c[sfd].x_ssl) {
			s = SSL_write(c[sfd].ssl, b, strlen(b));
		} else {
			s = sys_write(sfd, b, strlen(b));
		}
		free(b);
	}

	/* send last chunk */
	if (r < 1 || s < 1) {
		if (c[sfd].x_ssl) {
			s = SSL_write(c[sfd].ssl, http_fv_lch,
			    strlen(http_fv_lch));
		} else {
			s = sys_write(sfd, http_fv_lch, strlen(http_fv_lch));
		}
		close(c[sfd].pfdn[hr]);
		c[sfd].x_chk[hr] = 0;
		c[sfd].pfdo--;
	}
}

/*
 * send_file()
 *	send a file in blocks, the size is delivered in the header
 * Return:
 *	none
 */
void
send_file(const int sfd, const int hr)
{
	int		r, s;
	static char	fbuf[BS];

	r = s = 0;

	/* read block from file */
	r = read(c[sfd].pfdn[hr], fbuf, BS);

	/* send a block */
	if (r > 0) {
		if (c[sfd].x_ssl) {
			s = SSL_write(c[sfd].ssl, fbuf, r);
		} else {
			s = sys_write(sfd, fbuf, r);
		}
		if (s != -1)
			c[sfd].pfds[hr] += s;
	}

	/* file sent, write access log, cleanup */
	if (r < 1 || s < 1) {
		http_alog(sfd, hr);
		close(c[sfd].pfdn[hr]);
		c[sfd].pfdn[hr] = 0;
		c[sfd].pfds[hr] = 0;
		c[sfd].pfdo--;
		free(c[sfd].plreq[hr]);
		free(c[sfd].plref[hr]);
		free(c[sfd].plagt[hr]);
		free(c[sfd].pllog[hr]);
	}
}

/*
 * main()
 *	Nostromo webserver
 * Return:
 *	0 = terminated successfull, 1 = terminated with error
 */
int
main(int argc, char *argv[])
{
	int			sdlis1, sdlis2, sdlis3, sdlis4;
	int			sdmax, sdnew, sdnow;
	int			i, j, k, r, s, ch, rd, rp, rt, wr, post, size;
	int			fd, erro, mode, opt_c, opt_r, opt_4, opt_6;
	char			in[HDN * HDS + 1], tmp[HDS + 1];
	char			header[HDN][HDS + 1];
	char			*b, *h, *body = NULL;
	extern char		**environ;
	struct passwd		*pwd = NULL;
	struct stat		st;
	struct hostent		*host;
	struct sockaddr_in	sa, ca;
	struct sockaddr_in6	sa6, ca6;
	struct in6_addr		in6addr_any = IN6ADDR_ANY_INIT;
	struct timeval		tv;
	socklen_t		ssize;
	fd_set			master_r, master_w, set_r, set_w, seto_r;
	time_t			tnow;
	SSL_CTX			*ssl_ctx = NULL;
	FILE			*file;

	r = s = rd = sdmax = 0;
	sdlis1 = sdlis2 = sdlis3 = sdlis4 = -1;

	/*
	 * Command line arguments
	 */
	opt_4 = 1;
	opt_6 = opt_c = opt_r = 0;
	while ((ch = getopt(argc, argv, "hrv46c:")) != -1) {
		switch (ch) {
		case 'h':
			usage(1);
			break;
		case 'r':
			opt_r = 1;
			break;
		case 'v':
			usage(2);
			break;
		case '4':
			opt_6 = 1;
			break;
		case '6':
			opt_4 = 0;
			opt_6 = 1;
			break;
		case 'c':
			strlcpy(config.file, optarg, sizeof(config.file));
			opt_c = 1;
			break;
		default:
			usage(0);
			/* NOTREACHED */
		}
	}
	/* if no custom configuration file is given take the default */
	if (!opt_c)
		strlcpy(config.file, configfile, sizeof(config.file));
	/* check if configuration file exists */
	if ((fd = open(config.file, O_RDONLY)) == -1)
		err(1, config.file);
	else
		close(fd);

	/*
	 * Parse configuration file
	 */
	/* mandatory configuration parameters */
	if (fparse(config.servername, "servername", config.file,
	    sizeof(config.servername)) == -1) {
		fprintf(stderr, "Missing <servername> in config file.\n");
		r = -1;
	} else {
		/* get servers port */
		if (strcuts(config.serverportc, config.servername, ':', '\0',
		    sizeof(config.serverportc)) == -1) {
			/* default port */
			config.serverport = PRT;
			snprintf(config.serverportc, sizeof(config.serverportc),
			    "%d", PRT);
		} else {
			/* custom port */
			config.serverport = atoi(config.serverportc);
		}
		/* remove port from servername */
		strcuts(config.servername, config.servername, '\0', ':',
		    sizeof(config.servername));
	}
	if (fparse(config.serveradmin, "serveradmin", config.file,
	    sizeof(config.serveradmin)) == -1) {
		fprintf(stderr, "Missing <serveradmin> in config file.\n");
		r = -1;
	}
	if (fparse(config.serverroot, "serverroot", config.file,
	    sizeof(config.serverroot)) == -1) {
		fprintf(stderr, "Missing <serverroot> in config file.\n");
		r = -1;
	}
	if (fparse(config.servermimes, "servermimes", config.file,
	    sizeof(config.servermimes)) == -1) {
		fprintf(stderr, "Missing <servermimes> in config file.\n");
		r = -1;
	}
	if (fparse(config.logserver, "logserver", config.file,
	    sizeof(config.logserver)) == -1) {
		fprintf(stderr, "Missing <logserver> in config file.\n");
		r = -1;
	}
	if (fparse(config.logpid, "logpid", config.file,
	    sizeof(config.logpid)) == -1) {
		fprintf(stderr, "Missing <logpid> in config file.\n");
		r = -1;
	}
	if (fparse(config.logaccess, "logaccess", config.file,
	    sizeof(config.logaccess)) == -1) {
		fprintf(stderr, "Missing <logaccess> in config file.\n");
		r = -1;
	}
	if (fparse(config.docroot, "docroot", config.file,
	    sizeof(config.docroot)) == -1) {
		fprintf(stderr, "Missing <docroot> in config file.\n");
		r = -1;
	}
	if (fparse(config.docindex, "docindex", config.file,
	    sizeof(config.docindex)) == -1) {
		fprintf(stderr, "Missing <docindex> in config file.\n");
		r = -1;
	}
	if (fparse(config.cgiroot, "cgiroot", config.file,
	    sizeof(config.cgiroot)) == -1) {
		fprintf(stderr, "Missing <cgiroot> in config file.\n");
		r = -1;
	}
	if (fparse(config.cgialias, "cgialias", config.file,
	    sizeof(config.cgialias)) == -1) {
		fprintf(stderr, "Missing <cgialias> in config file.\n");
		r = -1;
	}
	/* optional configuration parameters */
	if (fparse(config.user, "user", config.file, sizeof(config.user))
	    == -1)
		strlcpy(config.user, "0", sizeof(config.user));
	if (fparse(config.htaccess, "htaccess", config.file,
	    sizeof(config.htaccess)) == -1)
		strlcpy(config.htaccess, "0", sizeof(config.htaccess));
	if (fparse(config.htpasswd, "htpasswd", config.file,
	    sizeof(config.htpasswd)) == -1)
		strlcpy(config.htpasswd, "0", sizeof(config.htpasswd));
	if (fparse(config.sslportc, "sslport", config.file,
	    sizeof(config.sslportc)) == -1) {
		config.ssl = 0;
	} else {
		config.ssl = 1;
		config.sslport = atoi(config.sslportc);
		if (fparse(config.sslcert, "sslcert", config.file,
		    sizeof(config.sslcert))  == -1) {
			fprintf(stderr, "Missing <sslcert> in config file.\n");
			r = -1;
		}
		if (fparse(config.sslcertkey, "sslcertkey", config.file,
		    sizeof(config.sslcertkey)) == -1) {
			fprintf(stderr, "Missing <sslcertkey> in config file."
			    "\n");
			r = -1;
		}
	}
	if (fparse(config.c401, "custom_401", config.file, sizeof(config.user))
	    == -1)
		strlcpy(config.c401, "0", sizeof(config.c401));
	if (fparse(config.c403, "custom_403", config.file, sizeof(config.user))
	    == -1)
		strlcpy(config.c403, "0", sizeof(config.c403));
	if (fparse(config.c404, "custom_404", config.file, sizeof(config.user))
	    == -1)
		strlcpy(config.c404, "0", sizeof(config.c404));
	/* we need all mandatory parameters otherwise no start */
	if (r == -1)
		exit(1);

	/*
	 * Change to serverroot directory
	 */
	if (chdir(config.serverroot) == -1)
		err(1, "chdir");

	/*
	 * Get UID and GID
	 */
	if (config.user[0] != '0') {
		if ((pwd = getpwnam(config.user)) == NULL)
			errx(1, "%s: no such user.", config.user);
	}

	/*
	 * Map mime types file
	 */
	stat(config.servermimes, &st);
	mimes_size = st.st_size;
	if ((fd = open(config.servermimes, O_RDONLY, 0)) == -1) {
		err(1, config.servermimes);
	} else {
		if ((mimes = mmap(NULL, mimes_size, PROT_READ,
		    MAP_FILE|MAP_SHARED, fd, 0)) == MAP_FAILED)
			err(1, "mmap");
		close(fd);
	}

	/*
	 * Assemble some data
	 */
	/* servers full url */
	if (config.serverport == PRT) {
		snprintf(http_url, sizeof(http_url), "%s", config.servername);
	} else {
		snprintf(http_url, sizeof(http_url), "%s:%d", config.servername,
		    config.serverport);
	}
	/* servers connection timeout */
	snprintf(http_fv_alv, sizeof(http_fv_alv), "timeout=%d, max=%d", TON,
	    TON);
	/* servers default content type */
	sys_mime(http_fv_cte, sizeof(http_fv_cte), mimes, mimes_size, "html");

	/*
	 * Clear all environment variables and save PATH
	 */
	strlcpy(http_path, getenv("PATH"), sizeof(http_path));
	memset(environ, 0, sizeof(environ));

	/*
	 * Get servers IP adress(es)
	 */
	if (opt_4) {
		if ((host = gethostbyname2(config.servername, AF_INET)) ==
		    NULL) {
			herror(config.servername);
			exit(1);
		}
		inet_ntop(AF_INET, host->h_addr, config.serverip4,
		    sizeof(config.serverip4));
	}
	if (opt_6) {
		if ((host = gethostbyname2(config.servername, AF_INET6)) ==
		    NULL) {
			herror(config.servername);
			exit(1);
		}
		inet_ntop(AF_INET6, host->h_addr, config.serverip6,
		    sizeof(config.serverip6));
	}

	/*
	 * SSL
	 */
	if (config.ssl) {
		/* initialize SSL library */
		SSL_load_error_strings();
		SSL_library_init();

		/* create SSL context */
		if ((ssl_ctx = SSL_CTX_new(SSLv2_server_method())) == NULL)
			err(1, "SSL_CTX_new");

		/* load certificate and private key and check them */
		if (SSL_CTX_use_certificate_file(ssl_ctx, config.sslcert,
		    SSL_FILETYPE_PEM) != 1)
			err(1, "SSL_CTX_use_certificate_file");
		if (SSL_CTX_use_PrivateKey_file(ssl_ctx, config.sslcertkey,
		    SSL_FILETYPE_PEM) != 1)
			err(1, "SSL_CTX_use_PrivateKey_file");
		if (SSL_CTX_check_private_key(ssl_ctx) != 1)
			err(1, "SSL_CTX_check_private_key");
	}

	/*
	 * Initialize select sets
	 */
	FD_ZERO(&master_r);
	FD_ZERO(&master_w);
	FD_ZERO(&set_r);
	FD_ZERO(&set_w);

	/*
	 * IPv4
	 */
	if (opt_4) {
		/* socket */
		if ((sdlis1 = socket(AF_INET, SOCK_STREAM, 0)) == -1)
			err(1, "socket");

		/* increase socket send buffer */
		size = HDN * BS;
		if (setsockopt(sdlis1, SOL_SOCKET, SO_SNDBUF, &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");
		/* allow rapid reuse of the address */
		size = 1;
		if (setsockopt(sdlis1, SOL_SOCKET, SO_REUSEADDR, (char *) &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");

		/* bind */
		sa.sin_family = AF_INET;
		sa.sin_port = htons(config.serverport);
		sa.sin_addr.s_addr = INADDR_ANY;
		memset(&sa.sin_zero, 0, sizeof(sa.sin_zero));
		if (bind(sdlis1, (struct sockaddr *)&sa, sizeof(sa)) == -1)
			err(1, "bind");

		/* listen */
		if (listen(sdlis1, CON) == -1)
			err(1, "listen");

		/* add listener socket to master set */
		FD_SET(sdlis1, &master_r);

		/* keep track of the biggest socket descriptor */
		sdmax = sdlis1;
	}

	/*
	 * IPv4 SSL
	 */
	if (opt_4 && config.ssl) {
		/* socket */
		if ((sdlis2 = socket(AF_INET, SOCK_STREAM, 0)) == -1)
			err(1, "socket");

		/* increase socket send buffer */
		size = HDN * BS;
		if (setsockopt(sdlis2, SOL_SOCKET, SO_SNDBUF, &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");
		/* allow rapid reuse of the address */
		size = 1;
		if (setsockopt(sdlis2, SOL_SOCKET, SO_REUSEADDR, (char *) &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");

		/* bind */
		sa.sin_family = AF_INET;
		sa.sin_port = htons(config.sslport);
		sa.sin_addr.s_addr = INADDR_ANY;
		memset(sa.sin_zero, 0, sizeof(sa.sin_zero));
		if (bind(sdlis2, (struct sockaddr *)&sa, sizeof(sa)) == -1)
			err(1, "bind");

		/* listen */
		if (listen(sdlis2, CON) == -1)
			err(1, "listen");

		/* add ssl listener socket to master set */
		FD_SET(sdlis2, &master_r);

		/* keep track of the biggest socket descriptor */
		sdmax = sdlis2;
	}

	/*
	 * IPv6
	 */
	if (opt_6) {
		/* socket */
		if ((sdlis3 = socket(AF_INET6, SOCK_STREAM, 0)) == -1)
			err(1, "socket");

		/* increase socket send buffer */
		size = HDN * BS;
		if (setsockopt(sdlis3, SOL_SOCKET, SO_SNDBUF, &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");
		/* allow rapid reuse of the address */
		size = 1;
		if (setsockopt(sdlis3, SOL_SOCKET, SO_REUSEADDR, (char *) &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");

		/* bind */
		#ifndef __linux__
		sa6.sin6_len = sizeof(sa6);
		#endif
		sa6.sin6_family = AF_INET6;
		sa6.sin6_port = htons(config.serverport);
		sa6.sin6_flowinfo = 0;
		sa6.sin6_scope_id = 0;
		memcpy(sa6.sin6_addr.s6_addr, &in6addr_any,
		    sizeof(in6addr_any));
		if (bind(sdlis3, (struct sockaddr *)&sa6, sizeof(sa6)) == -1)
			err(1, "bind");

		/* listen */
		if (listen(sdlis3, CON) == -1)
			err(1, "listen");

		/* add ipv6 listener to master set */
		FD_SET(sdlis3, &master_r);

		/* keep track of the biggest socket descriptor */
		sdmax = sdlis3;
	}

	/*
	 * IPv6 SSL
	 */
	if (opt_6 && config.ssl) {
		/* socket */
		if ((sdlis4 = socket(AF_INET6, SOCK_STREAM, 0)) == -1)
			err(1, "socket");

		/* increase socket send buffer */
		size = HDN * BS;
		if (setsockopt(sdlis4, SOL_SOCKET, SO_SNDBUF, &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");
		/* allow rapid reuse of the address */
		size = 1;
		if (setsockopt(sdlis4, SOL_SOCKET, SO_REUSEADDR, (char *) &size,
		    sizeof(size)) == -1)
			err(1, "setsockopt");

		/* bind */
		#ifndef __linux__
		sa6.sin6_len = sizeof(sa6);
		#endif
		sa6.sin6_family = AF_INET6;
		sa6.sin6_port = htons(config.sslport);
		sa6.sin6_flowinfo = 0;
		sa6.sin6_scope_id = 0;
		memcpy(sa6.sin6_addr.s6_addr, &in6addr_any,
		    sizeof(in6addr_any));
		if (bind(sdlis4, (struct sockaddr *)&sa6, sizeof(sa6)) == -1)
			err(1, "bind");

		/* listen */
		if (listen(sdlis4, CON) == -1)
			err(1, "listen");

		/* add ipv6 listener to master set */
		FD_SET(sdlis4, &master_r);

		/* keep track of the biggest socket descriptor */
		sdmax = sdlis4;
	}

	/*
	 * Signal Handler
	 */
	signal(SIGCHLD, sig_handler);
	signal(SIGTERM, sig_handler);
	signal(SIGINT, sig_handler);
	signal(SIGPIPE, sig_handler);
	signal(SIGHUP, sig_handler);
	signal(SIGQUIT, sig_handler);
	signal(SIGALRM, sig_handler);

	/*
	 * Daemonize
	 */
	ppid = getpid();
	ppid = fork();
	if (ppid == -1)
		err(1, "fork");
	if (ppid > 0)
		exit(0);
	setsid();
	ppid = fork();
	if (ppid == -1)
		err(1, "fork");
	if (ppid > 0)
		exit(0);
	ppid = getpid();
	umask(022);
	close(0);
	close(1);
	close(2);
	if (open("/dev/null", O_RDWR) == -1) {
		flogd(config.logserver, "[error] open: %s\n", strerror(errno));
		exit(1);
	}
	dup(0);
	dup(0);

	/*
	 * Chroot to serverroot directory
	 */
	if (opt_r) {
		if (chroot(config.serverroot) == -1) {
			flogd(config.logserver, "[error] chroot: %s\n",
			    strerror(errno));
			exit(1);
		}
		strcuti(config.file, config.file, strlen(config.serverroot),
		    strlen(config.file), sizeof(config.file));
	}

	/*
	 * Set UID and GID
	 */
	if (config.user[0] != '0') {
		if (setgid(pwd->pw_gid) == -1) {
			flogd(config.logserver, "[error] setgid: %s\n",
			    strerror(errno));
			exit(1);
		}
		if (setuid(pwd->pw_uid) == -1) {
			flogd(config.logserver, "[error] setuid: %s\n",
			    strerror(errno));
			exit(1);
		}
	}
	/* dont run as root */
	if (getuid() == 0) {
		flogd(config.logserver,
		    "[error] we dont run as root, choose another user\n");
		exit(1);
	}

	/*
	 * PID file creation
	 */
	if ((file = fopen(config.logpid, "w")) != NULL) {
		fclose(file);
		flog(config.logpid, "%d\n", ppid);
	} else {
		flogd(config.logserver, "[error] fopen: %s\n", strerror(errno));
		exit(1);
	}

	/*
	 * Nostromo ready
	 */
	flogd(config.logserver, "[info] nhttpd started\n");
	if (config.ssl)
		flogd(config.logserver, "[info] SSL enabled on port %d\n",
		    config.sslport);
	if (opt_r)
		flogd(config.logserver, "[info] chroot to %s\n",
		    config.serverroot);
	if (opt_4 && opt_6)
		flogd(config.logserver, "[info] IPv4 and IPv6 enabled\n");
	if (!opt_4 && opt_6)
		flogd(config.logserver, "[info] IPv6 only enabled\n");

	/*
	 * Main loop
	 */
	while (1) {
		/* copy master set */
		set_r = master_r;
		set_w = master_w;

		/* select is waiting for some action */
		tv.tv_sec = TON;
		tv.tv_usec = 0;
		if (select(sdmax + 1, &set_r, &set_w, NULL, &tv) == -1) {
			if (errno != EINTR)
				flogd(config.logserver, "[error] select: %s\n",
				    strerror(errno));
			continue;
		}

		/* check all existing connections */
		for (sdnow = 3; sdnow <= sdmax; sdnow++) {
			/*
			 * Unused connection
			 */
			if (c[sdnow].to == 0 && sdnow != sdlis1 &&
			    sdnow != sdlis2 && sdnow != sdlis3 &&
			    sdnow != sdlis4)
				continue;

			/* check connection read and write state */
			rd = FD_ISSET(sdnow, &set_r);
			wr = FD_ISSET(sdnow, &set_w);

			/*
			 * Existing connection idles, check for timeout
			 */
			if (rd == 0 && wr == 0) {
				if (sdnow != sdlis1 && sdnow != sdlis2 &&
				    sdnow != sdlis3 && sdnow != sdlis4 &&
				    c[sdnow].pfdo == 0 &&
				    (time(&tnow) - c[sdnow].to) > TON) {
					goto quit;
				}
				continue;
			}

			/*
			 * New connection
			 */
			if (rd > 0 && (sdnow == sdlis1 || sdnow == sdlis2 ||
			    sdnow == sdlis3 || sdnow == sdlis4)) {
				if (sdnow == sdlis3 || sdnow == sdlis4) {
					ssize = sizeof(ca6);
					if ((sdnew = accept(sdnow,
					    (struct sockaddr *)&ca6, &ssize)) ==
					    -1)
						continue;
				} else {
					ssize = sizeof(ca);
					if ((sdnew = accept(sdnow,
					    (struct sockaddr *)&ca, &ssize)) ==
					    -1)
						continue;
				}

				/* SSL ready */
				if (sdnow == sdlis2 || sdnow == sdlis4) {
					c[sdnew].ssl = SSL_new(ssl_ctx);
					SSL_set_fd(c[sdnew].ssl, sdnew);
					c[sdnew].x_ssl = 1;
				}

				/* save ip and port */
				if (sdnow == sdlis3 || sdnow == sdlis4) {
					inet_ntop(AF_INET6, &ca6.sin6_addr,
					    c[sdnew].ip, sizeof(c[sdnew].ip));
					snprintf(c[sdnew].pt,
					    sizeof(c[sdnew].pt), "%d",
					    ca6.sin6_port);
					c[sdnew].x_ip6 = 1;
				} else {
					strlcpy(c[sdnew].ip,
					    inet_ntoa(ca.sin_addr),
					    sizeof(c[sdnew].ip));
					snprintf(c[sdnew].pt,
					    sizeof(c[sdnew].pt), "%d",
					    ca.sin_port);
				}

				/* check connection limit */
				if (sdnew > CON) {
					h = http_head(http_s_503, "-",
					    c[sdnew].ip);
					b = http_body(http_s_503, "", h);
					if (c[sdnew].x_ssl) {
						SSL_write(c[sdnew].ssl, b,
						    strlen(b));
						SSL_free(c[sdnew].ssl);
						c[sdnew].x_ssl = 0;
					} else {
						sys_write(sdnew, b, strlen(b));
					}
					free(h);
					free(b);
					close(sdnew);
					continue;
				}

				/* add new connection to read fdset */
				FD_SET(sdnew, &master_r);

				/* set highest socket */
				if (sdnew > sdmax)
					sdmax = sdnew;

				/* set timestamp for connection timeout */
				c[sdnew].to = time(&tnow);

				continue;
			}

			/*
			 * SSL handshake
			 */
			if (c[sdnow].x_ssl == 1) {
				mode = fcntl(sdnow, F_GETFL);
				fcntl(sdnow, F_SETFL, O_NONBLOCK);
				r = SSL_accept(c[sdnow].ssl);
				fcntl(sdnow, F_SETFL, mode);
				if (r == 1) {
					c[sdnow].x_ssl = 2;
					continue;
				}
				erro = SSL_get_error(c[sdnow].ssl, r);
				if (erro == SSL_ERROR_WANT_READ)
					continue;
				if (erro == SSL_ERROR_WANT_WRITE)
					continue;
				/* SSL handshake error */
				goto quit;
			}

			/*
			 * Active connection wants partial file send
			 */
			if (c[sdnow].pfdo > 0) {
				for (i = 0; i < HDN; i++) {
					if (c[sdnow].pfdn[i] == 0)
						continue;
					/* send data chunked */
					if (c[sdnow].x_chk[i])
						send_chunk(sdnow, i);
					/* send data block */
					if (!c[sdnow].x_chk[i])
						send_file(sdnow, i);
				}
				/* all partial files sent */
				if (c[sdnow].pfdo == 0) {
					FD_CLR(sdnow, &master_w);
					if (c[sdnow].state == 0)
						goto quit;
				}
				continue;
			}

			/*
			 * Active connection sends request
			 */
			memset(tmp, 0, sizeof(tmp));

			/* receive data */
			if (c[sdnow].x_ssl)
				rt = SSL_read(c[sdnow].ssl, in, sizeof(in) - 1);
			else
				rt = sys_read(sdnow, in, sizeof(in) - 1);
			if (rt < 1)
				goto quit;
			in[rt] = '\0';

			/* set timestamp for connection timeout */
			c[sdnow].to = time(&tnow);

			/* receive everything */
			FD_ZERO(&seto_r);
			FD_SET(sdnow, &seto_r);
			while ((size = http_header_comp(in)) == 0) {
				tv.tv_sec = 1;
				tv.tv_usec = 0;
				r = select(sdnow + 1, &seto_r, NULL, NULL, &tv);
				if (r == -1) {
					if (errno == EINTR)
						continue;
					goto quit;
				}
				if (r == 0)
					goto quit;
				/* size check for data */
				if (rt == sizeof(in) - 1) {
					h = http_head(http_s_413, "-",
					    c[sdnow].ip);
					b = http_body(http_s_413, "", h);
					if (c[sdnow].x_ssl) {
						SSL_write(c[sdnow].ssl, b,
						    strlen(b));
					} else {
						sys_write(sdnow, b, strlen(b));
					}
					free(h);
					free(b);
					goto quit;
				}
				if (c[sdnow].x_ssl) {
					rp = SSL_read(c[sdnow].ssl, in + rt,
					    (sizeof(in) - rt) - 1);
				} else {
					rp = sys_read(sdnow, in + rt,
					    (sizeof(in) - rt) - 1);
				}
				if (rp < 1)
					goto quit;
				rt += rp;
			}
			in[rt] = '\0';

			/* size check for header count */
			if (size > HDN) {
				h = http_head(http_s_413, "-", c[sdnow].ip);
				b = http_body(http_s_413, "", h);
				if (c[sdnow].x_ssl) {
					SSL_write(c[sdnow].ssl, b, strlen(b));
				} else {
					sys_write(sdnow, b, strlen(b));
				}
				free(h);
				free(b);
				goto quit;
			}

			/* post */
			if (strncmp("POST ", in, 5) == 0)
				post = 1;
			else
				post = 0;

			/* separate every header */
			for (i = 0, j = 0, k = 0; i < rt; i++) {
				/* size check for header */
				if (j == sizeof(header[k]) - 1) {
					h = http_head(http_s_413, "-",
					    c[sdnow].ip);
					b = http_body(http_s_413, "", h);
					if (c[sdnow].x_ssl) {
						SSL_write(c[sdnow].ssl, b,
						    strlen(b));
					} else {
						sys_write(sdnow, b, strlen(b));
					}
					free(h);
					free(b);
					goto quit;
				}
				if (in[i] == '\n' && in[i + 2] == '\n') {
					tmp[j] = in[i];
					i = i + 2;
					strlcpy(header[k], tmp,
					    sizeof(header[k]));
					k++;
					j = 0;
					memset(tmp, 0, sizeof(tmp));
				} else {
					tmp[j] = in[i];
					j++;
				}
				/* we got the post header */
				if (post == 1 && k == 1)
					break;
			}

			/* for post get initial body content */
			if (post) {
				body = in + (strlen(header[0]) + 2);
				size = rt - (strlen(header[0]) + 2);
			}

			/* and process every single header */
			for (i = 0; i < k; i++) {
				r = http_verify(header[i], c[sdnow].ip, sdnow);
				if (r == 1)
					r = http_proc(header[i], body, i, size,
					    sdnow);
				/* on error abort processing */
				if (r == -1) {
					strcutl(tmp, header[i], 1, sizeof(tmp));
					h = http_head(http_s_500, tmp,
					    c[sdnow].ip);
					b = http_body(http_s_500, "", h);
					if (c[sdnow].x_ssl) {
						SSL_write(c[sdnow].ssl, b,
						    strlen(b));
					} else {
						sys_write(sdnow, b, strlen(b));
					}
					free(h);
					free(b);
					goto quit;
				}
			}

			/* connection has open files, add to write fdset */
			if (c[sdnow].pfdo > 0)
				FD_SET(sdnow, &master_w);

			/* check keep-alive */
			if (r != 1) {
				quit:
				if (c[sdnow].x_ssl)
					SSL_free(c[sdnow].ssl);
				c[sdnow].to = 0;
				c[sdnow].x_ip6 = 0;
				c[sdnow].x_ssl = 0;
				close(sdnow);
				FD_CLR(sdnow, &master_r);
			}
		}
	}

	return 0;
}
