/*
 * Copyright (c) 2004, 2005 Marcus Glocker <marcus@nazgul.ch>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/stat.h>

#include <errno.h>
#include <dirent.h>
#include <fcntl.h>
#include <openssl/ssl.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include "../libmy/str.h"
#include "../libmy/file.h"
#ifdef __linux__
#include "../libbsd/strlcpy.h"
#include "../libbsd/strlcat.h"
#endif
#include "config.h"
#include "proto.h"
#include "extern.h"

/*
 * Global vars local
 */
char	log_1[1024];
char	log_2[1024];
char	logn[1024];

static const struct {
	char	hex[3];
	char	sign;
} hexset[] = {
	{ "00", '\x00' }, { "01", '\x01' }, { "02", '\x02' }, { "03", '\x03' },
	{ "04", '\x04' }, { "05", '\x05' }, { "06", '\x06' }, { "07", '\x07' },
	{ "08", '\x08' }, { "09", '\x09' }, { "0a", '\x0a' }, { "0b", '\x0b' },
	{ "0c", '\x0c' }, { "0d", '\x0d' }, { "0e", '\x0e' }, { "0f", '\x0f' },
	{ "10", '\x10' }, { "11", '\x11' }, { "12", '\x11' }, { "13", '\x13' },
	{ "14", '\x14' }, { "15", '\x15' }, { "16", '\x16' }, { "17", '\x17' },
	{ "18", '\x18' }, { "19", '\x19' }, { "1a", '\x1a' }, { "1b", '\x1b' },
	{ "1c", '\x1c' }, { "1d", '\x1d' }, { "1e", '\x1e' }, { "1f", '\x1f' },
	{ "20", '\x20' }, { "21", '\x21' }, { "22", '\x22' }, { "23", '\x23' },
	{ "24", '\x24' }, { "25", '\x25' }, { "26", '\x26' }, { "27", '\x27' },
	{ "28", '\x28' }, { "29", '\x29' }, { "2a", '\x2a' }, { "2b", '\x2b' },
	{ "2c", '\x2c' }, { "2d", '\x2d' }, { "2e", '\x2e' }, { "2f", '\x2f' },
	{ "30", '\x30' }, { "31", '\x31' }, { "32", '\x32' }, { "33", '\x33' },
	{ "34", '\x34' }, { "35", '\x35' }, { "36", '\x36' }, { "37", '\x37' },
	{ "38", '\x38' }, { "39", '\x39' }, { "3a", '\x3a' }, { "3b", '\x3b' },
	{ "3c", '\x3c' }, { "3d", '\x3d' }, { "3e", '\x3e' }, { "3f", '\x3f' },
	{ "40", '\x30' }, { "40", '\x40' }, { "41", '\x41' }, { "42", '\x42' },
	{ "43", '\x43' }, { "44", '\x44' }, { "45", '\x45' }, { "46", '\x46' },
	{ "47", '\x47' }, { "48", '\x48' }, { "49", '\x49' }, { "4a", '\x4a' },
	{ "4b", '\x4b' }, { "4c", '\x4c' }, { "4d", '\x4d' }, { "4e", '\x4e' },
	{ "4f", '\x4f' }, { "50", '\x50' }, { "51", '\x51' }, { "52", '\x52' },
	{ "53", '\x53' }, { "54", '\x54' }, { "55", '\x55' }, { "56", '\x57' },
	{ "58", '\x58' }, { "59", '\x59' }, { "61", '\x61' }, { "62", '\x62' },
	{ "63", '\x64' }, { "65", '\x66' }, { "67", '\x68' }, { "69", '\x69' },
	{ "6a", '\x6a' }, { "6b", '\x6c' }, { "6d", '\x6e' }, { "6f", '\x6f' },
	{ "70", '\x70' }, { "71", '\x71' }, { "72", '\x72' }, { "73", '\x73' },
	{ "74", '\x75' }, { "75", '\x75' }, { "76", '\x76' }, { "77", '\x77' },
	{ "78", '\x78' }, { "79", '\x79' }, { "7a", '\x7a' }, { "7b", '\x7b' },
	{ "7c", '\x7c' }, { "7d", '\x7d' }, { "7e", '\x7e' }, { "7f", '\x7f' }
};

static const char *icd = "<img src=/icons/dir.gif>";
static const char *icf = "<img src=/icons/file.gif>";

/*
 * http_verify()
 *	verify if incoming header is valid
 * Return:
 *	0 = invalid header, 1 = valid header
 */
int
http_verify(const char *header, const char *cip, const int sfd)
{
	int		r, gmt;
	char		*h, *b, line[1024];
	time_t		tnow;
	struct tm	*t;

	r = 0;

	/* check for valid method */
	if (strcutl(line, header, 1, sizeof(line)) > 0) {
		if (strncmp("GET ", line, 4) == 0)
			r = 1;
		else if (strncmp("POST ", line, 5) == 0)
			r = 1;
		else if (strncmp("HEAD ", line, 5) == 0)
			r = 1;
	}
	
	if (r == 1) {
		/* access log 1 client ip, date, request string */
		time(&tnow);
		t = gmtime(&tnow);
		gmt = t->tm_hour;
		t = localtime(&tnow);
		snprintf(log_1, sizeof(log_1), "%s - - %s \"%s\" ", cip,
		    sys_date(t, gmt), line);
	} else {
		h = http_head(http_s_501, line, cip);
		b = http_body(http_s_501, "", h);
		if (c[sfd].x_ssl) {
			SSL_write(c[sfd].ssl, b, strlen(b));
		} else {
			sys_write(sfd, b, strlen(b));
		}
		free(h);
		free(b);
	}

	return r;
}

/*
 * http_proc()
 *	main function to process incoming header
 * Return:
 *	0 = close connection, 1 = keep alive connection, -1 = error
 */
int
http_proc(const char *header, char *body, const int hr, const int blen,
    const int sfd)
{
	DIR		*odir;
	int		cpid, file, fds1[2], fds2[2];
	int		i, r, s, bs, rd, sp, st, len, size;
	us_int		x_nph, x_fork, x_cpage;
	char		*h = NULL, *x = NULL;
	char		ch[1], image[128], tmp[1024], full[1024], status[1024];
	char		b[BS], cpage[1024];
	struct tm	*t = NULL;
	struct stat	sta;
	struct dirent	*rdir;
	struct header	*rh;
	struct timeval	tv;
	fd_set		o_read;

	r = sp = st = x_nph = x_fork = x_cpage = 0;
	bs = BS;

	/* parse request header */
	if ((rh = http_header(header, NULL, 0, blen, sfd)) == NULL)
		return -1;

	/* check parse response */
	if (rh->x_con)
		r = 1;
	if (rh->rp_status == 301) {
		x = http_body(http_s_301, "", rh->rp_header);
		if (c[sfd].x_ssl) {
			SSL_write(c[sfd].ssl, x, strlen(x));
		} else {
			sys_write(sfd, x, strlen(x));
		}
		free(x);
	}
	if (rh->rp_status == 304) {
		if (c[sfd].x_ssl) {
			SSL_write(c[sfd].ssl, rh->rp_header,
			    strlen(rh->rp_header));
		} else {
			sys_write(sfd, rh->rp_header, strlen(rh->rp_header));
		}
	}
	if (rh->rp_status == 401) {
		x = http_body(http_s_401, "", rh->rp_header);
		if (config.c401[0] != '0') {
			if (rh->x_dov) {
				snprintf(cpage, sizeof(cpage), "%s/%s",
				    rh->rq_docrootv, config.c401);
			} else {
				snprintf(cpage, sizeof(cpage), "%s/%s",
				    config.docroot, config.c401);
			}
			x_cpage = 1;
		} else {
			if (c[sfd].x_ssl) {
				SSL_write(c[sfd].ssl, x, strlen(x));
			} else {
				sys_write(sfd, x, strlen(x));
			}
			free(x);
		}
	}
	if (rh->rp_status == 403) {
		x = http_body(http_s_403, "", rh->rp_header);
		if (config.c403[0] != '0') {
			if (rh->x_dov) {
				snprintf(cpage, sizeof(cpage), "%s/%s",
				    rh->rq_docrootv, config.c403);
			} else {
				snprintf(cpage, sizeof(cpage), "%s/%s",
				    config.docroot, config.c403);
			}
			x_cpage = 1;
		} else {
			if (c[sfd].x_ssl) {
				SSL_write(c[sfd].ssl, x, strlen(x));
			} else {
				sys_write(sfd, x, strlen(x));
			}
			free(x);
		}
	}
	if (rh->rp_status == 404) {
		x = http_body(http_s_404, "", rh->rp_header);
		if (config.c404[0] != '0') {
			if (rh->x_dov) {
				snprintf(cpage, sizeof(cpage), "%s/%s",
				    rh->rq_docrootv, config.c404);
			} else {
				snprintf(cpage, sizeof(cpage), "%s/%s",
				    config.docroot, config.c404);
			}
			x_cpage = 1;
		} else  {
			if (c[sfd].x_ssl) {
				SSL_write(c[sfd].ssl, x, strlen(x));
			} else {
				sys_write(sfd, x, strlen(x));
			}
			free(x);
		}
	}

	/*
	 * Custom response
	 */
	if (x_cpage) {
		/* open custom file */
		if ((file = open(cpage, O_RDONLY, 0)) == -1) {
			/* can not open custom file, send default page */
			if (c[sfd].x_ssl) {
				SSL_write(c[sfd].ssl, x, strlen(x));
			} else {
				sys_write(sfd, x, strlen(x));
			}
			free(x);
		} else {
			free(x);
			/* custom file opened, send it */
			stat(cpage, &sta);
			memset(b, 0, sizeof(b));
			read(file, b, bs);
			x = http_chunk(b);

			size = rh->rp_hsize + strlen(x) + strlen(http_fv_lch)
			    + 1;

			h = malloc(size);
			strlcpy(h, rh->rp_header, size);
			strlcat(h, x, size);
			if (sta.st_size <= bs) {
				strlcat(h, http_fv_lch, size);
			} else {
				c[sfd].pfdn[hr] = file;
				c[sfd].x_chk[hr] = 1;
				c[sfd].pfdo++;
			}

			if (c[sfd].x_ssl) {
				SSL_write(c[sfd].ssl, h, strlen(h));
			} else {
				sys_write(sfd, h, strlen(h));
			}

			free(h);
			free(x);
			close(file);
			r = 1;
		}
	}

	/*
	 * File
	 */
	if ((rh->rp_status == 200 || rh->rp_status == 206) && rh->x_cgi == 0 &&
	    rh->x_idx == 0 && strcmp(rh->rq_method, "HEAD") != 0) {
		/* open requested file */
		if ((file = open(rh->rq_filef, O_RDONLY, 0)) == -1) {
			free(rh);
			return -1;
		}

		/* allocate memory for header and data */
		if ((h = malloc(rh->rp_hsize + bs)) == NULL) {
			free(rh);
			close(file);
			return -1;
		}

		/* send header and a file block */
		strlcpy(h, rh->rp_header, rh->rp_hsize + bs);
		if (rh->rp_status == 206)
			lseek(file, rh->rp_foffs, SEEK_SET);
		rd = read(file, h + strlen(h), bs);
		if (c[sfd].x_ssl) {
			if ((sp = SSL_write(c[sfd].ssl, h, rh->rp_hsize + rd))
			    == -1) {
				free(h);
				free(rh);
				close(file);
				return -1;
			}
		} else {
			if ((sp = sys_write(sfd, h, rh->rp_hsize + rd)) == -1) {
				free(h);
				free(rh);
				close(file);
				return -1;
			}
		}
		st += sp - rh->rp_hsize;
		free(h);

		/* check if whole file was sent or not */
		if (rh->rp_fsize <= bs) {
			/* whole file sent, close it, write access log */
			close(file);
		} else {
			/* first file block sent, save infos and return */
			c[sfd].pfdo++;
			c[sfd].state = rh->x_con;
			c[sfd].pfdn[hr] = file;
			c[sfd].pfds[hr] = st;
			c[sfd].status[hr] = rh->rp_status;
			c[sfd].plreq[hr] = malloc(128);
			c[sfd].plref[hr] = malloc(128);
			c[sfd].plagt[hr] = malloc(128);
			c[sfd].pllog[hr] = malloc(128);
			snprintf(c[sfd].plreq[hr], 128, "%s %s %s",
			    rh->rq_method, rh->rq_uri, rh->rq_protocol);
			if (rh->x_ref == 0) {
				c[sfd].plref[hr][0] = '-';
				c[sfd].plref[hr][1] = '\0';
			} else {
				strlcpy(c[sfd].plref[hr], rh->rq_fv_ref, 128);
			}
			if (rh->x_agt == 0) {
				c[sfd].plref[hr][0] = '-';
				c[sfd].plref[hr][1] = '\0';
			} else {
				strlcpy(c[sfd].plagt[hr], rh->rq_fv_agt, 128);
			}
			strlcpy(c[sfd].pllog[hr], logn, 128);

			free(rh);
			return 1;
		}
	}

	/*
	 * File header
	 */
	if ((rh->rp_status == 200 || rh->rp_status == 206) && rh->x_cgi == 0 &&
	    rh->x_idx == 0 && strcmp(rh->rq_method, "HEAD") == 0) {
		/* send header */
		sys_write(sfd, rh->rp_header, rh->rp_hsize);

		/* no body has been sent */
		st = 0;
	}

	/*
	 * CGI
	 */
	if (rh->rp_status == 200 && rh->x_cgi == 1) {
		/* fork child for nhttpd */
		if ((cpid = fork()) == -1) {
			flogd(config.logserver, "[error] can't fork nhttpd "
			    "child for cgi: fork: %s\n", strerror(errno));
			free(rh);
			return -1;
		}
		if (cpid > 0) {
			free(rh);
			return 0;
		}
		x_fork = 1;

		/* close all copied fds from parent */
		sys_close_except(sfd);

		/* create pipes to communicate with cgi */
		pipe(fds1);
		pipe(fds2);

		/* fork child for cgi */
		if ((cpid = fork()) == -1) {
			flogd(config.logserver, "[error] can't fork cgi: "
			    "fork: %s\n", strerror(errno));
			exit(1);
		}

		/* cgi */
		if (cpid == 0) {
			/* child dont need those fds */
			close(fds1[1]);
			close(fds2[0]);

			chdir(rh->rq_filep);
			dup2(fds1[0], STDIN_FILENO);
			dup2(fds2[1], STDOUT_FILENO);
			execl(rh->rq_filef, rh->rq_files, (char *)NULL);
			exit(0);
		}

		/* parent dont need those fds */
		close(fds1[0]);
		close(fds2[1]);

		/* if post send data to cgis stdin */
		if (strcmp(rh->rq_method, "POST") == 0)
			sys_write(fds1[1], body, blen);
		/* close fd to cgi stdin */
		close(fds1[1]);

		/* initialize */
		memset(b, 0, sizeof(b));

		/* parse cgi header */
		for (i = 0; i < sizeof(b); i++) {
			if ((rd = read(fds2[0], ch, 1)) < 1) {
				flogd(config.logserver, "[error] %s "
				    "sent a bad cgi header\n", rh->rq_uri);
				exit(1);
			}
			b[i] = ch[0];
			/* cgi header received */
			if (b[i] == '\n' && (b[i - 1] == '\n' ||
			    b[i - 2] == '\n')) {
				if ((s = http_cgi_header(b, status, sizeof(b),
				    sizeof(status))) > 0) {
					free(rh);
					rh = http_header(header, status, s,
					    blen, sfd);
				}
				len = rh->rp_hsize + strlen(b) + 1;
				if ((x = malloc(len)) == NULL) {
					flogd(config.logserver,
					    "[error] can't malloc memory for "
					    "cgi header: malloc: %s",
					    strerror(errno));
					exit(1);
				}
				/* check if it is a nph cgi */
				if (strncasecmp("nph-", rh->rq_files, 4) == 0) {
					strlcpy(x, b, len);
					x_nph = 1;
				} else {
					strlcpy(x, rh->rp_header, len);
					strlcat(x, b, len);
				}
				if (c[sfd].x_ssl) {
					if ((sp = SSL_write(c[sfd].ssl, x,
					    strlen(x))) < 1)
						exit(1);
				} else {
					if ((sp = sys_write(sfd, x, strlen(x)))
					    < 1)
						exit(1);
				}
				free(x);
				break;
			}
		}

		/* select */
		FD_ZERO(&o_read);
		FD_SET(fds2[0], &o_read);

		/* cgi loop */
		while (1) {
			tv.tv_sec = TOC;
			tv.tv_usec = 0;
			if (select(fds2[0] + 1, &o_read, NULL, NULL, &tv) != 1)
				/* timeout or error */
				break;

			/* data */
			memset(b, 0, sizeof(b));
			if ((rd = read(fds2[0], b, sizeof(b))) < 1)
				break;
			if (x_nph)
				x = b;
			else
				x = http_chunk(b);
			if (c[sfd].x_ssl) {
				if ((sp = SSL_write(c[sfd].ssl, x, strlen(x)))
				    < 1)
					break;
			} else {
				if ((sp = sys_write(sfd, x, strlen(x))) < 1)
					break;
			}
			st += sp;
			if (!x_nph)
				free(x);
		}

		/* send last chunk */
		if (!x_nph) {
			if (c[sfd].x_ssl) {
				if ((sp = SSL_write(c[sfd].ssl, http_fv_lch,
			    	    strlen(http_fv_lch))) != -1)
				st += sp;
			} else {
				if ((sp = sys_write(sfd, http_fv_lch,
			    	    strlen(http_fv_lch))) != -1)
				st += sp;
			}
		}

		/* close fd to cgis stdout */
		close(fds2[0]);

		/* be sure that cgi is gone */
		kill(cpid, SIGKILL);
	}

	/*
	 * Directory listing
	 */
	if (rh->rp_status == 200 && rh->x_idx == 1) {
		/* fork child for nhttpd */
		if ((cpid = fork()) == -1) {
			flogd(config.logserver, "[error] can't fork nhttpd "
			    "child for dirlist: fork: %s\n", strerror(errno));
			free(rh);
			return -1;
		}
		if (cpid > 0) {
			free(rh);
			return 0;
		}
		x_fork = 1;

		/* close all copied fds from parent */
		sys_close_except(sfd);

		/* send header */
		if (c[sfd].x_ssl) {
			SSL_write(c[sfd].ssl, rh->rp_header,
			    strlen(rh->rp_header));
		} else {
			sys_write(sfd, rh->rp_header, strlen(rh->rp_header));
		}

		/* open directory */
		if ((odir = opendir(rh->rq_filep)) == NULL)
			exit(1);

		/* initialize */
		memset(b, 0, sizeof(b));

		/* create html title */
		snprintf(b, sizeof(b), "<html>\n<head><title>Index of %s"
		    "</title></head>\n<body>\n<h1>Index of %s</h1>\n<hr>\n"
		    "<table cellpadding=2 cellspacing=5>\n"
		    "<tr><td><b>Type</b></td><td><b>Filename</b></td>"
		    "<td><b>Last Modified</b></td><td><b>Size</b></td></tr>\n",
		    rh->rq_index, rh->rq_index);

		/* list directory content */
		while ((rdir = readdir(odir)) != NULL) {
			/* we dont list hidden files */
			if (rdir->d_name[0] == '.')
				continue;

			/* create full file path */
			snprintf(full, sizeof(full), "%s%s", rh->rq_filep,
			    rdir->d_name);

			/* get file status */
			stat(full, &sta);
			/* status: directory or file */
			if (sta.st_mode & S_IFDIR)
				strlcpy(image, icd, sizeof(image));
			else
				strlcpy(image, icf, sizeof(image));
			/* status: last modification time */
			t = localtime(&sta.st_mtime);

			/* create html file entry */
			snprintf(tmp, sizeof(tmp),
			    "<tr><td>%s</td><td><a href=\"%s\">%s</a>"
			    "</td><td>%s</td><td>%d</td></tr>\n",
			    image, rdir->d_name, rdir->d_name, http_date(t),
			    (int)sta.st_size);

			/* buffer full? send it! */
			if (strlen(tmp) > bs - strlen(b)) {
				x = http_chunk(b);
				if (c[sfd].x_ssl) {
					if ((sp = SSL_write(c[sfd].ssl, x,
					    strlen(x))) == -1)
						break;
				} else {
					if ((sp = sys_write(sfd, x, strlen(x)))
					     == -1)
						break;
				}
				st += sp;
				free(x);
				memset(b, 0, sizeof(b));
			}

			/* fill buffer */
			strlcat(b, tmp, sizeof(b));
		}

		/* close directory */
		closedir(odir);

		if (sp != -1) {
			/* flush buffer */
			x = http_chunk(b);
			if (c[sfd].x_ssl) {
				if ((sp = SSL_write(c[sfd].ssl, x, strlen(x)))
					!= 1)
					st += sp;
			} else {
				if ((sp = sys_write(sfd, x, strlen(x))) != 1)
					st += sp;
			}
			memset(b, 0, sizeof(b));
			free(x);

			/* create html footer */
			snprintf(b, bs + 1, "</table>\n<hr>\n%s\n</body>\n"
			    "</html>", getenv("SERVER_SIGNATURE"));
			x = http_chunk(b);
			if (c[sfd].x_ssl) {
				if ((sp = SSL_write(c[sfd].ssl, x, strlen(x)))
				    != -1)
					st += sp;
			} else {
				if ((sp = sys_write(sfd, x, strlen(x))) != -1)
					st += sp;
			}
			free(x);

			/* send last chunk */
			if (c[sfd].x_ssl) {
				if ((sp = SSL_write(c[sfd].ssl, http_fv_lch,
				    strlen(http_fv_lch))) != -1)
					st += sp;
			} else {
				if ((sp = sys_write(sfd, http_fv_lch,
				    strlen(http_fv_lch))) != -1)
					st += sp;
			}
		}
	}

	/* access log 1 */
	if (rh->rp_status == 200 || rh->rp_status == 206) {
		snprintf(tmp, sizeof(tmp), "%d %d", rh->rp_status, st);
		strlcat(log_1, tmp, sizeof(log_1));
	} else { 
		snprintf(tmp, sizeof(tmp), "%d -", rh->rp_status);
		strlcat(log_1, tmp, sizeof(log_1));
	}

	/* access log 2 referer */
	if (rh->x_ref == 0)
		snprintf(log_2, sizeof(log_2), "\"-\" ");
	else
		snprintf(log_2, sizeof(log_2), "\"%s\" ", rh->rq_fv_ref);

	/* access log 2 user agent */
	if (rh->x_agt == 0) {
		strlcat(log_2, "\"-\"", sizeof(log_2));
	} else {
		snprintf(tmp, sizeof(tmp), "\"%s\"", rh->rq_fv_agt);
		strlcat(log_2, tmp, sizeof(log_2));
	}

	/* access log write */
	flog(logn, "%s %s\n", log_1, log_2);

	/* nhttpd child exits */
	if (x_fork)
		exit(0);

	free(rh);
	return r;
}

/*
 * http_cgi_getexec()
 * 	extracts requested cgi program path and options from URI
 * Return:
 *	0 = success without option, 1 = success with option, -1 = no prg found
 */
int
http_cgi_getexec(char *dst1, char *dst2, const char *src, const int dsize1,
    const int dsize2)
{
	char		*p, *source;
	char		file[1024], tmp[1024];
	struct stat	s;

	memset(file, 0, sizeof(file));

	/* we dont want to change src */
	if ((source = strdup(src)) == NULL)
		return -1;

	for (p = strtok(source, "/"); p; p = strtok(NULL, "/")) {
		snprintf(tmp, sizeof(tmp), "/%s", p);
		strlcat(file, tmp, sizeof(file));
		stat(file, &s);
		if (s.st_mode & S_IFDIR) {
			continue;
		} else {
			strlcpy(dst1, file, dsize1);
			if (strcuti(dst2, src, strlen(file), strlen(src),
			    dsize2) == 0) {
				free(source);
				return 1;
			} else {
				free(source);
				return 0;
			}
		}
	}

	free(source);

	return -1;
}

/*
 * http_cgi_header()
 *	parses header created by executed cgi and formats it.
 *	scans header for the 'status' option.
 * Return:
 *	0 = success, <status> = success and status field found
 */
int
http_cgi_header(char *header_cgi, char *status, const int dsize1,
    const int dsize2)
{
	int	i, j, r;
	char	line[1024], option[1024], tmp[1024], header_cgi_new[1024];
	
	r = 0;

	memset(header_cgi_new, 0, sizeof(header_cgi_new));

	j = strcutl(line, header_cgi, 1, sizeof(line));

	for (i = 1; i <= j; i++) {
		strcutl(line, header_cgi, i, sizeof(line));
		strcutw(option, line, 1, sizeof(option));

		/* cut out Status: */
		if (strcasecmp(option, "Status:") != 0) {
			snprintf(tmp, sizeof(tmp), "%s\r\n", line);
			strlcat(header_cgi_new, tmp, sizeof(header_cgi_new));
		} else {
			strcuts(status, line, ' ', '\0', dsize2);
			r = 1;
		}
	}

	/* returns formated cgi header */
	strlcpy(header_cgi, header_cgi_new, dsize1);

	/* convert status number to integer */
	if (r == 1) {
		strcutw(tmp, status, 1, sizeof(tmp));
		r = atoi(tmp);
	}

	return r;
}

/*
 * http_header_comp()
 * 	check if received headers arrived complete
 * Return:
 * 	0 = headers not complete, <nr of headers> = headers complete
 */
int
http_header_comp(char *header)
{
	int	r, foot, head;
	char	*p;

	r = foot = head = 0;

	/* count all headers */
	p = header;
	while ((p = strstr(p, "HTTP/")) != NULL) {
		head++;
		p = p + 5;
	}

	/* count all footers */
	p = header;
	while ((p = strstr(p, "\r\n\r\n")) != NULL) {
		foot++;
		p = p + 4;
		if (strncmp("POST ", header, 5) == 0)
			break;
	}

	/* headers match footers */
	if (head == foot)
		r = head;

	return r;
}

/*
 * http_body_comp()
 *	check if received body arrived complete
 * Return:
 *	0 = body not complete, <bytes of body> = body complete
 */
int
http_body_comp(const char *header, char *body, const int blen)
{
	int	i, r, hlen;
	char	line[1024], option[1024], clength[16];
	
	r = 0;

	/* get content length from header */
	for (i = 2; strcutl(line, header, i, sizeof(line)) != -1; i++) {
		strcutw(option, line, 1, sizeof(option));
		if (strcasecmp(option, http_fn_clt) == 0) {
			strcuts(clength, line, ' ', '\0', sizeof(clength));
			break;
		}
	}
	hlen = atoi(clength);

	/* handle the post body termination mess */
	if (blen >= hlen) {
		if (body[blen - 2] == '\r')
			r = blen - 2;
		else if (body[blen - 1] == '\n')
			r = blen - 1;
		else
			r = blen;
	}

	return r;
}

/*
 * http_access_htaccess()
 *	searches for htaccess file in every directory of *rootdir
 *	if found the full htaccess location is returned to *dst
 * Return:
 *	0 = htaccess not found, 1 = htaccess found, -1 = error
 */
int
http_access_htaccess(char *dst, const char *rootdir, const int dsize)
{
	char		*dir, *rootdir_copy;
	char		path[1024], file[1024], tmp[1024];
	struct stat	s;

	/* config: htaccess */
	if (config.htaccess[0] == '0')
		return 0;

	memset(path, 0, sizeof(path));

	/* we dont want to change rootdir */
	if ((rootdir_copy = strdup(rootdir)) == NULL)
		return -1;

	/* search htaccess file */
	for (dir = strtok(rootdir_copy, "/"); dir; dir = strtok(NULL, "/")) {
		snprintf(tmp, sizeof(tmp), "/%s", dir);
		strlcat(path, tmp, sizeof(path));
		snprintf(file, sizeof(file), "%s/%s", path, config.htaccess);
		if (stat(file, &s) == 0) {
			strlcpy(dst, file, dsize);
			free(rootdir_copy);
			return 1;
		}
	}

	free(rootdir_copy);
	return 0;
}

/*
 * http_alog()
 *	write access log after partitial file send
 * Return:
 *	0 = success
 */
int
http_alog(const int sfd, const int hr)
{
	int		gmt;
	time_t		tnow;
	struct tm	*t;

	time(&tnow);
	t = gmtime(&tnow);
	gmt = t->tm_hour;
	t = localtime(&tnow);

	snprintf(log_1, sizeof(log_1), "%s - - %s \"%s\" %d %d",
	    c[sfd].ip, sys_date(t, gmt), c[sfd].plreq[hr], c[sfd].status[hr],
	    c[sfd].pfds[hr]);
	snprintf(log_2, sizeof(log_2), "\"%s\" \"%s\"",
	    c[sfd].plref[hr], c[sfd].plagt[hr]);

	flog(c[sfd].pllog[hr], "%s %s\n", log_1, log_2);

	return 0;
}

/*
 * http_chunk()
 *	add chunk information to a data block
 * Return:
 *	pointer to chunked data block = success, NULL = error
 */
char *
http_chunk(const char *string)
{
	int	len_string, len_hex, len_total;
	char	*chunk, hex[128];

	len_string = strlen(string);

	snprintf(hex, sizeof(hex), "%x", len_string);

	len_hex = strlen(hex);
	len_total = len_string + len_hex + 5;

	if ((chunk = malloc(len_total)) == NULL)
		return NULL;

	snprintf(chunk, len_total, "%s\r\n%s\r\n", hex, string);

	return chunk; 
}

/*
 * http_date()
 *	converts struct tm to a RFC1123 conform date string
 * Return:
 *	pointer to date string = success, NULL = error
 */
char *
http_date(struct tm *t)
{
	int		hour, min, sec, year, mon, mday, wday;
	char		*d;
	static char	date[64];

	hour = t->tm_hour;
	min = t->tm_min;
	sec = t->tm_sec;
	year = t->tm_year;
	mon = t->tm_mon;
	mday = t->tm_mday;
	wday = t->tm_wday;

	snprintf(date, 64, "%s, %02d %s %d %02d:%02d:%02d %s",
	    day[wday], mday, month[mon], year+1900, hour, min, sec, t->tm_zone);

	d = date;

	return d;
}

/*
 * http_uridecode()
 *	decodes an encoded URI
 * Return:
 *	pointer to decoded uri = success, NULL = error or nothing to decode
 */
char *
http_uridecode(const char *uri)
{
	int	i, j, k, found;
	char	*dst, hex[3];
	
	found = 0;

	if ((dst = malloc(strlen(uri) + 1)) == NULL)
		return NULL;

	memset(dst, 0, strlen(uri) + 1);
	memset(hex, 0, sizeof(hex));

	for (i = 0, j = 0; uri[i] != '\0'; i++, j++) {
		if (uri[i] == '%') {
			i++;
			hex[0] = uri[i];
			i++;
			hex[1] = uri[i];
			for (k = 0; k < 128; k++) {
				if (strcasecmp(hexset[k].hex, hex) == 0) {
					dst[j] = hexset[k].sign;
					found = 1;
					break;
				}
			}
		} else {
			dst[j] = uri[i];
		}
	}

	if (found) {
		return dst;
	} else {
		free(dst);
		return NULL;
	}
}

/*
 * http_head()
 *	creates a defined header and write access log
 * Return:
 *	pointer to header = success, NULL = error
 */
char *
http_head(const char *status, const char *request, const char *cip)
{
	int		gmt, status_nr;
	char		*h, header[8192], tmp[1024], date[128];
	time_t		tnow;
	struct tm	*t;

	/* convert status to int */
	strcutw(tmp, status, 1, sizeof(tmp));
	status_nr = atoi(status);

	/* current date GMT */
	time(&tnow);
	t = gmtime(&tnow);
	strlcpy(date, http_date(t), sizeof(date));

	/* status Date: Server: */
	snprintf(header, sizeof(header), "%s %s\r\n%s %s\r\n%s %s\r\n",
	    http_fv_pro, status, http_fn_dat, date, http_fn_srv, http_fv_srv);
	/* Connection: */
	snprintf(tmp, sizeof(tmp), "%s ", http_fn_con);
	strlcat(header, tmp, sizeof(header));
	if (status_nr == 413 || status_nr == 500) {
		/* close */
		snprintf(tmp, sizeof(tmp), "%s\r\n", http_fv_con_cls);
		strlcat(header, tmp, sizeof(header));
	} else {
		/* keep-alive */
		snprintf(tmp, sizeof(tmp), "%s\r\n%s %s\r\n", http_fv_con_alv,
		    http_fn_alv, http_fv_alv);
		strlcat(header, tmp, sizeof(header));
	}
	/* Content-Type: */
	snprintf(tmp, sizeof(tmp), "%s %s\r\n", http_fn_cte, http_fv_cte);
	strlcat(header, tmp, sizeof(header));
	/* Transfer-Encoding: */
	snprintf(tmp, sizeof(tmp), "%s %s\r\n", http_fn_teg, http_fv_teg);
	strlcat(header, tmp, sizeof(header));
	/* end of header */
	strlcat(header, "\r\n", sizeof(header));

	if ((h = strdup(header)) == NULL)
		return NULL;

	/* get date and time */
	time(&tnow);
	t = gmtime(&tnow);
	gmt = t->tm_hour;
	t = localtime(&tnow);

	/* access log 1 */
	snprintf(log_1, sizeof(log_1), "%s - - %s \"%s\" %d -", cip,
	    sys_date(t, gmt), request, status_nr);

	/* access log 2 */
	strlcpy(log_2, "\"-\" \"-\"", sizeof(log_2));

	/* access log write */
	flog(logn, "%s %s\n", log_1, log_2);

	return h;
}

/*
 * http_body()
 *	creates a defined, chunked body and merges it with a header
 * Return:
 *	pointer to response = success, NULL = error
 */
char *
http_body(const char *title, const char *msg, const char *header)
{
	int	size;
	char	*rp, *body_chunk, body[8192];

	/* html */
	snprintf(body, sizeof(body), "<html>\n<head>\n<title>%s</title>\n"
	    "</head>\n<body>\n<h1>%s</h1>\n%s\n<hr>\n%s\n</body>\n</html>",
	    title, title, msg, getenv("SERVER_SIGNATURE"));

	/* chunk html */
	body_chunk = http_chunk(body);

	/* allocate memory for the response */
	size = strlen(header) + strlen(body_chunk) + strlen(http_fv_lch) + 1;
	if ((rp = malloc(size)) == NULL)
		return NULL;
	memset(rp, 0, size);

	/* create response */
	strlcpy(rp, header, size);
	strlcat(rp, body_chunk, size);
	strlcat(rp, http_fv_lch, size);

	free(body_chunk);

	return rp;
}

/*
 * http_header()
 *	parses incoming request header and returns our response in a structure.
 *	set environment vars for cgis.
 * Return:
 *	pointer to structure header = success, NULL = error
 */
struct header *
http_header(const char *header_data, const char *force_status,
    const int force_status_nr, const int blen, const int sfd)
{
	int		fsize, fd, i, r;
	char		line[1024], option[1024];
	char		alias[1024], cgi_path[1024], cgi_full[1024];
	char		docroot[1024], file_path[1024];
	char		acc_base64[1024], acc_file[1024];
	char		acc_userpw[1024], acc_user[1024], acc_pw[1024];
	char		tmp[1024], status[128];
	char		*x, *type = NULL;
	time_t		tnow;
	struct tm 	*t;
	struct stat	s;
	struct header	*h = NULL;

	fsize = 0;

	if ((h = malloc(sizeof(struct header))) == NULL)
		return NULL;
	h->x_opt = 0;
	h->x_qry = 0;
	h->x_sct = 0;
	h->x_ims = 0;
	h->x_ref = 0;
	h->x_agt = 0;
	h->x_con = 0;
	h->x_cok = 0;
	h->x_clt = 0;
	h->x_hos = 0;
	h->x_aut = 0;
	h->x_ran = 0;
	h->x_cgi = 0;
	h->x_idx = 0;
	h->x_dov = 0;
	h->rp_status = 200;
	strlcpy(status, http_s_200, sizeof(status));

	/* current date GMT */
	time(&tnow);
	t = gmtime(&tnow);
	strlcpy(h->rp_fv_dat, http_date(t), sizeof(h->rp_fv_dat));

	/* parse request line */
	strcutl(line, header_data, 1, sizeof(line));
	strcutw(h->rq_method, line, 1, sizeof(h->rq_method));
	strcutw(h->rq_uri, line, 2, sizeof(h->rq_uri));
	strcutw(h->rq_protocol, line, 3, sizeof(h->rq_protocol));

	/* decode uri if encoded */
	if (strchr(h->rq_uri, '%') != NULL) {
		if ((x = http_uridecode(h->rq_uri)) != NULL) {
			strlcpy(h->rq_uri, x, sizeof(h->rq_uri));
			free(x);
		}
	}
	
	/* is there a query string */
	if (strcuts(h->rq_query, h->rq_uri, '?', '\0', sizeof(h->rq_query))
	    != -1) {
		strcuts(h->rq_uri, line, ' ', '?', sizeof(h->rq_uri));
		h->x_qry = 1;
	}

	/* parse request options */
	for (i = 2; strcutl(line, header_data, i, sizeof(line)) != -1; i++) {
		strcutw(option, line, 1, sizeof(option));
		/* If-Modified-Since: */
		if (strcasecmp(option, http_fn_ims) == 0) {
			h->x_ims = 1;
			strcuts(h->rq_fv_ims, line, ' ', '\0',
			    sizeof(h->rq_fv_ims));
		}
		/* Referer: */
		if (strcasecmp(option, http_fn_ref) == 0) {
			h->x_ref = 1;
			strcuts(h->rq_fv_ref, line, ' ', '\0',
			    sizeof(h->rq_fv_ref));
		}
		/* User-Agent: */
		if (strcasecmp(option, http_fn_agt) == 0) {
			h->x_agt = 1;
			strcuts(h->rq_fv_agt, line, ' ', '\0',
			    sizeof(h->rq_fv_agt));
		}
		/* Connection: */
		if (strcasecmp(option, http_fn_con) == 0) {
			strcuts(h->rq_fv_con, line, ' ', '\0',
			    sizeof(h->rq_fv_con));
			if (strcasecmp(h->rq_fv_con, http_fv_con_alv) == 0) {
				h->x_con = 1;
			}
		}
		/* Cookie: */
		if (strcasecmp(option, http_fn_cok) == 0) {
			h->x_cok = 1;
			strcuts(h->rq_fv_cok, line, ' ', '\0',
			    sizeof(h->rq_fv_cok));
		}
		/* Content-Length: */
		if (strcasecmp(option, http_fn_clt) == 0) {
			h->x_clt = 1;
			strcuts(h->rq_fv_clt, line, ' ', '\0',
			    sizeof(h->rq_fv_clt));
		}
		/* Host: */
		if (strcasecmp(option, http_fn_hos) == 0) {
			h->x_hos = 1;
			strcuts(h->rq_fv_hos, line, ' ', '\0',
			    sizeof(h->rq_fv_hos));
		}
		/* Authorization: */
		if (strcasecmp(option, http_fn_aut) == 0) {
			h->x_aut = 1;
			strcuts(h->rq_fv_aut, line, ' ', '\0',
			    sizeof(h->rq_fv_aut));
		}
		/* Range: */
		if (strcasecmp(option, http_fn_ran) == 0) {
			h->x_ran = 1;
			strcuts(h->rq_fv_ran, line, ' ', '\0',
			    sizeof(h->rq_fv_ran));
		}
	}

	/* set default docroot and access log */
	strlcpy(h->rq_docrootv, config.docroot, sizeof(h->rq_docrootv));
	strlcpy(logn, config.logaccess, sizeof(logn));

	/* virtual hosts */
	if (h->x_hos == 1 && strcmp(http_url, h->rq_fv_hos) != 0) {
		/* search for virutal host in configuration */
		if (fparse(h->rq_docrootv, h->rq_fv_hos, config.file,
		    sizeof(h->rq_docrootv)) > 0) {
			/* set own access_log for virutal host */
			snprintf(logn, sizeof(logn), "%s_%s", config.logaccess,
			    h->rq_fv_hos);
			h->x_dov = 1;
		}
	}

	/* cut possible alias from uri */
	strcuts(alias, h->rq_uri, '\0', '/', sizeof(alias));

	/* cgi */
	if (strcmp(alias, config.cgialias) == 0) {
		/* remove alias from path */
		strcuti(cgi_path, h->rq_uri, strlen(alias), strlen(h->rq_uri),
		    sizeof(cgi_path));

		/* merge cgiroot with cgi_path */
		snprintf(cgi_full, sizeof(cgi_full), "%s%s",
		    config.cgiroot, cgi_path);

		/* separate path and options */
		if ((r = http_cgi_getexec(h->rq_filef, h->rq_option, cgi_full,
		    sizeof(h->rq_filef), sizeof(h->rq_option))) == 1) {
			/* options found */
			strcuti(h->rq_script, h->rq_uri, 0,
			    (strlen(h->rq_uri) - strlen(h->rq_option)) - 1,
			    sizeof(h->rq_script));

			/* set cgi option flag */
			h->x_opt = 1;
		} else if (r == 0) {
			/* no options found */
			strlcpy(h->rq_script, h->rq_uri, sizeof(h->rq_script));
		} else if (r == -1) {
			/* no cgi found */
			strlcpy(h->rq_filef, cgi_full, sizeof(h->rq_filef));
		}

		/* separate file and path */
		strcutf(h->rq_files, h->rq_filep, h->rq_filef,
		    sizeof(h->rq_files), sizeof(h->rq_filep));

		/* set cgi flag */
		h->x_cgi = 1;
	/* file */
	} else {
		/* check if uri contains an alias */
		if (fparse(docroot, alias, config.file, sizeof(tmp)) != -1) {
			/* yes cut it out */
			strcuti(file_path, h->rq_uri, strlen(alias),
			    strlen(h->rq_uri), sizeof(file_path));
		} else {
			/* no keep defaults */
			strlcpy(docroot, h->rq_docrootv, sizeof(docroot));
			strlcpy(file_path, h->rq_uri, sizeof(file_path));
		}

		/* directory? */
		if (h->rq_uri[strlen(h->rq_uri) - 1] == '/') {
			/* save original uri */
			strlcpy(h->rq_index, file_path, sizeof(h->rq_index));

			/* add default index to request uri */
			strlcat(file_path, config.docindex, sizeof(file_path));

			/* set index flag */
			h->x_idx = 1;
		}

		/* create full file path */
		snprintf(h->rq_filef, sizeof(h->rq_filef), "%s%s",
		    docroot, file_path);

		/* save file and path separated */
		strcutf(h->rq_files, h->rq_filep, h->rq_filef,
		    sizeof(h->rq_files), sizeof(h->rq_filep));
	}

	/* get content type from request uri */
	if (!h->x_cgi) {
		strlcpy(tmp, h->rq_filef, sizeof(tmp));
		for (x = strtok(tmp, "."); x; x = strtok(NULL, "."))
			type = x;
		strlcpy(h->rq_fv_cte, type, sizeof(h->rq_fv_cte));
		if (sys_mime(h->rp_fv_cte, sizeof(h->rp_fv_cte), mimes,
		    mimes_size, h->rq_fv_cte) == 0) {
			strlcpy(h->rp_fv_cte, http_fv_cte,
			    sizeof(h->rp_fv_cte));
		}
	}
	
	/* does the file exist */
	if (stat(h->rq_filef, &s) == -1) {
		/* no, but maybe its a directory without index file */
		if (h->x_idx == 1 && (fd = open(h->rq_filep, O_RDONLY)) > -1) {
			/* yes */
			close(fd);
		} else {
			/* nope */
			h->x_idx = 0;
			/* 404 */
			strlcpy(status, http_s_404, sizeof(status));
			strlcpy(h->rp_fv_cte, http_fv_cte,
			    sizeof(h->rp_fv_cte));
			h->rp_status = 404;
		}
	} else if (access(h->rq_filef, R_OK) == -1) {
		/* yes but we cant read it */
		h->x_idx = 0;
		/* 403 */
		strlcpy(status, http_s_403, sizeof(status));
		strlcpy(h->rp_fv_cte, http_fv_cte, sizeof(h->rp_fv_cte));
		h->rp_status = 403;
	} else {
		/* yes */
		h->x_idx = 0;
		/* is it a file or a directory */
		if (s.st_mode & S_IFDIR) {
			/* it is a directory without '/', send 301 */
			strlcpy(status, http_s_301, sizeof(status));
			/* create exact location uri */
			if (c[sfd].x_ssl) {
				snprintf(h->rp_fv_loc, sizeof(h->rp_fv_loc),
				    "https://%s%s/", http_url, h->rq_uri);
			} else {
				snprintf(h->rp_fv_loc, sizeof(h->rp_fv_loc),
				    "http://%s%s/", http_url, h->rq_uri);
			}
			/* set 301 headers content type */
			strlcpy(h->rp_fv_cte, http_fv_cte,
			    sizeof(h->rp_fv_cte));
			/* 301 */
			h->rp_status = 301;
		} else {
			/* it is a file, get modification date and size */
			t = gmtime(&s.st_mtime);
			strlcpy(h->rp_fv_dam, http_date(t),
			    sizeof(h->rp_fv_dat));
			h->rp_fsize = (int)s.st_size;
		}
	}

	/* was the file modified */
	if (h->x_ims == 1 && strcmp(h->rq_fv_ims, h->rp_fv_dam) == 0) {
		strlcpy(status, http_s_304, sizeof(status));
		h->rp_status = 304;
	}

	/* directory access control */
	if (http_access_htaccess(acc_file, h->rq_filep, sizeof(acc_file))
	    == 1) {
		/* authenticate option is set */
		if (h->x_aut) {
			/* decode base64 encoded user:password string */
			strcutw(acc_base64, h->rq_fv_aut, 2,
			    sizeof(acc_base64));
			strb64d(acc_userpw, acc_base64, sizeof(acc_userpw));
			/* separate user and password */
			strcuts(acc_user, acc_userpw, '\0', ':',
			    sizeof(acc_user));
			strcuts(acc_pw, acc_userpw, ':', '\0', sizeof(acc_pw));
			/* authenticate */
			if ((r = sys_access_auth(acc_user, acc_pw)) != 1)
				/* failed */
				h->x_aut = 0;
		}
		/* authenticate option is not set or failed */
		if (!h->x_aut) {
			/* get access realm */
			if (fparse(h->rp_fv_auw, "realm", acc_file,
			    sizeof(h->rp_fv_auw)) == -1) {
				/* realm could not be parsed */
				strlcpy(h->rp_fv_auw, "unknown realm",
				    sizeof(h->rp_fv_auw));
			}
			/* 401 */
			strlcpy(status, http_s_401, sizeof(status));
			strlcpy(h->rp_fv_cte, http_fv_cte,
			    sizeof(h->rp_fv_cte)); 
			h->x_cgi = 0;
			h->rp_status = 401;
		}
	}

	/* 206 */
	if (h->rp_status == 200 && h->x_ran == 1) {
		strcuts(tmp, h->rq_fv_ran, '=', '-', sizeof(tmp));
		h->rp_foffs = atoi(tmp);
		/* dont get beyond end of file */
		if (h->rp_foffs >= h->rp_fsize)
			return NULL;
		fsize = h->rp_fsize;
		h->rp_fsize = fsize - h->rp_foffs;
		h->rp_status = 206;
		strlcpy(status, http_s_206, sizeof(status));
	}

	/* forced response status */
	if (force_status != NULL) {
		strlcpy(status, force_status, sizeof(status));
		if (force_status_nr == 0)
			h->rp_status = 0;
		else
			h->rp_status = force_status_nr;
	}

	/*
	 * Build response header
	 */
	/* Status, Date:, Server: */
	snprintf(h->rp_header, sizeof(h->rp_header),
	    "%s %s\r\n%s %s\r\n%s %s\r\n", http_fv_pro, status, http_fn_dat,
	    h->rp_fv_dat, http_fn_srv, http_fv_srv);
	/* Connection: */
	snprintf(tmp, sizeof(tmp), "%s ", http_fn_con);
	strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	if (h->x_con == 0) {
		/* close */
		snprintf(tmp, sizeof(tmp), "%s\r\n", http_fv_con_cls);
		strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	}
	if (h->x_con == 1) {
		/* keep-alive */
		snprintf(tmp, sizeof(tmp), "%s\r\n%s %s\r\n", http_fv_con_alv,
		    http_fn_alv, http_fv_alv);
		strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	}
	/* Location: */
	if (h->rp_status == 301) {
		snprintf(tmp, sizeof(tmp), "%s %s\r\n", http_fn_loc,
		    h->rp_fv_loc);
		strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	}
	/* Last-Modified:, Content-Length: */
	if ((h->rp_status == 200 || h->rp_status == 206) && h->x_cgi == 0 &&
	    h->x_idx == 0) {
		snprintf(tmp, sizeof(tmp), "%s %s\r\n%s %d\r\n",
		    http_fn_lmd, h->rp_fv_dam, http_fn_clt, h->rp_fsize);
		strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	}
	/* Content-Range: */
	if (h->rp_status == 206 && h->x_cgi == 0 && h->x_idx == 0) {
		snprintf(tmp, sizeof(tmp), "%s bytes %d-%d/%d\r\n",
		    http_fn_rac, h->rp_foffs, fsize - 1, fsize);
		strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	}
	/* Content-Type: */
	if (h->rp_status != 304 && h->x_cgi == 0) {
		snprintf(tmp, sizeof(tmp), "%s %s\r\n", http_fn_cte,
		    h->rp_fv_cte);
		strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	}
	/* Transfer-Encoding: */
	if (h->rp_status == 404 || h->rp_status == 301 || h->rp_status == 401 ||
	    h->rp_status == 403 || h->x_cgi == 1 || h->x_idx == 1) {
		snprintf(tmp, sizeof(tmp), "%s %s\r\n", http_fn_teg,
		    http_fv_teg);
		strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	}
	/* WWW-Authenticate: */
	if (h->rp_status == 401) {
		snprintf(tmp, sizeof(tmp), "%s Basic realm=\"%s\"\r\n",
		    http_fn_auw, h->rp_fv_auw);
		strlcat(h->rp_header, tmp, sizeof(h->rp_header));
	}
	/* end of header */
	if (h->x_cgi == 0 || h->rp_status == 403 || h->rp_status == 404)
		strlcat(h->rp_header, "\r\n", sizeof(h->rp_header));

	/* return the size of our header */
	h->rp_hsize = strlen(h->rp_header);

	/* set dynamic cgi environment vars */
	if (h->x_cgi) {
		/* cleanup first */
		unsetenv("CONTENT_LENGTH");
		unsetenv("DOCUMENT_ROOT");
		unsetenv("HTTP_COOKIE");
		unsetenv("HTTP_USER_AGENT");
		unsetenv("HTTP_CONNECTION");
		unsetenv("HTTP_HOST");
		unsetenv("HTTPS");
		unsetenv("PATH_INFO");
		unsetenv("QUERY_STRING");
		unsetenv("REMOTE_ADDR");
		unsetenv("REMOTE_PORT");
		unsetenv("REMOTE_USER");
		unsetenv("REQUEST_METHOD");
		unsetenv("REQUEST_URI");
		unsetenv("SCRIPT_FILENAME");
		unsetenv("SCRIPT_NAME");
		unsetenv("SERVER_ADDR");
		unsetenv("SERVER_PORT");

		/* mandatory */
		setenv("DOCUMENT_ROOT", docroot, 1);
		setenv("REMOTE_ADDR", c[sfd].ip, 1);
		setenv("REMOTE_PORT", c[sfd].pt, 1);
		setenv("REQUEST_METHOD", h->rq_method, 1);
		setenv("REQUEST_URI", h->rq_uri, 1);
		setenv("SCRIPT_FILENAME", h->rq_filef, 1);
		setenv("SCRIPT_NAME", h->rq_script, 1);
		if (c[sfd].x_ip6)
			setenv("SERVER_ADDR", config.serverip6, 1);
		else
			setenv("SERVER_ADDR", config.serverip4, 1);
		if (c[sfd].x_ssl)
			setenv("SERVER_PORT", config.sslportc, 1);
		else
			setenv("SERVER_PORT", config.serverportc, 1);

		/* optional */
		if (h->x_clt) {
			snprintf(tmp, sizeof(tmp), "%d", blen);
			setenv("CONTENT_LENGTH", tmp, 1);
		}
		if (h->x_cok)
			setenv("HTTP_COOKIE", h->rq_fv_cok, 1);
		if (h->x_con)
			setenv("HTTP_CONNECTION", h->rq_fv_con, 1);
		if (h->x_hos)
			setenv("HTTP_HOST", h->rq_fv_hos, 1);
		if (h->x_agt)
			setenv("HTTP_USER_AGENT", h->rq_fv_agt, 1);
		if (c[sfd].x_ssl)
			setenv("HTTPS", "on", 1);
		if (h->x_opt)
			setenv("PATH_INFO", h->rq_option, 1);
		if (h->x_qry)
			setenv("QUERY_STRING", h->rq_query, 1);
		if (h->x_aut)
			setenv("REMOTE_USER", acc_user, 1);
	}

	/* set signature variable */
	if (h->x_cgi || h->x_idx || (h->rp_status != 200 &&
	    h->rp_status != 206 && h->rp_status != 314)) {
		snprintf(tmp, sizeof(tmp),
		    "<address>%s at %s Port %s</address>", http_fv_srv,
		    h->rq_fv_hos, c[sfd].x_ssl ? config.sslportc :
		    config.serverportc);
		setenv("SERVER_SIGNATURE", tmp, 1);
	}
		    
	return h;
}
