/*
 * net6emu r1: net6emu.c
 * Emulation code for the IPv6 socket API extensions.
 *
 * Copyright (c) 2002 Christoph Pfisterer
 * Based on emulation code from OpenSSH, reused and improved under
 * OpenSSH's BSD-style license.
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE. 
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>

#include <netdb.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "net6emu.h"

/*
 * part 1: getaddrinfo() and friends
 */
#ifndef N6E_HAVE_GETADDRINFO

char *gai_strerror(int ecode)
{
  switch (ecode) {
  case EAI_NODATA:
    return "No address associated with hostname";
  case EAI_MEMORY:
    return "Memory allocation failure";
  default:
    return "Unknown error";
  }
}

void freeaddrinfo(struct addrinfo *ai)
{
  struct addrinfo *next;

  do {
    next = ai->ai_next;
    free(ai);
  } while (NULL != (ai = next));
}

static struct addrinfo *malloc_ai(int port, u_long addr)
{
  struct addrinfo *ai;

  ai = malloc(sizeof(struct addrinfo) + sizeof(struct sockaddr_in));
  if (ai == NULL)
    return(NULL);

  memset(ai, 0, sizeof(struct addrinfo) + sizeof(struct sockaddr_in));

  ai->ai_family = AF_INET;
  ai->ai_socktype = SOCK_STREAM;
  ai->ai_addr = (struct sockaddr *)(ai + 1);
  ai->ai_addrlen = sizeof(struct sockaddr_in);
  ai->ai_addr->sa_family = ai->ai_family = AF_INET;

  ((struct sockaddr_in *)(ai)->ai_addr)->sin_port = port;
  ((struct sockaddr_in *)(ai)->ai_addr)->sin_addr.s_addr = addr;

  return(ai);
}

int getaddrinfo(const char *hostname, const char *servname, 
                const struct addrinfo *hints, struct addrinfo **res)
{
  struct addrinfo *cur, *prev = NULL;
  struct hostent *hp;
  struct in_addr in;
  int i, port;

  if (hints && hints->ai_family != 0 &&
      hints->ai_family != PF_UNSPEC && hints->ai_family != PF_INET)
    return EAI_NODATA;

  if (servname)
    port = htons(atoi(servname));
  else
    port = 0;

  if (hints && hints->ai_flags & AI_PASSIVE) {
    if (NULL != (*res = malloc_ai(port, htonl(0x00000000))))
      return 0;
    else
      return EAI_MEMORY;
  }

  if (!hostname) {
    if (NULL != (*res = malloc_ai(port, htonl(0x7f000001))))
      return 0;
    else
      return EAI_MEMORY;
  }

  if (inet_aton(hostname, &in)) {
    if (NULL != (*res = malloc_ai(port, in.s_addr)))
      return 0;
    else
      return EAI_MEMORY;
  }

  hp = gethostbyname(hostname);
  if (hp && hp->h_name && hp->h_name[0] && hp->h_addr_list[0]) {
    for (i = 0; hp->h_addr_list[i]; i++) {
      cur = malloc_ai(port, ((struct in_addr *)hp->h_addr_list[i])->s_addr);
      if (cur == NULL) {
	if (*res)
	  freeaddrinfo(*res);
	return EAI_MEMORY;
      }

      if (prev)
	prev->ai_next = cur;
      else
	*res = cur;

      prev = cur;
    }
    return 0;
  }

  return EAI_NODATA;
}

#endif /* !N6E_HAVE_GETADDRINFO */


/*
 * part 2: getnameinfo()
 */
#ifndef N6E_HAVE_GETNAMEINFO

int getnameinfo(const struct sockaddr *sa, size_t salen, char *host, 
                size_t hostlen, char *serv, size_t servlen, int flags)
{
  struct sockaddr_in *sin = (struct sockaddr_in *)sa;
  struct hostent *hp;
  char tmpserv[16];

  if (serv) {
    if (sa->sa_family != AF_INET)
      return EAI_NODATA;

    snprintf(tmpserv, sizeof(tmpserv), "%d", ntohs(sin->sin_port));
    if (strlen(tmpserv) >= servlen)
      return EAI_MEMORY;
    else
      strcpy(serv, tmpserv);
  }

  if (host) {
    if (sa->sa_family != AF_INET)
      return EAI_NODATA;

    if (flags & NI_NUMERICHOST) {
      if (strlen(inet_ntoa(sin->sin_addr)) >= hostlen)
	return EAI_MEMORY;

      strcpy(host, inet_ntoa(sin->sin_addr));
      return 0;
    } else {
      hp = gethostbyaddr((char *)&sin->sin_addr, 
			 sizeof(struct in_addr), AF_INET);
      if (hp == NULL)
	return EAI_NODATA;

      if (strlen(hp->h_name) >= hostlen)
	return EAI_MEMORY;

      strcpy(host, hp->h_name);
      return 0;
    }
  }
  return 0;
}

#endif /* !N6E_HAVE_GETNAMEINFO */
