/*
 * Copyright (c) 2007 Alexey Vatchenko <av@bsdua.org>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */
#include <sys/types.h>
#include <sys/event.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <sys/un.h>

#include <netinet/in.h>
#include <netdb.h>

#include <termios.h>

#include <err.h>
#include <errno.h>
#include <fcntl.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>

#define _MAX_DESCRIPTORS	2048
#define _BUFFER_SIZE		1024

extern char *__progname;

static __dead void usage(void);

/*
 * Logging stuff
 */
struct nf_log {
	char	*fmt;
	size_t	fmtsize;
	int	issyslog;

	void (*info)(struct nf_log *ctx, const char *msg, va_list ap);
	void (*warn)(struct nf_log *ctx, int level, int err, const char *msg,
	    va_list ap);
};

#define _NF_LOG_FMTSIZE		100

void	_nf_log_xnothing(struct nf_log *ctx, const char *msg, va_list ap);
void	_nf_log_nothing(struct nf_log *ctx, int level, int err, const char *msg,
	    va_list ap);
void	_nf_log_stdout(struct nf_log *ctx, const char *msg, va_list ap);
void	_nf_log_warn(struct nf_log *ctx, int level, int err, const char *msg,
	    va_list ap);
void	_nf_log_xsyslog(struct nf_log *ctx, const char *msg, va_list ap);
void	_nf_log_syslog(struct nf_log *ctx, int level, int err, const char *msg,
	    va_list ap);
void	nf_log_open(struct nf_log *ctx, int isdaemon, int isquiet,
	    int isverbose);
void	nf_log_close(struct nf_log *ctx);
void	nf_log_info(struct nf_log *ctx, const char *msg, ...);
void	nf_log_warn(struct nf_log *ctx, int err, const char *msg, ...);
void	nf_log_error(struct nf_log *ctx, int err, const char *msg, ...);

/*
 * Network stuff
 */
int	create_inet_server(struct addrinfo *ai);
int	create_inet_client(struct addrinfo *ai, int *inprogress);
int	complete_inet_client(int sock, struct addrinfo *ai);
int	create_unix_server(const char *path, int socktype);
int	create_unix_client(const char *path, int socktype);

/*
 * TTY stuff
 */
struct tty_settings {
	speed_t	baudrate;
	u_int	databits;
	char	parity;
	u_int	stopbits;
	char	flowcontrol;
};

int	tty_parse_settings(const char *str, struct tty_settings *cs);
int	create_tty_server(const char *path, struct tty_settings *cs);
int	create_tty_client(const char *path, struct tty_settings *cs);

/*
 * Connection context.
 */
struct nf_conn {
	int	sock;	/* socket of associated connection */
	char	buf[_BUFFER_SIZE];	/* data read from current connection */
	size_t	buflen;
	u_int	flags;
#define NF_CONN_ACCEPT		0x01	/* socket for accepting, not IO */
#define NF_CONN_CONNECTOR	0x02	/* server socket, not STREAM */
#define NF_CONN_CONNECTED	0x04	/* socket is ready to read/write */
#define NF_CONN_CLOSING		0x08	/* conn is waiting 2nd to close */
#define NF_CONN_CLOSED		0x10	/* closed conn, ignore its IO */
#define NF_CONN_LOCAL		0x20	/* local socket */

	int	af;
	int	proto;

	/* Peer address for ``inet'' and ``unix/stream'' protocols */
	struct sockaddr_storage	peer;
	socklen_t		peerlen;

	/*
	 * Save current address for ``inet/tcp'' connection. Used for
	 * reconnecting if primary address is not accessible.
	 */
	struct addrinfo *ai;
};

/*
 * Events.
 */
struct nf_events {
	int	kfd;
	struct kevent *evlist;
	struct kevent *ev;
	u_int	size;
	u_int	num;
};

int	_evlist_alloc(struct nf_events *events, u_int num, u_int *idx);
int	evlist_init(struct nf_events *events);
void	evlist_final(struct nf_events *events);
void	evlist_delete(struct nf_events *events, int sock);
void	evlist_enable(struct nf_events *events, int fd, int isread, int enable);
int	evlist_add(struct nf_events *events, int src_sock, int dst_sock,
	    int isconnected);
int	evlist_add_server(struct nf_events *events, int sock);
void	evlist_change(struct nf_events *events, int sock, int isconnected,
	    int old);

/* Address family */
#define NF_AF_INET		0
#define NF_AF_INET4		1
#define NF_AF_INET6		2
#define NF_AF_UNIX		3
#define NF_AF_TTY		4

/* Protocols */
#define NF_PROTO_STREAM		0
#define NF_PROTO_DGRAM		1
#define NF_PROTO_TTY		2

/* Address of communication point */
struct net_fwd_addr {
	int af;
	int proto;
	union {
		struct addrinfo	*ai0;		/* TCP/UDP */
		char		path[104];	/* unix sockaddr */
		struct {
			char path[MAXPATHLEN];
			struct tty_settings cfg;
		} tty;				/* TTY address and settings */
	} spec;
};

int	nf_addr_init(struct net_fwd_addr *addr, int af, int proto,
	    const char *host, const char *port);
void	nf_addr_final(struct net_fwd_addr *addr);

/* Socket table */
struct net_fwd {
	struct net_fwd_addr local;
	struct net_fwd_addr remote;

	struct nf_conn *conn;
	u_int	n_conn;
	struct nf_events ev;
};

u_int	fix_rlimit(void);

int	_nf_add_listener(struct net_fwd *nf, int fd, int af);
int	_nf_add_connector(struct net_fwd *nf, int af1, int proto1, int fd1,
	    int af2, int proto2, int fd2, int isconnected);
void	_nf_change_connector(struct net_fwd *nf, int fd1, int fd2,
	    int isconnected, int old);

int	nf_init(struct net_fwd *nf, int family, int proto, const char *host,
	    const char *port, int rfamily, int rproto, const char *rhost,
	    const char *rport);
int	nf_remote_conn(struct net_fwd *nf, int fd1);
int	nf_create_listener(struct net_fwd *nf);
int	nf_create_connector(struct net_fwd *nf);
int	nf_loop(struct net_fwd *nf);
int	nf_loop_events(struct net_fwd *nf, int n);
int	nf_accept_conn(struct net_fwd *nf, int sock, int num);
int	nf_complete_conn(struct net_fwd *nf, int sock);
ssize_t	nf_read_conn(struct net_fwd *nf, int fd, int size);
ssize_t	nf_write_conn(struct net_fwd *nf, int fd);
int	nf_close_conn(struct net_fwd *nf, int sock, int isread);

/* Global logger */
struct nf_log nflog;

static __dead void
usage(void)
{
	fprintf(stderr,
	    "usage: %s [-d] [-q | -v] local_address remote_address\n"
	    "Local address:\n"
	    "\t- [-s bind_address] [inet|inet6] tcp|udp port\n"
	    "\t- unix stream|dgram path\n"
	    "\t- tty path tty_settings\n"
	    "Remote address:\n"
	    "\t- [inet|inet6] tcp|udp host port\n"
	    "\t- unix stream|dgram path\n"
	    "\t- tty path tty_settings\n"
	    "TTY settings:\n"
	    "\tbaudrate,databits,parity,stopbits,flowcontrol\n"
	    "\t- baudrate: any reasonable value like 9600, 19200, 115200\n"
	    "\t- databits: 5|6|7|8\n"
	    "\t- parity: E|O|N (E-even, O-odd, N-none)\n"
	    "\t- stopbits: 1|1.5|2\n"
	    "\t- flowcontrol: X|H|N (X-Xon/Xoff, H-hardware, N-none)\n",
	    __progname);
	exit(1);
}

/* ARGSUSED */
void
_nf_log_xnothing(struct nf_log *ctx, const char *msg, va_list ap)
{
	/* nothing */
}

/* ARGSUSED */
void
_nf_log_nothing(struct nf_log *ctx, int level, int err, const char *msg,
    va_list ap)
{
	/* nothing */
}

/* ARGSUSED */
void
_nf_log_stdout(struct nf_log *ctx, const char *msg, va_list ap)
{
	vfprintf(stdout, msg, ap);
	fprintf(stdout, "\n");
}

/* ARGSUSED */
void
_nf_log_warn(struct nf_log *ctx, int level, int err, const char *msg,
    va_list ap)
{
	int olderrno;

	if (err != 0) {
		olderrno = errno;
		errno = err;
		vwarn(msg, ap);
		errno = olderrno;
	} else
		vwarnx(msg, ap);
}

/* ARGSUSED */
void
_nf_log_xsyslog(struct nf_log *ctx, const char *msg, va_list ap)
{
	vsyslog(LOG_INFO, msg, ap);
}

void
_nf_log_syslog(struct nf_log *ctx, int level, int err, const char *msg,
    va_list ap)
{
	int olderrno;
	size_t len;
	char *p;

	if (err == 0 || ctx->fmt == NULL) {
		vsyslog(level, msg, ap);
		return;
	}

	len = strlen(msg);
	/* we need to add ": %m" to the end of ``msg'' */
	if (len + 5 > ctx->fmtsize) {
		p = (char *)realloc(ctx->fmt, len + 5);
		if (p == NULL) {
			/* fallback to ``msg'' */
			vsyslog(level, msg, ap);
			return;
		}
		ctx->fmt = p;
		ctx->fmtsize = len + 5;
	}
	strlcpy(ctx->fmt, msg, ctx->fmtsize);
	strlcat(ctx->fmt, ": %m", ctx->fmtsize);
	olderrno = errno;
	errno = err;
	vsyslog(level, ctx->fmt, ap);
	errno = olderrno;
}

void
nf_log_open(struct nf_log *ctx, int isdaemon, int isquiet, int isverbose)
{
	ctx->fmt = NULL;
	ctx->fmtsize = 0;
	ctx->issyslog = 0;

	if (isdaemon) {
		openlog(__progname, LOG_PID, LOG_DAEMON);
		ctx->issyslog = 1;
	}

	if (isquiet)
		ctx->warn = _nf_log_nothing;
	else if (isdaemon) {
		ctx->warn = _nf_log_syslog;

		ctx->fmt = (char *)malloc(_NF_LOG_FMTSIZE);
		if (ctx->fmt == NULL) {
			warnx("can't allocate memory for syslog errno "
			    "messages");
		} else
			ctx->fmtsize = _NF_LOG_FMTSIZE;
	} else
		ctx->warn = _nf_log_warn;

	if (!isverbose)
		ctx->info = _nf_log_xnothing;
	else if (isdaemon)
		ctx->info = _nf_log_xsyslog;
	else
		ctx->info = _nf_log_stdout;
}

void
nf_log_close(struct nf_log *ctx)
{
	if (ctx->issyslog)
		closelog();
	if (ctx->fmt != NULL)
		free(ctx->fmt);
}

void
nf_log_info(struct nf_log *ctx, const char *msg, ...)
{
	va_list ap;

	va_start(ap, msg);
	ctx->info(ctx, msg, ap);
	va_end(ap);
}

void
nf_log_warn(struct nf_log *ctx, int err, const char *msg, ...)
{
	va_list ap;

	va_start(ap, msg);
	ctx->warn(ctx, LOG_WARNING, err, msg, ap);
	va_end(ap);
}

void
nf_log_error(struct nf_log *ctx, int err, const char *msg, ...)
{
	va_list ap;

	va_start(ap, msg);
	ctx->warn(ctx, LOG_ERR, err, msg, ap);
	va_end(ap);
}

int
create_inet_server(struct addrinfo *ai)
{
	char host[NI_MAXHOST], hostport[NI_MAXSERV];
	int sock, yes;
	int error;

	sock = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol);
	if (sock == -1) {
		nf_log_error(&nflog, errno, "can't create server socket");
		return (-1);
	}

	yes = 1;
	setsockopt(sock, SOL_SOCKET, SO_REUSEPORT, &yes, sizeof(yes));

	error = bind(sock, ai->ai_addr, ai->ai_addrlen);
	if (error == -1) {
		nf_log_error(&nflog, errno,
		    "can't bind socket to local address");
		close(sock);
		return (-1);
	}

	if (ai->ai_socktype == SOCK_STREAM) {
		error = listen(sock, SOMAXCONN);
		if (error == -1) {
			nf_log_error(&nflog, errno, "listen");
			close(sock);
			return (-1);
		}
	}

	if (getnameinfo(ai->ai_addr, ai->ai_addrlen, host, sizeof(host),
	    hostport, sizeof(hostport), NI_NUMERICHOST | NI_NUMERICSERV) != 0) {
		strlcpy(host, "unknown", sizeof(host));
		strlcpy(hostport, "unknown", sizeof(hostport));
	}
	nf_log_info(&nflog, "Listen on %s port %s", host, hostport);

	return (sock);
}

int
create_inet_client(struct addrinfo *ai, int *inprogress)
{
	char host[NI_MAXHOST], hostport[NI_MAXSERV];
	int sock, fl;
	int error;

	sock = socket(ai->ai_family, ai->ai_socktype, ai->ai_protocol);
	if (sock == -1) {
		nf_log_error(&nflog, errno, "can't create socket");
		return (-1);
	}

	*inprogress = 0;

	if (ai->ai_socktype == SOCK_STREAM) {
		error = -1;
		fl = fcntl(sock, F_GETFL);
		if (fl != -1)
			error = fcntl(sock, F_SETFL, fl | O_NONBLOCK);
		if (error == -1)
			nf_log_warn(&nflog, errno,
			    "can't set ``non-block'' flag, connect will block");
	}

	error = connect(sock, ai->ai_addr, ai->ai_addrlen);
	if (error == -1 && errno == EINPROGRESS)
		*inprogress = 1;
	else if (error == -1) {
		nf_log_error(&nflog, errno, "can't connect to remote host");
		close(sock);
		return (-1);
	} else {
		/* Revert to blocking mode */
		if (ai->ai_socktype == SOCK_STREAM) {
			error = -1;
			fl = fcntl(sock, F_GETFL);
			if (fl != -1)
				error = fcntl(sock, F_SETFL, fl & ~O_NONBLOCK);
			if (error == -1)
				nf_log_warn(&nflog, errno,
				    "can't revert from ``non-blocking''");
		}
	}

	if (!*inprogress) {
		if (getnameinfo(ai->ai_addr, ai->ai_addrlen, host, sizeof(host),
		    hostport, sizeof(hostport),
		    NI_NUMERICHOST | NI_NUMERICSERV) != 0) {
			strlcpy(host, "unknown", sizeof(host));
			strlcpy(hostport, "unknown", sizeof(hostport));
		}
		nf_log_info(&nflog,
		    "Connection with %s port %s is established", host,
		    hostport);
	}

	return (sock);
}

int
complete_inet_client(int sock, struct addrinfo *ai)
{
	char host[NI_MAXHOST], hostport[NI_MAXSERV];
	int fl, error;

	error = connect(sock, ai->ai_addr, ai->ai_addrlen);
	if (error == -1 && errno != EISCONN) {
		nf_log_error(&nflog, errno, "can't connect to remote host");
		return (-1);
	}

	error = -1;
	fl = fcntl(sock, F_GETFL);
	if (fl != -1)
		error = fcntl(sock, F_SETFL, fl & ~O_NONBLOCK);
	if (error == -1)
		nf_log_warn(&nflog, errno,
		    "can't revert from ``non-blocking''");

	if (getnameinfo(ai->ai_addr, ai->ai_addrlen, host, sizeof(host),
	    hostport, sizeof(hostport), NI_NUMERICHOST | NI_NUMERICSERV) != 0) {
		strlcpy(host, "unknown", sizeof(host));
		strlcpy(hostport, "unknown", sizeof(hostport));
	}
	nf_log_info(&nflog, "Connection with %s port %s is established", host,
	    hostport);

	return (0);
}

int
create_unix_server(const char *path, int socktype)
{
	struct sockaddr_un addr;
	socklen_t addrlen;
	int sock;
	size_t len;
	int error;

	sock = socket(PF_LOCAL, socktype, 0);
	if (sock == -1) {
		nf_log_error(&nflog, errno, "can't create server socket");
		return (-1);
	}

	/* emulate SO_REUSEADDR :) */
	unlink(path);

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_LOCAL;
	len = strlcpy(addr.sun_path, path, sizeof(addr.sun_path));
	if (len >= sizeof(addr.sun_path)) {
		nf_log_error(&nflog, 0, "socket name is too long");
		close(sock);
		return (-1);
	}
	addrlen = sizeof(addr);

	error = bind(sock, (struct sockaddr *)&addr, addrlen);
	if (error == -1) {
		nf_log_error(&nflog, errno,
		    "can't bind socket to local address");
		close(sock);
		return (-1);
	}

	if (socktype == SOCK_STREAM) {
		error = listen(sock, SOMAXCONN);
		if (error == -1) {
			nf_log_error(&nflog, errno, "listen");
			close(sock);
			return (-1);
		}
	}

	nf_log_info(&nflog, "Listen on %s", path);

	return (sock);
}

int
create_unix_client(const char *path, int socktype)
{
	struct sockaddr_un addr;
	socklen_t addrlen;
	int sock;
	size_t len;
	int error;

	sock = socket(PF_LOCAL, socktype, 0);
	if (sock == -1) {
		nf_log_error(&nflog, errno, "can't create server socket");
		return (-1);
	}

	memset(&addr, 0, sizeof(addr));
	addr.sun_family = AF_LOCAL;
	len = strlcpy(addr.sun_path, path, sizeof(addr.sun_path));
	if (len >= sizeof(addr.sun_path)) {
		nf_log_error(&nflog, 0, "socket name is too long");
		close(sock);
		return (-1);
	}
	addrlen = sizeof(addr);

	error = connect(sock, (struct sockaddr *)&addr, addrlen);
	if (error == -1) {
		nf_log_error(&nflog, errno, "can't connect to remote address");
		close(sock);
		return (-1);
	}

	nf_log_info(&nflog, "Socket \"%s\" is opened", path);

	return (sock);
}

int
tty_parse_settings(const char *str, struct tty_settings *cs)
{
	char *p;
	u_int i, cell;
	char ch;
	size_t len;
	int quit;

	quit = 0;

	/* Set default values */
	cs->baudrate = 115200;
	cs->databits = 8;
	cs->parity = 'N';
	cs->stopbits = 1;
	cs->flowcontrol = 'H';

	/*
	 * Baudrate.
	 */

	p = strchr(str, ',');
	if (p == NULL) {
		len = strlen(str);
		p = (char *)str + len;
		quit = 1;
	}

	len = p - str;
	if (len > 8) {	/* it's more than enough */
		nf_log_error(&nflog, 0, "baudrate value is too long");
		return (-1);
	}

	if (len > 0) {
		cs->baudrate = 0;
		cell = 1;
		for (i = 0; i < len; i++) {
			ch = *(p - 1 - i);
			if (ch < '0' || ch > '9') {
				nf_log_error(&nflog, 0, "invalid baudrate");
				return (-1);
			}
			cs->baudrate += (*(p - 1 - i) - '0') * cell;
			cell *= 10;
		}
	}

	if (quit)
		return (0);

	/*
	 * Databits.
	 */

	str = p + 1;

	p = strchr(str, ',');
	if (p == NULL) {
		len = strlen(str);
		p = (char *)str + len;
		quit = 1;
	}

	len = p - str;
	if (len > 1) {
		nf_log_error(&nflog, 0, "invalid databits parameter");
		return (-1);
	} else if (len == 0)
		;	/* we use default value */
	else if (*str < '5' || *str > '8') {
		nf_log_error(&nflog, 0, "invalid databits parameter");
		return (-1);
	} else
		cs->databits = *str - '0';

	if (quit)
		return (0);

	/*
	 * Parity.
	 */

	str = p + 1;

	p = strchr(str, ',');
	if (p == NULL) {
		len = strlen(str);
		p = (char *)str + len;
		quit = 1;
	}

	len = p - str;
	if (len > 1) {
		nf_log_error(&nflog, 0, "invalid parity parameter");
		return (-1);
	} else if (len > 0) {
		switch (*str) {
		case 'E':
		case 'e':
			cs->parity = 'E';
			break;
		case 'O':
		case 'o':
			cs->parity = 'O';
			break;
		case 'N':
		case 'n':
			cs->parity = 'N';
			break;
		default:
			nf_log_error(&nflog, 0, "invalid parity parameter");
			return (-1);
		}
	}

	if (quit)
		return (0);

	/*
	 * Stopbits.
	 *
	 * Do not validate stopbits accordance to databits
	 * (e.g. 1.5 stopbits -> 5 databits). Let the hardware do this.
	 */

	str = p + 1;

	p = strchr(str, ',');
	if (p == NULL) {
		len = strlen(str);
		p = (char *)str + len;
		quit = 1;
	}

	len = p - str;
	if (len == 1) {
		if (*str == '1')
			cs->stopbits = 1;
		else if (*str == '2')
			cs->stopbits = 2;
		else {
			nf_log_error(&nflog, 0, "invalid stopbits parameter");
			return (-1);
		}
	} else if (len == 3) {
		if (str[0] == '1' && str[1] == '.' && str[2] == '5') {
			cs->stopbits = 2;
		} else {
			nf_log_error(&nflog, 0, "invalid stopbits parameter");
			return (-1);
		}
	} else if (len == 0)
		;	/* we use default value */
	else {
		nf_log_error(&nflog, 0, "invalid stopbits parameter");
		return (-1);
	}

	if (quit)
		return (0);

	/*
	 * Flow control.
	 */

	str = p + 1;

	len = strlen(str);
	if (len > 1) {
		nf_log_error(&nflog, 0, "invalid flow control parameter");
		return (-1);
	} else if (len > 0) {
		switch (*str) {
		case 'X':
		case 'x':
			cs->flowcontrol = 'X';
			break;
		case 'H':
		case 'h':
			cs->flowcontrol = 'H';
			break;
		case 'N':
		case 'n':
			cs->flowcontrol = 'N';
			break;
		default:
			nf_log_error(&nflog, 0,
			    "invalid flow control parameter");
			return (-1);
		}
	}

	return (0);
}

int
create_tty_server(const char *path, struct tty_settings *cs)
{
	int fd;
	struct termios tc;
	int error;

	fd = open(path, O_RDWR);
	if (fd == -1) {
		nf_log_error(&nflog, errno, "can't open tty device");
		return (-1);
	}

	error = tcgetattr(fd, &tc);
	if (error == -1) {
		nf_log_error(&nflog, errno, "can't get tty device settings");
		close(fd);
		return (-1);
	}

	error = cfsetspeed(&tc, cs->baudrate);
	if (error == -1) {
		nf_log_error(&nflog, 0, "can't set baudrate");
		close(fd);
		return (-1);
	}

	tc.c_cflag &= ~CSIZE;
	switch (cs->databits) {
	case 5:
		tc.c_cflag |= CS5;
		break;
	case 6:
		tc.c_cflag |= CS6;
		break;
	case 7:
		tc.c_cflag |= CS7;
		break;
	case 8:
		tc.c_cflag |= CS8;
		break;
	}

	switch (cs->parity) {
	case 'E':
		tc.c_cflag |= PARENB;
		tc.c_cflag &= ~PARODD;
		break;
	case 'O':
		tc.c_cflag |= PARENB;
		tc.c_cflag |= PARODD;
		break;
	case 'N':
		tc.c_cflag &= ~PARENB;
		break;
	}

	if (cs->stopbits == 2)
		tc.c_cflag |= CSTOPB;
	else
		tc.c_cflag &= ~CSTOPB;

	switch (cs->flowcontrol) {
	case 'X':
	case 'N':
		tc.c_cflag &= ~CHWFLOW;
		/* XXX - Xon/Xoff currently not supported */
		break;
	case 'H':
		tc.c_cflag |= CHWFLOW;
		break;
	}

	tc.c_cflag |= CLOCAL;
	tc.c_iflag &= ~(ISTRIP|ICRNL);
	tc.c_oflag &= ~OPOST;
	tc.c_lflag &= ~(ICANON|ISIG|IEXTEN|ECHO);
	tc.c_cc[VMIN] = 1;
	tc.c_cc[VTIME] = 0;

	error = tcsetattr(fd, TCSAFLUSH, &tc);
	if (error == -1) {
		nf_log_error(&nflog, errno, "can't configure tty device");
		close(fd);
		return (-1);
	}

	return (fd);
}

int
create_tty_client(const char *path, struct tty_settings *cs)
{
	return create_tty_server(path, cs);
}

int
_evlist_alloc(struct nf_events *events, u_int num, u_int *idx)
{
	struct kevent *old;
	u_int total;

	total = num + events->num;
	if (total > events->size) {
		old = events->evlist;
		events->evlist = (struct kevent *)realloc(events->evlist,
		    total * sizeof(*events->evlist));
		if (events->evlist == NULL) {
			nf_log_error(&nflog, errno, "realloc");
			events->evlist = old;
			return (-1);
		}

		events->size = total;
	}

	*idx = events->num;
	events->num += num;
	return (0);
}

int
evlist_init(struct nf_events *events)
{
	events->kfd = kqueue();
	if (events->kfd == -1) {
		nf_log_error(&nflog, errno, "kqueue");
		return (-1);
	}

	events->evlist = NULL;
	events->size = events->num = 0;

	events->ev = (struct kevent *)malloc(sizeof(*events->ev) * 100);
	if (events->ev == NULL) {
		nf_log_error(&nflog, errno, "malloc");
		return (-1);
	}

	return (0);
}

void
evlist_final(struct nf_events *events)
{
	close(events->kfd);
	free(events->ev);
	free(events->evlist);
}

void
evlist_delete(struct nf_events *events, int sock)
{
	struct kevent *p;
	u_int i;

	for (i = 0; i < events->num; ) {
		if ((int)events->evlist[i].ident == sock) {
			if (i != events->num - 1) {
				memmove(events->evlist + i,
				    events->evlist + i + 1,
				    events->num - i - 1);
			}
			events->num--;
		} else
			i++;
	}

	/*
	 * If size of evlist is greater than the number of actual evlist
	 * elements in 3 times, shrink it to 1.5 number of evlist elements.
	 * But do it if size is greater than 100.
 	 */
	if (events->size > 100 && events->num * 3 < events->size) {
		events->size = events->num + events->num / 2;
		p = (struct kevent *)realloc(events->evlist,
		    events->size * sizeof(*events->evlist));
		if (p != NULL)
			events->evlist = p;
	}
}

void
evlist_enable(struct nf_events *events, int fd, int isread, int enable)
{
	u_int i;

	for (i = 0; i < events->num; i++) {
		if ((int)events->evlist[i].ident == fd) {
			if (!isread)
				i++;
			if (enable) {
				events->evlist[i].flags &= ~EV_DISABLE;
				events->evlist[i].flags |= EV_ENABLE;
			} else {
				events->evlist[i].flags &= ~EV_ENABLE;
				events->evlist[i].flags |= EV_DISABLE;
			}
			if (isread == 2)
				isread = 1;
			else
				return;
		}
	}
}

int
evlist_add(struct nf_events *events, int src_sock, int dst_sock,
    int isconnected)
{
	u_int idx;
	int rd, wr;
	int error;

	error = _evlist_alloc(events, 4, &idx);
	if (error == -1)
		return (-1);

	EV_SET(events->evlist + idx + 0, src_sock, EVFILT_READ,
	    EV_ADD | EV_ENABLE, NOTE_LOWAT, 1, NULL);
	EV_SET(events->evlist + idx + 1, src_sock, EVFILT_WRITE,
	    EV_ADD | EV_DISABLE, NOTE_LOWAT, 1, NULL);

	if (isconnected) {
		rd = EV_ENABLE;
		wr = EV_DISABLE;
	} else {
		rd = EV_DISABLE;
		wr = EV_ENABLE;
	}

	/*
	 * Keep this order, other evlist_xxx functions rely on it:
	 * 1st - EVFILT_READ, 2nd - EVFILT_WRITE.
	 */
	EV_SET(events->evlist + idx + 2, dst_sock, EVFILT_READ,
	    EV_ADD | rd, NOTE_LOWAT, 1, NULL);
	EV_SET(events->evlist + idx + 3, dst_sock, EVFILT_WRITE,
	    EV_ADD | wr, NOTE_LOWAT, 1, NULL);
	return (0);
}

int
evlist_add_server(struct nf_events *events, int sock)
{
	u_int idx;
	int error;

	error = _evlist_alloc(events, 1, &idx);
	if (error == -1)
		return (-1);

	EV_SET(events->evlist + idx, sock, EVFILT_READ, EV_ADD | EV_ENABLE, 0,
	    0, NULL);
	return (0);
}

void
evlist_change(struct nf_events *events, int sock, int isconnected, int old)
{
	u_int i;
	int rd, wr;

	if (isconnected) {
		rd = EV_ENABLE;
		wr = EV_DISABLE;
	} else {
		rd = EV_DISABLE;
		wr = EV_ENABLE;
	}

	for (i = 0; i < events->num; i++) {
		if ((int)events->evlist[i].ident == old) {
			if (events->evlist[i].filter == EVFILT_READ) {
				EV_SET(events->evlist + i, sock, EVFILT_READ,
				    EV_ADD | rd, NOTE_LOWAT, 1, NULL);
			} else {
				EV_SET(events->evlist + i, sock, EVFILT_WRITE,
				    EV_ADD | wr, NOTE_LOWAT, 1, NULL);

				/* EVFILT_WRITE is the last for ``ident'' */
				return;
			}
		}
	}
}

int
nf_addr_init(struct net_fwd_addr *addr, int af, int proto, const char *host,
    const char *port)
{
	struct addrinfo hints, *ai0;
	size_t len;
	int error;

	memset(addr, 0, sizeof(*addr));

	addr->af = af;
	addr->proto = proto;

	if (af == NF_AF_INET || af == NF_AF_INET4 ||
	    af == NF_AF_INET6) {
		memset(&hints, 0, sizeof(hints));
		hints.ai_flags = AI_PASSIVE;
		switch (af) {
		case NF_AF_INET4:
			hints.ai_family = AF_INET;
			break;
		case NF_AF_INET6:
			hints.ai_family = AF_INET6;
			break;
		default:
			hints.ai_family = AF_UNSPEC;
			break;
		}
		if (proto == NF_PROTO_STREAM)
			hints.ai_socktype = SOCK_STREAM;
		else
			hints.ai_socktype = SOCK_DGRAM;
		error = getaddrinfo(host, port, &hints, &ai0);
		if (error != 0) {
			nf_log_error(&nflog, 0,
			    "can't resolve address \"%s\" or port \"%s\"",
			    (host == NULL) ? "" : host,
			    (port == NULL) ? "" : port);
			return (-1);
		}
		addr->spec.ai0 = ai0;
	} else if (af == NF_AF_UNIX) {
		len = strlcpy(addr->spec.path, port, sizeof(addr->spec.path));
		if (len >= sizeof(addr->spec.path)) {
			nf_log_error(&nflog, 0, "unix socket path is too long");
			return (-1);
		}
	} else if (af == NF_AF_TTY) {
		len = strlcpy(addr->spec.tty.path, host,
		    sizeof(addr->spec.tty.path));
		if (len >= sizeof(addr->spec.tty.path)) {
			nf_log_error(&nflog, 0, "device name is too long");
			return (-1);
		}

		error = tty_parse_settings(port, &addr->spec.tty.cfg);
		if (error == -1)
			return (-1);
	} else {
		nf_log_error(&nflog, 0,
		    "unsupported address family or protocol");
		return (-1);
	}
	return (0);
}

void
nf_addr_final(struct net_fwd_addr *addr)
{
	if ((addr->af == NF_AF_INET || addr->af == NF_AF_INET4 ||
	    addr->af == NF_AF_INET6) && addr->spec.ai0 != NULL)
		freeaddrinfo(addr->spec.ai0);
}

u_int
fix_rlimit(void)
{
	struct rlimit lim;
	rlim_t old;
	u_int rv;
	int error;

	error = getrlimit(RLIMIT_NOFILE, &lim);
	if (error == -1) {
		nf_log_warn(&nflog, 0, "can't get ``openfiles'' limit\n"
		    "not greater than %u file descriptors will be available",
		    _MAX_DESCRIPTORS);
		return (_MAX_DESCRIPTORS);
	}

	if (lim.rlim_cur == RLIM_INFINITY)
		return (_MAX_DESCRIPTORS);

	rv = _MAX_DESCRIPTORS;
	if (lim.rlim_max != RLIM_INFINITY &&
	    lim.rlim_cur < lim.rlim_max &&
	    lim.rlim_max < _MAX_DESCRIPTORS)
		rv = lim.rlim_max;

	if (lim.rlim_cur != rv) {
		/*
		 * Adjust soft limit for ``open file number'' and use this
		 * number as number of available sockets.
		 */
		old = lim.rlim_cur;
		lim.rlim_cur = rv;
		error = setrlimit(RLIMIT_NOFILE, &lim);
		if (error == -1) {
			nf_log_warn(&nflog, 0, "can't set ``openfiles'' limit. "
			    "%qu file descriptors will be available", old);
			return (old);
		}
	}
	return (rv);
}

int
_nf_add_listener(struct net_fwd *nf, int fd, int af)
{
	int error;

	error = evlist_add_server(&nf->ev, fd);
	if (error == -1)
		return (-1);

	memset(nf->conn + fd, 0, sizeof(nf->conn[fd]));
	nf->conn[fd].sock = -1;
	nf->conn[fd].flags = NF_CONN_ACCEPT;
	nf->conn[fd].af = af;
	nf->conn[fd].proto = NF_PROTO_STREAM;
	return (0);
}

int
_nf_add_connector(struct net_fwd *nf, int af1, int proto1, int fd1, int af2,
    int proto2, int fd2, int isconnected)
{
	int error;

	error = evlist_add(&nf->ev, fd1, fd2, isconnected);
	if (error == -1)
		return (-1);

	memset(nf->conn + fd1, 0, sizeof(nf->conn[fd1]));
	nf->conn[fd1].sock = fd2;
	nf->conn[fd1].flags = NF_CONN_CONNECTED | NF_CONN_LOCAL;
	nf->conn[fd1].af = af1;
	nf->conn[fd1].proto = proto1;

	memset(nf->conn + fd2, 0, sizeof(nf->conn[fd2]));
	nf->conn[fd2].sock = fd1;
	nf->conn[fd2].flags = (isconnected) ? NF_CONN_CONNECTED : 0;
	nf->conn[fd2].af = af2;
	nf->conn[fd2].proto = proto2;
	nf->conn[fd2].ai = nf->remote.spec.ai0;
	return (0);
}

void
_nf_change_connector(struct net_fwd *nf, int fd1, int fd2, int isconnected,
    int old)
{
	evlist_change(&nf->ev, fd2, isconnected, old);

	nf->conn[fd1].sock = fd2;

	memset(nf->conn + fd2, 0, sizeof(nf->conn[fd2]));
	nf->conn[fd2].sock = fd1;
	nf->conn[fd2].flags = (isconnected) ? NF_CONN_CONNECTED : 0;
	nf->conn[fd2].af = nf->conn[old].af;
	nf->conn[fd2].proto = nf->conn[old].proto;
	nf->conn[fd2].ai = nf->conn[old].ai;
}

int
nf_init(struct net_fwd *nf, int family, int proto, const char *host,
    const char *port, int rfamily, int rproto, const char *rhost,
    const char *rport)
{
	int error;

	memset(nf, 0, sizeof(*nf));

	nf->n_conn = fix_rlimit();
	nf->conn = (struct nf_conn *)calloc(nf->n_conn, sizeof(*nf->conn));
	if (nf->conn == NULL) {
		nf_log_error(&nflog, errno, "calloc");
		return (-1);
	}

	error = evlist_init(&nf->ev);
	if (error == 0) {
		error = nf_addr_init(&nf->local, family, proto, host, port);
		if (error == 0) {
			error = nf_addr_init(&nf->remote, rfamily, rproto,
			    rhost, rport);
			if (error == 0)
				return (0);

			nf_addr_final(&nf->local);
		}
		evlist_final(&nf->ev);
	}
	free(nf->conn);
	return (-1);
}

int
nf_remote_conn(struct net_fwd *nf, int fd1)
{
	int fd2;
	struct addrinfo *ai;
	int socktype, inprogress;
	int error;

	if (nf->remote.af == NF_AF_UNIX) {
		socktype = (nf->remote.proto == NF_PROTO_STREAM) ?
		    SOCK_STREAM : SOCK_DGRAM;
		inprogress = 0;
		fd2 = create_unix_client(nf->remote.spec.path, socktype);
	} else if (nf->remote.af == NF_AF_TTY) {
		inprogress = 0;
		socktype = SOCK_DGRAM;	/* XXX - to avoid shutdown() */
		fd2 = create_tty_client(nf->remote.spec.tty.path,
		    &nf->remote.spec.tty.cfg);
	} else {
		socktype = nf->remote.spec.ai0->ai_socktype;
		fd2 = -1;
		for (ai = nf->remote.spec.ai0; ai != NULL; ai = ai->ai_next) {
			fd2 = create_inet_client(ai, &inprogress);
			if (fd2 != -1)
				break;

			if (ai->ai_next != NULL)
				nf_log_info(&nflog,
				    "Have another address to reconnect");
		}

		if (fd2 != -1)
			nf->conn[fd2].ai = ai;
	}
	if (fd2 == -1)
		return (-1);

	error = _nf_add_connector(nf, nf->local.af, nf->local.proto, fd1,
	    nf->remote.af, nf->remote.proto, fd2, !inprogress);
	if (error == -1) {
		if (socktype == SOCK_STREAM)
			shutdown(fd2, SHUT_RDWR);
		close(fd2);
		return (-1);
	}

	return (0);
}

int
nf_create_listener(struct net_fwd *nf)
{
	int fd;
	int af;
	u_int n_srv;
	struct addrinfo *ai;
	int error;

	if (nf->local.proto != NF_PROTO_STREAM)
		return (0);	/* listening is not supported */

	if (nf->local.af == NF_AF_UNIX) {
		fd = create_unix_server(nf->local.spec.path, SOCK_STREAM);
		if (fd == -1)
			return (-1);

		error = _nf_add_listener(nf, fd, NF_AF_UNIX);
		if (error == -1)
			return (-1);

		return (0);
	}

	/* Internet listener here */
	n_srv = 0;
	for (ai = nf->local.spec.ai0; ai != NULL; ai = ai->ai_next) {
		fd = create_inet_server(ai);
		if (fd == -1)
			return (-1);

		af = NF_AF_INET4;
		if (ai->ai_family == AF_INET6)
			af = NF_AF_INET6;
		else if (ai->ai_family != AF_INET) {
			nf_log_warn(&nflog, 0, "unsupported address family");
			continue;
		}

		error = _nf_add_listener(nf, fd, af);
		if (error == -1)
			return (-1);

		n_srv++;
	}

	if (n_srv == 0) {
		nf_log_warn(&nflog, 0,
		    "there are no local addresses to listen");
		return (-1);
	}

	return (0);
}

int
nf_create_connector(struct net_fwd *nf)
{
	int fd1;
	int error;

	if (nf->local.proto == NF_PROTO_STREAM)
		return (0);

	if (nf->local.af == NF_AF_UNIX)
		fd1 = create_unix_server(nf->local.spec.path, SOCK_DGRAM);
	else if (nf->local.af == NF_AF_TTY)
		fd1 = create_tty_server(nf->local.spec.tty.path,
		    &nf->local.spec.tty.cfg);
	else
		fd1 = create_inet_server(nf->local.spec.ai0);
	if (fd1 == -1)
		return (-1);

	error = nf_remote_conn(nf, fd1);
	if (error == -1) {
		close(fd1);
		return (-1);
	}

	nf->conn[fd1].flags |= NF_CONN_CONNECTOR;
	return (0);
}

int
nf_loop(struct net_fwd *nf)
{
	int n;
	int error;

	for (;;) {
		n = kevent(nf->ev.kfd, nf->ev.evlist, nf->ev.num, nf->ev.ev,
		    100, NULL);
		if (n == -1) {
			nf_log_error(&nflog, errno, "kevent");
			return (-1);
		}

		error = nf_loop_events(nf, n);
		if (error == -1)
			return (-1);
	}
	/* NOTREACHED */
}

int
nf_loop_events(struct net_fwd *nf, int n)
{
	int sock, i;
	ssize_t sz;

	for (i = 0; i < n; i++) {
		sock = (int)nf->ev.ev[i].ident;

		if ((nf->conn[sock].flags & NF_CONN_CLOSED) != 0)
			continue;
		else if ((nf->conn[sock].flags & NF_CONN_ACCEPT) != 0)
			nf_accept_conn(nf, sock, nf->ev.ev[i].data);
		else if ((nf->conn[sock].flags & NF_CONN_CONNECTED) == 0)
			nf_complete_conn(nf, sock);
		else if (nf->ev.ev[i].filter == EVFILT_READ) {
			sz = nf_read_conn(nf, sock, nf->ev.ev[i].data);
			if (sz == -1)
				return (-1);
		} else if (nf->ev.ev[i].filter == EVFILT_WRITE) {
			sz = nf_write_conn(nf, sock);
			if (sz == -1)
				return (-1);
		}
	}
	return (0);
}

int
nf_accept_conn(struct net_fwd *nf, int sock, int num)
{
	int fd1;
	int i;
	char host[NI_MAXHOST], hostport[NI_MAXSERV];
	int error;

	for (i = 0; i < num; i++) {
		nf->conn[sock].peerlen = sizeof(nf->conn[sock].peer);
		fd1 = accept(sock, (struct sockaddr *)&nf->conn[sock].peer,
		    &nf->conn[sock].peerlen);
		if (fd1 == -1) {
			nf_log_error(&nflog, errno,
			    "can't accept incoming connection");
			continue;
		}

		if (nf->local.af != NF_AF_UNIX) {
			if (getnameinfo((struct sockaddr *)&nf->conn[sock].peer,
			    nf->conn[sock].peerlen, host, sizeof(host),
			    hostport, sizeof(hostport),
			    NI_NUMERICHOST | NI_NUMERICSERV) != 0) {
				strlcpy(host, "unknown", sizeof(host));
				strlcpy(hostport, "unknown", sizeof(hostport));
			}
			nf_log_info(&nflog, "Incoming connection from %s "
			    "port %s", host, hostport);
		} else
			nf_log_info(&nflog, "Incoming connection to socket");

		error = nf_remote_conn(nf, fd1);
		if (error == -1) {
			close(fd1);
			continue;
		}

		memcpy(&nf->conn[fd1].peer, &nf->conn[sock].peer,
		    nf->conn[sock].peerlen);
		nf->conn[fd1].peerlen = nf->conn[sock].peerlen;
	}

	return (0);
}

int
nf_complete_conn(struct net_fwd *nf, int sock)
{
	struct addrinfo *ai;
	int sock2, inprogress;
	int error;

	ai = nf->conn[sock].ai;
	if (ai == NULL) {
		nf_log_error(&nflog, 0,
		    "empty address for incomplete connection");
		nf_close_conn(nf, sock, 0);
		return (-1);
	}

	error = complete_inet_client(sock, ai);
	if (error == -1) {
		/*
		 * We shouldn't close ``sock'' here because if all other
		 * create_inet_client() calls fail it's very convenient to call
		 * nf_close_conn() to clean up connections' data.
 		 */
		sock2 = -1;
		for (ai = ai->ai_next; ai != NULL; ai = ai->ai_next) {
			nf_log_info(&nflog,
			    "Have another address to reconnect");

			sock2 = create_inet_client(ai, &inprogress);
			if (sock2 != -1)
				break;
		}

		nf->conn[sock].ai = ai;	/* save current address */

		if (sock2 != -1) {
			_nf_change_connector(nf, nf->conn[sock].sock, sock2,
			    !inprogress, sock);
			close(sock);
			return (0);
		}
	}

	if (error == -1)
		return nf_close_conn(nf, sock, 0);

	nf->conn[sock].flags |= NF_CONN_CONNECTED;
	evlist_enable(&nf->ev, sock, 1, 1);
	if (nf->conn[nf->conn[sock].sock].buflen == 0)
		evlist_enable(&nf->ev, sock, 0, 0);
	return (0);
}

ssize_t
nf_read_conn(struct net_fwd *nf, int fd, int size)
{
	ssize_t rdsz;
	size_t nbytes;
	struct nf_conn *conn2;

	if (size < 0) {
		nf_log_warn(&nflog, 0, "reading negative size");
		return (-1);
	}

	nbytes = sizeof(nf->conn[fd].buf) - nf->conn[fd].buflen;
	if (nbytes > (size_t)size)
		nbytes = size;

	do {
		if (nf->conn[fd].proto == NF_PROTO_DGRAM) {
			nf->conn[fd].peerlen = sizeof(nf->conn[fd].peer);
			rdsz = recvfrom(fd, nf->conn[fd].buf +
			    nf->conn[fd].buflen, nbytes, 0,
			    (struct sockaddr *)&nf->conn[fd].peer,
			    &nf->conn[fd].peerlen);
		} else {
			rdsz = read(fd, nf->conn[fd].buf + nf->conn[fd].buflen,
			    nbytes);
		}
	} while (rdsz == -1 && errno == EAGAIN);/* if still non-blocking */

	if (rdsz == -1 || rdsz == 0) {
		if (rdsz == -1) {
			nf_log_error(&nflog, errno,
			    "read error, broken connection");
		}
		return nf_close_conn(nf, fd, 1);
	}

	nf->conn[fd].buflen += rdsz;

	conn2 = nf->conn + nf->conn[fd].sock;

	/*
	 * Do not enable writing to UDP socket if we haven't get incoming
	 * packets (recvfrom() must give us peer's address).
	 */
	if (conn2->proto != NF_PROTO_DGRAM ||
	    (conn2->flags & NF_CONN_LOCAL) == 0 ||
	    conn2->peerlen != 0)
		evlist_enable(&nf->ev, nf->conn[fd].sock, 0, 1);

	/*
	 * We should enable fd's conn writer for UDP local socket if
	 * conn2 has data.
	 */
	if (nf->conn[fd].proto == NF_PROTO_DGRAM ||
	    (nf->conn[fd].flags & NF_CONN_LOCAL) != 0 || conn2->buflen > 0)
		evlist_enable(&nf->ev, fd, 0, 1);

	if (nf->conn[fd].buflen == sizeof(nf->conn[fd].buf))
		evlist_enable(&nf->ev, fd, 1, 0);

	return (rdsz);
}

ssize_t
nf_write_conn(struct net_fwd *nf, int fd)
{
	struct nf_conn *conn2;
	ssize_t wrsz;

	conn2 = nf->conn + nf->conn[fd].sock;

	if (conn2->buflen == 0)
		return (0);

	do {
		if (nf->conn[fd].proto == NF_PROTO_DGRAM &&
		    (nf->conn[fd].flags & NF_CONN_LOCAL) != 0) {
			if (nf->conn[fd].peerlen == 0)
				return (0);

			wrsz = sendto(fd, conn2->buf, conn2->buflen, 0,
			    (struct sockaddr *)&nf->conn[fd].peer,
			    nf->conn[fd].peerlen);
		} else
			wrsz = write(fd, conn2->buf, conn2->buflen);
	} while (wrsz == -1 && errno == EAGAIN);/* if still non-blocking */

	if (wrsz == -1 || wrsz == 0) {
		if (wrsz == -1) {
			nf_log_error(&nflog, errno,
			    "write error, broken connection");
		}
		return nf_close_conn(nf, fd, 0);
	}

	conn2->buflen -= wrsz;
	if (conn2->buflen > 0)
		memmove(conn2->buf, conn2->buf + wrsz, conn2->buflen);

	evlist_enable(&nf->ev, nf->conn[fd].sock, 1, 1);
	if (conn2->buflen == 0) {
		evlist_enable(&nf->ev, fd, 0, 0);
		if ((conn2->flags & NF_CONN_CLOSING) != 0)
			return nf_close_conn(nf, fd, 0);
	}
	return (wrsz);
}

int
nf_close_conn(struct net_fwd *nf, int sock, int isread)
{
	int sock2;
	struct nf_conn *conn2, *conndbg;
	char host[NI_MAXHOST], hostport[NI_MAXSERV];

	sock2 = nf->conn[sock].sock;
	conn2 = nf->conn + sock2;
	if (isread) {
		/* do not receive data from the socket */
		evlist_enable(&nf->ev, sock, 1, 0);

		if ((conn2->flags & NF_CONN_CLOSING) == 0 &&
		    nf->conn[sock].buflen > 0) {
			nf->conn[sock].flags |= NF_CONN_CLOSING;
			return (0);
		}
	}

	evlist_delete(&nf->ev, sock2);
	evlist_delete(&nf->ev, sock);

	nf->conn[sock].flags |= NF_CONN_CLOSED;
	conn2->flags |= NF_CONN_CLOSED;

	conndbg = ((conn2->flags & NF_CONN_LOCAL) != 0) ? conn2 :
	    (nf->conn + sock);
	if ((conndbg->flags & NF_CONN_CONNECTED) != 0) {
		switch (nf->local.af) {
		case NF_AF_INET:
		case NF_AF_INET4:
		case NF_AF_INET6:
			if (getnameinfo((struct sockaddr *)&conndbg->peer,
			    conndbg->peerlen, host, sizeof(host), hostport,
			    sizeof(hostport),
			    NI_NUMERICHOST | NI_NUMERICSERV) != 0) {
				strlcpy(host, "unknown", sizeof(host));
				strlcpy(hostport, "unknown", sizeof(hostport));
			}
			nf_log_info(&nflog, "Closing incoming connection "
			    "from %s port %s", host, hostport);
			break;
		case NF_AF_UNIX:
			nf_log_info(&nflog,
			    "Closing incoming connection to socket");
			break;
		}

		if (nf->conn[conndbg->sock].proto == NF_PROTO_STREAM)
			shutdown(conndbg->sock, SHUT_RDWR);
	}
	close(conndbg->sock);

	conndbg = nf->conn + conndbg->sock;
	if ((conndbg->flags & NF_CONN_CONNECTED) != 0) {
		switch (nf->remote.af) {
		case NF_AF_INET:
		case NF_AF_INET4:
		case NF_AF_INET6:
			if (getnameinfo(
			    (struct sockaddr *)nf->remote.spec.ai0->ai_addr,
			    nf->remote.spec.ai0->ai_addrlen, host, sizeof(host),
			    hostport, sizeof(hostport),
			    NI_NUMERICHOST | NI_NUMERICSERV) != 0) {
				strlcpy(host, "unknown", sizeof(host));
				strlcpy(hostport, "unknown", sizeof(hostport));
			}
			nf_log_info(&nflog, "Closing outgoing connection "
			    "to %s port %s", host, hostport);
			break;
		case NF_AF_UNIX:
			nf_log_info(&nflog, "Closing outgoing connection "
			    "to \"%s\" socket", nf->remote.spec.path);
			break;
		}

		if (nf->conn[conndbg->sock].proto == NF_PROTO_STREAM)
			shutdown(conndbg->sock, SHUT_RDWR);
	}
	close(conndbg->sock);

	return nf_create_connector(nf);
}

int
main(int argc, char * const *argv)
{
	struct net_fwd nf;
	char *srcaddr, *srcport;
	char *dstaddr, *dstport;
	int idx, ch;
	int local_af, remote_af, local_proto, remote_proto;
	int odaemon, overbose, oquiet;
	int error;

	odaemon = 0;
	overbose = 0;
	oquiet = 0;
	srcaddr = NULL;

	while ((ch = getopt(argc, argv, "dhqs:v")) != -1) {
		switch (ch) {
		case 'd':
			odaemon = 1;
			break;
		case 'q':
			/* Absolute silence */
			oquiet = 1;
			overbose = 0;
			break;
		case 's':
			srcaddr = optarg;
			break;
		case 'v':
			oquiet = 0;
			overbose = 1;
			break;
		case 'h':
		default:
			usage();
			/* NOTREACHED */
		}
	}

	argc -= optind;
	argv += optind;

	if (argc < 2) {
		usage();
		/* NOTREACHED */
	}

	local_af = remote_af = NF_AF_INET;
	idx = 1;

	/*
	 * Local.
	 */
	if (strcasecmp(*argv, "inet") == 0)
		local_af = NF_AF_INET4;
	else if (strcasecmp(*argv, "inet6") == 0)
		local_af = NF_AF_INET6;
	else if (strcasecmp(*argv, "unix") == 0)
		local_af = NF_AF_UNIX;
	else if (strcasecmp(*argv, "tty") == 0)
		local_af = NF_AF_TTY;
	else
		idx = 0;

	if (idx > 0 && argc < 3) {
		usage();
		/* NOTREACHED */
	}

	srcport = NULL;

	switch (local_af) {
	case NF_AF_INET:
	case NF_AF_INET4:
	case NF_AF_INET6:
		if (strcasecmp(argv[idx], "tcp") == 0)
			local_proto = NF_PROTO_STREAM;
		else if (strcasecmp(argv[idx], "udp") == 0)
			local_proto = NF_PROTO_DGRAM;
		else {
			if (!oquiet) {
				if (local_af == NF_AF_INET)
					warnx("unsupported local address "
					    "family");
				else
					warnx("unsupported local protocol");
			}
			usage();
			/* NOTREACHED */
		}

		srcport = argv[idx + 1];
		break;
	case NF_AF_UNIX:
		if (strcasecmp(argv[idx], "stream") == 0)
			local_proto = NF_PROTO_STREAM;
		else if (strcasecmp(argv[idx], "dgram") == 0)
			local_proto = NF_PROTO_DGRAM;
		else {
			if (!oquiet)
				warnx("unsupported local protocol");
			usage();
			/* NOTREACHED */
		}

		srcport = argv[idx + 1];
		break;
	case NF_AF_TTY:
		local_proto = NF_PROTO_TTY;
		srcaddr = argv[idx];
		srcport = argv[idx + 1];
		break;
	default:
		if (!oquiet)
			warnx("unsupported local address family");
		usage();
		/* NOTREACHED */
	}

	argc -= idx + 2;
	argv += idx + 2;

	if (argc < 1) {
		usage();
		/* NOTREACHED */
	}

	idx = 1;

	/*
	 * Remote.
	 */
	if (strcasecmp(*argv, "inet") == 0)
		remote_af = NF_AF_INET4;
	else if (strcasecmp(*argv, "inet6") == 0)
		remote_af = NF_AF_INET6;
	else if (strcasecmp(*argv, "unix") == 0)
		remote_af = NF_AF_UNIX;
	else if (strcasecmp(*argv, "tty") == 0)
		remote_af = NF_AF_TTY;
	else
		idx = 0;

	dstaddr = dstport = NULL;

	switch (remote_af) {
	case NF_AF_INET:
	case NF_AF_INET4:
	case NF_AF_INET6:
		if (argc < 3 + idx) {
			usage();
			/* NOTREACHED */
		}

		if (strcasecmp(argv[idx], "tcp") == 0)
			remote_proto = NF_PROTO_STREAM;
		else if (strcasecmp(argv[idx], "udp") == 0)
			remote_proto = NF_PROTO_DGRAM;
		else {
			if (!oquiet) {
				if (remote_af == NF_AF_INET)
					warnx("unsupported remote address "
					    "family");
				else
					warnx("unsupported remote protocol");
			}
			usage();
			/* NOTREACHED */
		}

		dstaddr = argv[idx + 1];
		dstport = argv[idx + 2];
		break;
	case NF_AF_UNIX:
		if (argc < 3) {
			usage();
			/* NOTREACHED */
		}

		if (strcasecmp(argv[1], "stream") == 0)
			remote_proto = NF_PROTO_STREAM;
		else if (strcasecmp(argv[1], "dgram") == 0)
			remote_proto = NF_PROTO_DGRAM;
		else {
			if (!oquiet)
				warnx("unsupported remote protocol");
			usage();
			/* NOTREACHED */
		}

		dstport = argv[2];
		break;
	case NF_AF_TTY:
		if (argc < 3) {
			usage();
			/* NOTREACHED */
		}

		remote_proto = NF_PROTO_TTY;
		dstaddr = argv[1];
		dstport = argv[2];
		break;
	default:
		if (!oquiet)
			warnx("unsupported remote address family");
		usage();
		/* NOTREACHED */
	}

	if (odaemon) {
		error = daemon(0, 0);
		if (error == -1) {
			if (!oquiet)
				warnx("can't become daemon");
			exit(1);
		}
	}

	/*
	 * Probably we are in daemon mode here. So we need our logger to
	 * be initialized and used.
	 */
	nf_log_open(&nflog, odaemon, oquiet, overbose);

	signal(SIGPIPE, SIG_IGN);

	error = nf_init(&nf, local_af, local_proto, srcaddr, srcport,
	    remote_af, remote_proto, dstaddr, dstport);
	if (error == -1)
		exit(1);

	error = nf_create_listener(&nf);
	if (error == -1)
		exit(1);

	error = nf_create_connector(&nf);
	if (error == -1)
		exit(1);

	nf_loop(&nf);
	exit(1);
}
