/*

Copyright (C) 2004 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#include "nd_ta.h"

#include <nd_trace.h>
#include <nd_trace_registry.h>
#include <nd_dialog.h>
#include <nd_prefs.h>

#include "interface.h"
#include "callbacks.h"

static GtkWidget *dialog;

static void
ta_sync_trace_list_cb(LND_Trace *trace, GList **items)
{
  GtkWidget *item; 

  item = gtk_list_item_new_with_label(libnd_trace_get_name(trace));
  gtk_widget_show(item);
  *items = g_list_append(*items, item);
}

static void
ta_sync_trace_list(GtkWidget *dialog)
{
  GtkWidget *list;
  GList *items = NULL;

  ND_GTK_GET(list, dialog, "trace_list");
  gtk_list_clear_items(GTK_LIST(list), 0, -1);
  nd_trace_registry_foreach((ND_TraceFunc) ta_sync_trace_list_cb, &items);

  gtk_list_append_items(GTK_LIST(list), items);
}

static ND_TAData *
ta_data_new(void)
{
  ND_TAData *data;

  if (! (data = g_new0(ND_TAData, 1)))
    {
      D(("Out of memory.\n"));
      return NULL;
    }
  
  if (! (data->traces = libnd_traceset_new()))
    {
      D(("Out of memory.\n"));
      g_free(data);
      return NULL;
    }

  return data;
}

static void
ta_data_free(ND_TAData *data)
{
  if (!data)
    return;

  libnd_traceset_free(data->traces);

  g_free(data->output_dir);
  g_free(data);
}


static void
ta_get_traces_from_app_cb(LND_Trace *trace, ND_TAData *data)
{
  if (! libnd_traceset_add_trace(data->traces, trace))
    {
      D(("Error when adding trace %s\n", libnd_trace_get_name(trace)));
    }
}

static gboolean
ta_get_traces_from_app(ND_TAData *data)
{
  nd_trace_registry_foreach((ND_TraceFunc) ta_get_traces_from_app_cb, data);
  return TRUE;
}

static gboolean
ta_get_traces_from_file(ND_TAData *data)
{
  char *textfile;
  char  msg[MAXPATHLEN];
  GtkWidget *w;
  
  ND_GTK_GET(w, dialog, "text_in_entry");
  textfile = gtk_entry_get_text(GTK_ENTRY(w));

  if (! libnd_traceset_add_trace_name_list(data->traces, textfile))
    {
      g_snprintf(msg, MAXPATHLEN, _("Could not open input file '%s'."), textfile);
      nd_dialog_message("Input Traces Problem", textfile, TRUE);
      return FALSE;
    }

  return (libnd_traceset_get_size(data->traces) > 0);
}


static void
ta_browse_cb(const char *filename, void *user_data)
{
  GtkWidget *dialog = GTK_WIDGET(user_data);
  GtkWidget *in_entry, *w;

  if (!libnd_misc_can_read(filename))
    {
      nd_dialog_message("File error",
			"Could not read the file you selected.\n",
			TRUE);
      return;
    }
  
  nd_dialog_filesel_close();

  ND_GTK_GET(w, dialog, "ok_button");
  gtk_widget_set_sensitive(w, TRUE);
  
  ND_GTK_GET(w, dialog, "text_in_radiobutton");
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
  
  ND_GTK_GET(in_entry, dialog, "text_in_entry");
  gtk_entry_set_text(GTK_ENTRY(in_entry), filename);
}

void                 
nd_ta_browse_input(void)
{
  nd_dialog_filesel(_("Select Trace List File"),
		    NULL, ta_browse_cb, dialog);  
}

static void
ta_browse_output_cb(const char *filename, void *user_data)
{
  GtkWidget *w;
  char msg[MAXPATHLEN];

  ND_GTK_GET(w, dialog, "out_dir_entry");

  if (libnd_misc_is_dir(filename)    &&
      libnd_misc_can_read(filename)  &&
      libnd_misc_can_write(filename) &&
      libnd_misc_can_exec(filename))
    {
      gtk_entry_set_text(GTK_ENTRY(w), filename);
      nd_dialog_filesel_close();
      return;
    }

  g_snprintf(msg, MAXPATHLEN,
	     _("'%s'\n"
	       "must be a directory you can write to."),
	     filename);
  nd_dialog_message("Directory error", msg, TRUE);

  return;
  TOUCH(user_data);
}


static void
ta_data_do_none(ND_TAData *data)
{
  if (!data)
    return;
  
  data->do_basic     = FALSE;
  data->do_ip        = FALSE;
  data->do_ports     = FALSE;
  data->do_src_ports = FALSE;
  data->do_dst_ports = FALSE;
  data->do_flows     = FALSE;
}


static gboolean
ta_do_per_file_cb(LND_TraceSet *set, LND_Trace *trace, ND_TAData *data)
{
  FILE *f;
  LND_TAnalysis *an;
  char out[MAXPATHLEN];
  int attempt;
  GtkWidget *w;
  
  if (! (an = libnd_ta_new()))
    return FALSE;

  ND_GTK_GET(w, dialog, "ta_progressbar");
  
  libnd_ta_add_trace(an, trace);
  
  an->do_basic     = data->do_basic;
  an->do_ip        = data->do_ip;
  an->do_ports     = data->do_ports;
  an->do_src_ports = data->do_src_ports;
  an->do_dst_ports = data->do_dst_ports;
  an->do_flows     = data->do_flows;
  
  libnd_ta_analyze(an);
  
  /* Now write output with only a single todo-flag set,
   * cycling over all flags.
   */      
  if (data->do_basic)
    {
      attempt = 0;
      do {
	g_snprintf(out, MAXPATHLEN, "%s/nd_ta_%s_%i_basic.txt",
		   data->output_dir,
		   g_basename(libnd_trace_get_name(trace)),
		   attempt++);
      } while (libnd_misc_exists(out));
      
      if ( (f = fopen(out, "w")))
	{
	  libnd_ta_do_none(an);
	  an->do_basic = TRUE;
	  libnd_ta_write_results(an, f);
	  fclose(f);
	}
    }
  
  if (data->do_ip)
    {
      attempt = 0;
      do {
	g_snprintf(out, MAXPATHLEN, "%s/nd_ta_%s_%i_ip.txt",
		   data->output_dir,
		   g_basename(libnd_trace_get_name(trace)),
		   attempt++);
      } while (libnd_misc_exists(out));
      
      if ( (f = fopen(out, "w")))
	{
	  libnd_ta_do_none(an);
	  an->do_ip = TRUE;
	  libnd_ta_write_results(an, f);
	  fclose(f);
	}
    }
  
  if (data->do_ports)
    {
      attempt = 0;
      do {
	g_snprintf(out, MAXPATHLEN, "%s/nd_ta_%s_%i_ports.txt",
		   data->output_dir,
		   g_basename(libnd_trace_get_name(trace)),
		   attempt++);
      } while (libnd_misc_exists(out));
      
      if ( (f = fopen(out, "w")))
	{
	  libnd_ta_do_none(an);
	  an->do_ports = TRUE;
	  libnd_ta_write_results(an, f);
	  fclose(f);
	}
    }

  if (data->do_src_ports)
    {
      attempt = 0;
      do {
	g_snprintf(out, MAXPATHLEN, "%s/nd_ta_%s_%i_src_ports.txt",
		   data->output_dir,
		   g_basename(libnd_trace_get_name(trace)),
		   attempt++);
      } while (libnd_misc_exists(out));
      
      if ( (f = fopen(out, "w")))
	{
	  libnd_ta_do_none(an);
	  an->do_src_ports = TRUE;
	  libnd_ta_write_results(an, f);
	  fclose(f);
	}
    }

  if (data->do_dst_ports)
    {
      attempt = 0;
      do {
	g_snprintf(out, MAXPATHLEN, "%s/nd_ta_%s_%i_dst_ports.txt",
		   data->output_dir,
		   g_basename(libnd_trace_get_name(trace)),
		   attempt++);
      } while (libnd_misc_exists(out));
      
      if ( (f = fopen(out, "w")))
	{
	  libnd_ta_do_none(an);
	  an->do_dst_ports = TRUE;
	  libnd_ta_write_results(an, f);
	  fclose(f);
	}
    }
  
  if (data->do_flows)
    {
      attempt = 0;
      do {
	g_snprintf(out, MAXPATHLEN, "%s/nd_ta_%s_%i_flows.txt",
		   data->output_dir,
		   g_basename(libnd_trace_get_name(trace)),
		   attempt++);
      } while (libnd_misc_exists(out));
      
      if ( (f = fopen(out, "w")))
	{
	  libnd_ta_do_none(an);
	  an->do_flows = TRUE;
	  libnd_ta_write_results(an, f);
	  fclose(f);
	}
    }
  
  libnd_ta_free(an);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(w),
			  0.5 * ((double) ++data->it_count) / libnd_traceset_get_size(set));
  
  return TRUE;
}


static void
ta_do_per_file(ND_TAData *data)
{
  if (!data)
    return;

  if ((! data->do_basic)     &&
      (! data->do_ip)        &&
      (! data->do_ports)     &&
      (! data->do_src_ports) &&
      (! data->do_dst_ports) &&
      (! data->do_flows))
    {
      D(("Nothing to do!\n"));
      return;
    }

  data->it_count = 0;
  libnd_traceset_foreach(data->traces, (LND_TraceSetCB) ta_do_per_file_cb, data);
}

static void
ta_progress_cb(LND_TAnalysis *ta, int done, void *user_data)
{
  GtkWidget *w;

  ND_GTK_GET(w, dialog, "ta_progressbar");
  gtk_progress_bar_update(GTK_PROGRESS_BAR(w),
			  0.5 + 0.5 * ((double) done) / libnd_traceset_get_size(ta->traces));
  return;
  TOUCH(user_data);
}

static void
ta_do_for_all(ND_TAData *data)
{
  LND_TAnalysis *an;
  int attempt = 0;
  char out[MAXPATHLEN];
  FILE *f;

  if (!data)
    return;

  if ((! data->do_basic)     &&
      (! data->do_ip)        &&
      (! data->do_ports)     &&
      (! data->do_src_ports) &&
      (! data->do_dst_ports) &&
      (! data->do_flows))
    {
      D(("Nothing to do!\n"));
      return;
    }

  if (! (an = libnd_ta_new()))
    return;

  an->do_basic     = data->do_basic;
  an->do_ip        = data->do_ip;
  an->do_ports     = data->do_ports;
  an->do_src_ports = data->do_src_ports;
  an->do_dst_ports = data->do_dst_ports;
  an->do_flows     = data->do_flows;

  libnd_ta_set_progress_callback(an, ta_progress_cb, NULL);

  D(("Analysis for all: basic[%c] ip[%c] ports[%c] sports[%c] dports[%c] flows[%x]\n",
     an->do_basic ? 'x':' ',
     an->do_ip    ? 'x':' ',
     an->do_ports ? 'x':' ',
     an->do_src_ports ? 'x':' ',
     an->do_dst_ports ? 'x':' ',
     an->do_flows ? 'x':' '));

  if (! libnd_ta_set_traceset(an, data->traces))
    {
      D(("Out of memory.\n"));
      libnd_ta_free(an);
      return;
    }
  
  libnd_ta_analyze(an);
  
  do {
    g_snprintf(out, MAXPATHLEN, "%s/nd_ta_%i.txt",
	       data->output_dir,
	       attempt++);
  } while (libnd_misc_exists(out));
	  
  if ( (f = fopen(out, "w")))
    {
      libnd_ta_write_results(an, f);
      fclose(f);
    }

  libnd_ta_free(an);
}


void                 
nd_ta_browse_output(void)
{
  GtkWidget *w;
  const char *output;

  ND_GTK_GET(w, dialog, "out_dir_entry");
  output = gtk_entry_get_text(GTK_ENTRY(w));

  if ((! libnd_misc_is_dir(output))    ||
      (! libnd_misc_can_read(output))  ||
      (! libnd_misc_can_write(output)) ||
      (! libnd_misc_can_exec(output)))
    {
      D(("%s is not rwx or not a dir -- not using.\n", output));
      output = NULL;
    }

  nd_dialog_filesel(_("Select Output Directory"),
		    output, ta_browse_output_cb, dialog);  
}

void                 
nd_ta_run(void)
{
  GtkWidget *w;
  const char *output;
  ND_TAData *data;
  gboolean do_basic, do_ip, do_ports, do_src_ports, do_dst_ports, do_flows;
  gboolean do_basic_all = FALSE, do_ip_all = FALSE;
  gboolean do_ports_all = FALSE, do_src_ports_all = FALSE, do_dst_ports_all = FALSE;
  gboolean do_flows_all = FALSE;
  
  ND_GTK_GET(data, dialog, "ta_data");
  if (data)
    ta_data_free(data);
  
  data = ta_data_new();
  
  /* Check which input we use */
  ND_GTK_GET(w, dialog, "trace_in_radiobutton");
  
  /* Obtain the list of trace files */
  if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
    {
      if (! ta_get_traces_from_app(data))
	{
	  ta_data_free(data);
	  return;
	}
    }
  else
    {
      if (! ta_get_traces_from_file(data))
	{
	  ta_data_free(data);
	  return;
	}
    }

  /* Make sure we have a usable output directory. */
  ND_GTK_GET(w, dialog, "out_dir_entry");
  output = gtk_entry_get_text(GTK_ENTRY(w));

  if ((! libnd_misc_is_dir(output))    ||
      (! libnd_misc_can_read(output))  ||
      (! libnd_misc_can_write(output)) ||
      (! libnd_misc_can_exec(output)))
    {
      nd_dialog_message(_("Output Directory Error"),
			_("Make sure the output file is a directory\n"
			  "and that you have write access to it."),
			TRUE);
      return;
    }

  data->output_dir = g_strdup(output);

  /* Find out what user wants analysed: */
  ND_GTK_GET(w, dialog, "basic_checkbutton");
  do_basic = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  ND_GTK_GET(w, dialog, "ip_checkbutton");
  do_ip = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  ND_GTK_GET(w, dialog, "ports_checkbutton");
  do_ports = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  ND_GTK_GET(w, dialog, "sports_checkbutton");
  do_src_ports = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  ND_GTK_GET(w, dialog, "dports_checkbutton");
  do_dst_ports = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  ND_GTK_GET(w, dialog, "flows_checkbutton");
  do_flows = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w));
  
  ta_data_do_none(data);

  /* Find out what user wants per-file: */
  if (do_basic)
    {
      ND_GTK_GET(w, dialog, "basic_per_file_checkbutton");
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	data->do_basic = TRUE;
      else
	do_basic_all = TRUE;
    }
  
  if (do_ip)
    {
      ND_GTK_GET(w, dialog, "ip_per_file_checkbutton");
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	data->do_ip = TRUE;
      else
	do_ip_all = TRUE;
    }
  
  if (do_ports)
    {
      ND_GTK_GET(w, dialog, "ports_per_file_checkbutton");
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	data->do_ports = TRUE;
      else
	do_ports_all = TRUE;
    }

  if (do_src_ports)
    {
      ND_GTK_GET(w, dialog, "sports_per_file_checkbutton");
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	data->do_src_ports = TRUE;
      else
	do_src_ports_all = TRUE;
    }

  if (do_dst_ports)
    {
      ND_GTK_GET(w, dialog, "dports_per_file_checkbutton");
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	data->do_dst_ports = TRUE;
      else
	do_dst_ports_all = TRUE;
    }
  
  if (do_flows)
    {
      ND_GTK_GET(w, dialog, "flows_per_file_checkbutton");
      if (gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)))
	data->do_flows = TRUE;
      else
	do_flows_all = TRUE;
    }

  /* Disable Close button while scanning */
  ND_GTK_GET(w, dialog, "cancel_button");
  gtk_widget_set_sensitive(w, FALSE);

  /* Reset the progressbar in the dialog to the total number
   * of traces, times two -- we scan the list twice.
   */
  ND_GTK_GET(w, dialog, "ta_progressbar");
  gtk_progress_bar_update(GTK_PROGRESS_BAR(w), 0.0);

  /* Now do the ones selected per-file.
   */
  ta_do_per_file(data);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(w), 0.5);


  /* Now do the remaining ones wanted for all traces.
   */
  data->do_basic     = do_basic_all;
  data->do_ip        = do_ip_all;
  data->do_ports     = do_ports_all;
  data->do_src_ports = do_src_ports_all;
  data->do_dst_ports = do_dst_ports_all;
  data->do_flows     = do_flows_all;
  
  ta_do_for_all(data);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(w), 1.0);

  ND_GTK_GET(w, dialog, "cancel_button");
  gtk_widget_set_sensitive(w, TRUE);

  ND_GTK_SET(data, dialog, "ta_data");
}


void                 
nd_ta_save(void)
{
}


const char *
name(void)
{
  return "/Statistics/Traffic Analyzer";
}

/* A description of the plugin and what it does. */
char *
description(void)
{
  return _("The plugin scans the packets in a trace file and collects "
	   "statistics about the IP protocol numbers found as well as "
	   "the source and destination ports of TCP and UDP packets. "
	   "The results are saved to one or more text files for "
	   "further analysis, visualization, etc.");
}

const char *
author(void)
{
  return "Christian Kreibich, <christian.kreibich-AT-cl.cam.ac.uk>";
}

const char *
version(void)
{
  return VERSION;
}


gboolean
init(void)
{
  /* We need libnetdude's traffic analyzer plugin -- this is after
   * all just a GUI front end for its API.
   */
  if (! libnd_plugin_find("Traffic-Analyzer"))
    {
      nd_dialog_message(_("Plugin Initialization Failed."),
			_("The Traffic Analyzer plugin could not initialize\n"
			  "properly. Make sure you have the required analysis\n"
			  "plugin for libnetdude installed.\n"),
			FALSE);

      return FALSE;
    }

  return TRUE;
}


void
run(LND_Trace *trace)
{
  GtkWidget *w;
  char *load_dir_tmp = "/tmp";
  char *load_dir = load_dir_tmp;

  if (!dialog)
    dialog = create_analyzer_dialog();
  
  ta_sync_trace_list(dialog);

  /* Disable Start button until users selects trace input */
  ND_GTK_GET(w, dialog, "ok_button");
  gtk_widget_set_sensitive(w, FALSE);

  /* Display Netdude's current save directory in the output
   * directory selector:
   */

  if (libnd_prefs_get_str_item(ND_DOM_NETDUDE, "load_dir", &load_dir))
    load_dir = libnd_misc_add_slash(g_strdup(load_dir));

  ND_GTK_GET(w, dialog, "out_dir_entry");
  gtk_entry_set_text(GTK_ENTRY(w), load_dir);

  if (load_dir != load_dir_tmp)
    g_free(load_dir);		    

  /* Finally, show the thing to the user.
   */
  gtk_widget_show(dialog);

  return;
  TOUCH(trace);
}
