/*

Copyright (C) 2000 - 2007 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include <nd.h>
#include <nd_gui.h>
#include <nd_prefs.h>
#include <nd_trace_registry.h>
#include <nd_recent.h>


static void
recent_change_menuitem(int item, const char *label)
{
  GtkWidget  *w, *w2;
  char        s[MAXPATHLEN];
  char       *s2;

  if (item < 0 || item >= NUM_RECENT)
    return;

  sprintf(s, "recent_item%i", item);
  ND_GTK_GET(w, nd_toplevel_window, s);

  if (!label)
    {
      gtk_widget_hide(w);
      return;
    }

  s2 = strrchr(label, '/');
  if (s2)
    g_snprintf(s, MAXPATHLEN, "%i. %s", item+1, s2+1);
  else
    g_snprintf(s, MAXPATHLEN, "%i. %s", item+1, label);

  ND_GTK_GET(w2, nd_toplevel_window, "tooltips");
  gtk_tooltips_set_tip(GTK_TOOLTIPS(w2), w, label, NULL);
  
  gtk_widget_show(w);
  gtk_label_set_text(GTK_LABEL(GTK_BIN(w)->child), s);

  ND_GTK_GET(w, nd_toplevel_window, "recent_separator");
  gtk_widget_show(w);
}


void
nd_recent_init(void)
{
  GtkWidget * w;

  D_ENTER;

  ND_GTK_GET(w, nd_toplevel_window, "recent_separator");
  gtk_widget_hide(w);

  ND_GTK_GET(w, nd_toplevel_window, "recent_item0");
  gtk_widget_hide(w);
  ND_GTK_GET(w, nd_toplevel_window, "recent_item1");
  gtk_widget_hide(w);
  ND_GTK_GET(w, nd_toplevel_window, "recent_item2");
  gtk_widget_hide(w);
  ND_GTK_GET(w, nd_toplevel_window, "recent_item3");
  gtk_widget_hide(w);
  ND_GTK_GET(w, nd_toplevel_window, "recent_item4");
  gtk_widget_hide(w);

  nd_recent_reload_entries();
  D_RETURN;
}


void
nd_recent_reload_entries(void)
{
  GtkWidget *w;
  char       key[1024];
  char      *recent_file;
  int        i, last;

  ND_GTK_GET(w, nd_toplevel_window, "recent_separator");
  gtk_widget_hide(w);

  for (i = 0, last = 0; i < NUM_RECENT; i++)
    {
      recent_change_menuitem(i, NULL);
      sprintf(key, "recent_%i", i);

      if (libnd_prefs_get_str_item(ND_DOM_NETDUDE, key, &recent_file))
	{
	  if (libnd_misc_is_tcpdump_file(recent_file))
	    {
	      recent_change_menuitem(last, recent_file);
	      sprintf(key, "recent_%i", last);
	      libnd_prefs_set_str_item(ND_DOM_NETDUDE, key, recent_file);
	      last++;
	    }
	}
    }

  /* Delete all filenames remaining */
  for ( ; last < NUM_RECENT; last++)
    {
      sprintf(key, "recent_%i", last);
      libnd_prefs_del_item(ND_DOM_NETDUDE, key);
    }
}


void 
nd_recent_add_file(const char *filename)
{
  char       *file;
  char        key[MAXPATHLEN];
  int         i, j;

  if (!filename || filename[0] == '\0')
    return;

  /* Check all existing entries -- if the filename exists
     already, don't add it. */
  for (i = 0; i < NUM_RECENT; i++)
    {
      g_snprintf(key, MAXPATHLEN, "recent_%i", i);
      
      if (libnd_prefs_get_str_item(ND_DOM_NETDUDE, key, &file))
	{
	  if (!strcmp(filename, file))
	    {
	      /* Okay -- we've already used this file.
		 Move it to front of list: */
	      if (i == 0)
		return;

	      for (j = i - 1; j >= 0; j--)
		{
		  g_snprintf(key, MAXPATHLEN, "recent_%i", j);

		  if (libnd_prefs_get_str_item(ND_DOM_NETDUDE, key, &file))
		    {
		      recent_change_menuitem(j + 1, file);
		      g_snprintf(key, MAXPATHLEN, "recent_%i", j + 1);
		      libnd_prefs_set_str_item(ND_DOM_NETDUDE, key, file);
		    }
		}

	      recent_change_menuitem(0, filename);
	      libnd_prefs_set_str_item(ND_DOM_NETDUDE, "recent_0", filename);      
	      libnd_prefs_save();
	      return;
	    }
	}
    }
  
  /* New file -- forget oldest file, move everything down
     one slot and add new file at the top of the list */
  for (i = NUM_RECENT - 2; i >= 0; i--)
    {
      g_snprintf(key, MAXPATHLEN, "recent_%i", i);
      
      if (libnd_prefs_get_str_item(ND_DOM_NETDUDE, key, &file))
	{
	  recent_change_menuitem(i+1, file);
	  g_snprintf(key, MAXPATHLEN, "recent_%i", i+1);
	  libnd_prefs_set_str_item(ND_DOM_NETDUDE,key, file);
	}
    }
  
  libnd_prefs_set_str_item(ND_DOM_NETDUDE, "recent_0", (void*) filename);
  recent_change_menuitem(0, filename);
  libnd_prefs_save();
}


void 
nd_recent_load(int index)
{
  char  key[MAXPATHLEN];
  char *file;

  if (index < 0 || index >= NUM_RECENT)
    return;

  g_snprintf(key, MAXPATHLEN, "recent_%i", index);

  if (libnd_prefs_get_str_item(ND_DOM_NETDUDE, key, &file))
    {
      LND_Trace *trace;

      trace = nd_trace_new(file);
      D_ASSERT_PTR(trace);

      nd_trace_registry_add(trace);
    }
}
