/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.xml.wsdl.ui.netbeans.module;

import java.awt.datatransfer.Transferable;
import java.io.IOException;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.netbeans.modules.xml.wsdl.model.WSDLComponent;
import org.netbeans.modules.xml.wsdl.ui.view.treeeditor.WSDLElementNode;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.Nameable;
import org.netbeans.modules.xml.xam.Named;
import org.openide.nodes.Node;
import org.openide.nodes.NodeTransfer;
import org.openide.util.datatransfer.PasteType;

/**
 * Paste type for WSDL components. Expects WSDLElementNodes in the
 * Transferable and determines what is allowed to be pasted where,
 * delegating most of the work to the WSDL model.
 *
 * @author Nathan Fiedler
 */
public class ComponentPasteType {
    // Based on ActionPasteType in core/windows

    /**
     * Create the PasteType to receive the transferable into the given
     * WSDL component.
     *
     * @param  target    the target component.
     * @param  transfer  the component(s) being pasted.
     * @param  type      type of the component to allow (e.g. PortType.class),
     *                   or null to allow all types.
     * @return  the new paste type.
     */
    public static PasteType getPasteType(WSDLComponent target,
            Transferable transfer, Class<? extends WSDLComponent> type) {
        int[] operations = new int[] {
            NodeTransfer.COPY,
            NodeTransfer.MOVE
        };
        return getPasteType(target, transfer, type, operations, -1);
    }

    /**
     * Create the PasteType to receive the transferable into the given
     * WSDL component.
     *
     * @param  target    the target component.
     * @param  transfer  the component(s) being pasted.
     * @param  type      type of the component to allow (e.g. Element.class),
     *                   or null to allow all types.
     * @param  action    the NodeTransfer constant for cut/copy.
     * @param  index     index at which to paste the component (-1 to append).
     * @return  the new paste type.
     */
    public static PasteType getDropType(WSDLComponent target,
            Transferable transfer, Class<? extends WSDLComponent> type,
            int action, int index) {
// The action value passed to AbstractNode.getDropType() is completely
// useless, so we must ignore it and use our own values.
//        int[] operations = new int[] { action };
        int[] operations = new int[] {
            NodeTransfer.COPY,
            NodeTransfer.MOVE
        };
        return getPasteType(target, transfer, type, operations, index);
    }

    /**
     * Create the PasteType to receive the transferable into the given
     * WSDL component.
     *
     * @param  target      the target component.
     * @param  transfer    the component(s) being pasted.
     * @param  type        type of the component to allow (e.g. Element.class),
     *                     or null to allow all types.
     * @param  operations  set of NodeTransfer constants for cut/copy.
     * @param  index       index at which to paste the component (-1 to append).
     * @return  the new paste type.
     */
    private static PasteType getPasteType(WSDLComponent target,
            Transferable transfer, Class<? extends WSDLComponent> type,
            int[] operations, int index) {
        PasteType retVal = null;
        // Check each operation until a supported one is found.
        for (int oper : operations) {
            // Attempt to retrieve the nodes from transferable.
            Node[] nodes = NodeTransfer.nodes(transfer, oper);
            if (nodes != null) {
                // Can any of these be pasted into the target?
                if (canBePasted(nodes, target, oper, type)) {
                    retVal = new PasteTypeImpl(Arrays.asList(nodes), target,
                            oper, index);
                    break;
                }
            }
        }
        return retVal;
    }

    /**
     * Determine if all of the given nodes can be pasted into the component.
     *
     * @param  nodes      the nodes being pasted.
     * @param  target     the target component.
     * @param  operation  the NodeTransfer constant for cut/copy.
     * @param  type       type of the component to allow (e.g. PortType.class),
     *                    or null to allow all types.
     * @return  true if the nodes can be pasted.
     */
    private static boolean canBePasted(Node[] nodes, WSDLComponent target,
            int operation, Class<? extends WSDLComponent> type) {
        Set<Node> pasteables = new HashSet<Node>();
        for (Node node : nodes) {
            WSDLElementNode scn = (WSDLElementNode) node.
                    getLookup().lookup(WSDLElementNode.class);
            if (scn != null) {
                // Must be a WSDL component node, else we cannot handle it.
                WSDLComponent component = scn.getWSDLComponent();
                // Check that the target can receive this component.
                // Ensure that the model is still valid, in case the
                // component was deleted or moved elsewhere.
                if ((type == null || type.isAssignableFrom(component.getClass())) &&
                        component.getModel() != null && target.canPaste(component)) {
                    boolean isCopyPaste = (operation & NodeTransfer.COPY) != 0;
                    // Prevent cutting and pasting into the same component.
                    boolean isCutPaste = (operation & NodeTransfer.MOVE) != 0 &&
                            !(component.getParent() == target) && scn.canDestroy();
                    if (isCopyPaste || isCutPaste) {
                        // We could check for duplicates here, but at this
                        // time we are allowing them.
                        pasteables.add(node);
                    }
                }
            }
        }
        return pasteables.size() == nodes.length;
    }

    /**
     * Our PasteType implementation for WSDL component nodes.
     */
    private static class PasteTypeImpl extends PasteType {
        /** The component to receive the paste. */
        private WSDLComponent target;
        /** The nodes being pasted. */
        private List<Node> nodes;
        /** NodeTransfer constant (e.g. COPY, MOVE). */
        private int operation;
        /** Position at which to insert item (-1 to append). */
        private int index;

        /**
         * Creates a new instance of PasteTypeImpl.
         *
         * @param  nodes      those which are to be pasted.
         * @param  target     the paste recipient.
         * @param  operation  NodeTransfer constant indicating cut/copy.
         * @param  index      position to paste, or -1 to append.
         */
        private PasteTypeImpl(List<Node> nodes, WSDLComponent target,
                int operation, int index) {
            this.target = target;
            this.nodes = nodes;
            this.operation = operation;
            this.index = index;
        }

        public Transferable paste() throws IOException {
            // Perform the cut or copy to the target component.
            if (target != null && nodes.size() > 0) {
                Model model = target.getModel();
                Model srcModel = ((WSDLElementNode) nodes.get(0).
                        getLookup().lookup(WSDLElementNode.class)).
                        getWSDLComponent().getModel();
                // Keep everything in a single transaction so the undo/redo
                // acts on the entire set rather than individual nodes.
                // This makes the assumption that the source nodes are all
                // coming from a single model, which should always be true.
                model.startTransaction();
                try {
                    for (Node node : nodes) {
                        // The type of the nodes has already been established,
                        // so we can assume they are WSDL component nodes.
                        WSDLElementNode scn = (WSDLElementNode) node.
                                getLookup().lookup(WSDLElementNode.class);
                        WSDLComponent child = scn.getWSDLComponent();
                        // Always make a clone of the component, even for the cut
                        // operation, since it converts global to local and vice
                        // versa, and we want to be consistent with the copy
                        // operation in that respect.
                        WSDLComponent copy = (WSDLComponent) child.copy(target);
                        // 'copy' will be null if the copy was unsuccessful.
                        // Better to fail silently than throw assertion errors,
                        // so make sure the child is non-null before proceeding.
                        if (copy != null) {
                            if ((operation & NodeTransfer.MOVE) != 0) {
                                // For cut, remove the component from its model.
                                // This should allow it to be collected.
                                boolean srcInTransaction = srcModel.isIntransaction();
                                try {
                                    if (!srcInTransaction) {
                                        // Must be separate models, in which
                                        // case create a new transaction.
                                        srcModel.startTransaction();
                                    }
                                    srcModel.removeChildComponent(child);
                                } finally {
                                    if (!srcInTransaction) {
                                        srcModel.endTransaction();
                                    }
                                }
                            }
                            // Add child to target model under component.
                            // Must add before changing the name, as some
                            // WSDL components throw an NPE in getName().
                            model.addChildComponent(target, copy, index);
                            // Ensure the name of the copy is unique within
                            // the target component, if it is nameable.
                            if (copy instanceof Nameable) {
                                String name = ((Nameable) copy).getName();
                                String preferredName = name;
                                HashSet<String> nameSet = new HashSet<String>();
                                for (WSDLComponent sibling : target.getChildren()) {
                                    if (sibling != copy && sibling instanceof Named) {
                                        nameSet.add(((Named) sibling).getName());
                                    }
                                }
                                int unique = 1;
                                while (nameSet.contains(name)) {
                                    name = preferredName + unique;
                                    unique++;
                                }
                                ((Nameable) copy).setName(name);
                            }
                        }
                    }
                } finally {
                    model.endTransaction();
                }
            }
            return null;
        }

        public String toString() {
            return "PasteTypeImpl=[operation=" + operation + ",index=" + index + "]";
        }
    }
}
