/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.

 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

/*
 * CompletionQueryHelper.java
 *
 * Created on June 9, 2006, 3:14 PM
 *
 * To change this template, choose Tools | Template Manager
 * and open the template in the editor.
 */

package org.netbeans.modules.xml.schema.completion.util;

import java.io.File;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import javax.xml.XMLConstants;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.TokenItem;
import org.netbeans.modules.xml.axi.AXIModel;
import org.netbeans.modules.xml.axi.Element;
import org.netbeans.modules.xml.schema.completion.CompletionResultItem;
import org.netbeans.modules.xml.text.api.XMLDefaultTokenContext;
import org.netbeans.modules.xml.text.syntax.SyntaxElement;
import org.netbeans.modules.xml.text.syntax.XMLSyntaxSupport;
import org.netbeans.modules.xml.text.syntax.dom.StartTag;
import org.w3c.dom.Attr;
import org.w3c.dom.NamedNodeMap;

/**
 * Helps in populating the completion list.
 *
 * @author Samaresh (Samaresh.Panda@Sun.Com)
 */
public class CompletionQueryHelper {
    
    /**
     * Creates a new instance of CompletionQueryHelper
     */
    public CompletionQueryHelper(CatalogModelHelper helper,
            XMLSyntaxSupport support, int offset) {
        try {
            this.helper = helper;
            this.xmlFileLocation = helper.getFileLocation();
            this.document = support.getDocument();
            this.element = support.getElementChain(offset);
            this.token = support.getPreviousToken(offset);
            this.docRoot = CompletionUtil.getRoot(element);                    
            populateNamespaces();
        } catch(Exception ex) {
            //in the worst case, there will not be
            //any code completion help.
        }
    }
    
    /**
     * Keeps all namespaces along with their prefixes in a HashMap.
     * This is obtained from the root element's attributes, with
     * the attribute value(namespace) as the key and name with prefix
     * as the value.
     * For example the hashmap may look like this
     *  KEY                                    VALUE
     *  http://www.camera.com                  xmlns:c
     *  http://www.nikon.com                   xmlns:n
     */
    private void populateNamespaces() {
        if(docRoot == null)
            return;
        
        //Check if the tag has any prefix. If yes, the defaultNamespace
        //is the one with this prefix.
        String tagName = docRoot.getTagName();
        String tns = XMLConstants.XMLNS_ATTRIBUTE;
        if(tagName.indexOf(":") != -1) {
            tns = tns + ":" + tagName.substring(0, tagName.indexOf(":"));
        }
        
        if(namespaces == null)
            namespaces = new HashMap<String, List<String>>();
        
        NamedNodeMap attributes = docRoot.getAttributes();
        for(int index=0; index<attributes.getLength(); index++) {
            Attr attr = (Attr)attributes.item(index);
            if(attr.getName().endsWith(XSI_SCHEMALOCATION)) {
                schemaLocation = attr.getValue().trim();
                continue;
            }            
            if(attr.getName().equals(tns)) {
                defaultNamespace = attr.getValue();
            }
            List<String> values = namespaces.get(attr.getValue());
            if(values != null) {
                values.add(attr.getName());
                continue;
            }
            values = new ArrayList<String>();
            values.add(attr.getName());
            namespaces.put(attr.getValue(), values);
        }
    }
        
    /**
     *
     */
    public boolean isSchemaBasedCompletion() {
        if(element == null || token == null || docRoot == null) {
            return false;
        }
        
        this.schemaLocation = findSchemaLocation();
        if(schemaLocation == null)
            return false;
        
        this.model = helper.getModel(schemaLocation);
        if(model == null)
            return false;
        
        return true;
    }
    
    /*
     * An instance document that conforms to a schema may have the root
     * element declaration as follows:
     *    <purchaseOrder xmlns='http://xml.netbeans.org/examples/targetNS'
     *       xmlns:xsi='http://www.w3.org/2001/XMLSchema-instance'
     *       xsi:schemaLocation='http://xml.netbeans.org/examples/targetNS po.xsd'>
     */
    private String findSchemaLocation() {
        if(defaultNamespace == null || schemaLocation == null) {
            return null;
        }
        String realSchemaLocation = CompletionUtil.
                getRealSchemaLocation(schemaLocation, defaultNamespace);
        if(realSchemaLocation == null)
            return null;
        
        //absolute location
        if(realSchemaLocation.startsWith("/") ||
           realSchemaLocation.startsWith(FILE_STR_WINDOWS)) {
            try {
                URI uri = URI.create(realSchemaLocation);
                return uri.toString();
            } catch(Exception ex) {
                return null;
            }
        }
        
        //relative location
        realSchemaLocation = xmlFileLocation + realSchemaLocation;
        File file = new File(realSchemaLocation);
        return file.toURI().toString();
    }
            
    /**
     * At a given context, that is, at the current cursor location
     * in the document, finds the type of query that needs to be
     * carried out and finds the path from root.
     */
    private void initContext() {
        int id = token.getTokenID().getNumericID();
        switch ( id) {
            //user enters < character
            case XMLDefaultTokenContext.TEXT_ID:
                String chars = token.getImage().trim();
                if(chars.equals("") &&
                   token.getPrevious().getImage().trim().equals(">")) {
                    completionType = COMPLETION_TYPE_UNKNOWN;
                    break;
                }
                if(!chars.equals("<") &&
                   token.getPrevious().getImage().trim().equals(">")) {
                    completionType = COMPLETION_TYPE_UNKNOWN;
                    break;
                }
                completionType = COMPLETION_TYPE_ELEMENT;
                pathFromRoot = CompletionUtil.getPathFromRoot(element);
                break;
                
            //start tag of an element
            case XMLDefaultTokenContext.TAG_ID:
                if(element instanceof StartTag) {
                    StartTag tag = (StartTag)element;
                    typedChars = tag.getTagName();
                }
                completionType = COMPLETION_TYPE_ELEMENT;
                pathFromRoot = CompletionUtil.getPathFromRoot(element.getPrevious());
                break;
                
            //user enters an attribute name
            case XMLDefaultTokenContext.ARGUMENT_ID:
                completionType = COMPLETION_TYPE_ATTRIBUTE;
                typedChars = token.getImage();
                pathFromRoot = CompletionUtil.getPathFromRoot(element);
                break;
                
            //not sure
            case XMLDefaultTokenContext.CHARACTER_ID:
                break;
                
            //user enters = character, we should ignore all other operators
            case XMLDefaultTokenContext.OPERATOR_ID:
            //user enters either ' or "
            case XMLDefaultTokenContext.VALUE_ID:
                completionType = COMPLETION_TYPE_UNKNOWN;
                break;
                
            //user enters white-space character
            case XMLDefaultTokenContext.WS_ID:
                completionType = COMPLETION_TYPE_UNKNOWN;
                TokenItem prev = token.getPrevious();
                while( prev != null &&
                       (prev.getTokenID().getNumericID() == XMLDefaultTokenContext.WS_ID) ) {
                        prev = prev.getPrevious();
                }                    
                if( (prev.getTokenID().getNumericID() == XMLDefaultTokenContext.VALUE_ID) ||
                    (prev.getTokenID().getNumericID() == XMLDefaultTokenContext.TAG_ID) ) {
                    completionType = COMPLETION_TYPE_ATTRIBUTE;
                    pathFromRoot = CompletionUtil.getPathFromRoot(element);
                }
                break;
                
            default:
                completionType = COMPLETION_TYPE_UNKNOWN;
                pathFromRoot = CompletionUtil.getPathFromRoot(element);
                break;
        }
    }
    
    public List<CompletionResultItem> getCompletionItems() {
        initContext();

        switch (getCompletionType()) {
            case COMPLETION_TYPE_ELEMENT:
                completionItems = queryElements();
                break;
                
            case COMPLETION_TYPE_ATTRIBUTE:
                completionItems = queryAttributes();
                break;
            
            case COMPLETION_TYPE_VALUE:
                completionItems = queryValues();
                break;            
            
            case COMPLETION_TYPE_ENTITY:
                completionItems = queryEntities();
                break;
            
            case COMPLETION_TYPE_NOTATION:
                completionItems = queryNotations();
                break;
                
            default:
                break;
        }
        
        return completionItems;
    }
    
    private List<CompletionResultItem> queryEntities() {
        return null;
    }
    
    private List<CompletionResultItem> queryElements() {
        Element axiElement = CompletionUtil.findElementAtContext(
                model.getRoot(), getPathFromRoot());
        return CompletionUtil.getElements(axiElement , typedChars, namespaces);
    }
    
    private List<CompletionResultItem> queryAttributes() {
        Element axiElement = CompletionUtil.findElementAtContext(
                model.getRoot(), getPathFromRoot());        
        return CompletionUtil.getAttributes(axiElement , typedChars, namespaces);
    }
    
    private List<CompletionResultItem> queryValues() {
        return null;
    }
    
    private List<CompletionResultItem> queryNotations() {
        return null;
    }    
        
    private TokenItem getToken() {
        return token;
    }
    
    private SyntaxElement getSyntaxElement() {
        return element;
    }
    
    private int getCompletionType() {
        return completionType;
    }
        
    private List<StartTag> getPathFromRoot() {
        return pathFromRoot;
    }
    
    private String getSchemaLocation() {
        return schemaLocation;
    }
    
    private String xmlFileLocation;
    private String typedChars;
    private TokenItem token;
    private SyntaxElement element;
    private StartTag docRoot;
    private HashMap<String, List<String>> namespaces;
    private int completionType;
    private List<StartTag> pathFromRoot;
    private List<CompletionResultItem> completionItems;
    private String schemaLocation;
    private String defaultNamespace;
    private AXIModel model;
    private BaseDocument document;
    private CatalogModelHelper helper;
    
    public final static int COMPLETION_TYPE_UNKNOWN     = 0;
    public final static int COMPLETION_TYPE_ATTRIBUTE   = 1;
    public final static int COMPLETION_TYPE_VALUE       = 2;
    public final static int COMPLETION_TYPE_ELEMENT     = 3;
    public final static int COMPLETION_TYPE_ENTITY      = 4;
    public final static int COMPLETION_TYPE_NOTATION    = 5;
    public final static int COMPLETION_TYPE_DTD         = 6;
    
    public static final String XSI_SCHEMALOCATION   = "schemaLocation";
    public static final String FILE_STR_WINDOWS     = "file:";
}
