/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.xml.refactoring.impl;

import java.io.IOException;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.undo.AbstractUndoableEdit;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.UndoableEdit;
import javax.swing.undo.UndoableEditSupport;
import org.netbeans.modules.xml.refactoring.FileRenameRequest;
import org.netbeans.modules.xml.refactoring.RefactorRequest;
import org.netbeans.modules.xml.refactoring.spi.ChangeExecutor;
import org.netbeans.modules.xml.refactoring.spi.SharedUtils;
import org.netbeans.modules.xml.refactoring.spi.UIHelper;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.Referenceable;

/**
 * Change executor for refactoring actions that are not specific to any particular
 * XML document models.
 *
 * @author Nam Nguyen
 */
public class GeneralChangeExecutor extends ChangeExecutor {
    private UndoableEditSupport ues;
    
    /** Creates a new instance of GenericChangeExecutor */
    public GeneralChangeExecutor() {
        ues = new UndoableEditSupport(this);
    }
    
    public <T extends RefactorRequest> boolean canChange(Class<T> changeType, Referenceable target) {
        if (changeType == FileRenameRequest.class && target instanceof Model) {
            return true;
        }
        return false;
    }
    
    /**
     * Perform the change specified by the refactor request.  Any errors that would
     * fail the overall refactoring should be reported throught #RefactoringRequest.addError
     * Implementation should quietly ignore unsupported refactoring type.
     */
    public void doChange(RefactorRequest request) throws IOException {
        if (request instanceof FileRenameRequest) {
            FileRenameRequest frr = (FileRenameRequest)request;
            SharedUtils.renameFile(frr);
            FileRenameUndoable ue = new FileRenameUndoable(frr);
            fireUndoEvent(ue);
        }
    }
    
    /**
     * Returns UI helper in displaying the usages.  Implementation could override
     * the default UI to help display usages in a more intuitive way than the
     * generic helper.
     */
    public UIHelper getUIHelper() {
        return new UIHelper();
    }
    
    public synchronized void addUndoableEditListener(UndoableEditListener l) {
        ues.addUndoableEditListener(l);
    }
    
    public synchronized void removeUndoableEditListener(UndoableEditListener l) {
        ues.removeUndoableEditListener(l);
    }
    
    protected void fireUndoEvent(UndoableEdit edit) {
	    UndoableEditEvent ue = new UndoableEditEvent(this, edit);
	    for (UndoableEditListener l:ues.getUndoableEditListeners()) {
            l.undoableEditHappened(ue);
	    }
    }
    
    public static class FileRenameUndoable extends AbstractUndoableEdit {
        private static final long serialVersionUID = -1L;
        private FileRenameRequest request;
        
        public FileRenameUndoable(FileRenameRequest request) {
            this.request = request;
        }
        
        public void undo() throws CannotUndoException {
            try {
                SharedUtils.undoRenameFile(request);
                super.undo();
            } catch(IOException ioe) {
                CannotUndoException cue = new CannotUndoException();
                cue.initCause(ioe);
                throw cue;
            }
        }
        
        public void redo() throws CannotRedoException {
            try {
                SharedUtils.renameFile(request);
                super.redo();
            } catch(IOException ioe) {
                CannotUndoException cue = new CannotUndoException();
                cue.initCause(ioe);
                throw cue;
            }
        }
    }
}
