/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.

 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.xml.refactoring;

import java.util.ArrayList;
import java.util.List;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.modules.xml.refactoring.spi.RefactoringEngine;
import org.netbeans.modules.xml.xam.Component;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.Referenceable;
import org.openide.filesystems.FileObject;

/**
 * This class represent a usage search result within a model or a model source.
 *
 * @author Nam Nguyen
 */

// TODO soft references to target instead ?

public class UsageGroup {
    private RefactoringEngine engine;
    private Model model;
    private FileObject source; //TODO maybe remove dependency on NB ?
    private Referenceable target;
    private SourceGroup sourceGroup;
    
    private List<Usage> items = new ArrayList<Usage>();
    private List<ErrorItem> errors = new ArrayList<ErrorItem>();
    
    public UsageGroup(RefactoringEngine engine, Model model, Referenceable target) {
        this(engine, target);
        this.model = model;
    }
    
    public UsageGroup(RefactoringEngine engine, FileObject src, Referenceable target) {
        this(engine, target);
        this.source = src;
    }
    
    private UsageGroup(RefactoringEngine engine, Referenceable target) {
        if (engine == null || target == null) {
            throw new IllegalArgumentException(
               "Usage constructor needs non-null engine and target arguments"); //NOI18N
        }
        this.engine = engine;
        this.target = target;
    }

    /**
     * Returns the engine that generated this usage search result.
     */
    public RefactoringEngine getEngine() {
        return engine;
    }
    
    /**
     * Returns the model in which this usage search is done.
     */
    public Model getModel() { 
        return model; 
    }
    
    /**
     * Returns target of the usage search.
     */
    public Referenceable getTarget() { 
        return target;
    }
    
    /**
     * Returns target component of the usage search.
     */
    public Component getTargetComponent() { 
        if (target instanceof Component) {
            return (Component) target; 
        }
        return null;
    }
    
    /**
     * Returns target component of the usage search.
     */
    public Model getTargetModel() { 
        if (target instanceof Component) {
            return ((Component) target).getModel(); 
        } else if (target instanceof Model) {
            return (Model) target;
        }
        return null;
    }
    
    /**
     * Returns the usage search result items.
     */
    public List<Usage> getItems() {
        assert model != null : "Try to retrieve items from an Usage created with null model"; //NOI18N
        return items;
    }

    /**
     * Returns the usage search result items.
     */
    public List<Component> getUsageComponents() {
        assert model != null : "Try to retrieve items from an Usage created with null model"; //NOI18N
        List<Component> ret = new ArrayList<Component>();
        for(Usage i : items) {
            ret.add(i.getComponent());
        }
        return ret;
    }
    
    /**
     * Returns the usage items selected for refactoring.
     */
    public List<Component> getRefactorComponents() {
        assert model != null : "Try to retrieve items from an Usage created with null model"; //NOI18N
        List<Component> ret = new ArrayList<Component>();
        for(Usage i : items) {
            if (i.isIncludedInRefactoring()) {
                ret.add(i.getComponent());
            }
        }
        return ret;
    }
    
    public void addItem(Component usageComponent) {
        assert model != null : "Cannot add item to Usage created with null model"; //NOI18N
        items.add(new Usage(usageComponent, this));
    }
    
    /**
     * Returns the list of errors seen during the search.
     */
    public List<ErrorItem> getErrors() {
        return errors;
    }
    
    public void addError(Component errorComponent, String message) {
        errors.add(new ErrorItem(errorComponent, message));
    }
    
    public void addError(ErrorItem error) {
        errors.add(error);
        for (Usage u : items) {
            if (u.getComponent().equals(error.getComponent())) {
                u.setIncludedInRefactoring(false);
                break;
            }
        }
    }

    public boolean isEmpty() {
        return getItems().isEmpty() && getErrors().isEmpty();
    }
    
    protected void setSourceGroup(SourceGroup group) {
        sourceGroup = group;
    }
    
    public SourceGroup getSourceGroup() {
        return sourceGroup;
    }
    
    public FileObject getFolder() {
        return getFileObject().getParent();
    }
    
    public FileObject getFileObject() {
        if (source == null) {
            assert model != null : "Invalid Usage object, expecting non-null model."; //NOI18N
            source = (FileObject) getModel().getModelSource().getLookup().lookup(FileObject.class);
            assert source != null : "ModelSource should have FileObject in lookup"; //NOI18N
        }
        return source;
    }
    
}
