/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.

 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.xml.refactoring;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import javax.swing.undo.CannotRedoException;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.UndoManager;
import org.netbeans.modules.xml.refactoring.impl.GeneralChangeExecutor;
import org.netbeans.modules.xml.refactoring.impl.RefactoringUtil;
import org.netbeans.modules.xml.refactoring.spi.ChangeExecutor;
import org.netbeans.modules.xml.refactoring.spi.RefactoringEngine;
import org.netbeans.modules.xml.refactoring.spi.UIHelper;
import org.netbeans.modules.xml.xam.Component;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.Referenceable;
import org.openide.filesystems.FileObject;
import org.openide.util.Cancellable;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;

/**
 * Singleton refactoring manager, ensuring atomic refactoring change across
 * affected models.  Provide undo/redo capability of each set of refactoring
 * changes.
 *
 * @author Nam Nguyen
 */
public class RefactoringManager implements Cancellable, PropertyChangeListener {
    private static RefactoringManager manager = new RefactoringManager();
    private List<RefactoringEngine> engines;
    private List<ChangeExecutor> executors;
    private RefactorRequest lastRefactorRequest;
    private Map<Model,UndoManager> undoManagers;
    private UndoManager genericChangeUndoManager;
    private enum Status { IDLE, RUNNING, CANCELLING }
    private Status status = Status.IDLE;
    private FindUsageResult findUsageResult;
    
    /** Creates a new instance of RefactoringManager */
    private RefactoringManager() {
    }
    
    /**
     * Returns Refactoring manager default instance.
     */
    public static RefactoringManager getInstance() {
        return manager;
    }
    
    /**
     * Return all usages of the given target component within the specified search scope.
     */
    
    /**
     * Return all usages of the given target component this refactoring manager can find.
     */
    public FindUsageResult findUsages(Referenceable target) {
        if (target == null) {
            throw new IllegalArgumentException("Calling find usages with null target"); //NOI18N
        }
        return new FindUsageResult(target);
    }
    
    /**
     * Return local usages of the given target component.
     */
    public FindUsageResult findLocalUsages(Referenceable target) {
        if (target == null || ! (target instanceof Component)) {
            throw new IllegalArgumentException("Calling find local usages with non-component target"); //NOI18N
        }
        Component root = RefactorRequest.getRootOf(target);
        return new FindUsageResult(target, Collections.singleton(root));
    }
    
    /**
     * Return all usages of the given target component this refactoring manager can find.
     */
    public FindUsageResult findUsages(Referenceable target, Set<Component> searchRoots) {
        return new FindUsageResult(target, searchRoots);
    }
    
    /**
     * Return all usages of the given target component this refactoring manager can find.
     */
    public FindUsageResult findUsages(Referenceable target, Component searchRoot) {
        return new FindUsageResult(target, searchRoot);
    }
    
    public boolean canChange(Class<? extends RefactorRequest> type, Referenceable target) {
        for (ChangeExecutor executor : getExecutors()) {
            if (executor.canChange(type, target)) {
                return true;
            }
        }
        return false;
    }
    
    public RefactorRequest getLastRefactorRequest() {
        return lastRefactorRequest;
    }
    
    public synchronized boolean canUndo() {
        if (undoManagers == null || undoManagers.isEmpty()) {
            return false;
        }
        for (UndoManager um : undoManagers.values()) {
            if (! um.canUndo()) {
                return false; 
            }
        }
        if (genericChangeUndoManager != null && ! genericChangeUndoManager.canUndo()) {
            return false;
        }
        return true;
    }
    
    public synchronized void undo() throws CannotUndoException {
        if (! canUndo()) {
            throw new CannotUndoException();
        }
        try {
            setStatus(Status.RUNNING);
            for (UndoManager um : undoManagers.values()) {
                while (um.canUndo()) {
                    um.undo();
                }
            }
            if (genericChangeUndoManager != null && genericChangeUndoManager.canUndo()) {
                genericChangeUndoManager.undo();
            }
        } finally {
            setStatus(Status.IDLE);
        }
    }
    
    private void _undo() {
        if (undoManagers == null) return;
        for (UndoManager um : undoManagers.values()) {
            while (um.canUndo()) {
                um.undo();
            }
        }
        if (genericChangeUndoManager != null && genericChangeUndoManager.canUndo()) {
            genericChangeUndoManager.undo();
        }
    }
    
    public synchronized boolean canRedo() {
        if (undoManagers == null || undoManagers.isEmpty()) {
            return false;
        }
        for (UndoManager um : undoManagers.values()) {
            if (! um.canRedo()) {
                return false; 
            }
        }
        if (genericChangeUndoManager != null && ! genericChangeUndoManager.canRedo()) {
            return false;
        }
        return true;
    }
    
    public synchronized void redo() throws CannotRedoException {
        if (! canRedo()) {
            throw new CannotRedoException();
        }
        try {
            setStatus(Status.RUNNING);
            for (UndoManager um : undoManagers.values()) {
                while (um.canRedo()) {
                    um.redo();
                }
            }
            if (genericChangeUndoManager != null && genericChangeUndoManager.canRedo()) {
                genericChangeUndoManager.redo();
            }
        } finally {
            setStatus(Status.IDLE);
        }
    }
    
    protected List<ChangeExecutor> getExecutors() {
        if (executors == null) {
            executors = new ArrayList<ChangeExecutor>();
            Lookup.Result results = Lookup.getDefault().lookup(
                    new Lookup.Template(ChangeExecutor.class));
            for (Object service : results.allInstances()){
                executors.add((ChangeExecutor)service);
            }
            executors.add(new GeneralChangeExecutor());
        }
        return executors;
    }
    
    protected List<RefactoringEngine> getEngines() {
        if (engines == null) {
            engines = new ArrayList<RefactoringEngine>();
            Lookup.Result results = Lookup.getDefault().lookup(
                    new Lookup.Template(RefactoringEngine.class));
            for (Object service : results.allInstances()){
                engines.add((RefactoringEngine)service);
            }
        }
        return engines;
    }
    
    private synchronized void addUndoableRefactorListener(Model model) {
        if (undoManagers == null) {
            undoManagers = new HashMap<Model,UndoManager>();
        }
        // checking against source to eliminate embedded model case
        FileObject source = (FileObject) model.getModelSource().getLookup().lookup(FileObject.class);
        if (source == null) {
            throw new IllegalArgumentException("Could not get source file from provided model"); //NOI18N
        }
        for (Model m : undoManagers.keySet()) {
            FileObject s = (FileObject) m.getModelSource().getLookup().lookup(FileObject.class);
            if (source.equals(s)) {
                return;
            }
        }
        
        if (undoManagers.get(model) == null) {
            UndoManager um = new UndoManager();
            model.addUndoableRefactorListener(um);
            undoManagers.put(model, um);
        }
    }
    
    private synchronized void removeRefactorUndoEventListeners() {
        if (undoManagers == null) return;
        for(Model model : undoManagers.keySet()) {
            model.removeUndoableRefactorListener(undoManagers.get(model));
        }
        if (lastRefactorRequest != null) {
            removeUndoableListener(lastRefactorRequest.getChangeExecutor());
        }
    }

    private synchronized void addUndoableListener(GeneralChangeExecutor executor) {
        genericChangeUndoManager = new UndoManager();
        executor.addUndoableEditListener(genericChangeUndoManager);
    }
    
    private synchronized void removeUndoableListener(ChangeExecutor exec) {
        if (! (exec instanceof GeneralChangeExecutor) || 
            genericChangeUndoManager == null || exec == null) {
            return;
        }
        
        GeneralChangeExecutor executor = (GeneralChangeExecutor) exec;
        executor.removeUndoableEditListener(genericChangeUndoManager);
    }
    
    /**
     * Returns UI helper to display find usage or refactoring of the given 
     * target component.
     */
    public UIHelper getTargetComponentUIHelper(Referenceable target) {
        for (ChangeExecutor ce : RefactoringManager.getInstance().getExecutors()) {
            if (ce.canChange(RenameRequest.class, target) ||
                ce.canChange(DeleteRequest.class, target)) {
                return ce.getUIHelper();
            }
        }
        return null;
    }

    /**
     * Pre-checking if all the pre-conditions are met for the request to be processed.
     * @param request the refactor request.
     * @return list of errors prevent the request from completedly processed;
     * return null if precheck is cancelled.
     */
    public synchronized List<ErrorItem> precheckChange(RefactorRequest request) {
        if (status != Status.IDLE) {
            throw new IllegalStateException("Invalid state "+status); //NOI18N
        }
        setStatus(Status.RUNNING);
        try {
            request.precheckChange();
            for (ChangeExecutor ce : getExecutors()) {
                if (! isRunning()) return null;
                if (ce.canChange(request.getType(), request.getTarget())) {
                    request.setChangeExecutor(ce);
                    ce.precheck(request);
                    break;
                }
            }
            return request.getErrors();
        } finally {
            setStatus(Status.IDLE);
        }
    }

    public synchronized List<ErrorItem> precheckUsages(RefactorRequest request) {
        if (status != Status.IDLE) {
            throw new IllegalStateException("Invalid state "+status);
        }
        setStatus(Status.RUNNING);
        
        try {
            if (request.getUsages() == null) {
                findUsageResult = findUsages(request.getTarget(), request.getScope());
                request.setUsages(findUsageResult.get());
            }
            request.precheckUsages();
            for (RefactoringEngine engine : getEngines()) {
                if (! isRunning()) return null;
                engine.precheck(request);
            }
        } catch (InterruptedException e) {
            request.addError(e.getLocalizedMessage());
        } catch (ExecutionException e) {
            request.addError(e.getLocalizedMessage());
        } finally {
            setStatus(Status.IDLE);
            findUsageResult = null;
        }

        return request.getErrors();
    }
    
    /**
     * Process refactor request.  The processing of the latest request 
     * could be cancelled by calling #cancel().
     * @param request the refactor request.
     */
     public synchronized void process(RefactorRequest request) throws IOException {
        if (request.hasFatalErrors()) {
            throw new IllegalStateException(
                    "Cannot process a request with errors"); //NOI18N
        }
        
        boolean isLocal = request.isScopeLocal();
        setStatus(Status.RUNNING);
        try {
            clearLastUndoSupport();
            lastRefactorRequest = request;
            
            if (request.getChangeExecutor() == null) {
                for (ChangeExecutor ce : getExecutors()) {
                    if (ce.canChange(request.getType(), request.getTarget())) {
                        request.setChangeExecutor(ce);
                        break;
                    }
                }
            }
            
            if (! isRunning()) return;
            Set<Model> excludedFromSave = request.getDirtyModels();
            UsageSet usageSet = request.getUsages();
            
            if (! isLocal) {
                if (request.getChangeExecutor() instanceof GeneralChangeExecutor) {
                    addUndoableListener((GeneralChangeExecutor) request.getChangeExecutor());
                } else {
                    addUndoableRefactorListener(request.getTargetModel());
                }
                request.getChangeExecutor().doChange(request);
                if (! request.confirmChangePerformed()) {
                    return;
                }
                
                for (UsageGroup u : usageSet.getUsages()) {
                    if (u.getItems() != null && ! u.getItems().isEmpty()) {
                        addUndoableRefactorListener(u.getModel());
                    }
                }
                
                for (RefactoringEngine engine : getEngines()) {
                    if (! isRunning()) return;
                    engine.refactorUsages(request);
                }
            } else { // isLocal
                Model model = request.getTargetModel();
                if (model != null) {
                    try {
                        model.startTransaction();
                        request.getChangeExecutor().doChange(request);
            
                        if (! request.confirmChangePerformed()) {
                            return;
                        }
                        
                        for (UsageGroup u : usageSet.getUsages()) {
                            if (! isRunning()) return;
                            if (u.getModel().equals(model)) {
                                u.getEngine().refactorUsages(request);
                            }
                        }

                    } finally {
                        if (model.isIntransaction()) {
                            model.endTransaction();
                        }
                    }
                }
            }
            
            if (request.getAutosave()) {
                RefactoringUtil.save(request, excludedFromSave);
            }
            
            usageSet.addPropertyChangeListener(this);
            
        } catch (RuntimeException t) {
            _undo();
            throw t;
        } catch (IOException ioe) {
            _undo();
            throw ioe;
        } finally {
            if (! isLocal) {
                removeRefactorUndoEventListeners();
                removeUndoableListener(request.getChangeExecutor());
            }
            if (! isRunning()) { // cancelled
                _undo();
                clearLastUndoSupport();
            }
            setStatus(Status.IDLE);
        }
    }
    
    /**
     * Execute the refactor request quietely if there are no errors.
     *
     * @params failsOnUsages if true the execution will fail when there are usages.
     * @exceptions CannotRefactorException when there are fatal errors or usages and failsOnUsages is set to true.
     */
    public void execute(RefactorRequest request, boolean failsOnUsages) throws CannotRefactorException, IOException {
        precheckChange(request);
        if (request.hasFatalErrors()) {
            throw new CannotRefactorException(request.getErrors().get(0).getMessage());
        }
        precheckUsages(request);
        if (failsOnUsages && ! request.getUsages().isEmpty()) {
            throw new CannotRefactorException(NbBundle.getMessage(RefactoringUtil.class, "MSG_HasUsages"));
        }
        if (request.hasFatalErrors()) {
            throw new CannotRefactorException(request.getErrors().get(0).getMessage());
        }
        process(request);
    }
    
    public synchronized boolean cancel() {
        if (! isRunning()) {
            return false;
        }
        setStatus(Status.CANCELLING);
        findUsageResult.cancel();
        return true;
    }
    
    public boolean isRunning() {
        return getStatus() == Status.RUNNING;
    }
    
    public boolean isRunning(UsageSet usages) {
        return isRunning() && 
               lastRefactorRequest != null && lastRefactorRequest.getUsages() == usages;
    }

    private synchronized Status getStatus() {
        return status;
    }
    
    private synchronized void setStatus(Status s) {
        status = s;
    }
    
    synchronized void clearLastUndoSupport() {
        undoManagers = null;
        genericChangeUndoManager = null;
        if (lastRefactorRequest != null && lastRefactorRequest.getUsages() != null) {
            lastRefactorRequest.getUsages().removePropertyChangeListener(this);
        }
        lastRefactorRequest = null;
    }
    
    public void propertyChange(PropertyChangeEvent evt) {
        if (UsageSet.VALID_PROPERTY.equals(evt.getPropertyName()) &&
                Boolean.FALSE.equals(evt.getNewValue())) {
            clearLastUndoSupport();
        }
    }

}
