/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.xml.refactoring;

import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.modules.xml.refactoring.impl.RefactoringUtil;
import org.netbeans.modules.xml.refactoring.spi.RefactoringEngine;
import org.netbeans.modules.xml.xam.Component;
import org.netbeans.modules.xml.xam.Model;
import org.netbeans.modules.xml.xam.Referenceable;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;
import org.openide.loaders.DataObject;
import org.openide.util.Cancellable;

/**
 *
 * @author Nam Nguyen
 */
public class FindUsageResult implements Cancellable, Future<UsageSet> {
    private List<SourceGroup> sourceGroups;
    private Referenceable target;
    private Set<Component> searchRoots;
    private UsageSet usageSet;
    private boolean done;
    private boolean cancelled;
    
    /** Creates a new instance of RefactorResult */
    public FindUsageResult(Referenceable target) {
        this.target = target;
        cancelled = false;
    }
    
    public FindUsageResult(Referenceable target, Component searchRoot) {
        this(target, Collections.singleton(searchRoot));
    }
    
    public FindUsageResult(Referenceable target, Set<Component> searchRoots) {
        this(target);
        this.searchRoots = searchRoots;
    }
    
    public FindUsageResult(Referenceable target, UsageSet usages) {
        this(target);
        this.usageSet = usages;
        done = true;
    }
    
    public UsageSet get(long timeout, TimeUnit unit) throws InterruptedException,
        ExecutionException, TimeoutException {
        throw new UnsupportedOperationException("Timeout not supported."); //NOI18N
    }
    
    public boolean cancel(boolean mayInterruptIfRunning) {
        if (done || cancelled) {
            return false;
        }
        //TODO something with mayInterruptIfRunning
        setCancelled(true);
        return true;
    }
    
    void setDone(boolean v) {
        done = v;
    }
    
    public boolean isDone() {
        return done;
    }
    
    void setCancelled(boolean v) {
        cancelled = v;
    }
    
    public boolean isCancelled() {
        return cancelled;
    }
    
    public UsageSet get() throws InterruptedException, ExecutionException {
        if (usageSet == null) {
            usageSet = new UsageSet(target);
            doSearch(getSearchRoots());
            done = true;
        }
        return usageSet;
    }
    
    protected void doSearch(Set<Component> searchRoots) {
        for (RefactoringEngine engine : RefactoringManager.getInstance().getEngines()) {
            for (Component root : searchRoots) {
                if (isCancelled()) return;
                List<UsageGroup> founds = findUsages(engine, target, root);
                if (founds != null) {
                    for (UsageGroup ug : founds) {
                        usageSet.addUsage(ug);
                        ug.setSourceGroup(getSourceGroup(root));
                    }
                }
            }
        }
    }
    
    private List<UsageGroup> findUsages(RefactoringEngine engine, Referenceable target, Component searchRoot) {
        if (target instanceof Model) {
            return engine.findUsages((Model) target, searchRoot);
        } else if (target instanceof Component) {
            return engine.findUsages((Component) target, searchRoot);
        } else {
            return null;
        }
    }
    
    public SourceGroup getSourceGroup(Component component) {
        if (component == null) return null;
        FileObject fobj = (FileObject) 
            component.getModel().getModelSource().getLookup().lookup(FileObject.class);
        if (fobj == null) return null;
        for (SourceGroup sg : getSourceGroups()) {
            FileObject rootFolder = sg.getRootFolder();
            if (fobj != rootFolder && !FileUtil.isParentOf(rootFolder, fobj)) {
                continue;
            }
            if (sg.contains(fobj)) {
                return sg;
            }
        }
        return null;
    }
    
    private List<SourceGroup> getSourceGroups() {
        if (sourceGroups == null) {
            Project project = RefactoringUtil.findCurrentProject(target);
            if (project == null) {
                sourceGroups = Collections.emptyList();
            } else {
                sourceGroups = RefactoringUtil.findSourceRoots(project);
            }
        }
        return sourceGroups;
    }
    
    protected Set<FileObject> getSearchFiles() {
        HashSet<FileObject> files = new HashSet<FileObject>();
        for (SourceGroup sourceGroup : getSourceGroups()) {
            files.addAll(RefactoringUtil.findSourceFiles(sourceGroup.getRootFolder()));
        }
        // make sure target source is also included in search in case outside projects
        DataObject dobj = RefactoringUtil.getDataObject(RefactorRequest.getModel(target));
        if (dobj != null) {
            files.add(dobj.getPrimaryFile());
        }
        return files;
    }
    
    protected Set<Component> getSearchRoots() {
        if (searchRoots != null) {
            return searchRoots;
        }
        searchRoots = new HashSet<Component>();
        for (FileObject file : getSearchFiles()) {
            for (RefactoringEngine engine : RefactoringManager.getInstance().getEngines()) {
                try {
                    if (isCancelled()) {
                        searchRoots.clear();
                        break;
                    }
                    Component root = engine.getSearchRoot(file);
                    searchRoots.add(root);
                } catch (IOException ex) {
                    ErrorManager.getDefault().log(ErrorManager.INFORMATIONAL, ex.getMessage());
                }
            }
        }
        return searchRoots;
    }
    
    public boolean cancel() {
        return cancel(true);
    }
}
