/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.websvc.wsitconf;
import java.awt.Container;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import javax.jmi.reflect.JmiException;
import javax.swing.JEditorPane;
import javax.swing.SwingUtilities;
import javax.swing.text.BadLocationException;
import org.netbeans.api.java.classpath.ClassPath;

import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.JMManager;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.jmiimpl.javamodel.JavaClassImpl;
import org.openide.ErrorManager;
import org.openide.cookies.EditorCookie;
import org.openide.cookies.EditorCookie;
import org.openide.cookies.SaveCookie;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.nodes.Node;
import org.openide.nodes.Node.Cookie;
import org.openide.text.CloneableEditorSupport;
import org.openide.text.PositionBounds;
import org.openide.text.PositionRef;
import org.openide.util.RequestProcessor;
import org.openide.windows.TopComponent;

/**
 * Helper for JMI-related code operations
 *
 * @author Martin Adamek
 * @author Pavel Fiala
 */
public class JMIUtils {

    private JMIUtils() {}

    public static JavaClass findClass(String className, ClassPath classPath) {
        JavaClass result = (JavaClass) resolveType(className, classPath);
        return result instanceof UnresolvedClass ? null : result;
    }

    public static JavaClass getJavaClassFromNode(Node node) {
        beginJmiTransaction();
        JavaClass jc = null;
        try {
            DataObject dobj = (DataObject) node.getCookie(DataObject.class);
            if (dobj != null) {
                Resource res = JavaModel.getResource(dobj.getPrimaryFile());
                if (res != null) {
                    JavaModel.setClassPath(res);
                    List/*<JavaClass>*/ classes = res.getClassifiers();
                    if (classes.size() == 1) {
                        jc = (JavaClass)classes.get(0);
                    }
                }
            } else {
                Feature feature = (Feature)node.getLookup().lookup(Feature.class);
                if (feature != null) {
                    jc = feature instanceof JavaClass ? (JavaClass)feature : (JavaClass) feature.getDeclaringClass();
                }
            }
            return jc;
        } finally {
            endJmiTransaction();
        }
    }

    public static void beginJmiTransaction(boolean writeAccess) {
        JavaModel.getJavaRepository().beginTrans(writeAccess);
    }

    public static void beginJmiTransaction() {
        beginJmiTransaction(false);
    }

    public static void endJmiTransaction(boolean rollback) {
        JavaModel.getJavaRepository().endTrans(rollback);
    }

    public static void endJmiTransaction() {
        endJmiTransaction(false);
    }

    /**
     * Finds Java class on default classpath for the passed FileObject
     *
     * @param className fully-qualified class name
     * @param fo file for resolving classpath
     * @return found class or null if not found
     */
    public static JavaClass findClass(String className, FileObject fo) {
        JavaClass result = (JavaClass) resolveType(className, fo);
        return result instanceof UnresolvedClass ? null : result;
    }

    public static JavaClass findClass(String className) {
        JavaClass result = (JavaClass) resolveType(className);
        return result instanceof UnresolvedClass ? null : result;
    }

    /**
     * Returns JavaClass for given Feature. This method was introduced, because
     * calling getDeclaringClass on JavaClass caused null result
     *
     * @param feature element to find declaring class for
     * @return Returns the declaring class for this feature
     */
    public static JavaClass getDeclaringClass(Feature feature) {
        if (feature == null) {
            return null;
        }
        return feature instanceof JavaClass ? (JavaClass) feature : (JavaClass) feature.getDeclaringClass();
    }

    /**
     * Checks whether given <code>typeName</code> represents resolvable 
     * <code>Type</code>, i.e. Type other than <code>UnresolvableClass</code>.
     * @param typeName String representing Type. 
     * @return true if given <code>typeName</code> doesn't represent <code>UnresolvableClass</code>.
     */
    public static boolean isResolvableType(String typeName){
        if (null == typeName || "".equals(typeName.trim())){
            return false;
        }
        if (typeName.endsWith("...")){
            typeName = typeName.substring(0, typeName.length() - 3);
        }
        return !(resolveType(typeName) instanceof UnresolvedClass);
    }
    
    public static Type resolveType(String typeName) {
        Type type = JavaModel.getDefaultExtent().getType().resolve(typeName);
        if (type instanceof UnresolvedClass) {
            Type basicType = JavaModel.getDefaultExtent().getType().resolve("java.lang." + typeName);  // NOI18N;
            if (!(basicType instanceof UnresolvedClass)) {
                return basicType;
            }
        }
        return type;
    }

    public static Type resolveType(String typeName, ClassPath classPath) {
        JavaModel.setClassPath(classPath);
        return resolveType(typeName);
    }

    /**
     * Resolves type on default classpath for the passed FileObject
     *
     * @param typeName fully-qualified type name
     * @param fo file for resolving classpath
     * @return resolved type
     */
    public static Type resolveType(String typeName, FileObject fo) {
        JavaModel.setClassPath(fo);
        return resolveType(typeName);
    }

    public static Method[] getMethods(JavaClass jc) {
        List/*<Method>*/ result = new LinkedList();
        if (jc != null) {
            List features = jc.getFeatures();
            for (Iterator it = features.iterator(); it.hasNext();) {
                Object o =  it.next();
                if (o instanceof Method) {
                    result.add(o);
                }
            }
        }
        return (Method[]) result.toArray(new Method[result.size()]);
    }

    public static List<Field> getFields(JavaClass jc) {
        List<Field> result = new ArrayList<Field>();
        if (jc != null) {
            List<Feature> features = jc.getFeatures();
            for (Feature feature : features) {
                if (feature instanceof Field) {
                    result.add((Field) feature);
                }
            }
        }
        return result;
    }

    /**
     * Returns first found field of given type or null if not such field exists.
     * Finds also fields with broken imports, so be aware that type of the field can be unresolved.
     */
    public static Field findFieldByType(JavaClass jc, String fqn) {
        if (jc == null || fqn == null) {
            throw new IllegalArgumentException("Tried to pass null argument");
        }
        String simple = fqn.substring(fqn.lastIndexOf(".") + 1);
        List<Field> fields = getFields(jc);
        for (Field field : fields) {
            String fieldTypeName = field.getTypeName().getName();
            if (fqn.equals(field.getType().getName()) || fqn.equals(fieldTypeName) || simple.equals(fieldTypeName)) {
                return field;
            }
        }
        return null;
    }
    
    public static Cookie getCookie(Feature feature, Class c) {
        if (feature == null) {
            return null;
        }
        DataObject dataObject = null;
        try {
            dataObject = DataObject.find(JavaModel.getFileObject(feature.getResource()));
        } catch (DataObjectNotFoundException dnfe) {
        }
        return dataObject == null ? null : dataObject.getCookie(c);
    }

    private static final TopComponent getTopComponent(Container temp) {
        while (!(temp instanceof TopComponent)) {
            temp = temp.getParent();
        }
        return (TopComponent) temp;
    }

    /** Checks if signature of a 'method1' matches signature of 'method2' */
    public static boolean equalMethods(Method method1, Method method2) {
        // TODO - check return types and exception
        if (!method1.getName().equals(method2.getName())) {
            return false;
        }

        List params1 = method1.getParameters();
        List params2 = method2.getParameters();
        if (params1.size() != params2.size()) {
            return false;
        }

        Iterator it1 = params1.iterator();
        Iterator it2 = params2.iterator();
        for (int i=0; i < params1.size(); i++) {
            Parameter param1 = (Parameter)it1.next();
            Parameter param2 = (Parameter)it2.next();
            if (!param1.getType().getName().equals(param2.getType().getName())) {
                return false;
            }
        }
        return true;
    }

    /**
     * gets short name of the element
     * @param el element
     * @return shart variant of element's name
     * @throws JmiException
     */
    public static String elementName(NamedElement el) throws JmiException {
        if (el instanceof JavaClass) {
            return ((JavaClass) el).getSimpleName();
        }

        if (el instanceof Constructor) {
            JavaClass jc = (JavaClass) ((Constructor) el).getDeclaringClass();
            return jc.getSimpleName();
        }

        return el.getName();
    }

    public static void addInterface(JavaClass javaClass, String interfaceName) {
        List interfaceNames = javaClass.getInterfaceNames();
        for (Iterator it = interfaceNames.iterator(); it.hasNext();) {
            if (interfaceName.equals(((MultipartId) it.next()).getElement().getName())) {
                return;
            }
        }
        interfaceNames.add(createMultipartId(javaClass, interfaceName));
    }

    public static void removeInterface(JavaClass javaClass, String interfaceName) {
        if (javaClass == null) {
            return;
        }
        for (Iterator it = javaClass.getInterfaceNames().iterator(); it.hasNext();) {
            String name = ((MultipartId) it.next()).getElement().getName();
            if (interfaceName.equals(name)) {
                it.remove();
            }
        }
    }

    public static Type getPrimitiveTypeWrapper(PrimitiveType type) {
        PrimitiveTypeKind kind = type.getKind();
        if (PrimitiveTypeKindEnum.BOOLEAN.equals(kind)) return resolveType("java.lang.Boolean"); // NOI18N
        if (PrimitiveTypeKindEnum.BYTE.equals(kind)) return resolveType("java.lang.Byte"); // NOI18N
        if (PrimitiveTypeKindEnum.CHAR.equals(kind)) return resolveType("java.lang.Character"); // NOI18N
        if (PrimitiveTypeKindEnum.DOUBLE.equals(kind)) return resolveType("java.lang.Double"); // NOI18N
        if (PrimitiveTypeKindEnum.FLOAT.equals(kind)) return resolveType("java.lang.Float"); // NOI18N
        if (PrimitiveTypeKindEnum.INT.equals(kind)) return resolveType("java.lang.Integer"); // NOI18N
        if (PrimitiveTypeKindEnum.LONG.equals(kind)) return resolveType("java.lang.Long"); // NOI18N
        if (PrimitiveTypeKindEnum.SHORT.equals(kind)) return resolveType("java.lang.Short"); // NOI18N
        return null;
    }

    public static Method findInClass(Method me, JavaClass jc) {
        List parameterTypes = new ArrayList();
        for (Iterator it = me.getParameters().iterator(); it.hasNext();) {
            Parameter parameter = (Parameter) it.next();
            // TODO: this is ugly hack. There is problem when method has parameter
            // of type String and in class is same method with parameter of type
            // java.lang.String, this method is not found with javaClass.GetMethod()
            Type type = resolveType(parameter.getType().getName());
            if (type instanceof UnresolvedClass) {
                type = parameter.getType();
            }
            parameterTypes.add(type);
        }
        return jc.getMethod(me.getName(), parameterTypes, false);
    }

    public static MultipartId createMultipartId(Feature feature, String name) {
        return getJavaModelPackage(feature).getMultipartId().createMultipartId(name, null, null);
    }

    /**
     * just for testing
     */
    public static String printParameters(Method method) {
        if (method == null) {
            return "### null";
        }
        StringBuffer result = new StringBuffer("### " + method.getName() + "(");
        for (Iterator it = method.getParameters().iterator(); it.hasNext();) {
            Parameter parameter = (Parameter) it.next();
            result.append(parameter.getType().getName());
            if (it.hasNext()) {
                result.append(",");
            }
        }
        result.append(")");
        return result.toString();
    }

    private static JavaModelPackage getJavaModelPackage(Feature feature) {
        return feature != null ? (JavaModelPackage) feature.refImmediatePackage() : JavaModel.getDefaultExtent();
    }

    /**
     * Checks whether the signatures of given methods are equal. This method 
     * differs <code>Method#signatureEquals</code> in that it performs an extra
     * check to see whether parameter lists of methods' were empty or null. This is
     * useful for example when comparing equality of a newly created method's 
     * signature. 
     *@param method1 must not be null
     *@param method2 must not be null
     *@return true if signatures of given methods were equal.
     */
    public static boolean signaturesEqual(Method method1, Method method2){
        assert method1 != null && method2 != null;
        if (isEmptyOrNull(method1.getParameters()) && isEmptyOrNull(method2.getParameters())){
            return true;
        }
        return method1.signatureEquals(method2);
    }
    
    private static boolean isEmptyOrNull(Collection collection){
        return collection == null ? true : collection.isEmpty();
    }   
    
    /**
     * Generates unique member name in class-scope
     * @param javaClass scope for uniqueness
     * @param memberName suggested member name
     * @return given member name if no such member exists or given member name extended with unique number
     */
    public static String uniqueMemberName(JavaClass javaClass, String memberName){
        List<Feature> existing = javaClass.getFeatures();
        List<String> existingMethodNames = new ArrayList<String>();
        for (Feature f : existing) {
            existingMethodNames.add(f.getName());
        }
        int uniquefier = 1;
        String newName = memberName;
        while (existingMethodNames.contains(newName)){
            newName = memberName + uniquefier++;
        }
        return newName;
    }
}
