/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 *
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.j2ee.dd.impl.webservices;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.project.JavaProjectConstants;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.AttributeValue;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.StringLiteral;
import org.netbeans.modules.j2ee.dd.api.ejb.DDProvider;
import org.netbeans.modules.j2ee.dd.api.ejb.Ejb;
import org.netbeans.modules.j2ee.dd.api.ejb.EjbJar;
import org.netbeans.modules.j2ee.dd.api.ejb.EnterpriseBeans;
import org.netbeans.modules.j2ee.dd.api.ejb.Session;
import org.netbeans.modules.j2ee.dd.api.webservices.PortComponent;
import org.netbeans.modules.j2ee.dd.api.webservices.ServiceImplBean;
import org.netbeans.modules.j2ee.dd.api.webservices.WebserviceDescription;
import org.netbeans.modules.j2ee.dd.api.webservices.Webservices;
import org.netbeans.modules.j2ee.metadata.NNModelBuilder;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.schema2beans.BaseBean;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;

/**
 *
 * @author Martin Adamek
 */
public class WsAnnotationListener extends NNModelBuilder {
    
    private static final String JAVAX_EJB_WEBSERVICE = "javax.jws.WebService";
    
    private static final Collection<String> primaryAnnotations = Arrays.asList(new String[]{
        JAVAX_EJB_WEBSERVICE
    });
    
    /** A map storing annotations to model structure mapping. */
    private Map<String, String> nn2ModelMapping;
    private Webservices ws;
    private ClassPath cp;
    
    public WsAnnotationListener(Webservices ws, ClassPath cp) {
        super((BaseBean) ws);
        this.ws = ws;
        this.cp = cp;
        initNN2ModelMapping();
    }
    
    protected Map<String, String> getAnnotation2ModelMapping() {
        return nn2ModelMapping;
    }
    
    public ClassPath getClassPath() {
        return cp;
    }
    
    public Collection<String> getPrimaryAnnotations() {
        return primaryAnnotations;
    }
    
    public void classRemoved(String fqn) {
        WebserviceDescription wsd = findWebserviceByServiceEndpointInterface(fqn);
        if (wsd != null) {
            ws.removeWebserviceDescription(wsd);
        }
//         System.out.println("");
//        ((BaseBean) ws).dumpXml();
    }
    
    
    @Override()
    public void removeClassAnnotation(JavaClass javaClass, Annotation nn, AnnotationType at) {
        FileObject fo = JavaModel.getFileObject(javaClass.getResource());
        if (isInSourceGroup(fo)) { // excluding files from build/generated
            String annotationClass = at.getName();
            if (JAVAX_EJB_WEBSERVICE.equals(annotationClass)) {
                String wsName = getWSName(javaClass, nn);
                WebserviceDescription wsd = getWebservice(wsName);
                if(wsd != null) {
                    //we found matching WSD, lets delete it
                    ws.removeWebserviceDescription(wsd);
                }
            }
        }
//        System.out.println("");
//        ((BaseBean) ws).dumpXml();
        
    }
    
    @Override()
    public void addClassAnnotation(JavaClass javaClass, Annotation annotation, AnnotationType annotationType) {
        FileObject fo = JavaModel.getFileObject(javaClass.getResource());
        if (isInSourceGroup(fo)) { // excluding files from build/generated
            String annotationClass = annotationType.getName();
            if (JAVAX_EJB_WEBSERVICE.equals(annotationClass)) {
                // find WS name (from annotation attribute or default one)
                String wsName = getWSName(javaClass, annotation);
                WebserviceDescription wsd = getWebservice(wsName);
                
                if(wsd == null) {
                    //create the WebserviceDescription
                    wsd = ws.newWebserviceDescription();
                    wsd.setWebserviceDescriptionName(wsName);
                    ws.addWebserviceDescription(wsd);
                }
                
                // try to find PortComponent
                boolean isNewPortComponent = false;
                PortComponent pc = (PortComponent) CommonDDAccess.findBeanByName((BaseBean) wsd, WebserviceDescription.PORT_COMPONENT, PortComponent.PORT_COMPONENT_NAME, wsName);
                // if PortComponent doesn't exist, create one
                if (pc == null) {
                    pc = wsd.newPortComponent();
                    pc.setPortComponentName(getPCName(javaClass, annotation));
                    isNewPortComponent = true;
                }
                // service-endpoint-interface (it is not needed,
                // but we will use it for removing ws from model on class removal event
                pc.setServiceEndpointInterface(javaClass.getName());
                // try to find ServiceImplBean
                boolean isNewServiceImplBean = false;
                ServiceImplBean sib = pc.getServiceImplBean();
                if (sib == null) {
                    sib = pc.newServiceImplBean();
                    isNewServiceImplBean = true;
                }
                // find if this WS is session bean
                org.netbeans.modules.j2ee.api.ejbjar.EjbJar apiEjbJar = org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(fo);
                if (apiEjbJar != null) {
                    EjbJar ejbJar = null;
                    try {
                        ejbJar = DDProvider.getDefault().getMergedDDRoot(apiEjbJar.getMetadataUnit());
                    } catch (IOException ex) {
                        ErrorManager.getDefault().notify(ex);
                    }
                    if (ejbJar != null) {
                        EnterpriseBeans ebs = ejbJar.getEnterpriseBeans();
                        if (ebs != null) {
                            Session session = (Session) ebs.findBeanByName(EnterpriseBeans.SESSION, Ejb.EJB_CLASS, javaClass.getName());
                            if (session != null) {
                                sib.setEjbLink(session.getEjbName());
                            }
                        }
                    }
                }
                // if it is not session bean, just use simple class name
                // TODO: define servlet name other than default simple class name?
                if (sib.getEjbLink() == null) {
                    sib.setServletLink(javaClass.getSimpleName());
                }
                
                if (isNewServiceImplBean) {
                    pc.setServiceImplBean(sib);
                }
                if (isNewPortComponent) {
                    wsd.addPortComponent(pc);
                }
            }
//            System.out.println("");
//            ((BaseBean) ws).dumpXml();
        }
    }
    
    // private -----------------------------------------------------------------
    
    private String getWSName(JavaClass javaClass, Annotation annotation) {
        AttributeValue nameAV = findAttribute(annotation, "serviceName"); //NOI18N
        String wsName = null;
        // there is "name" attribute
        if (nameAV != null) {
            Object o = nameAV.getValue();
            if (o instanceof StringLiteral) {
                wsName = ((StringLiteral) o).getValue();
            }
        }
        // no "name" attribute, use simple class name
        if (wsName == null) {
            wsName = javaClass.getSimpleName() + "Service"; //NOI18N
        }
        return wsName;
    }
    
    private String getPCName(JavaClass javaClass, Annotation annotation) {
        AttributeValue nameAV = findAttribute(annotation, "name"); //NOI18N
        String wsName = null;
        // there is "name" attribute
        if (nameAV != null) {
            Object o = nameAV.getValue();
            if (o instanceof StringLiteral) {
                wsName = ((StringLiteral) o).getValue();
            }
        }
        // no "name" attribute, use simple class name
        if (wsName == null) {
            wsName = javaClass.getSimpleName();
        }
        return wsName;
    }
    
    private boolean isInSourceGroup(FileObject fo) {
        Project project = FileOwnerQuery.getOwner(fo);
        assert project!=null;
        SourceGroup[] sourceGroups = ProjectUtils.getSources(project).getSourceGroups(JavaProjectConstants.SOURCES_TYPE_JAVA);
        for(int i = 0; i < sourceGroups.length; i++) {
            if(FileUtil.isParentOf(sourceGroups[i].getRootFolder(), fo)) return true;
        }
        return false;
    }
    
    private WebserviceDescription getWebservice(String name) {
        return (WebserviceDescription) ws.findBeanByName(Webservices.WEBSERVICE_DESCRIPTION, WebserviceDescription.WEBSERVICE_DESCRIPTION_NAME, name);
    }
    
    private WebserviceDescription findWebserviceByServiceEndpointInterface(String fqn) {
        if (fqn == null) {
            return null;
        }
        for (WebserviceDescription wsd : ws.getWebserviceDescription()) {
            for (PortComponent pc : wsd.getPortComponent()) {
                if (fqn.equals(pc.getServiceEndpointInterface())) {
                    return wsd;
                }
            }
        }
        return null;
    }
    
    private void initNN2ModelMapping() {
        nn2ModelMapping = new HashMap();
        Map imaps = new HashMap();
        
        // following list represents requirements for this listener:
        //
        //    *  DConfigBean bindings (e.g. these control display of UI panels and are very important)
        //          o /webservices
        //          o /webservices/webservice-description
        //    * Other bindings (e.g. for subpanel UI or validity constraints -- not necessarily as important as above)
        //          o /webservices/webservice-description/webservice-description-name
        //          o /webservices/webservice-description/port-component
        //          o /webservices/webservice-description/port-component/port-component-name
        //          o /webservices/webservice-description/port-component/service-impl-bean/servlet-link
        //          o /webservices/webservice-description/port-component/service-impl-bean/ejb-link
        //
        
        nn2ModelMapping.put(JAVAX_EJB_WEBSERVICE, "WebserviceDescription#!serviceName=WebserviceDescriptionName;WebserviceDescriptionName=$JAVACLASS_SIMPLE");
        
    }
    
    private static AttributeValue findAttribute(Annotation a, String attributeName) {
        List<AttributeValue> attributeValues = a.getAttributeValues();
        for (AttributeValue av : attributeValues) {
            if (av.getName().equals(attributeName)) {
                return av;
            }
        }
        return null;
    }
    
}
