/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.websvc.jaxwsmodel.project.metadata;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Timer;
import java.util.TimerTask;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.api.java.project.JavaProjectConstants;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.api.project.ProjectUtils;
import org.netbeans.api.project.SourceGroup;
import org.netbeans.jmi.javamodel.Annotation;
import org.netbeans.jmi.javamodel.AnnotationType;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.modules.j2ee.metadata.NNListener;
import org.netbeans.modules.j2ee.metadata.Utils;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.websvc.api.jaxws.project.WSUtils;
import org.netbeans.modules.websvc.api.jaxws.project.config.JaxWsModel;
import org.netbeans.modules.websvc.api.jaxws.project.config.Service;
import org.netbeans.modules.websvc.api.jaxws.project.config.ServiceAlreadyExistsExeption;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileUtil;

/**
 *
 * @author Marek Fukala
 */
public class JAXWSNNListener implements NNListener {
    
    private static final ArrayList<String> supportedAnnotations;
    static {
        supportedAnnotations = new ArrayList();
        supportedAnnotations.add("javax.jws.WebService");//NOI18N
    }
    
    private static final String WS_NAME = "name";//NOI18N
    private static final String WSDL_LOCATION_NN_ATTRIBUTE = "wsdlLocation"; //NOI18N
    
    private JaxWsModel jaxWSmodel;
    private ClassPath cp;
    
    //jax-ws.xml update task timer
    private Timer timer = new Timer();
    private TimerTask ttask = createTimerTask();
    
    /** Creates a new instance of JAXWSNNListener */
    public JAXWSNNListener(JaxWsModel jaxWSmodel, ClassPath cp) {
        this.jaxWSmodel = jaxWSmodel;
        this.cp = cp;
    }
    
    public ClassPath getClassPath() {
        return cp;
    }
    
    public Collection<String> getSupportedAnnotations() {
        return supportedAnnotations;
    }
    
    public Collection<String> getPrimaryAnnotations() {
        return Collections.emptyList();
    }
    
    public void addClassAnnotation(JavaClass javaClass, Annotation annotation, AnnotationType annotationType) {
        //do not update the jaxwsmodel if the annotation contains wsdlLocation attribute
        //it is necessary for special handling of WS generated from WSDL file
        if(Utils.getNNAttributes(annotation).get(WSDL_LOCATION_NN_ATTRIBUTE) == null) {
            String wsName = getWSName(javaClass, annotation);
            String className = javaClass.getName();
            FileObject fo = JavaModel.getFileObject(javaClass.getResource());
            if(isInSourceGroup(fo) && !javaClass.isInterface()) {
                try {
                    Service service = jaxWSmodel.findServiceByImplementationClass(className);
                    if (service == null) {
                        jaxWSmodel.addService(wsName, className);
                        requestModelUpdate();
                    }
                }catch(ServiceAlreadyExistsExeption e) {
                    //the service name already exists - we need to check whether the impl. class is also the same
                    Service service = jaxWSmodel.findServiceByImplementationClass(className);
                    if(service == null || !wsName.equals(service.getName())) {
                        //there is no service for the impl. class with the wsName name => generate new wsName
                        String newName = WSUtils.findProperServiceName(wsName, jaxWSmodel);
                        try {
                            jaxWSmodel.addService(newName, className);
                        } catch (ServiceAlreadyExistsExeption ex1) {
                            //this shouldn't happen
                        }
                    }
                }
            }
        }
    }
    
    public void removeClassAnnotation(JavaClass javaClass, org.netbeans.jmi.javamodel.Annotation annotation, AnnotationType annotationType) {
        //do not update the jaxwsmodel if the annotation contains wsdlLocation attribute
        //it is necessary for special handling of WS generated from WSDL file
        FileObject fo = JavaModel.getFileObject(javaClass.getResource());
        if(fo == null || isInSourceGroup(fo)) {
            Service serviceToRemove = jaxWSmodel.findServiceByImplementationClass(javaClass.getName());
            if (serviceToRemove != null && serviceToRemove.getWsdlUrl() == null) {
                jaxWSmodel.removeServiceByClassName(javaClass.getName());
                requestModelUpdate();
            }
        }
    }
    
    public void classRemoved(String fqn) {
        //do not update the jaxwsmodel if the annotation contains wsdlLocation attribute
        //it is necessary for special handling of WS generated from WSDL file
        // TODO: check if service was in source group, but how (file doesn't exist anymore)?
        Service serviceToRemove = jaxWSmodel.findServiceByImplementationClass(fqn);
        if (serviceToRemove != null && serviceToRemove.getWsdlUrl() == null) {
            jaxWSmodel.removeServiceByClassName(fqn);
            requestModelUpdate();
        }
    }
    
    public void addClassAttribute(JavaClass javaClass, Annotation annotation, AnnotationType t, String attributeName, String attributeValue) {
    }
    
    public void removeClassAttribute(JavaClass javaClass, Annotation annotation, AnnotationType t, String attributeName) {
    }
    
    public void addMemberAttribute(JavaClass javaClass, Element member, Annotation annotation, AnnotationType t, String attributeName, String attributeValue) {
    }
    
    public void removeMemberAttribute(JavaClass javaClass, Element memberName, Annotation annotation, AnnotationType t, String attributeName) {
    }
    
    public void addMemberAnnotation(boolean field, JavaClass javaClass, Element member, Annotation annotation, AnnotationType t) {
    }
    
    public void removeMemberAnnotation(boolean field, JavaClass javaClass, String memberName, String memberType, Annotation annotation, AnnotationType t) {
    }
    
    private boolean isInSourceGroup(FileObject fo) {
        Project project = FileOwnerQuery.getOwner(jaxWSmodel.getJaxWsFile());
        SourceGroup[] sourceGroups = ProjectUtils.getSources(project).getSourceGroups(JavaProjectConstants.SOURCES_TYPE_JAVA);
        for(int i = 0; i < sourceGroups.length; i++) {
            if(FileUtil.isParentOf(sourceGroups[i].getRootFolder(), fo)) return true;
        }
        return false;
    }
    
    private void requestModelUpdate() {
        synchronized (ttask) {
            ttask.cancel();
            //create a new task a schedule it
            ttask = createTimerTask();
            timer.schedule(ttask, 100);
        }
    }
    
    private TimerTask createTimerTask() {
        return new TimerTask() {
            public void run() {
                synchronized (ttask) {
                    try {
                        //try to ensure
                        jaxWSmodel.write();
                    }catch(IOException ioe) {
                        //probably cannot get exclusive access to the file -> try it later
                        requestModelUpdate();
                    }
                }
            }
        };
    }
    
    private String getWSName(JavaClass javaClass, Annotation annotation) {
        String wsname = (String)Utils.getNNAttributes(annotation).get(WS_NAME);
        if(wsname == null) {
            //use default name
            wsname = javaClass.getSimpleName();
        }
        return wsname;
    }

}
