/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.websvc.editor.completion;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import javax.swing.text.BadLocationException;
import org.netbeans.editor.BaseDocument;
import org.netbeans.editor.SyntaxSupport;
import org.netbeans.editor.TokenItem;
import org.netbeans.editor.ext.java.JavaSyntaxSupport;
import org.netbeans.editor.ext.java.JavaTokenContext;
import org.openide.ErrorManager;

/**
 * Builds an annotations tree containg NN name and attribs map. Supports nested annotations.
 *
 * @author Marek Fukala
 */
public class NNParser {
    
    //parser states
    private static final int INIT = 0;
    private static final int NN = 1; //@
    private static final int ERROR = 2;
    private static final int NNNAME = 3; //@Table
    private static final int INNN = 4; //@Table(
    private static final int ATTRNAME = 5; //@Table(name
    private static final int EQ = 6; //@Table(name=
    private static final int ATTRVALUE = 7; //@Table(name="hello" || @Table(name=@
    
    private JavaSyntaxSupport sup;
    
    public NNParser(BaseDocument bdoc) {
        SyntaxSupport ssup = bdoc.getSyntaxSupport();
        if(!(ssup instanceof JavaSyntaxSupport)) throw new IllegalArgumentException("Only java files are supported!");
        sup = (JavaSyntaxSupport)ssup;
    }
    
    public NN parseAnnotation(int offset) {
        int nnStart = findAnnotationStart(offset);
        if(nnStart == -1) {
            return null;
        } else {
            return parseAnnotationOnOffset(nnStart);
        }
    }
    
    /** very simple annotations parser */
    private NN parseAnnotationOnOffset(int offset) {
        try {
            int parentCount = -1;
            int state = INIT;
            TokenItem ti = sup.getTokenChain(offset, offset+1);
            
            assert ti.getTokenID() == JavaTokenContext.ANNOTATION;
            
            int nnstart = offset;
            int nnend = -1;
            String nnName = null;
            String currAttrName = null;
            Object currAttrValue = null;
            Map attrs = new HashMap();
            
            do {
                int tid = ti.getTokenID().getNumericID();
                //ignore whitespaces
                if(tid == JavaTokenContext.WHITESPACE_ID) {
                    ti = ti.getNext();
                    continue;
                }
                
                switch(state) {
                    case INIT:
                        switch(tid) {
                            case JavaTokenContext.ANNOTATION_ID:
                                state = NN;
                                break;
                            default:
                                state = ERROR;
                        }
                        break;
                    case NN:
                        switch(tid) {
                            case JavaTokenContext.IDENTIFIER_ID:
                                state = NNNAME;
                                nnName = ti.getImage();
//                                debug("parsing annotation " + nnName);
                                break;
                            default:
                                state = ERROR;
                        }
                        break;
                    case NNNAME:
                        switch(tid) {
                            case JavaTokenContext.LPAREN_ID:
                                state = INNN;
                                break;
                            default:
                                //we need to handle situations when the annotation is not completely valid
                                //->the cases when the CC is invoked like:
                                //@Table(type="something", name=|
                                state = ERROR;
                        }
                        break;
                    case INNN:
                        switch(tid) {
                            case JavaTokenContext.IDENTIFIER_ID:
                                currAttrName = ti.getImage();
//                                debug("parsing attribute " + currAttrName);
                                state = ATTRNAME;
                                break;
                            default:
                                state = ERROR;
                        }
                        break;
                    case ATTRNAME:
                        switch(tid) {
                            case JavaTokenContext.EQ_ID:
                                state = EQ;
                                break;
                            default:
                                state = ERROR;
                        }
                        break;
                    case EQ:
                        switch(tid) {
                            case JavaTokenContext.STRING_LITERAL_ID:
                                state = ATTRVALUE;
                                currAttrValue = Utils.unquote(ti.getImage());
//                                debug("found atribute value " + currAttrValue);
                                attrs.put(currAttrName, currAttrValue);
                                break;
                            case JavaTokenContext.ANNOTATION_ID:
                                //nested annotation
                                NN nestedNN = parseAnnotationOnOffset(ti.getOffset());
                                attrs.put(currAttrName, nestedNN);
                                state = ATTRVALUE;
                                //I need to skip what was parsed in the nested annotation in this parser
                                ti = sup.getTokenChain(nestedNN.getEndOffset(), nestedNN.getEndOffset()+1);
                                continue; //next loop
                            default:
//                                debug("currently we support just annotation and string attribute value types!");
                                state = ERROR;
                        }
                    case ATTRVALUE:
                        switch(tid) {
                            case JavaTokenContext.COMMA_ID:
                                state = INNN;
                                break;
                            case JavaTokenContext.RPAREN_ID:
                                //we reached end of the annotation
                                nnend = ti.getOffset() + ti.getImage().length();
                                NN newNN = new NN(nnName, attrs, nnstart, nnend);
                                return newNN;
                        }
                        break;
                }
                
                if(state == ERROR) break;
                
                ti = ti.getNext();//get next token
                
            } while(ti != null);
            
            //we get to the end of the file without finding NN end - this may happen when the source is broken
            if(nnName != null) {
                NN newNN = new NN(nnName, attrs, nnstart, offset);
                return newNN;
            }
            
        }catch(BadLocationException e) {
            ErrorManager.getDefault().notify(ErrorManager.WARNING, e);
        }
        
        return null;
    }
   
    
    private int  findAnnotationStart(int offset) {
        int parentCount = 0;
        try {
            TokenItem ti = sup.getTokenChain(offset - 1, offset);
            while(ti != null) {
//                debug(ti);
                if(ti.getTokenID() == JavaTokenContext.RPAREN) {
                    parentCount++;
                } else if(ti.getTokenID() == JavaTokenContext.LPAREN) {
                    parentCount--;
                } else if(ti.getTokenID() == JavaTokenContext.ANNOTATION) {
                    if(parentCount == -1) {
//                        debug("found outer annotation: " + ti.getImage());
                        return ti.getOffset();
                    }
                }
                ti = ti.getPrevious();
            }
            
        }catch(BadLocationException e) {
            ErrorManager.getDefault().notify(ErrorManager.WARNING, e);
        }
        
        return -1;
    }
       
//    private static void debug(Object message) {
//        System.out.println(message.toString());
//    }
    
   public class NN {
        
        private String name;
        private Map attributes;
        private int startOffset, endOffset;
        
        public NN(String name, Map attributes, int startOffset, int endOffset) {
            this.name = name;
            this.attributes = attributes;
            this.startOffset = startOffset;
            this.endOffset = endOffset;
        }
        
        public String getName() {
            return name;
        }
        
        public Map getAttributes() {
            return attributes;
        }
        
        public int getStartOffset() {
            return startOffset;
        }
        
        public int getEndOffset() {
            return endOffset;
        }
        
        public String toString() {
            //just debug purposes -> no need for superb performance
            String text = "@" + getName() + " [" + getStartOffset() + " - " + getEndOffset() + "](";
            Iterator i = getAttributes().keySet().iterator();
            while(i.hasNext()) {
                String key = (String)i.next();
                Object value = getAttributes().get(key);
                text+=key+"="+value.toString()+(i.hasNext() ? "," : "");
            }
            text+=")";
            return text;
        }
        
    }
    
}
