/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.proxy;

import javax.net.SocketFactory;
import java.io.*;
import java.net.*;
import java.util.regex.*;
import java.nio.channels.SocketChannel;

/**
 * This class is responsible for establishing socket connection to a given IP address. Whether the
 * connection is direct or routes via some type of proxy depends on current ConnectivitySettings that must
 * be provided by the caller. Proxy protocols and authentication methods are handled internally and transparently
 * to the caller.
 *
 * @author Maros Sandor
 */
public class ClientSocketFactory extends SocketFactory {
    private static final int CONNECT_TIMEOUT = 1000 * 20; /// 20 seconds timeout

    private static final String AUTH_NONE = "<none>";
    private static final String AUTH_BASIC = "Basic";
    private static final Pattern sConnectionEstablishedPattern = Pattern.compile("HTTP\\/\\d+\\.\\d+\\s+200\\s+");
    private static final Pattern sProxyAuthRequiredPattern = Pattern.compile("HTTP\\/\\d+\\.\\d+\\s+407\\s+");

    private ConnectivitySettings mSettings;

    /**
     * Creates a new socket IP connection factory.
     *
     * @param settings the connectivity settings to use when connecting to a remote IP address
     */
    public ClientSocketFactory(ConnectivitySettings settings) {
        mSettings = settings;
    }

    /**
     * Creates probe socket that supports only
     * connect(SocketAddressm, int timeout).
     */
    public Socket createSocket() throws IOException {
        return new Socket() {
            public void connect(SocketAddress endpoint, int timeout) throws IOException {
                Socket s = createSocket((InetSocketAddress)endpoint, timeout);
                s.close();
            }

            public void bind(SocketAddress bindpoint) {
                throw new UnsupportedOperationException();
            }

            protected Object clone() {
                throw new UnsupportedOperationException();
            }

            public synchronized void close() {
            }

            public void connect(SocketAddress endpoint) {
                throw new UnsupportedOperationException();
            }

            public SocketChannel getChannel() {
                throw new UnsupportedOperationException();
            }

            public InetAddress getInetAddress() {
                throw new UnsupportedOperationException();
            }

            public InputStream getInputStream() {
                throw new UnsupportedOperationException();
            }

            public boolean getKeepAlive() {
                throw new UnsupportedOperationException();
            }

            public InetAddress getLocalAddress() {
                throw new UnsupportedOperationException();
            }

            public int getLocalPort() {
                throw new UnsupportedOperationException();
            }

            public SocketAddress getLocalSocketAddress() {
                throw new UnsupportedOperationException();
            }

            public boolean getOOBInline() {
                throw new UnsupportedOperationException();
            }

            public OutputStream getOutputStream() {
                throw new UnsupportedOperationException();
            }

            public int getPort() {
                throw new UnsupportedOperationException();
            }

            public synchronized int getReceiveBufferSize() {
                throw new UnsupportedOperationException();
            }

            public SocketAddress getRemoteSocketAddress() {
                throw new UnsupportedOperationException();
            }

            public boolean getReuseAddress() {
                throw new UnsupportedOperationException();
            }

            public synchronized int getSendBufferSize() {
                throw new UnsupportedOperationException();
            }

            public int getSoLinger() {
                throw new UnsupportedOperationException();
            }

            public synchronized int getSoTimeout() {
                throw new UnsupportedOperationException();
            }

            public boolean getTcpNoDelay() {
                throw new UnsupportedOperationException();
            }

            public int getTrafficClass() {
                throw new UnsupportedOperationException();
            }

            public boolean isBound() {
                throw new UnsupportedOperationException();
            }

            public boolean isClosed() {
                throw new UnsupportedOperationException();
            }

            public boolean isConnected() {
                throw new UnsupportedOperationException();
            }

            public boolean isInputShutdown() {
                throw new UnsupportedOperationException();
            }

            public boolean isOutputShutdown() {
                throw new UnsupportedOperationException();
            }

            public void sendUrgentData(int data) {
                throw new UnsupportedOperationException();
            }

            public void setKeepAlive(boolean on) {
                throw new UnsupportedOperationException();
            }

            public void setOOBInline(boolean on) {
                throw new UnsupportedOperationException();
            }

            public synchronized void setReceiveBufferSize(int size) {
                throw new UnsupportedOperationException();
            }

            public void setReuseAddress(boolean on) {
                throw new UnsupportedOperationException();
            }

            public synchronized void setSendBufferSize(int size) {
                throw new UnsupportedOperationException();
            }

            public void setSoLinger(boolean on, int linger) {
                throw new UnsupportedOperationException();
            }

            public synchronized void setSoTimeout(int timeout) {
                throw new UnsupportedOperationException();
            }

            public void setTcpNoDelay(boolean on) {
                throw new UnsupportedOperationException();
            }

            public void setTrafficClass(int tc) {
                throw new UnsupportedOperationException();
            }

            public void shutdownInput() {
                throw new UnsupportedOperationException();
            }

            public void shutdownOutput() {
                throw new UnsupportedOperationException();
            }
        };
    }

    public Socket createSocket(String host, int port) throws IOException, UnknownHostException {
        return createSocket(new InetSocketAddress(host, port), CONNECT_TIMEOUT);
    }

    public Socket createSocket(InetAddress inetAddress, int port) throws IOException {
        return createSocket(new InetSocketAddress(inetAddress, port), CONNECT_TIMEOUT);
    }

    public Socket createSocket(String s, int i, InetAddress inetAddress, int i1) throws IOException,
            UnknownHostException {
        throw new IOException("Unsupported operation");
    }

    public Socket createSocket(InetAddress inetAddress, int i, InetAddress inetAddress1, int i1) throws IOException {
        throw new IOException("Unsupported operation");
    }

    /**
     * Connects to the remote machine by establishing a tunnel through a HTTP proxy. It issues a CONNECT request and
     * eventually authenticates with the HTTP proxy. Supported authentication methods include: Basic.
     *
     * @param address remote machine to connect to
     * @return a TCP/IP socket connected to the remote machine
     * @throws UnknownHostException if the proxy host name cannot be resolved
     * @throws IOException          if an I/O error occurs during handshake (a network problem)
     */
    private Socket getHttpsTunnelSocket(InetSocketAddress address, int timeout) throws UnknownHostException, IOException {
        Socket proxy = new Socket();
        proxy.connect(new InetSocketAddress(mSettings.getProxyHost(), mSettings.getProxyPort()), timeout);
        BufferedReader r = new BufferedReader(new InputStreamReader(new InterruptibleInputStream(proxy.getInputStream())));
        DataOutputStream dos = new DataOutputStream(proxy.getOutputStream());

        dos.writeBytes("CONNECT ");
        dos.writeBytes(address.getHostName() + ":" + address.getPort());
        dos.writeBytes(" HTTP/1.0\r\n");
        dos.writeBytes("Connection: Keep-Alive\r\n\r\n");
        dos.flush();

        String line;
        line = r.readLine();

        if (sConnectionEstablishedPattern.matcher(line).find()) {
            for (; ;) {
                line = r.readLine();
                if (line.length() == 0) break;
            }
            return proxy;
        } else if (sProxyAuthRequiredPattern.matcher(line).find()) {
            boolean authMethodSelected = false;
            String authMethod = AUTH_NONE;
            for (; ;) {
                line = r.readLine();
                if (line.length() == 0) break;
                if (line.startsWith("Proxy-Authenticate:") && !authMethodSelected) {
                    authMethod = line.substring(19).trim();
                    if (authMethod.equals(AUTH_BASIC)) {
                        authMethodSelected = true;
                    }
                }
            }
            // TODO: need to read full response before closing connection?
            proxy.close();

            if (authMethod.startsWith(AUTH_BASIC)) {
                return authenticateBasic(address);
            } else {
                throw new IOException("Unsupported authentication method: " + authMethod);
            }
        } else {
            proxy.close();
            throw new IOException("HTTP proxy does not support CONNECT command. Received reply: " + line);
        }
    }

    /**
     * Connects to the remote machine by establishing a tunnel through a HTTP proxy with Basic authentication.
     * It issues a CONNECT request and authenticates with the HTTP proxy with Basic protocol.
     *
     * @param address remote machine to connect to
     * @return a TCP/IP socket connected to the remote machine
     * @throws IOException     if an I/O error occurs during handshake (a network problem)
     */
    private Socket authenticateBasic(InetSocketAddress address) throws IOException {
        Socket proxy = new Socket(mSettings.getProxyHost(), mSettings.getProxyPort());
        BufferedReader r = new BufferedReader(new InputStreamReader(new InterruptibleInputStream(proxy.getInputStream())));
        DataOutputStream dos = new DataOutputStream(proxy.getOutputStream());

        String username = mSettings.getProxyUsername() == null ? "" : mSettings.getProxyUsername();
        String password = mSettings.getProxyPassword() == null ? "" : String.valueOf(mSettings.getProxyPassword());
        String credentials = username + ":" + password;
        String basicCookie = Base64Encoder.encode(credentials.getBytes("US-ASCII"));

        dos.writeBytes("CONNECT ");
        dos.writeBytes(address.getHostName() + ":" + address.getPort());
        dos.writeBytes(" HTTP/1.0\r\n");
        dos.writeBytes("Connection: Keep-Alive\r\n");
        dos.writeBytes("Proxy-Authorization: Basic " + basicCookie + "\r\n");
        dos.writeBytes("\r\n");
        dos.flush();

        String line = r.readLine();
        if (sConnectionEstablishedPattern.matcher(line).find()) {
            for (; ;) {
                line = r.readLine();
                if (line.length() == 0) break;
            }
            return proxy;
        }
        throw new IOException("Basic authentication failed: " + line);
    }

    private Socket getSocks4TunnelSocket(InetSocketAddress address, int timeout) throws IOException {
        boolean success = false;
        Socket proxy = new Socket();
        proxy.connect(new InetSocketAddress(mSettings.getProxyHost(), mSettings.getProxyPort()), timeout);
        try {
            DataInputStream din = new DataInputStream(new InterruptibleInputStream(proxy.getInputStream()));
            DataOutputStream dos = new DataOutputStream(proxy.getOutputStream());

            dos.writeByte(4);				// protocol
            dos.writeByte(1);				// connect command
            dos.writeShort(address.getPort());
            InetAddress addr = address.getAddress();
            if (addr == null) throw new UnknownHostException(address.getHostName());
            byte[] byteAddress = addr.getAddress();
            for (int i = 0; i < byteAddress.length; i++) {
                dos.writeByte(byteAddress[i]);
            }
            String uname = mSettings.getProxyUsername();
            if (uname != null) {
                byte[] unamebytes = uname.getBytes();
                for (int i = 0; i < unamebytes.length; i++) {
                    dos.writeByte(unamebytes[i]);
                }
            }
            dos.writeByte(0);

            int replyVersion = din.read();
            if (replyVersion != 0) throw new IOException("socks4.not.available." + replyVersion);
            int retCode = din.read();
            if (retCode != 90) throw new IOException("socks4.error." + retCode);
            while (din.available() > 0) din.read();
            success = true;
            return proxy;
        } finally {
            if (!success) proxy.close();
        }
    }

    private Socket getSocks5TunnelSocket(InetSocketAddress address, int timeout) throws IOException {
        boolean success = false;
        int tmp;

        Socket proxy = new Socket();
        proxy.connect(new InetSocketAddress(mSettings.getProxyHost(), mSettings.getProxyPort()), timeout);
        try {
            DataInputStream din = new DataInputStream(new InterruptibleInputStream(proxy.getInputStream()));
            DataOutputStream dos = new DataOutputStream(proxy.getOutputStream());

// protocol, # of supported auth methods, no auth, user/pass auth   
            dos.write(new byte[]{5, 2, 0, 2});

            int serverVersion = din.read();
            if (serverVersion != 5) throw new IOException("SOCKS5 protocol error: version: " + serverVersion);
            int authMethod = din.read();
            if (authMethod == 0xFF) throw new IOException("SOCKS5 authentication failure: no supported method acccepted by server");

            if (authMethod == 2) {
                dos.writeByte(1);		// negotiation version
                String uname = mSettings.getProxyUsername();
                byte[] unamebytes = (uname == null) ? new byte[]{} : uname.getBytes();
                dos.writeByte(unamebytes.length);
                for (int i = 0; i < unamebytes.length; i++) {
                    dos.writeByte(unamebytes[i]);
                }
                String pwd = null;
                if (mSettings.getProxyPassword() != null) {
                    pwd = new String(mSettings.getProxyPassword());
                }
                byte[] pwdbytes = (pwd == null) ? new byte[]{} : pwd.getBytes();
                dos.writeByte(pwdbytes.length);
                for (int i = 0; i < pwdbytes.length; i++) {
                    dos.writeByte(pwdbytes[i]);
                }

                tmp = din.read();
                if (tmp != 1) throw new IOException("socks5.auth.error." + tmp);
                tmp = din.read();
                if (tmp != 0) throw new IOException("socks5.auth.error." + tmp);
            }

            String hostName = address.getHostName();
// protocol, CONNECT, <reserved>, domain name follows, domain name length
            dos.write(new byte[]{5, 1, 0, 3, (byte) hostName.length()});
            dos.writeBytes(hostName);
            dos.writeShort(address.getPort());

            serverVersion = din.read();
            if (serverVersion != 5) throw new IOException("SOCKS5 protocol error: version: " + serverVersion);
            tmp = din.read();
            if (tmp != 0) throw new IOException("SOCKS5 protocol error: " + tmp);
            tmp = din.read();
            if (tmp != 0) throw new IOException("SOCKS5 protocol error: " + tmp);
            int addrType = din.read();
            if (addrType == -1) throw new IOException("SOCKS5 protocol error: " + addrType);
            // address
            for (int i = 0; i < 4; i++) {
                tmp = din.read();
                if (tmp == -1) throw new IOException("SOCKS5 error: " + tmp);
            }
            // port
            tmp = din.read();
            tmp = din.read();
            success = true;
            return proxy;
        } finally {
            if (!success) proxy.close();
        }
    }

    /**
     * Creates a new Socket connected to the given IP address. The method uses connection settings supplied
     * in the constructor for connecting the socket.
     *
     * @param address the IP address to connect to
     * @return connected socket
     * @throws java.net.UnknownHostException  if the hostname of the address or the proxy cannot be resolved
     * @throws java.io.IOException            if an I/O error occured while connecting to the remote end or to the proxy
     */
    private Socket createSocket(InetSocketAddress address, int timeout) throws UnknownHostException, IOException {
        String socksProxyHost = System.getProperty("socksProxyHost");
        System.getProperties().remove("socksProxyHost");
        try
        {
        switch (mSettings.getConnectionType()) {
        case ConnectivitySettings.CONNECTION_VIA_SOCKS:
            try {
                return getSocks5TunnelSocket(address, timeout);
            } catch (IOException e) {
                return getSocks4TunnelSocket(address, timeout);
            }

        case ConnectivitySettings.CONNECTION_DIRECT:
            Socket s = new Socket();
            s.connect(address, timeout);
            return s;

        case ConnectivitySettings.CONNECTION_VIA_HTTPS:
            return getHttpsTunnelSocket(address, timeout);

        default:
            throw new IOException("Illegal connection type: " + mSettings.getConnectionType());
        }
        } finally {
            if (socksProxyHost != null) {
                System.setProperty("socksProxyHost", socksProxyHost);
    }
        }
    }

}
