/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.search.types;

import java.util.regex.Pattern;
import java.util.regex.PatternSyntaxException;
import java.util.regex.Matcher;


/**
 * Abstract search type for searching in text.
 * The text can be searched either for a substring or for a regular expression
 * pattern. When a {@linkplain #setMatchString substring is set},
 * a previously {@linkplain #setRe set regular expression}, is cleared
 * and vice versa.
 *
 * @see  #match  match(String)
 * @author  Petr Kuzel
 * @author  Marian Petras
 */
public abstract class TextType extends DataObjectType {

    private static final long serialVersionUID = 3L;
    //private static final long serialVersionUID = -3260233821452714775L;

    /** set of word separator characters */
    private static final String WORD_SEPARATORS
            = " \t,;.:+-*/\\()[]{}<>=&|\"\'`~!?@#%^\n\r";               //NOI18N

    /** flag - be case-sensitive when searching? */
    protected boolean caseSensitive;
    /**
     * flag - match only whole words?
     *
     * @see  #setWholeWords
     */
    protected boolean wholeWords;

    /**
     * uppercased search pattern - substring.
     * It is used for case-insensitive searching.
     *
     * @see  #matchString
     */
    protected transient String ciMatchString;
    /**
     * search pattern - substring
     *
     * @see  #reString
     */
    protected String matchString;
    /**
     * search pattern - regular expression
     *
     * @see  #matchString
     */
    protected String reString;
    /**
     * compiled search pattern - regular expression
     *
     * @see  #reString
     */
    protected transient Pattern pattern;

    private transient Matcher matcher;

    /** reads a serialized form - used during deserialization */
    private void readObject(java.io.ObjectInputStream ois)
            throws java.io.IOException, ClassNotFoundException {
        ois.defaultReadObject();
        if (!caseSensitive && matchString != null) {
            ciMatchString = matchString.toUpperCase();
        }
        if (reString != null) {
            setRe(reString);
        }
    }

    // Match methods
    /**
     * Checks whether the specified string matches the specified pattern.
     * If a substring is set, the text is searched for the substring,
     * otherwise the text is searched for a regular expression.
     *
     * @param  text  text to check agains the pattern
     * @return  <code>true</code> if the text matches the pattern;
     *          <code>false</code> otherwise
     * @see  #setMatchString
     * @see  #setRe
     */
    protected boolean match(String text) {
        if (matchString != null) {
            if (!caseSensitive) {
                text = text.toUpperCase();
            }
            return matchString(text, 0) >= 0;
        } else {
            return matchRE(text);
        }
    }

    /**
     * Checks whether the specified string matches the specified regular
     * expression.
     *
     * @param  line  text to check agains the pattern
     * @return  <code>true</code> if the text matches the pattern;
     *          <code>false</code> otherwise
     * @see  #setRe
     */
    protected boolean matchRE(String line) {
        matcher = pattern.matcher(line);
        return matcher.find();
    }

    /** Returs matches that was used by last matchRE operation. */
    protected Matcher getMatcher() {
        return matcher;
    }

    /**
     * Searches part of text for a substring.
     *
     * @param  text  text to search
     *               - must be all in uppercase for case-insensitive search
     * @param  fromIndex  offset to start searching at
     *                    (first character &sim; <code>0</code>)
     * @return  position of the substring within the (whole) text, starting from
     *          <code>0</code>; or <code>-1</code> if the text
     *          does not contain the substring
     * @see  #setMatchString
     */
    protected int matchString(String text, int fromIndex) {
        int index = text.indexOf(caseSensitive ? matchString : ciMatchString,
                                 fromIndex);
        if (wholeWords && index >= 0) { // test "whole words only" condition
            if (index > 0 && WORD_SEPARATORS.indexOf(text.charAt(index-1)) < 0) {
                index = -1;
            } else {
                int matchLen = matchString.length();
                if (index + matchLen < text.length()
                        && WORD_SEPARATORS.indexOf(text.charAt(index + matchLen)) < 0) {
                    index = -1;
                }
            }
        }

        return index;
    }


    // Properties
    /** Getter for <code>matchString</code> property. */
    /**
     * Returns a substring to search for.
     *
     * @return  substring to search for; or an empty string if a regular
     *          expression will be used for searching
     * @see  #setMatchString  setMatchString(...)
     */
    public String getMatchString() {
        if (matchString == null) {
            return ""; // NOI18N
        } else {
            return matchString;
        }
    }

    /**
     * Sets a substring to search for.
     * Calling this method clears a previously set regular expression
     * (see {@link #setRe setRe(...)}), unless an exception is thrown.
     *
     * @param  substring  substring to search for
     * @exception  java.lang.IllegalArgumentException
     *             if the specified substring is <code>null</code>
     */
    public void setMatchString(String substring) {
        if (substring == null) {
            setValid(false);
            throw new IllegalArgumentException();
        }

        if (substring.length() == 0) {
            substring = null;
        } else if (!caseSensitive) {
            ciMatchString = substring.toUpperCase();
        }
        this.matchString = substring;
        pattern = null;
        reString = null;

        setValid(substring != null);
    }

    /**
     * Returns a regular expression that will be used for searching.
     *
     * @return  regular expression; or an empty string if text will be searched
     *          for a substring (not for a regular expression)
     * @see  #setRe  setRe(...)
     */
    public String getRe() {
        if (reString == null) {
            return "";                                                  //NOI18N
        } else {
            return reString;
        }
    }

    /**
     * Sets a regular expresion to use for searching.
     * Calling this method clears a previously set substring
     * (see {@link #setMatchString setMatchString(...)}), unless an exception
     * is thrown.
     *
     * @param  exp  regular expression to use
     * @exception  java.lang.IllegalArgumentException
     *             if the passed string was <code>null</code> or if it was not
     *             a regular expression accepted by class
     *             {@link java.util.regex.Pattern Pattern}
     */
    public void setRe(String re) {
        setReImpl(re);
    }

    /**
     * Sets a regular expresion to use for searching.
     *
     * @param  exp  regular expression to use
     * @exception  java.lang.IllegalArgumentException
     *             if the passed string was <code>null</code> or if it was not
     *             a regular expression accepted by class
     *             {@link java.util.regex.Pattern Pattern}
     */
    private void setReImpl(String exp) {
        if (exp == null) {
            setValid(false);
            throw new IllegalArgumentException();
        }

        if (exp.length() == 0) {
            exp = null;
        } else {
            try {
                pattern = Pattern.compile(exp);
            } catch (PatternSyntaxException ex) {
                setValid(false);
                throw new IllegalArgumentException();
            }
        }
        reString = exp;
        matchString = null;

        setValid(exp != null);
    }

    /**
     * Will searching be case-sensitive?
     *
     * @return  <code>true</code> if case-sensitive search will be used;
     *          <code>false</code> otherwise
     */
    public boolean isCaseSensitive() {
        return caseSensitive;
    }

    /**
     * Specifies whether case-sensitive or case-insensitive search should be
     * used.
     *
     * @param  caseSensitive  use <code>true</code> for case-sensitive,
     *                        <code>false</code> for case-insensitive
     */
    public void setCaseSensitive(boolean caseSensitive) {
        if (caseSensitive == this.caseSensitive) {
            return;
        }
        this.caseSensitive = caseSensitive;

        if (matchString != null) {
            ciMatchString = caseSensitive ? null : matchString.toUpperCase();
        }
    }

    /**
     * Are only whole words allowed in matching results?
     *
     * @return  <code>true</code> if only whole words are allowed;
     *          <code>false</code> if parts are allowed, too
     */
    public boolean getWholeWords() {
        return wholeWords;
    }

    /**
     * Specifies whether only whole words are allowed in matching results.
     *
     * @param wholeWords  use <code>true</code> if only whole words are allowed;
     *                    <code>false</code> if parts are allowed, too
     */
    public void setWholeWords(boolean wholeWords) {
        if (wholeWords == this.wholeWords) {
            return;
        }
        this.wholeWords = wholeWords;
    }

}
