/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */


package org.netbeans.modules.search;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import org.openide.loaders.DataObject;
import org.openide.nodes.Node;
import org.openide.util.HelpCtx;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;
import org.openide.util.actions.NodeAction;
import org.openidex.search.SearchInfo;
import org.openidex.search.SearchType;


/**
 *
 * @author  Petr Kuzel
 * @author  Marian Petras
 * @see org.openide.actions.FindAction
 */
public final class SearchPerformer extends NodeAction {
    
    /**
     */
    public HelpCtx getHelpCtx() {
        return HelpCtx.DEFAULT_HELP;        //PENDING
    }
    
    /**
     */
    public String getName() {
        return NbBundle.getMessage(SearchPerformer.class,
                                   "TEXT_FIND_ACTION_IMPL_NAME");       //NOI18N
    }

    /**
     * Collects a list of search types that are able to search the specified
     * nodes.
     *
     * @param  nodes  nodes to be searched
     * @return  list of <code>SearchType</code>s being able to search
     *          the specified nodes, or an empty list if <code>nodes</code>
     *          is <code>null</code> or empty; or if no applicable search type
     *          is found for the specified set of nodes
     * @see  Utils#getSearchTypes()
     * @see  SearchType#enabled
     */
    static List getTypes(Node[] nodes) {
        if (nodes == null || nodes.length == 0) {
            return Collections.EMPTY_LIST;
        }
        Iterator it = Utils.getSearchTypes().iterator();
        if (!it.hasNext()) {
            return Collections.EMPTY_LIST;
        }
        List result = new ArrayList(5);
        do {
            SearchType searchType = (SearchType) it.next();
            if (searchType.enabled(nodes) && !result.contains(searchType)) {
                result.add(searchType);
            }
        } while (it.hasNext());
        return result;
    }

    /**
     * Decides whether searching should be enabled with respect to a set
     * of selected nodes.
     * Searching is enabled if searching instructions
     * (<code>SearchInfo</code> object) are available for all selected nodes
     * and at least one registered search type is able to search all the
     * selected nodes.
     *
     * @param  nodes  selected nodes
     * @return  <code>true</code> if searching the selected nodes should be
     *          enabled; <code>false</code> otherwise
     * @see  SearchInfo
     * @see  SearchType
     */
    public boolean enable(Node[] nodes) {
        //System.out.println("enable(...)");
        //System.out.println("  - nodes count: " + nodes.length);
        if (nodes == null || nodes.length == 0) {
            return false;
        }

        for (int i = 0; i < nodes.length; i++) {
            if (!canSearch(nodes[i])) {
                return false;
            }
        }

        Iterator it = Utils.getSearchTypes().iterator();
        
        while (it.hasNext()) {
            SearchType searchType = (SearchType) it.next();

            if (searchType.enabled(nodes)) {
                return true;
            }
        }

        return false;
    }

    /**
     */
    private static boolean canSearch(Node node) {
        Lookup nodeLookup = node.getLookup();
        
        /* 1st try - is the SearchInfo object in the node's lookup? */
        SearchInfo searchInfo = (SearchInfo)
                                nodeLookup.lookup(SearchInfo.class);
        if (searchInfo != null) {
            return searchInfo.canSearch();
        }
    
        /* 2nd try - does the node represent a DataObject.Container? */
        return nodeLookup.lookup(DataObject.Container.class) != null;
    }
    
    /**
     */
    protected boolean asynchronous() {
        return false;
    }

    /**
     * Performs search on nodes. Displays <code>CriteriaView</code> with predefined search types set.
     *
     * @param nodes currently selected nodes
     * @see CriteriaView
     */
    public void performAction(Node[] nodes) {

        /* Get a list of applicable search types: */
        List searchTypeList = getTypes(nodes);
        if (searchTypeList.isEmpty()) {
            return;
        }
        
        /* Clone the list (deep copy): */
        List clonedSearchTypeList = new ArrayList(searchTypeList.size());
        for (Iterator it = searchTypeList.iterator(); it.hasNext(); ) {
            clonedSearchTypeList.add(((SearchType) it.next()).clone());
        }
        
        SearchPanel searchPanel = new SearchPanel(clonedSearchTypeList);
        
        /* a node may bear title of the Find dialog: */
        if (nodes.length == 1) {
            Object title = nodes[0].getValue(SearchPanel.PROP_DIALOG_TITLE);
            if (title != null && title instanceof String) {
                searchPanel.setTitle((String) title);
            }
        }

        searchPanel.showDialog();
        if (searchPanel.getReturnStatus() != SearchPanel.RET_OK) {
            return;
        }
        
        ResultView resultView = ResultView.getInstance();
        resultView.rememberInput(nodes,
                                 Utils.cloneSearchTypes(clonedSearchTypeList));
        resultView.open();
        resultView.requestActive();
        
        Manager.getInstance().scheduleSearchTask(
                new SearchTask(nodes,
                               clonedSearchTypeList,
                               searchPanel.getCustomizedSearchTypes()));
    }
    
}
