/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.nbbuild;

import java.io.*;
import java.util.*;
import java.util.jar.*;
import java.util.zip.*;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.*;
import org.apache.tools.ant.types.*;
import org.apache.tools.ant.DirectoryScanner;
import org.apache.tools.ant.BuildException;

/**
 * This task builds <l10n> NBMs
 * @author  Rudolf Balada
 */
 
public class BuildLocalizedNBMs extends Task {
    private File nbSourceZipFile = null;
    private File nbBuildDir = null;
    private File tmpDir = null;
    private File nbTargetDir = null;
    private ModuleTracking mt = null;
    
    public void setTmpDir (String tmpd) throws BuildException {
        File testNbzf1 = new File(this.getProject().getBaseDir(), tmpd);
        File testNbzf2 = new File(tmpd);
        if ((testNbzf1.exists()) && (testNbzf1.isDirectory())) {
            this.tmpDir = testNbzf1;
        } else if ((testNbzf2.exists()) && (testNbzf2.isDirectory())) {
            this.tmpDir = testNbzf2;
        } else {
            if (testNbzf1.mkdir()) {
                this.tmpDir = testNbzf1;
            } else {
                throw new BuildException("Unable to find temporary directory "+tmpd+" for unzipping NetBeans build.", this.getLocation());
            }
        }
    }
    
    public void setBuildDir (String bd) throws BuildException {
        File testNbzf1 = new File(this.getProject().getBaseDir(), bd);
        File testNbzf2 = new File(bd);
        if ((testNbzf1.exists()) && (testNbzf1.isDirectory())) {
            this.nbBuildDir = testNbzf1;
        } else if ((testNbzf2.exists()) && (testNbzf2.isDirectory())) {
            this.nbBuildDir = testNbzf2;
        } else {
            if (testNbzf1.mkdir()) {
                this.nbBuildDir = testNbzf1;
            } else {
                throw new BuildException("Unable to find directory'"+bd+"' with unpacked NetBeans build.", this.getLocation());
            }
        }
    }
    
    /** 
     * Set the source zipfile with NetBeans build to be checked
     */
    public void setSourceZipFile (String nbzf) throws BuildException {
        File testNbzf1 = new File(this.getProject().getBaseDir(), nbzf);
        File testNbzf2 = new File(nbzf);
        if ((testNbzf1.exists()) && (testNbzf1.isFile())) {
            this.nbSourceZipFile = testNbzf1;
        } else if ((testNbzf2.exists()) && (testNbzf2.isFile())) {
            this.nbSourceZipFile = testNbzf2;
        } else {
            throw new BuildException("Unable to find sourcezipfile with NetBeans build referred by \""+nbzf+"\" string.", this.getLocation());
        }
    }
    
    /** 
     * Set the target zipfile with NetBeans build to be produced while checking
     */
    public void setTargetDir (String nbtd) throws BuildException {
        File absolute = new File(nbtd);
        File relative = new File(this.getProject().getBaseDir(), nbtd);
        if( absolute.isAbsolute() )
            nbTargetDir = absolute;
        else
            nbTargetDir = relative;
    }

    public void execute() throws BuildException {
    	// check inputs
        if ((nbSourceZipFile != null) && (nbBuildDir != null)) {
            throw new BuildException("Source zipfile and build dir are mutually exclusive attributes. Provide only one of them.", this.getLocation());
        } else if ((nbSourceZipFile == null) && (nbBuildDir == null)) {
            throw new BuildException("Set one of 'sourcezipfile' or 'builddir' attributes to define input.", this.getLocation());
        }
        // define temp dir
        if (tmpDir == null) {
            try {
                tmpDir = java.io.File.createTempFile("tmpdir_", "_dir", getProject().getBaseDir()); //NOI18N
                tmpDir.mkdirs(); tmpDir.mkdir();
            } catch (java.io.IOException ioe) {
                throw new BuildException ("I/O Error while establishing temporary directory",ioe, this.getLocation());
            }
        }
        // unpack
        if (nbBuildDir == null) {
            log("Unpacking source zipfile '"+nbSourceZipFile.getAbsolutePath()+"'",Project.MSG_VERBOSE);
            Expand unzip = (Expand) getProject().createTask("unzip"); //NOI18N
            unzip.setSrc(nbSourceZipFile);
            nbBuildDir = new File(tmpDir, "build");
            nbBuildDir.mkdirs();
            nbBuildDir.mkdir();
            unzip.setDest(nbBuildDir);
            unzip.execute();
            nbBuildDir = new File(tmpDir, "build"+File.separator+"netbeans");
            if (!(nbBuildDir.exists())) {
            	throw new BuildException("Directory '"+nbBuildDir.getAbsolutePath()+"' does not exist. Supplied source zipfile '"+nbSourceZipFile.getAbsolutePath()+"' probably does not contain root 'netbeans' directory.", this.getLocation());
            }
        } else if (!(nbBuildDir.getName().equals("netbeans"))) {
            throw new BuildException("Supplied NetBeans build directory name '"+nbBuildDir.getAbsolutePath()+"' does not contain 'netbeans' substring.", this.getLocation());
        }
        this.getProject().addTaskDefinition("locmakenbm", LocMakeNBM.class); //NOI18N
        log("Loading module tracking data from directory '"+nbBuildDir.getAbsolutePath()+"'", Project.MSG_VERBOSE);
        mt = new ModuleTracking( nbBuildDir.getAbsolutePath() );
        Map/*<String,ModuleTracking.Module>*/ htModules = mt.getModules();
        Iterator enMods = htModules.values().iterator();
        log("Iterating through modules", Project.MSG_VERBOSE);
        while (enMods.hasNext()) {
            ModuleTracking.Module mod = (ModuleTracking.Module) enMods.next();
            buildOneModuleNBM(mod);
        }
        Delete deltask = (Delete) this.getProject().createTask("delete");
        deltask.setDir(tmpDir);
        //deltask.execute();
        
    }    
    
    private void buildOneModuleNBM(ModuleTracking.Module mod) throws BuildException {
        log("Building localized NBM(s) for module '"+mod.getCodeName()+"' ("+mod.getName()+")", Project.MSG_VERBOSE);
        // copy files to temporary place
        File moduleDir = new File(tmpDir, "module"+File.separator+"netbeans");
        
        // clean up before copying
        Delete deltask = (Delete) this.getProject().createTask("delete");
        deltask.setDir(moduleDir);
        deltask.execute();
        
        // make the directory again
        moduleDir.mkdirs(); moduleDir.mkdir();

        // copy module files to working directory
        log("Copying module files for module '"+mod.getName()+"'", Project.MSG_VERBOSE);
        List files = mod.getFiles();
        Iterator it = files.iterator();
        String fname;
        int slashPos;
        boolean failOnError;
        Copy copytask = (Copy) this.getProject().createTask("copy");
        while (it.hasNext()) {
            
            fname = (String) it.next();
            log("copying file '"+fname+"'", Project.MSG_VERBOSE);
            slashPos = fname.lastIndexOf("/");
            if (fname.lastIndexOf("config/Modules")<0) {
                failOnError = true;
                log("failonerror=true", Project.MSG_VERBOSE);
            } else {
                failOnError = false;
                log("failonerror=false", Project.MSG_VERBOSE);
            }
            copytask.setFailOnError(failOnError);
            copytask.setFile(new File(nbBuildDir,mod.getPath()+File.separator+fname.replace('/', File.separatorChar)));
            copytask.setTofile(new File(moduleDir, fname.replace('/', File.separatorChar)));
            copytask.execute();
        }
        
        // find module jarfile
        log("looking for module jarfile '"+mod.getName()+"'", Project.MSG_VERBOSE);
        FileSet fs = new FileSet();
//        fs.setDir(new File(nbBuildDir,mod.getPath()));
        fs.setDir(moduleDir);
        fs.setIncludes("**/*.jar"); //NOI18N
        DirectoryScanner ds = fs.getDirectoryScanner(this.getProject());
        String[] jarfiles = ds.getIncludedFiles();
        String moduleJar = "";
        String moduleName = null;
        String moduleLocBundle = "";
        String moduleSpecVer = null;
        for (int i=0; i< jarfiles.length; i++) {
            log("Examining file '"+jarfiles[i]+"'", Project.MSG_VERBOSE);
            // open manifest and look for OpenIDE-Module attribute
            File f = new File(nbBuildDir, mod.getPath()+File.separator+jarfiles[i].replace('/', File.separatorChar));
            JarFile jf;
            try {
                jf = new JarFile(f);
            } catch (java.io.IOException ioe) {
                throw new BuildException("I/O Error when accessing jarfile '"+f.getAbsolutePath()+"'",ioe,this.getLocation());
            }
            java.util.jar.Manifest mani;
            try {
                mani = jf.getManifest();
            } catch (java.io.IOException ioe) {
                throw new BuildException("I/O Error when accessing manifest in jarfile '"+f.getAbsolutePath()+"'",ioe,this.getLocation());
            }
            if (mani == null) { continue; }
            
            java.util.jar.Attributes attr = mani.getMainAttributes ();
            if ((attr.containsKey ("OpenIDE-Module")) || (attr.containsValue(mod.getCodeName()))) {
                // it's NetBeaans module jarfile
                moduleJar = f.getName();
                moduleSpecVer = (String) attr.getValue("OpenIDE-Module-Specification-Version");
                log("Found module specification version '"+moduleSpecVer+"'", Project.MSG_VERBOSE);

//                moduleLocBundle = (String) attr.getValue("OpenIDE-Module-Localizing-Bundle");
                
//                log("Found localizing Bundle '"+moduleLocBundle+"'", Project.MSG_VERBOSE);
                break;
            } else {
                log("Jarfile '"+f.getAbsolutePath()+"' is not NetBeans module jar ("+mod.getCodeName()+")", Project.MSG_VERBOSE);
            }
        }
        if (moduleName == null) {
            moduleName = mod.getCodeName();
        }
        if (moduleSpecVer == null) {
            throw new BuildException("Unable to determine module specification version for '"+moduleName+"'", this.getLocation());
        }
        log("Setting up <locmakenbm> task", Project.MSG_VERBOSE);
        LocMakeNBM lmntask = (LocMakeNBM) this.getProject().createTask("locmakenbm");
        String nbmFileName = mod.getNbmFileName();
//        if ((nbmFileName == null) || (nbmFileName.equals(""))) {
            nbmFileName = moduleJar.substring(0,moduleJar.length()-4) + ".nbm";
//        }
        log("NBM file name '"+nbmFileName+"'", Project.MSG_VERBOSE);
        lmntask.setFile(nbmFileName);
        log("Module dir '"+moduleDir.getAbsolutePath()+"'", Project.MSG_VERBOSE);
        lmntask.setTopDir(moduleDir);
//        lmntask.setLocBundle(new File(moduleDir))
//        int sPos = moduleName.lastIndexOf('/');
//        lmntask.setModInfo(moduleName+'/'+moduleSpecVer);
        String modInfo = mod.getCodeName()+'/'+moduleSpecVer;
        log("Module info '"+modInfo+"'", Project.MSG_VERBOSE);
        lmntask.setModInfo(modInfo);
        
        lmntask.execute();
        log("Module '"+moduleName+"' is done", Project.MSG_VERBOSE);

    }
}
