/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.tasklist.usertasks;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;

import javax.swing.tree.TreePath;

import org.netbeans.modules.tasklist.core.filter.Filter;
import org.netbeans.modules.tasklist.usertasks.treetable.AdvancedTreeTableNode;
import org.netbeans.modules.tasklist.usertasks.treetable.BooleanComparator;
import org.netbeans.modules.tasklist.usertasks.treetable.DefaultMutableTreeTableNode;
import org.netbeans.modules.tasklist.usertasks.treetable.DefaultTreeTableModel;
import org.netbeans.modules.tasklist.usertasks.treetable.FilterIntf;
import org.netbeans.modules.tasklist.usertasks.treetable.NotComparator;
import org.netbeans.modules.tasklist.usertasks.treetable.SortingModel;
import org.netbeans.modules.tasklist.usertasks.treetable.StringIgnoreCaseComparator;
import org.openide.text.Line;
import org.netbeans.modules.tasklist.usertasks.model.UserTask;
import org.netbeans.modules.tasklist.usertasks.model.UserTaskList;

/**
 * TT model for user tasks
 */
public class UserTasksTreeTableModel extends DefaultTreeTableModel {
    public static final int SUMMARY = 0;
    public static final int PRIORITY = 1;
    public static final int DONE = 2;
    public static final int PERCENT_COMPLETE = 3;
    public static final int EFFORT = 4;
    public static final int REMAINING_EFFORT = 5;
    public static final int SPENT_TIME = 6;
    public static final int DETAILS = 7;
    public static final int FILE_BASE_NAME = 8;
    public static final int LINE_NUMBER = 9;
    public static final int CATEGORY = 10;
    public static final int CREATED = 11;
    public static final int LAST_EDITED = 12;
    public static final int COMPLETED_DATE = 13;
    public static final int DUE_DATE = 14;
    public static final int OWNER = 15;
    public static final int START = 16;
    public static final int SPENT_TIME_TODAY = 17;

    /**
     * Compares lines of 2 errors
     */
    public static class LinesComparator implements Comparator {
        public int compare(Object o1, Object o2) {
            Line f1 = (Line) o1;
            Line f2 = (Line) o2;
            if (f1 == null && f2 == null)
                return 0;
            if (f1 == null)
                return -1;
            if (f2 == null)
                return 1;
            return f1.getLineNumber() - f2.getLineNumber();
        }
    }
    
    /**
     * Comparator for the "effort" column
     */
    private static class EffortComparator implements Comparator {
        public int compare(Object o1, Object o2) {
            if (o1 == null && o2 == null)
                return 0;
            if (o1 == null)
                return -1;
            if (o2 == null)
                return 1;
            UserTask ut1 = (UserTask) o1;
            UserTask ut2 = (UserTask) o2;
            return ut1.getEffort() - ut2.getEffort();
        }
    }

    /**
     * Comparator for the "category" column
     */
    private static class CategoryComparator implements Comparator {
        public int compare(Object o1, Object o2) {
            if (o1 == null && o2 == null)
                return 0;
            if (o1 == null)
                return -1;
            if (o2 == null)
                return 1;
            UserTask ut1 = (UserTask) o1;
            UserTask ut2 = (UserTask) o2;
            return ut1.getCategory().compareTo(ut2.getCategory());
        }
    }

    /**
     * Comparator for the "owner" column
     */
    private static class OwnerComparator implements Comparator {
        public int compare(Object o1, Object o2) {
            if (o1 == null && o2 == null)
                return 0;
            if (o1 == null)
                return -1;
            if (o2 == null)
                return 1;
            UserTask ut1 = (UserTask) o1;
            UserTask ut2 = (UserTask) o2;
            return ut1.getOwner().compareTo(ut2.getOwner());
        }
    }

    
    public static final String[] COLUMN_PROPERTIES = {
        "summary", // NOI18N
        "priority", // NOI18N
        "done", // NOI18N
        "percentComplete", // NOI18N
        "effort", // NOI18N
        "remainingEffort", // NOI18N
        "spentTime", // NOI18N
        "details", // NOI18N
        "file", // NOI18N
        "line", // NOI18N
        "category", // NOI18N
        "created", // NOI18N
        "edited", // NOI18N
        "completedDate", // NOI18N
        "due", // NOI18N
        "owner", // NOI18N
        "start", // NOI18N
        "spentTimeToday" // NOI18N
    };
    
    private static final Comparator[] COMPARATORS = {
        new StringIgnoreCaseComparator(),
        new PriorityComparator(),
        new BooleanComparator(), 
        SortingModel.DEFAULT_COMPARATOR, 
        new EffortComparator(), 
        SortingModel.DEFAULT_COMPARATOR, 
        SortingModel.DEFAULT_COMPARATOR, 
        SortingModel.DEFAULT_COMPARATOR, 
        SortingModel.DEFAULT_COMPARATOR, 
        SortingModel.DEFAULT_COMPARATOR, 
        new CategoryComparator(),
        SortingModel.DEFAULT_COMPARATOR, 
        SortingModel.DEFAULT_COMPARATOR,
        SortingModel.DEFAULT_COMPARATOR,
        SortingModel.DEFAULT_COMPARATOR,
        new OwnerComparator(),
        SortingModel.DEFAULT_COMPARATOR,
        SortingModel.DEFAULT_COMPARATOR
    };
    
    private static final Class[] COLUMN_CLASS = {
        null, 
        Integer.class,
        Boolean.class,
        Integer.class,
        UserTask.class,
        Integer.class,
        Integer.class,
        String.class,
        String.class,
        Integer.class,
        UserTask.class,
        String.class,
        Long.class,
        Long.class,
        Long.class,
        String.class,
        Date.class,
        Integer.class
    };
    
    private static final String[] COLUMNS = {
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnSummary"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnPriority"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnDone"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnPercentComplete"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnEffort"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnRemEffort"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnSpentTime"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnDetails"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnFile"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnLine"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnCategory"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnCreated"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnEdited"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnCompletedDate"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnDue"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnOwner"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnStart"), // NOI18N
        org.openide.util.NbBundle.getMessage(UserTasksTreeTableModel.class, "ColumnSpentTimeToday") // NOI18N
    };
            
    private UserTaskList utl;
    
    /**
     * Creates a new instance of UserTasksTreeTableModel
     *
     * @param root root node
     * @param sm a sorting model
     * @param filter a filter
     */
    public UserTasksTreeTableModel(UserTaskList utl, SortingModel sm, 
    final Filter filter) {
        super(null, COLUMNS);
        this.utl = utl;
        for (int i = 0; i < COMPARATORS.length; i++) {
            sm.setColumnComparator(i, COMPARATORS[i]);
        }
        
        FilterIntf fi = null;
        if (filter != null) {
            fi = new FilterIntf() {
                public boolean accept(Object obj) {
                    if (obj instanceof UserTask) {
                        return filter.accept((UserTask) obj);
                    } else {
                        return true;
                    }                
                }
            };
        }
        this.root = new UserTaskListTreeTableNode(fi, this, utl, null);
        //this.root = new UTListDependenciesTreeTableNode(fi, this, utl, null);
    }
    
    /**
     * Should be called after this model was remove from a TreeTable.
     * Destroys all nodes
     */
    public void destroy() {
        ((AdvancedTreeTableNode) root).destroy();
    }
    
    /**
     * Returns the user task list associated with this model
     * 
     * @return user task list
     */
    public UserTaskList getUserTaskList() {
        return utl;
    }
    
    /**
     * Finds the path to the specified user task
     *
     * @param a user task from this model
     * @return path to the task or null
     */
    public TreePath findPathTo(UserTask ut) {
        List path = new ArrayList();
        while (ut != null) {
            path.add(0, ut);
            ut = (UserTask) ut.getParent();
        }
        List retp = new ArrayList();
        retp.add(getRoot());
        
        DefaultMutableTreeTableNode n = (DefaultMutableTreeTableNode) getRoot();
        for(int i = 0; i < path.size(); i++) {
            Object obj = path.get(i);
            boolean found = false;
            for (int j = 0; j < n.getChildCount(); j++) {
                if (((DefaultMutableTreeTableNode) n.getChildAt(j)).
                    getUserObject() == obj) {
                    found = true;
                    retp.add(n);
                    break;
                }
            }
            if (!found)
                return null;
        }
        
        return new TreePath(retp.toArray());
    }
    
    public Class getColumnClass(int column) {
        if (column == 0)
            return super.getColumnClass(0);
        return COLUMN_CLASS[column];
    }
    
    /**
     * Sorts the tree according to the specified sorting model
     */
    public void sort(SortingModel sm) {
        final int sortedColumn = sm.getSortedColumn();
        if (sortedColumn == -1) {
            ((UserTaskListTreeTableNode) getRoot()).setComparator(null);
            return;
        }
        
        Comparator c = COMPARATORS[sortedColumn];
        if (c == null)
            return;
        
        final Comparator c2;
        if (!sm.isSortOrderDescending())
            c2 = new NotComparator(c);
        else
            c2 = c;
        
        Comparator comparator = new Comparator() {
            public int compare(Object obj1, Object obj2) {
                UserTaskTreeTableNode n1 = (UserTaskTreeTableNode) obj1;
                UserTaskTreeTableNode n2 = (UserTaskTreeTableNode) obj2;
                return c2.compare(
                    n1.getValueAt(sortedColumn), n2.getValueAt(sortedColumn));
            }
        };

        ((UserTaskListTreeTableNode) getRoot()).setComparator(comparator);
    }
    
    public void fireTreeNodesChanged(Object source, Object[] path, 
    int[] childIndices, Object[] children) {
        UTUtils.LOGGER.fine("path=" + Arrays.asList(path).toString() + // NOI18N
            "children=" + Arrays.asList(children).toString()); // NOI18N
        super.fireTreeNodesChanged(source, path, childIndices, children);
    }
    
    public void fireTreeNodesInserted(Object source, Object[] path, 
    int[] childIndices, Object[] children) {
        UTUtils.LOGGER.fine("path=" + Arrays.asList(path).toString() + // NOI18N
            "children=" + Arrays.asList(children).toString()); // NOI18N
        super.fireTreeNodesInserted(source, path, childIndices, children);
    }
    
    public void fireTreeNodesRemoved(Object source, Object[] path, 
    int[] childIndices, Object[] children) {
        UTUtils.LOGGER.fine("path=" + Arrays.asList(path).toString() + // NOI18N
            "children=" + Arrays.asList(children).toString()); // NOI18N
        super.fireTreeNodesRemoved(source, path, childIndices, children);
    }
    
    public void fireTreeStructureChanged(Object source, Object[] path) {
        UTUtils.LOGGER.fine("path=" + Arrays.asList(path).toString()); // NOI18N
        if (UTUtils.LOGGER.isLoggable(Level.FINER))
            Thread.dumpStack();
        super.fireTreeStructureChanged(source, path);
    }
    
    public void fireTreeStructureChanged(Object source, Object[] path, 
    int[] childIndices, Object[] children) {
        UTUtils.LOGGER.fine("path=" + Arrays.asList(path).toString() +  // NOI18N
            "children=" + Arrays.asList(children).toString()); // NOI18N
        super.fireTreeStructureChanged(source, path, childIndices, children);
    }
}
