/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.tasklist.usertasks;

import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.UnsupportedFlavorException;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;

import org.openide.ErrorManager;
import org.openide.util.NbBundle;
import org.openide.util.datatransfer.ExClipboard;
import org.openide.util.datatransfer.ExTransferable;
import org.openide.util.datatransfer.MultiTransferObject;
import org.netbeans.modules.tasklist.usertasks.model.UserTask;

/** 
 * Utilities dealing with data transfer operations on todo items.
 *
 * @author Tor Norbye
 */
public final class UserTaskTransfer implements ExClipboard.Convertor {
    /** Flavor for tasks on the clipboard */    
    public static final DataFlavor TODO_FLAVOR = new DataFlavor(
        UserTask.class, 
        NbBundle.getMessage(UserTaskTransfer.class, "LBL_todo_flavor")); // NOI18N
    
    /** Construct a task transfer object */    
    public UserTaskTransfer() {}
    
    /** Convert a transferable.
     * If it has just a text selection, make a corresponding task
     * by parsing it.
     * If just a task, make a corresponding text selection by writing it out.
     * If it has a multiple selection all of which are todo items, make a text
     * selection with all of them concatenated (and leave the todoitems there too
     * obviously).
     * Otherwise leave it alone.
     * @param t The transferable to convert
     * @return The converted transferable */
    public Transferable convert(final Transferable t) {
        boolean supportsString = t.isDataFlavorSupported(DataFlavor.stringFlavor);
        boolean supportsTodo = t.isDataFlavorSupported(TODO_FLAVOR);
        if (supportsString && !supportsTodo) {
	    // Return a new TodoItem from a string
            ExTransferable t2 = ExTransferable.create(t);
            t2.put(new ExTransferable.Single(TODO_FLAVOR) {
                protected Object getData() throws IOException, UnsupportedFlavorException {
                    String text = (String)t.getTransferData(DataFlavor.stringFlavor);
		    return UserTask.parse(new StringReader(text));
                }
            });
            return t2;
        } else if (!supportsString && supportsTodo) {
	    // Return a new string from a todo item
            ExTransferable t2 = ExTransferable.create(t);
            t2.put(new ExTransferable.Single(DataFlavor.stringFlavor) {
                protected Object getData() throws IOException, UnsupportedFlavorException {
                    UserTask item = (UserTask)t.getTransferData(TODO_FLAVOR);
                    StringWriter wr = new StringWriter();
		    UserTask.generate(item, wr);
                    return wr.toString();
                }
            });
            return t2;
        } else if (t.isDataFlavorSupported(ExTransferable.multiFlavor)) {
            UTUtils.LOGGER.fine("multi selection"); // NOI18N
	    // Multiselection
            try {
                final MultiTransferObject mto = (MultiTransferObject)
                    t.getTransferData(ExTransferable.multiFlavor);
                boolean allSupportTodo = mto.areDataFlavorsSupported(
                    new DataFlavor[] {TODO_FLAVOR});
                if (allSupportTodo) {
                    UTUtils.LOGGER.fine("multi selection all supports todo"); // NOI18N
                    ExTransferable t2 = ExTransferable.create(t);
                    if (!supportsString) {
			// Create string representation
                        t2.put(new ExTransferable.Single(DataFlavor.stringFlavor) {
                            protected Object getData() throws IOException, UnsupportedFlavorException {
                                StringWriter wr = new StringWriter();
                                for (int i = 0; i < mto.getCount(); i++) {
                                    UserTask item =	
                                        (UserTask)mto.getTransferData(i, TODO_FLAVOR);
				    UserTask.generate(item, wr);
                                }
                                return wr.toString();
                            }
                        });
                    }
                    return t2;
                } // else: not all support todoitems - so don't do anything
            } catch (Exception e) {
                // Should not happen: IOException, UnsupportedFlavorException
                ErrorManager.getDefault().notify(ErrorManager.INFORMATIONAL, e);
            }
        }
        return t;
    }

}
