/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.tasklist.core.filter;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;
import javax.swing.*;

import org.netbeans.modules.tasklist.core.TaskListView;
import org.netbeans.modules.tasklist.core.TaskNode;
import org.openide.awt.Mnemonics;
import org.openide.explorer.view.BeanTreeView;
import org.openide.nodes.FilterNode;
import org.openide.nodes.Node;
import org.openide.util.NbBundle;

/**
 * Panel which is used to create a filter for the tasklist
 * The GUI is based on the one in Mozilla's mail tool.
 * Please read comment at the beginning of initA11y before editing
 * this file using the form builder.
 *
 * @author Tor Norbye
 */
public final class FilterPanel extends JPanel implements java.awt.event.ActionListener, java.beans.PropertyChangeListener, java.awt.event.FocusListener {

    private static final long serialVersionUID = 1;

    private FilteredTopComponent view;
    private Filter filter;

    // head of subpanels chain
    private FilterSubpanel subpanel;

    /**
     * Creates new form FilterPanel. 
     *
     * @param view view with the given filter
     * @param filter filter to be edited. Can be null and in that case
     *               all fields are disabled.
     */
    public FilterPanel(FilteredTopComponent view, Filter filter) {
        this.view = view;
        this.filter = filter;

        initComponents();
        initA11y();

        // it's not generated by form editor
        JPanel topAlign = new JPanel();
        topAlign.setLayout(new BorderLayout());
        topAlign.add(conditionsPanel, BorderLayout.NORTH);
        conditionsScrollPane.setViewportView(topAlign);
        // compute 80x10 chars space in scroll pane
        FontMetrics fm = getFontMetrics(getFont());
        int width = fm.charWidth('n') * 80;  // NOI18N
        int height = fm.getHeight() * 10;
        conditionsScrollPane.setPreferredSize(new java.awt.Dimension(width, height));

        Color background = (Color)UIManager.get("Table.background"); //NOI18N
        conditionsPanel.setBackground(background);
        topAlign.setBackground(background);

        moreButton.addActionListener(this);
        fewerButton.addActionListener(this);
        matchAllRadio.addActionListener(this);
        matchAnyRadio.addActionListener(this);

        showFilter(filter);
        updateSensitivity();
    }

    /**
     * Links to next filter customizer (or null).
     */
    public void initSubpanel(FilterSubpanel next) {
        assert subpanel == null;
        subpanel = next;
    }

    /**
     * Show the given filter in the GUI
     */
    private void showFilter(Filter filter) {
        if (filter != null) {
            if (filter.matchAll()) {
                matchAllRadio.setSelected(true);
            } else {
                matchAnyRadio.setSelected(true);
            }
            conditionsPanel.removeAll();
            List conditions = filter.getConditions();
            if (conditions.size() > 0) {
                Iterator it = conditions.iterator();
                while (it.hasNext()) {
                    AppliedFilterCondition cond = (AppliedFilterCondition) it.next();
                    addCondition(false, cond);
                }
            } else {
                addCondition(false, null);
            }
            filterName.setText(filter.getName());
        } else {
            this.setEnabled(false);
            namePanel.setEnabled(false);
            filterName.setEnabled(false);
            filterName.setText(NbBundle.getMessage(FilterPanel.class, "no-filter"));
            matchAnyRadio.setEnabled(false);
            matchAllRadio.setEnabled(false);
            conditionsPanel.setEnabled(false);
            moreButton.setEnabled(false);
            fewerButton.setEnabled(false);
        }
    }

    /**
     * This method is called from within the constructor to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        java.awt.GridBagConstraints gridBagConstraints;

        setLayout(new java.awt.GridBagLayout());

        addFocusListener(this);

        namePanel.setLayout(new java.awt.GridBagLayout());

        filterNameLabel.setLabelFor(filterName);
        /*
        org.openide.awt.Mnemonics.setLocalizedText(filterNameLabel, java.util.ResourceBundle.getBundle("org/netbeans/modules/tasklist/core/filter/Bundle").getString("LBL_Filter_Name"));
        */
        filterNameLabel.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/tasklist/core/filter/Bundle").getString("LBL_Filter_Name_Hint"));
        filterNameLabel.setAlignmentY(0.0F);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 0, 5);
        namePanel.add(filterNameLabel, gridBagConstraints);
        filterNameLabel.getAccessibleContext().setAccessibleName("<Name>");

        filterName.setText("jTextField1");
        filterName.setToolTipText(java.util.ResourceBundle.getBundle("org/netbeans/modules/tasklist/core/filter/Bundle").getString("LBL_Filter_Name_Hint"));
        filterName.setAlignmentX(0.0F);
        filterName.setAlignmentY(0.0F);
        filterName.setPreferredSize(new java.awt.Dimension(32767, 20));
        filterName.addFocusListener(this);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.ipadx = 3;
        gridBagConstraints.ipady = 5;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(5, 0, 5, 0);
        namePanel.add(filterName, gridBagConstraints);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 3;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(0, 0, 10, 0);
        add(namePanel, gridBagConstraints);

        matchGroup.add(matchAllRadio);
        /*
        org.openide.awt.Mnemonics.setLocalizedText(matchAllRadio, NbBundle.getMessage(FilterPanel.class, "MatchAll")); // NOI18N();
        */
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        add(matchAllRadio, gridBagConstraints);

        matchGroup.add(matchAnyRadio);
        matchAnyRadio.setSelected(true);
        /*
        org.openide.awt.Mnemonics.setLocalizedText(matchAnyRadio, NbBundle.getMessage(FilterPanel.class, "MatchAny")); // NOI18N();
        */
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        add(matchAnyRadio, gridBagConstraints);

        conditionsScrollPane.setVerticalScrollBarPolicy(javax.swing.ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);
        conditionsScrollPane.setAlignmentX(0.0F);
        conditionsScrollPane.setAlignmentY(0.0F);
        conditionsScrollPane.setPreferredSize(new java.awt.Dimension(32767, 32767));
        conditionsPanel.setLayout(new java.awt.GridBagLayout());

        conditionsScrollPane.setViewportView(conditionsPanel);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = java.awt.GridBagConstraints.BOTH;
        gridBagConstraints.weightx = 1.0;
        gridBagConstraints.weighty = 1.0;
        gridBagConstraints.insets = new java.awt.Insets(12, 0, 12, 0);
        add(conditionsScrollPane, gridBagConstraints);
        conditionsScrollPane.getAccessibleContext().setAccessibleName(org.openide.util.NbBundle.getBundle(FilterPanel.class).getString("LBL_Conditions"));
        conditionsScrollPane.getAccessibleContext().setAccessibleDescription(org.openide.util.NbBundle.getBundle(FilterPanel.class).getString("HNT_Conditions"));

        buttonsPanel.setLayout(new java.awt.FlowLayout(java.awt.FlowLayout.LEFT));

        /*
        org.openide.awt.Mnemonics.setLocalizedText(moreButton, NbBundle.getMessage(FilterPanel.class, "More")); // NOI18N();
        */
        buttonsPanel.add(moreButton);

        /*
        org.openide.awt.Mnemonics.setLocalizedText(fewerButton, NbBundle.getMessage(FilterPanel.class, "Fewer")); // NOI18N();
        */
        buttonsPanel.add(fewerButton);

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.gridwidth = java.awt.GridBagConstraints.REMAINDER;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        add(buttonsPanel, gridBagConstraints);

    }

    // Code for dispatching events from components to event handlers.

    public void focusGained(java.awt.event.FocusEvent evt) {
        if (evt.getSource() == FilterPanel.this) {
            FilterPanel.this.formFocusGained(evt);
        }
        else if (evt.getSource() == filterName) {
            FilterPanel.this.filterNameFocusGained(evt);
        }
    }

    public void focusLost(java.awt.event.FocusEvent evt) {
    }// </editor-fold>//GEN-END:initComponents

    private void filterNameFocusGained(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_filterNameFocusGained
        filterName.selectAll();        
    }//GEN-LAST:event_filterNameFocusGained

    private void formFocusGained(java.awt.event.FocusEvent evt) {//GEN-FIRST:event_formFocusGained
        // TODO add your handling code here:
        filterName.requestFocus();
    }//GEN-LAST:event_formFocusGained

    private void initA11y() {
        /*
          I couldn't figure out how to use Mnemonics.setLocalizedText
          to set labels and checkboxes with a mnemonic using the
          form builder, so the closest I got was to use "/*" and "* /
          as code pre-init/post-init blocks, such that I don't actually
          execute the bundle lookup code - and then call it explicitly
          below. (I wanted to keep the text on the components so that
          I can see them when visually editing the GUI.
        */

        Mnemonics.setLocalizedText(matchAllRadio, NbBundle.getMessage(
                FilterPanel.class, "MatchAll")); // NOI18N
        matchAllRadio.setToolTipText (NbBundle.getMessage(
                FilterPanel.class, "HINT_MatchAll")); // NOI18N
        Mnemonics.setLocalizedText(matchAnyRadio, NbBundle.getMessage(
                FilterPanel.class, "MatchAny")); // NOI18N
        matchAnyRadio.setToolTipText (NbBundle.getMessage(
                FilterPanel.class, "HINT_MatchAny")); // NOI18N
        Mnemonics.setLocalizedText(moreButton, NbBundle.getMessage(
                FilterPanel.class, "More")); // NOI18N
        moreButton.setToolTipText (NbBundle.getMessage(
                FilterPanel.class, "HINT_More")); // NOI18N
        Mnemonics.setLocalizedText(fewerButton, NbBundle.getMessage(
                FilterPanel.class, "Fewer")); // NOI18N
        fewerButton.setToolTipText (NbBundle.getMessage(
                FilterPanel.class, "HINT_Fewer")); // NOI18N
                
        Mnemonics.setLocalizedText(filterNameLabel, NbBundle.getMessage(
                FilterPanel.class, "LBL_Filter_Name"));
        filterNameLabel.setToolTipText(NbBundle.getMessage(
                FilterPanel.class, "LBL_Filter_Name_Hint"));              
        
        // accessible
        this.getAccessibleContext().setAccessibleDescription(
                NbBundle.getMessage(FilterPanel.class,
                                    "ACSD_Filter")); // NOI18N
        conditionsPanel.getAccessibleContext().setAccessibleName(
                NbBundle.getMessage(FilterPanel.class,
                                    "ACSN_Conditions")); // NOI18N
        conditionsPanel.getAccessibleContext().setAccessibleDescription(
                NbBundle.getMessage(FilterPanel.class,
                                    "ACSD_Conditions")); // NOI18N
    }

    /**
     * Adds a condition to the filter.
     *
     * @param revalidate true = invalidate() will be called
     * @param condition a condition or null
     */
    private ConditionPanel addCondition(boolean revalidate, AppliedFilterCondition condition) {
        GridBagConstraints gridBagConstraints = new GridBagConstraints();
        gridBagConstraints.gridwidth = GridBagConstraints.REMAINDER;
        gridBagConstraints.fill = GridBagConstraints.HORIZONTAL;
        gridBagConstraints.anchor = GridBagConstraints.NORTH;
        gridBagConstraints.insets = new Insets(6, 0, 0, 0);
        gridBagConstraints.weightx = 1.0;
        ConditionPanel panel = new ConditionPanel(filter, condition);
        panel.addPropertyChangeListener(FilterCondition.PROP_VALUE_VALID, this);
        panel.setBackground((Color)UIManager.get("Table.background")); //NOI18N
        conditionsPanel.add(panel, gridBagConstraints);
        if (revalidate) {
            invalidate();
            getParent().validate();
            repaint();
        }
        putClientProperty(FilterCondition.PROP_VALUE_VALID, Boolean.valueOf(isValueValid()));
        return panel;
    }

    /** @return aggregated PROP_VALUE_VALID value */
    public final boolean isValueValid() {
        Component[] cps = conditionsPanel.getComponents();
        for (int i = 0; i < cps.length; i++) {
            ConditionPanel condition = (ConditionPanel) cps[i];
            if (condition != null) {
                if (condition.isValueValid() == false) {
                    return false;
                }
            }
        }
        return true;
    }

    // forward aggregated PROP_VALUE_VALID value
    public void propertyChange(PropertyChangeEvent evt) {
        putClientProperty(FilterCondition.PROP_VALUE_VALID, Boolean.valueOf(isValueValid()));
    }

    /**
     * Invoked when an action occurs.
     */
    public void actionPerformed(ActionEvent evt) {
        Object source = evt.getSource();
        if (source == moreButton) {
            ConditionPanel comp = addCondition(true, null);
            JScrollBar vsb = conditionsScrollPane.getVerticalScrollBar();
            vsb.setValue(vsb.getMaximum());
            comp.focusPropertyCombo();
        } else if (source == fewerButton) {
            conditionsPanel.remove(conditionsPanel.getComponentCount() - 1);
            invalidate();
            getParent().validate();
            repaint();
        }
        updateSensitivity();
    }

    /**
     * Updates enabled/disabled state of the "fewer" button
     */
    private void updateSensitivity() {
        int n = conditionsPanel.getComponentCount();
        fewerButton.setEnabled(n > 0);
    }

    /**
     * Return a filter corresponding to what is in the GUI
     *
     * @return filter or null
     */
    public Filter getFilter() {
      if (filter != null) {
        // Get conditions
        Component[] cps = conditionsPanel.getComponents();
        ArrayList conditions = new ArrayList(cps.length);
        for (int i = 0; i < cps.length; i++) {
            AppliedFilterCondition condition = ((ConditionPanel)cps[i]).getCondition();
            if (condition != null) {
                conditions.add(condition);
            }
        }
        filter.setName(filterName.getText());
        filter.setConditions(conditions);
        filter.setMatchAll(matchAllRadio.isSelected());
        filter.setFlattened(true);
        if (subpanel != null) {
            subpanel.updateFilter(filter);
        }
      }
      return filter;

    }
    
    public JTextField getFilterNameField() {
        return filterName;
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    final javax.swing.JPanel buttonsPanel = new javax.swing.JPanel();
    final javax.swing.JPanel conditionsPanel = new javax.swing.JPanel();
    final javax.swing.JScrollPane conditionsScrollPane = new javax.swing.JScrollPane();
    final javax.swing.JButton fewerButton = new javax.swing.JButton();
    final javax.swing.JTextField filterName = new javax.swing.JTextField();
    final javax.swing.JLabel filterNameLabel = new javax.swing.JLabel();
    final javax.swing.JRadioButton matchAllRadio = new javax.swing.JRadioButton();
    final javax.swing.JRadioButton matchAnyRadio = new javax.swing.JRadioButton();
    final javax.swing.ButtonGroup matchGroup = new javax.swing.ButtonGroup();
    final javax.swing.JButton moreButton = new javax.swing.JButton();
    final javax.swing.JPanel namePanel = new javax.swing.JPanel();
    final javax.swing.ButtonGroup subtaskGroup = new javax.swing.ButtonGroup();
    // End of variables declaration//GEN-END:variables

}
