/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.j2ee.websphere6;

import java.io.*;

import javax.enterprise.deploy.model.*;
import javax.enterprise.deploy.spi.*;
import javax.enterprise.deploy.spi.exceptions.*;

import org.netbeans.modules.j2ee.deployment.plugins.api.*;

import org.netbeans.modules.j2ee.websphere6.util.WSDebug;

/**
 * A wrapper for the server's DeploymentConfiguration implementation. It is 
 * used mostly to solve the classloading problems, i.e. to substitute the
 * thread's context classloader with the WebSphere loader and back when 
 * WebSphere method returns
 *
 * @author Kirill Sorokin
 */
public class WSDeploymentConfiguration implements DeploymentConfiguration {
    
    /**
     * The WS implementation of DeploymentConfiguration, all calls are 
     * delegated to it
     */
    private DeploymentConfiguration configuration;
    
    /**
     * Wrapper for j2eeserver's DeployableObject, it is used for correctly 
     * getting the DConfigBeanRoot
     */
    private WSDeployableObject deployableObject;
    
    /**
     * Instance properties for the current server instance
     */
    private InstanceProperties instanceProperties;
    
    /**
     * Current clasloader used for WS classes
     */
    private WSClassLoader loader;
    
    /**
     * Creates a new instance of WSDeploymentConfiguration.
     * 
     * @param dm the DeploymentManager
     * @param deployableObject j2eeserver's DeployableObject
     * @param instanceProperties instance properties for the current server 
     *      instance
     * 
     * @throws InvalidModuleException never thrown in this implementation
     */
    public WSDeploymentConfiguration(DeploymentManager dm, 
            DeployableObject deployableObject, 
            InstanceProperties instanceProperties) 
            throws InvalidModuleException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "WSDeploymentConfiguration()"); // NOI18N
        
        // save the instance properties
        this.instanceProperties = instanceProperties;
        
        // get the correct loader instance
        loader = WSClassLoader.getInstance(instanceProperties.getProperty(
                WSDeploymentFactory.SERVER_ROOT_ATTR), 
                instanceProperties.getProperty(
                WSDeploymentFactory.DOMAIN_ROOT_ATTR));
        
        // create a proper wrapper for the supplied deployable object - either
        // a general deployable object or an application object
        if (deployableObject instanceof J2eeApplicationObject) {
            this.deployableObject = new WSJ2eeApplicationObject(
                    (J2eeApplicationObject) deployableObject);
        } else {
            this.deployableObject = new WSDeployableObject(deployableObject);
        }
        
        // create the server's DeploymentConfiguration and save it
        try {
            this.configuration = dm.createConfiguration(this.deployableObject);
        } catch (InvalidModuleException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public DConfigBeanRoot getDConfigBeanRoot(DDBeanRoot bean) 
            throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getDConfigBeanRoot(" + bean +  // NOI18N
                    ")");                                              // NOI18N
        
        // update the context classloader
        loader.updateLoader();
        
        // call the server's class
        DConfigBeanRoot dConfigBeanRoot = configuration.getDConfigBeanRoot(
                deployableObject.findDDBeanRoot(bean));
        
        // restore the context classloader
        loader.restoreLoader();
        
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "returning: " +                 // NOI18N
                    dConfigBeanRoot); 
        
        // return
        return dConfigBeanRoot;
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public DeployableObject getDeployableObject() {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "getDeployableObject()");       // NOI18N
        
        return configuration.getDeployableObject();
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public void removeDConfigBean(DConfigBeanRoot bean) 
            throws BeanNotFoundException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "removeDConfigBean(" + bean +   // NOI18N
                    ")");                                              // NOI18N
        
        try {
            configuration.removeDConfigBean(bean);
        } catch (BeanNotFoundException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public void restore(InputStream inputArchive) throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "restore(" + inputArchive +     // NOI18N
                    ")");                                              // NOI18N
        
        try {
            configuration.restore(inputArchive);
        } catch (ConfigurationException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public DConfigBeanRoot restoreDConfigBean(InputStream inputArchive, 
            DDBeanRoot bean) throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "restoreDConfigBean(" +         // NOI18N
                    inputArchive + ", " + bean + ")");                 // NOI18N
        
        try {
            return configuration.restoreDConfigBean(inputArchive, bean);
        } catch (ConfigurationException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
    
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public void save(OutputStream outputArchive) throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "save(" + outputArchive + ")"); // NOI18N
        
        // updte the context classloader
        loader.updateLoader();
        
        try {
            configuration.save(outputArchive);
        } catch (ConfigurationException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
        
        // restore the context classloader
        loader.restoreLoader();
    }
            
    /**
     * Delegates the call to the server's DeploymentConfiguration, handling
     * the classloading issues, if necessary.
     */
    public void saveDConfigBean(OutputStream outputArchive, 
            DConfigBeanRoot bean) throws ConfigurationException {
        if (WSDebug.isEnabled()) // debug output
            WSDebug.notify(getClass(), "saveDConfigBean(" +            // NOI18N
                    outputArchive + ", " + bean + ")");                // NOI18N
        
        try {
            configuration.saveDConfigBean(outputArchive, bean);
        } catch (ConfigurationException e) {
            if (WSDebug.isEnabled()) // debug output
                WSDebug.notify(e);
            
            throw e;
        }
    }
}