/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.test.refactoring;

import java.io.*;
import java.lang.reflect.Modifier;
import java.util.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.junit.diff.LineDiff;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.RefactoringElement;
import org.netbeans.modules.refactoring.api.RefactoringSession;

/** LogTestCase
 * @author Jan Becicka
 */
public class RefactoringTestCase extends LogTestCase {
    
    protected static TypeClass typeProxy;
    protected static JavaClass jc;
    
    public RefactoringTestCase(java.lang.String testName) {
        super(testName);
    }
    
    protected void setUp() {
        super.setUp();
        setJavaClass("java.lang.String");
    }
    
    protected void setJavaClass(String name) {
        Utility.beginTrans(true);
        try {
            jc = (JavaClass) Utility.findClass(name);
            typeProxy = ((JavaModelPackage) jc.refOutermostPackage()).getType();
        } finally {
            Utility.endTrans();
        }
    }
    
    /**
     * Stores problems into log file.
     */
    public void logProblems(Problem problem) {
        Problem p=problem;
        if (p != null) {
            ArrayList list=new ArrayList();
            while (p != null) {
                if (p.isFatal()) {
                    list.add("Problem fatal: "+p.getMessage());
                } else {
                    list.add("Problem: "+p.getMessage());
                }
                p=p.getNext();
            }
            Collections.sort(list);
            for (int i=0;i < list.size();i++) {
                log(list.get(i));
            }
        }
    }
    
    /**
     * Stores problems into ref file. Problems should be sorted.
     * @return true if problem is not null and one of them is fatal
     */
    public boolean refProblems(Problem problem) {
        Problem p=problem;
        boolean ret=false;
        if (p != null) {
            ArrayList list=new ArrayList();
            while (p != null) {
                if (p.isFatal()) {
                    ret=true;
                    list.add("Problem fatal: "+p.getMessage());
                } else {
                    list.add("Problem: "+p.getMessage());
                }
                p=p.getNext();
            }
            Collections.sort(list);
            for (int i=0;i < list.size();i++) {
                ref(list.get(i));
            }
        }
        return ret;
    }
    
    /**
     * Makes diffs of files in map (before refactoring) with refactored files.
     * @param files map of resource to temporary file
     * @param result usages after refactoring
     */
    protected void compareResources(HashMap files, RefactoringSession session, String name, String renamed) throws Exception {
        Collection result = session.getRefactoringElements();
        ref("Found "+String.valueOf(result.size())+" occurance(s).");
        File folder=getWorkDir();
        Object[] ress=files.keySet().toArray();
        Arrays.sort(ress);
        for (int i=0;i < ress.length;i++) {
            if (ress[i].equals("empty")) {
                continue;
            }
            Res res=(Res)(files.get(ress[i]));
            File oldfile=res.backup;
            if (oldfile == null || !oldfile.exists()) { //resource has new name
                log("compareResources: backuped old file is null - changed resource ");
            }
            File actualfile=new File(classPathWorkDir, (String)ress[i]);
            if (!actualfile.exists()) { //moved
                if ((name.replace('.','/')+".java").equals(ress[i])) { //class name
                    actualfile=new File(classPathWorkDir, renamed.replace('.','/')+".java");
                } else if (((String)ress[i]).indexOf(name.replace('.','/')) == 0) { //package name
                    String rsnm=(String)ress[i]; // a/b/c/A.java
                    String nm=name.replace('.','/');  // a/b/c
                    rsnm=renamed.replace('.','/')+rsnm.substring(nm.length()); // b/b/c/A.java
                    actualfile=new File(classPathWorkDir, rsnm);
                }
            }
            LineDiff ldiff=new LineDiff();
            File diff=File.createTempFile("xtest", "refactoring");
            ldiff.diff(actualfile, oldfile, diff);
            ref("\n"+ress[i]+"\n");
            Collections.sort(res.texts);
            for (int j=0;j < res.texts.size();j++) {
                ref("      "+res.texts.get(j));
            }
            ref("\nFile diff:\n");
            ref(diff);
            diff.delete();
            oldfile.delete();
            Utility.copyFile(actualfile, new File(getWorkDir(), actualfile.getName()));
        }
        ref("");
        Res empty=(Res)(files.get("empty"));
        if (empty != null) {
            for (int i=0;i < empty.texts.size();i++) {
                ref(empty.texts.get(i));
            }
        }
    }
    
    /**
     * makes map of files mapped to resources names of usages, files are backuped into tmp files
     */
    protected HashMap getResources(RefactoringSession session) throws Exception {
        Collection result = session.getRefactoringElements();
        ArrayList list=new ArrayList();
        Res empty=new Res();
        HashMap files=new HashMap();
        for (Iterator it=result.iterator();it.hasNext();) {
            Object o=it.next();
            if (o instanceof RefactoringElement) {
                RefactoringElement wue=(RefactoringElement) o;
                Element el = wue.getJavaElement();
                if (el != null && el.getResource() != null && el.getResource().isValid()) {
                    Res res;
                    if (!list.contains(el.getResource())) {
                        list.add(el.getResource());
                        res=new Res();
                        files.put(el.getResource().getName(), res);
                    } else {
                        res=(Res)(files.get(el.getResource().getName()));
                    }
                    res.texts.add(getDisplayText(wue));
                } else {
                    empty.texts.add(getDisplayText(wue));
                }
            }
        }
        files.put("empty", empty);
        Collections.sort(list, new Comparator() {
            public int compare(Object o1, Object o2) {
                return ((Resource)o1).getName().compareTo(((Resource)o2).getName());
            }
        });
        File folder=getWorkDir();
        for (int i=0;i < list.size();i++) {
            Resource r=(Resource)(list.get(i));
            Res res=(Res)(files.get(r.getName()));
            String name=r.getName().replace('/', '_')+".test";
            name=name.replace('\\', '_');
            File fl=new File(folder, name);
            if (!fl.getParentFile().exists()) {
                fl.getParentFile().mkdirs();
            }
            PrintStream ps=new PrintStream(new FileOutputStream(fl));
            ps.print(r.getSourceText());
            ps.close();
            res.backup=fl;
        }
        return files;
    }
    
    protected void refUsages(RefactoringSession session) {
        Collection result = session.getRefactoringElements();
        ArrayList list=new ArrayList();
        HashMap map=new HashMap();
        for (Iterator it=result.iterator();it.hasNext();) {
            Object o=it.next();
            if (o instanceof RefactoringElement) {
                RefactoringElement wue=(RefactoringElement) o;
                Element el = wue.getJavaElement();
                if (el != null && el.getResource() != null) {
                    String s;
                    s=el.getResource().getName().replace(File.separatorChar,'/');
                    list=(ArrayList)(map.get(s));
                    if (list == null) {
                        list=new ArrayList();
                        map.put(s, list);
                    }
                    list.add(getDisplayText(wue));
                } else {
                    log("refUsages without resource");
                    log(getDisplayText(wue));
                    map.put(getDisplayText(wue), "");
                }
            }
        }
        ref("Found "+String.valueOf(result.size())+" occurance(s).");
        Object[] keys=map.keySet().toArray();
        Arrays.sort(keys);
        for (int i=0;i < keys.length;i++) {
            ref("");
            if (map.get(keys[i]) instanceof ArrayList) {
                ref(keys[i]);
                list=(ArrayList)(map.get(keys[i]));
                Collections.sort(list);
                for (int j=0;j < list.size();j++) {
                    ref("      "+list.get(j));
                }
            } else {
                ref(keys[i]);
            }
        }
        ref("");
    }
    
    protected String getDisplayText(RefactoringElement elm) {
        String app="";
        if (elm.getStatus() == RefactoringElement.WARNING) {
            app=" [ warning! ]";
        } else if (elm.getStatus() == RefactoringElement.GUARDED) {
            app=" [ error: code is in guarded block ]";
        }
        return elm.getDisplayText()+app;
    }
    
    protected String getModifier(int modifier) {
        String ret="";
        if (Modifier.isPublic(modifier)) {
            ret+="public";
        } else if (Modifier.isProtected(modifier)) {
            ret+="protected";
        } else if (Modifier.isPrivate(modifier)) {
            ret+="private";
        }
        if (Modifier.isAbstract(modifier)) {
            ret+=" abstract";
        } else if (Modifier.isFinal(modifier)) {
            ret+=" final";
        } else if (Modifier.isInterface(modifier)) {
            ret+=" interface";
        } else if (Modifier.isNative(modifier)) {
            ret+=" native";
        } else if (Modifier.isStatic(modifier)) {
            ret+=" static";
        } else if (Modifier.isSynchronized(modifier)) {
            ret+=" synchronized";
        } else if (Modifier.isTransient(modifier)) {
            ret+=" transient";
        } else if (Modifier.isVolatile(modifier)) {
            ret+=" volatile";
        }
        return ret.trim();
    }
    
    class Res {
        public File backup;
        public ArrayList texts;
        
        public Res() {
            texts=new ArrayList();
        }
    }
}

