/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.ui;

import java.util.Iterator;
import javax.jmi.reflect.RefObject;
import org.netbeans.api.fileinfo.NonRecursiveFolder;
import org.netbeans.api.mdr.MDRepository;
import javax.swing.SwingUtilities;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.classpath.Util;
import org.netbeans.modules.refactoring.spi.ui.AbstractRefactoringAction;
import org.netbeans.modules.refactoring.spi.ui.RefactoringUI;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.java.JavaDataObject;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.classpath.RefactoringClassPathImplementation;
import org.openide.filesystems.FileObject;
import org.openide.filesystems.FileStateInvalidException;
import org.openide.filesystems.Repository;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.nodes.Node;
import org.openide.text.CloneableEditorSupport;
import org.openide.util.NbBundle;
import org.openide.windows.TopComponent;

/**
 *
 * @author Pavel Flaska, Martin Matula
 */
public class RenameAction extends AbstractRefactoringAction {
    /** Creates a new instance of RenameAction */
    public RenameAction() {
        super(NbBundle.getMessage(RenameAction.class, "LBL_RenameAction"), null);
        putValue("noIconInMenu", Boolean.TRUE); // NOI18N
    }
    
    protected RefactoringUI createRefactoringUI(Node[] nodes, org.netbeans.jmi.javamodel.Element selectedElement) {
        RefObject representedObject = selectedElement;
        if (representedObject!=null) {
            if (representedObject instanceof MultipartId) {
                RefObject temp = representedObject;
                RefObject last = temp;
                while (temp instanceof MultipartId && ((ElementReference)temp).getElement() instanceof JavaPackage) {
                    last = temp;
                    temp = (RefObject) temp.refImmediateComposite();
                }
                if (temp instanceof Resource || temp instanceof Import) {
                    representedObject = last;
                } else {
                    representedObject = (((ElementReference)temp).getElement() instanceof JavaClass) ? temp : last;
                }
            }
            if (representedObject instanceof ElementReference) {
                RefObject r = ((ElementReference)representedObject).getElement();
                if (r!=null) {
                    representedObject = r;
                }
            }
        } else {
            Node selNode = nodes[0]; 
            representedObject = (Element) selNode.getLookup().lookup(Element.class);
            
            if (representedObject == null) {
                Object ob = selNode.getCookie(JavaDataObject.class);
                if (ob != null) {
                    JavaDataObject o = (JavaDataObject) ob;
                    Resource cr = JavaModel.getResource(o.getPrimaryFile());
                    representedObject = cr;
                    if (cr != null) {
                        for (Iterator it = cr.getClassifiers().iterator(); it.hasNext();) {
                            Object obj = it.next();
                            if (!(obj instanceof JavaClass))
                                continue;
                            JavaClass cls = (JavaClass) obj;
                            representedObject = cls;
                            if (selNode.getName().equals(cls.getSimpleName())) {
                                break;
                            }
                        }
                    }
                } else {
                    ob = selNode.getCookie(DataFolder.class);
                    FileObject folder = ((DataFolder) ob).getPrimaryFile();
                    try {
                        if (folder.getFileSystem() != Repository.getDefault().getDefaultFileSystem()) {
                            ClassPath cp = ClassPath.getClassPath(folder, ClassPath.SOURCE);
                            representedObject = null;
                            if (cp!=null) {
                                String packageName = cp.getResourceName(folder, '.', false);
                                FileObject root = cp.findOwnerRoot(folder);
                                if (root != null) {
                                    //cannot refactor root
                                    JavaModelPackage pck = JavaModel.getJavaExtent(root);
                                    assert pck != null: "JavaModel.getJavaExtent(" + root.getPath() +") returned null";
                                    representedObject = pck.getJavaPackage().resolvePackage(packageName);
                                    boolean isRecursive = !(selNode.getLookup().lookup(NonRecursiveFolder.class) != null);
                                    return new RenameRefactoringUI(representedObject, folder, !isRecursive);
                                }
                            }
                        }
                    } catch (FileStateInvalidException e) {
                        // ignore
                    }
                }
            }
        }
        if (representedObject instanceof Constructor) 
            representedObject = ((Constructor)representedObject).getDeclaringClass();
        return new RenameRefactoringUI(representedObject);
    }
    
    protected boolean enabled(Node[] activatedNodes) {
        if (activatedNodes.length != 1) return false;
        DataObject dobj = (DataObject) activatedNodes[0].getCookie(DataObject.class);
        if (dobj!=null)
            if (!Util.isUnderSourceRootOfOpenProjects(dobj.getPrimaryFile()))
                return false;
        
        boolean isRecursive = !(activatedNodes[0].getLookup().lookup(NonRecursiveFolder.class) != null);
        if (dobj instanceof DataFolder) {
            if (!isRecursive) {
                if (activatedNodes[0].isLeaf())
                    return false;
            } else {
                if (Util.isClassPathRoot(dobj.getPrimaryFile())) {
                    return false;
                }
                if (!CheckUtils.hasChildren((DataFolder) dobj)) {
                    return false;
                }
            }
            return true;
        }
        return (dobj instanceof JavaDataObject);
    }
    
    // called from RefactoringOperationListener when a package is renamed manually (not by refactoring action)
    void performAction(final RefObject elem, final String name) {
        performAction(elem, name, null, false);
    }
    
    void performAction(FileObject folder, final String name, final boolean packageRename) {
        performAction(null, name, folder, packageRename);
    }

    private void performAction(final RefObject elem, final String name, final FileObject folder, final boolean packageRename) {
        final Runnable run = new Runnable() {
            public void run() {
                doRun(elem, name, folder, packageRename);
            }
        };
        
        if (SwingUtilities.isEventDispatchThread()) {
            JavaMetamodel.getManager().invokeAfterScanFinished(run, NbBundle.getMessage(RenameAction.class, "LBL_Rename"));
        } else {
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    JavaMetamodel.getManager().invokeAfterScanFinished(run, NbBundle.getMessage(RenameAction.class, "LBL_Rename"));
                }
            });
        }
    }
    
    
    private final void doRun(RefObject elem, String name, FileObject folder, boolean packageRename) {
        MDRepository rep = JavaModel.getJavaRepository();
        final RefactoringUI ui;
        
        rep.beginTrans(false);
        try {
            JavaModel.setClassPath(RefactoringClassPathImplementation.getDefault());
            ui = folder != null ? new RenameRefactoringUI(folder, name ,packageRename) : new RenameRefactoringUI(elem, name);
        } finally {
            rep.endTrans();
        }
        TopComponent activetc = TopComponent.getRegistry().getActivated();
        if (activetc instanceof CloneableEditorSupport.Pane) {
            new RefactoringPanel(ui, activetc);
        } else {
            new RefactoringPanel(ui);    
        }    
    }
    
}
