/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.refactoring.plugins;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.logging.Logger;
import org.netbeans.api.java.queries.SourceLevelQuery;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.ElementReference;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.JavaModelPackage;
import org.netbeans.jmi.javamodel.LocalVarDeclaration;
import org.netbeans.jmi.javamodel.LocalVariable;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.MethodInvocation;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.Parameter;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.jmi.javamodel.TypeReference;
import org.netbeans.jmi.javamodel.Variable;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.jmiimpl.javamodel.MethodImpl;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.RenameRefactoring;
import org.netbeans.modules.refactoring.api.UseSuperTypeRefactoring;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.util.NbBundle;


/*
 * UseSuperTypeRefactoringPlugin.java
 *
 * Created on June 22, 2005
 *
 * @author Bharath Ravi Kumar
 */
/**
 * The plugin that performs the actual work on
 * behalf of the use super type refactoring
 */
public class UseSuperTypeRefactoringPlugin extends JavaRefactoringPlugin {
    
    private final UseSuperTypeRefactoring refactoring;
    private final RenameRefactoring renameRefactoring = null;
    private final Logger log = Logger.getLogger("org.netbeans.modules.refactoring.plugins"); // NOI18N
    private ErrorManager errMngr =  ErrorManager.getDefault();
    private static final float ONE_DOT_FIVE = 1.5f;
    
    /**
     * Creates a new instance of UseSuperTypeRefactoringPlugin
     * @param refactoring The refactoring to be used by this plugin
     */
    public UseSuperTypeRefactoringPlugin(UseSuperTypeRefactoring refactoring) {
        this.refactoring = refactoring;
    }
    
    /**
     * Prepares the underlying where used query & checks
     * for the visibility of the target type.
     */
    public org.netbeans.modules.refactoring.api.Problem prepare(org.netbeans.modules.refactoring.spi.RefactoringElementsBag refactoringElements) {
        
        JavaClass superType = refactoring.getTargetSuperType();
        String superTypeName = superType.getName();
        JavaClass subClass = (JavaClass) refactoring.getTypeElement();
        fireProgressListenerStart(AbstractRefactoring.PREPARE, 1);
        Collection subClassUsages;
        try {
            subClassUsages = subClass.getReferences();
        } finally {
            fireProgressListenerStop();
        }
        
        Iterator iterator = subClassUsages.iterator();
        
        fireProgressListenerStart(AbstractRefactoring.PREPARE, subClassUsages.size());
        try{
            while(iterator.hasNext()) {
                ElementReference referringElement  = ((ElementReference) iterator.next());
                Element subTypeReferrer = (Element) referringElement.refImmediateComposite();
                
                //Check if the target element is a valid candidate element. Valid candidates are fields, local vars
                //and method return types
                
                boolean isValidElement = false;
                
                if(subTypeReferrer instanceof Method){
                    //Checks if the return type is exactly the subClass
                    //In addition, if the source level is >= 1.5 +, it could be a covariant return type.
                    //So, we check for that.
                    Method methodWithSubType = ((Method)subTypeReferrer);
                    TypeReference retTypeRef = methodWithSubType.getTypeName();
                    //Check if the method returns exactly the sub type
                    boolean isReturnTypeSubClass = retTypeRef.getElement().equals(subClass);
                    if(isReturnTypeSubClass){
                        FileObject referringFileObject = JavaModel.getFileObject(subTypeReferrer.getResource());
                        String srcLevelStr = SourceLevelQuery.getSourceLevel(referringFileObject);
                        if(Float.parseFloat(srcLevelStr) >= ONE_DOT_FIVE) {
                            Collection overriddenMethods = JavaModelUtil.getOverriddenMethods(methodWithSubType);
                            Iterator ovrmIterator = overriddenMethods.iterator();
                            boolean compatibleRetType = true;
                            while(ovrmIterator.hasNext() && compatibleRetType){
                                Method superMethod = (Method) ovrmIterator.next();
                                JavaClass returnType = (JavaClass) superMethod.getType();
                                compatibleRetType = CheckUtils.isSubTypeOf(superType,returnType);
                            }
                            isValidElement = compatibleRetType;
                        }//End  if source level >= 1.5
                    }//End  if(returnTypeSubClass)
                }//End  if instanceof Method
                else if ( subTypeReferrer instanceof Variable) {
                    isValidElement = checkValidInvocations(superType , ((Variable) subTypeReferrer).getReferences());
                } else if(subTypeReferrer instanceof LocalVarDeclaration){
                    List localVarList = ((LocalVarDeclaration) subTypeReferrer).getVariables();
                    Iterator locVarIterator = localVarList.iterator();
                    boolean validLocVarUsage = true;
                    while(locVarIterator.hasNext() && validLocVarUsage){
                        LocalVariable locVar = (LocalVariable) locVarIterator.next();
                        validLocVarUsage = checkValidInvocations(superType, locVar.getReferences());
                    }
                    isValidElement = validLocVarUsage;
                }
                
                //If it is not a valid target element, continue to the next element
                if(!isValidElement){
                    fireProgressListenerStep();
                    continue;
                }
                //Check if the super class Element is visible in the referring element's
                //surrounding scope.
                Element surroundingComposite = subTypeReferrer;
                while( surroundingComposite != null && !(surroundingComposite instanceof JavaClass)){
                    //This is a little iffy. Will there be a condition when the while loop
                    //runs forever?
                    if(surroundingComposite.equals(surroundingComposite.refImmediateComposite()))
                        break;
                    surroundingComposite = (Element) surroundingComposite.refImmediateComposite();
                }
                if(surroundingComposite != null){
                    boolean accessible = CheckUtils.isAccessible(refactoring.getTargetSuperType(), (JavaClass) surroundingComposite);
                    //TODO:Remove this statement & instead return a ProblemDetails Object
                    //created by the ProblemsFactory
                    if(!accessible){
                        String errorMsg = NbBundle.getMessage(UseSuperTypeRefactoringPlugin.class,"TXT_UseSuperTypeProblem",
                                refactoring.getTargetSuperType().getSimpleName(),surroundingComposite);
                        return new Problem(false, errorMsg);
                    }
                }
                
                //Now, add the RefactoringElement to the bag
                refactoringElements.add(refactoring , new UseSuperTypeElement((Element) subTypeReferrer, superType));
                fireProgressListenerStep();
            }
        } finally{
            fireProgressListenerStop();
        }
        return null;
    }
    
    /**
     *Checks whether the candidate element is a valid Type.
     *@return Problem The problem instance indicating that an invalid element was selected.
     */
    public org.netbeans.modules.refactoring.api.Problem preCheck() {
        Element subType = refactoring.getTypeElement();
        if(!(subType instanceof JavaClass)){
            String errMsg = NbBundle.getMessage(UseSuperTypeRefactoringPlugin.class,
                    "ERR_UseSuperType_InvalidElement"); // NOI18N
            return new Problem(true, errMsg);
        }
        return null;
    }
    
    /**
     * @return A problem indicating that no super type was selected.
     */
    public org.netbeans.modules.refactoring.api.Problem fastCheckParameters() {
        if (refactoring.getTargetSuperType() == null) {
            return new Problem(true, NbBundle.getMessage(UseSuperTypeRefactoringPlugin.class, "ERR_UseSuperTypeNoSuperType"));
        }
        return null;
    }
    
    /**
     * A no op. Returns null
     */
    public org.netbeans.modules.refactoring.api.Problem checkParameters() {
        return null;
    }
    
    //---------private helper methods follow--------
    
    private Problem checkSuperTypeVisibility(Resource resource){
        JavaModelPackage pkg = (JavaModelPackage) resource.refImmediatePackage();
        return null;
    }
    
    /**
     * Indicates whether the method invocations on a variable would be valid if the variable's
     * type was changed to the given super type.
     * @param superType The super type with which to check for valid method invocations on that variable
     * @param variableUsages The collection containing the usages of that variable
     * @return true if the invocations would be valid if the variable's type was changed to
     * the given super type. false otherwise
     */
    private boolean checkValidInvocations(JavaClass superType, Collection variableUsages) {
        boolean safeInvocationsOnSuper = true;
        Iterator varReferences = variableUsages.iterator();
        while(varReferences.hasNext()) {
            ElementReference varReferrer  = ((ElementReference) varReferences.next());
            Element varReferrerComposite = (Element) varReferrer.refImmediateComposite();
            if(varReferrerComposite instanceof MethodInvocation) {
                Method method = (Method) ((MethodInvocation) varReferrerComposite).getElement();
                // retrieve types of parameters
                List params = method.getParameters();
                List paramTypes = new ArrayList(params.size());
                for (Iterator iter = params.iterator(); iter.hasNext(); ) {
                    paramTypes.add(((Parameter)iter.next()).getType());
                }
                //Check if the method invoked on an object of the sub type is available in the
                //super type as well
                if(superType.getMethod(method.getName(), paramTypes, true) == null) {
                    safeInvocationsOnSuper = false;
                    break;
                }
            }
        }
        return safeInvocationsOnSuper;
    }
    
    //---------private classes follow--------
    private class UseSuperTypeElement extends SimpleRefactoringElementImpl {
        
        private final Element targetElement;
        private final JavaClass targetType;
        private String text;
        
        public UseSuperTypeElement(Element sourceElement, JavaClass targetType) {
            this.targetElement= sourceElement;
            this.targetType = targetType;
            if(sourceElement instanceof Variable){
                text = NbBundle.getMessage(UseSuperTypeRefactoringPlugin.class,
                        "TXT_UseSuperTypeVariable", ((Variable)sourceElement).getName(), targetType.getSimpleName()); // NOI18N
            } else if(sourceElement instanceof Method){
                text = NbBundle.getMessage(UseSuperTypeRefactoringPlugin.class,
                        "TXT_UseSuperTypeReturnType", ((Method)sourceElement).getName(), targetType.getSimpleName()); // NOI18N
            }else if(sourceElement instanceof LocalVarDeclaration){
                String variableNames = getLocalVariables(((LocalVarDeclaration) sourceElement));
                text = NbBundle.getMessage(UseSuperTypeRefactoringPlugin.class,
                        "TXT_UseSuperTypeLocalVar", variableNames, targetType.getSimpleName()); // NOI18N
            }
        }
        
        public void performChange() {
            if(targetElement instanceof Method){
                TypeReference typeRef = JavaModelUtil.resolveImportsForClass((Element) targetElement.refImmediateComposite(), targetType);
                Method method = ((Method)targetElement);
                method.setTypeName(typeRef);
            } else if(targetElement instanceof Variable){
                Variable param = ((Variable) targetElement);
                TypeReference typeRef = JavaModelUtil.resolveImportsForClass((Element) targetElement.refImmediateComposite(), targetType);
                param.setTypeName(typeRef);
            } else if(targetElement instanceof LocalVarDeclaration){
                LocalVarDeclaration localVar = ((LocalVarDeclaration) targetElement);
                TypeReference typeRef = JavaModelUtil.resolveImportsForClass((Element) targetElement.refImmediateComposite(), targetType);
                localVar.setTypeName(typeRef);
            } else
                ErrorManager.getDefault().log(ErrorManager.WARNING,"UseSupertype cannot associate" +
                        "any specific type with "+targetElement); // NOI18N
        }
        
        public String getText() {
            return text;
        }
        
        public org.openide.text.PositionBounds getPosition() {
            return JavaMetamodel.getManager().getElementPosition(targetElement);
        }
        
        public FileObject getParentFile() {
            return JavaMetamodel.getManager().getFileObject(targetElement.getResource());
        }
        
        public Element getJavaElement() {
            return JavaModelUtil.getDeclaringFeature(targetElement);
        }
        
        public String getDisplayText() {
            return getText();
        }
        //--Private helper methods follow--
        private String getLocalVariables(final LocalVarDeclaration sourceElement) {
            Iterator declaredVarIterator = sourceElement.getVariables().iterator();
            //We will show at most 2 local variables in the text
            int numVars = 0;
            StringBuffer localVarBuffer = new StringBuffer();
            while(declaredVarIterator.hasNext() && numVars < 2){
                localVarBuffer.append(((NamedElement) declaredVarIterator.next()).getName());
                ++numVars;
                if(declaredVarIterator.hasNext() && numVars < 2)
                    localVarBuffer.append(", "); // NOI18N
                else if(numVars == 2 && declaredVarIterator.hasNext())
                    localVarBuffer.append("..."); // NOI18N
            }
            String variableNames = localVarBuffer.toString();
            return variableNames;
        }
    }
    
}
