/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.plugins;

import java.io.IOException;
import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import java.util.*;
import javax.jmi.reflect.RefObject;
import org.netbeans.api.java.classpath.ClassPath;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.ClassIndex;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.javacore.internalapi.ProgressListener;
import org.netbeans.modules.javacore.jmiimpl.javamodel.MethodImpl;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.CommentRenameFinder;
import org.netbeans.modules.refactoring.RenameUsageElement;
import org.netbeans.modules.refactoring.api.*;
import org.netbeans.modules.refactoring.classpath.RefactoringClassPathImplementation;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataFolder;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;

/**
 *
 * @author Jan Becicka, Martin Matula, Pavel Flaska, Daniel Prusa
 */
public class RenameRefactoringPlugin extends JavaRefactoringPlugin implements ProgressListener {
    private RefObject jmiObject = null;
    private String newName = null;
    private Collection overriddenByMethods = null; // methods that override the method to be renamed
    private Collection overridesMethods = null; // methods that are overridden by the method to be renamed
    private boolean doCheckName = true;
    private FileObject originalFolder = null;
    private Set varNames;
    
    private RenameRefactoring refactoring;
    
    /** Creates a new instance of RenameRefactoring */
    public RenameRefactoringPlugin(RenameRefactoring rename) {
        this.refactoring = rename;
        jmiObject = (RefObject) rename.getRefactoredObject();
    }
    
    public Problem checkParameters() {
        return setParameters(refactoring.getNewName());
    }
    
    public Problem fastCheckParameters() {
        return checkParameters(refactoring.getNewName());
    }

    
    public Problem preCheck() {
        fireProgressListenerStart(refactoring.PRE_CHECK, 4);
        try {
            Problem result = isElementAvail((Element) jmiObject);
            if (result != null) {
                return result;
            }
            Resource res = (((Element) jmiObject)).getResource();
            if (res!=null && CheckUtils.isFromLibrary(res)) { //NOI18N
                return createProblem(result, true, getCannotRename(res));
            }
            
            if (res!=null && !CheckUtils.isElementInOpenProject((Element) jmiObject)) {
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "ERR_ProjectNotOpened"));
            }
            
            if (jmiObject instanceof UnresolvedClass) {
                return new Problem(true, NbBundle.getMessage(JavaRefactoringPlugin.class, "DSC_ElNotAvail")); // NOI18N
            }
            if (jmiObject instanceof Method) {
                fireProgressListenerStep();
                Method m = (Method) jmiObject;
                List argTypes = getParamTypes(m);
                fireProgressListenerStep();
                overriddenByMethods = overriddenBy(m, m.getName(), argTypes);
                fireProgressListenerStep();
                if (!overriddenByMethods.isEmpty()) {
                    String msg = new MessageFormat(getString("ERR_IsOverridden")).format(
                        new Object[] {getDefClassName(m.getDeclaringClass())}
                    );
                    result = createProblem(result, false, msg);
                }
                ClassDefinition[] par = new ClassDefinition[] {null, null};
                overridesMethods = overrides(m, m.getName(), argTypes, par);
                fireProgressListenerStep();
                if (!overridesMethods.isEmpty()) {
                    if (par[0] == null) {
                        boolean fatal = false;
                        for (Iterator iter = overridesMethods.iterator();iter.hasNext();) {
                            Method method = (Method) iter.next();
                            if (CheckUtils.isFromLibrary(method.getResource())) {
                                fatal = true;
                                break;
                            }
                        }
                        String msg = fatal?getString("ERR_Overrides_Fatal"):getString ("ERR_Overrides");
                        result = createProblem(result, fatal, msg);
                    } else {
                        String clsName = getDefClassName(par[0]);
                        String msg;
                        if (CheckUtils.isFromLibrary(par[0].getResource())) {
                            msg = getString("ERR_Overrides_Fatal");
                        } else {
                            msg = new MessageFormat(getString("ERR_Overrides_tree")).format(
                                    new Object[] {clsName, getDefClassName(par[1]), clsName}
                            );
                        }
                        result = createProblem(result, true, msg);
                    }
                }
            } else if (jmiObject instanceof Field) {
                fireProgressListenerStep();
                fireProgressListenerStep();
                Field f = (Field) jmiObject;
                Field hiddenField = hides(f, f.getName());
                fireProgressListenerStep();
                fireProgressListenerStep();
                if (hiddenField != null) {
                    String msg = new MessageFormat(getString("ERR_Hides")).format(
                        new Object[] {getDefClassName(hiddenField.getDeclaringClass())}
                    );
                    result = createProblem(result, false, msg);
                }
            } else if(jmiObject instanceof JavaPackage) {
                //TODO: any prechecks?
            } else if (jmiObject instanceof Variable) {
                //TODO: any prechecks for formal parametr or local variable?
            } else if (jmiObject instanceof Attribute) {
                //TODO: any prechecks for Attribute?
            } else if (jmiObject instanceof JavaClass) {
                //TODO: any prechecks for JavaClass?
            } else {
                fireProgressListenerStep();
                if (!((jmiObject instanceof Resource) && ((Resource)jmiObject).getClassifiers().isEmpty()))
                    result = createProblem(result, true, NbBundle.getMessage(RenameRefactoring.class, "ERR_RenameWrongType"));
            }

            return result;
        } finally {
            fireProgressListenerStop();
        }
    }
    
    private static final String getCannotRename(Resource r) {
        return new MessageFormat(NbBundle.getMessage(RenameRefactoring.class, "ERR_CannotRenameFile")).format(new Object[] {r.getName()});
    }
    
    private Problem checkParameters(String newName) {
        Problem result = null;
        
        if (getElementName((NamedElement) jmiObject).equals(newName)) {
            boolean nameNotChanged = true;
            if (jmiObject instanceof JavaClass) {
                Object comp = jmiObject.refImmediateComposite();
                if (comp instanceof Resource && isResourceClass((Resource)comp, jmiObject)) {
                    String dobjName = JavaMetamodel.getManager().getDataObject((Resource)comp).getName();
                    nameNotChanged = dobjName.equals(newName);
                }
            }
            if (nameNotChanged)
                return createProblem(result, true, getString("ERR_NameNotChanged"));
        } 

        if (!Utilities.isJavaIdentifier(newName)) {
            String s = (jmiObject instanceof JavaPackage)? getString("ERR_InvalidPackage"):getString("ERR_InvalidIdentifier"); //NOI18N
            String msg = new MessageFormat(s).format(
                new Object[] {newName}
            );
            result = createProblem(result, true, msg);
            return result;
        }

        if (jmiObject instanceof JavaPackage) {
            if (doCheckName) {
                JavaPackage pckg = (JavaPackage) jmiObject;
                String fullName = pckg.getName();
                String name;
                int last = fullName.lastIndexOf('.');
                if (last > 0) {
                    name = fullName.substring(0,last+1) + newName;
                } else {
                    name = newName;
                }
                if (originalFolder != null) {
                    ClassPath projectClassPath = ClassPath.getClassPath(originalFolder, ClassPath.SOURCE);
                    if (projectClassPath.findResource(name.replace('.','/'))!=null) {
                        String msg = new MessageFormat(getString("ERR_PackageExists")).format(
                            new Object[] {newName}
                        );
                        result = createProblem(result, true, msg);
                    }
                }
            }

            this.newName = newName;
            return result;
        }

        if (jmiObject instanceof JavaClass || jmiObject instanceof Resource) {
            if (doCheckName) {
                Object ref;
                if (jmiObject instanceof Resource) {
                    ref = jmiObject;
                } else {
                    ref = jmiObject.refImmediateComposite();
                    String pkgname = ((NamedElement) jmiObject).getName();
                    int i = pkgname.indexOf('.');
                    if (i>=0)
                        pkgname = pkgname.substring(0,i);
                    else 
                        pkgname = "";
                       
                    String fqn = "".equals(pkgname) ? newName : pkgname + '.' + newName;
                    FileObject fo = JavaModel.getFileObject(((Element)ref).getResource());
                    ClassPath cp = ClassPath.getClassPath(fo, ClassPath.SOURCE);
                    JavaClass clazz = ClassIndex.getClassByFqn( fqn, cp);
                    if (clazz != null && !jmiObject.equals(clazz)) {
                        String msg = new MessageFormat(getString("ERR_ClassClash")).format(
                                new Object[] {newName, pkgname}
                        );
                        return createProblem(result, true, msg);
                    }
                }
                if (ref instanceof Resource && !JavaMetamodel.getManager().getDataObject((Resource) ref).getName().equals(newName)) {
                    DataObject dobj = JavaMetamodel.getManager().getDataObject((Resource)ref);
                    FileObject primFile = dobj.getPrimaryFile();
                    FileObject folder = primFile.getParent();
                    FileObject[] children = folder.getChildren();
                    for (int x = 0; x < children.length; x++) {
                        if (children[x] != primFile && !children[x].isVirtual() && children[x].getName().equals(newName) && "java".equals(children[x].getExt())) { //NOI18N
                            String msg = new MessageFormat (getString ("ERR_ClassClash")).format (
                                new Object[] {newName, ((Resource) ref).getPackageName()}
                            );
                            result = createProblem(result, true, msg);
                            break;
                        }
                    } // for
                } // if
            } // if
        }

        if (jmiObject instanceof Feature) {
            String msg = clashes((Feature) jmiObject, newName);
            if (msg != null) {
                result = createProblem(result, true, msg);
                return result;
            }
        }
        if (jmiObject instanceof Variable) {
            String msg = variableClashes(newName,JavaModelUtil.getDeclaringFeature((Variable) jmiObject));
            if (msg != null) {
                result = createProblem(result, true, msg);
                return result;
            }
        }
        return result;
    }
    
    private Problem setParameters(String newName) {
        int steps = 0;
        if (overriddenByMethods != null)
            steps += overriddenByMethods.size();
        if (overridesMethods != null)
            steps += overridesMethods.size();
        
        fireProgressListenerStart(refactoring.PARAMETERS_CHECK, 8 + 3*steps);
        
        try {
            fireProgressListenerStep();
            Problem result = checkParameters(newName);
            if (result != null && result.isFatal())
                return result;
            fireProgressListenerStep();
            String msg;
            if (jmiObject instanceof Method) {
                Method m = (Method) jmiObject;
                result = checkMethodForOverriding(m, newName, result);
                for (Iterator iter = overridesMethods.iterator(); iter.hasNext();) {
                    m = (Method) iter.next();
                    msg = clashes(m, newName);
                    if (msg != null) {
                        result = createProblem(result, true, msg);
                    }
                    result = checkMethodForOverriding(m, newName, result);
                }
                for (Iterator iter = overriddenByMethods.iterator(); iter.hasNext();) {
                    m = (Method) iter.next();
                    msg = clashes(m, newName);
                    if (msg != null) {
                        result = createProblem(result, true, msg);
                    }
                    result = checkMethodForOverriding(m, newName, result);
                }
                fireProgressListenerStep();
                fireProgressListenerStep();
            } else if (jmiObject instanceof Field) {
                Field f = (Field) jmiObject;
                fireProgressListenerStep();
                fireProgressListenerStep();
                Field hiddenField = hides(f, newName);
                fireProgressListenerStep();
                fireProgressListenerStep();
                fireProgressListenerStep();
                if (hiddenField != null) {
                    msg = new MessageFormat(getString("ERR_WillHide")).format(
                    new Object[] {getDefClassName(hiddenField.getDeclaringClass())}
                    );
                    result = createProblem(result, false, msg);
                }
            }
            
            if (result == null || !result.isFatal()) {
                this.newName = newName; // [PENDING] ???
            }
            return result;
        } finally {
            fireProgressListenerStop();
        }
    }
    
    private Problem checkMethodForOverriding(Method m, String newName, Problem problem) {
        List argTypes = getParamTypes(m);
        fireProgressListenerStep();
        problem = willBeOverridden(m, newName, argTypes, problem);
        fireProgressListenerStep();        
        problem = willOverride(m, newName, argTypes, problem);
        fireProgressListenerStep();
        return problem;
    }
    
    public Problem prepare(RefactoringElementsBag elements) {
        varNames = null;
        CommentRenameFinder docFinder=null;
        
        if (newName == null) {
            return new Problem(true, getString("ERR_NameNotSet"));
        }
        
        int steps = 9;
        if (overriddenByMethods != null)
            steps += overriddenByMethods.size();
        if (overridesMethods != null)
            steps += overridesMethods.size();
        
        JavaMetamodel.getManager().getProgressSupport().addProgressListener(this);

        
        //fireProgressListenerStart(rename.PREPARE, steps);
        
        try {
            if (refactoring.isSearchInComments()) {
                docFinder=new CommentRenameFinder((Element)jmiObject,newName);
                elements.addAll(refactoring, docFinder.searchCommentsInResource(((Element)jmiObject).getResource()));
            }
            if (jmiObject instanceof JavaPackage) {
                //fireProgressListenerStep();
                
                referencesIterator = ((NamedElement) jmiObject).getReferences().iterator();
                while (referencesIterator.hasNext()) {
                    Element element = (Element) referencesIterator.next();
                    
                    if (cancelRequest) {
                        return null;
                    }
                    if (refactoring.isSearchInComments()) {
                        elements.addAll(refactoring, docFinder.searchCommentsInResource(element.getResource()));
                    }
                    elements.add(refactoring, new RenamePackageElement(jmiObject, element, newName));
                }
                DataFolder folder = originalFolder!=null ? DataFolder.findFolder(originalFolder) : getFolder(((JavaPackage)jmiObject).getName());
                if (folder != null) {
                    elements.add(refactoring, new RenameDataFolder(folder, newName));
                }
                return null;
            } else {
                //fireProgressListenerStep();
                addElementsForJmiObject(elements, jmiObject, docFinder);
                
                
                if (overridesMethods != null) {
                    for (Iterator iter = overridesMethods.iterator(); iter.hasNext();) {
                        if (cancelRequest) {
                            return null;
                        }
                        //fireProgressListenerStep();
                        Method m = (Method) iter.next();
                        if (m.getResource().getName().endsWith(".class")) {
                            return resourceNotAvailable(m);
                        }
                        elements.add(refactoring, new RenameDOElement(m));
                        //addElementsForJmiObject(elements, (RefObject) iter.next());
                    }
                }
                if (overriddenByMethods != null) {
                    for (Iterator iter = overriddenByMethods.iterator(); iter.hasNext();) {
                        if (cancelRequest) {
                            return null;
                        }
                        Method m = (Method) iter.next();
                        if (m.getResource().getName().endsWith(".class")) {
                            return resourceNotAvailable(m);
                        }
                        //fireProgressListenerStep();
                        elements.add(refactoring, new RenameDOElement(m));
                        //addElementsForJmiObject(elements, (RefObject) iter.next());
                    }
                }
                return null;
            }
        } finally {
            referencesIterator = null;
            JavaMetamodel.getManager().getProgressSupport().removeProgressListener(this);
        }
    }
    
    private static Problem resourceNotAvailable(Method m) {
        String resourceName = Utilities.replaceString(m.getResource().getName(), ".class", ".java"); //NOI18N
        return new Problem(true, new MessageFormat(getString("ERR_ResourceUnavailable")).format (new Object[] {m.getName(), resourceName}));
    }
    
    private void addElementsForJmiObject(RefactoringElementsBag elements, RefObject refObject, CommentRenameFinder docFinder) {
        elements.add(refactoring, new RenameDOElement(refObject));
        if (refObject instanceof Method) {
            referencesIterator = ((MethodImpl) refObject).findDependencies(true, true, false).iterator();
        } else {
            referencesIterator = ((NamedElement) refObject).getReferences().iterator();
        }
        while (referencesIterator.hasNext()) {
            Element element = (Element) referencesIterator.next();

            if (cancelRequest) {
                return;
            }
            if (refactoring.isSearchInComments()) {
                elements.addAll(refactoring, docFinder.searchCommentsInResource(element.getResource()));
            }
            String name = newName;
            if (jmiObject instanceof Field) {
                Feature f = JavaModelUtil.getDeclaringFeature(element);
                if (((VariableAccess)element).getParentClass()==null && variableClashes(newName,f)!=null) {
                    ClassDefinition decl = ((Field)jmiObject).getDeclaringClass() ;
                    if (f.getDeclaringClass().equals(decl)) {
                        name = "this." + newName; //NOI18N
                    } else {
                        if (decl instanceof NamedElement)
                            name = ((NamedElement) decl).getName() + ".this." + newName; //NOI18N
                    }
                }
            }
            elements.add(refactoring, new RenameUsageElement(refObject, element, name));
        }
    }
    
    private DataFolder getFolder(String name) {
        FileObject fo = RefactoringClassPathImplementation.getDefault().findResource(name.replace('.','/'));
        if (fo == null)
            return null;
        return DataFolder.findFolder(fo);
    }
    
    private Collection overriddenBy(Method method, String name, List argTypes) {
        if (!CheckUtils.isVirtual(method))
            return Collections.EMPTY_LIST;

        return ((MethodImpl) method).findDependencies(false, false, true);
    }
    
    private Collection overrides(Method method, String name, List argTypes, ClassDefinition[] cls_par) {
        if (!CheckUtils.isVirtual(method))
            return Collections.EMPTY_LIST;

        ClassDefinition javaClass = method.getDeclaringClass ();
        LinkedList supertypes = new LinkedList ();
        Collection result = new HashSet();
        Method last = null;

        supertypes.addAll (javaClass.getInterfaces());
        ClassDefinition jc = javaClass.getSuperClass ();
        if (jc != null)
            supertypes.add (jc);
        while (supertypes.size () > 0) {
            jc = (ClassDefinition) supertypes.removeFirst ();
            if (jc instanceof UnresolvedClass) {
                continue;
            }
            Method m = jc.getMethod (name, argTypes, false);
            if ((m != null) && CheckUtils.isVirtual(m)) {
                result.add(m);
                last = m;                
            }
            supertypes.addAll (jc.getInterfaces ());
            jc = jc.getSuperClass ();
            if (jc != null) {
                supertypes.add (jc);
            }
        }
        
        if (last != null) {
            ClassDefinition cd = last.getDeclaringClass();
            ClassDefinition implementor = findDifferentSubtype((JavaClass) cd, name, argTypes, javaClass);
            if (implementor != null) {
                cls_par[0] = cd;
                cls_par[1] = implementor;
            }
        }
        return result;
    }        
    
    private ClassDefinition findDifferentSubtype(JavaClass baseClass, String name, List argTypes, ClassDefinition subtype) {
        Set supertypes = new HashSet();
        LinkedList subtypes = new LinkedList();
        ClassDefinition jc = baseClass;
        
        collectSupertypes(supertypes, subtype);
        addSubtypes(jc, subtypes);
        while (subtypes.size() > 0) {
            jc = (ClassDefinition) subtypes.removeFirst();
            if (jc instanceof UnresolvedClass) {
                continue;
            }
            if (supertypes.contains(jc)) {
                continue;
            } else if (jc.getMethod(name, argTypes, false) != null) {
                return jc;
            }
            addSubtypes(jc, subtypes);
        }
        return null;
    }

    private void collectSupertypes(Set supertypes, ClassDefinition jc) {
        if (jc == null)
            return;
        supertypes.add(jc);
        collectSupertypes(supertypes, jc.getSuperClass());
        for (Iterator iter = jc.getInterfaces().iterator(); iter.hasNext();) {
            collectSupertypes(supertypes, (ClassDefinition)iter.next());
        }
    }
    
    private boolean isStatic(Feature feature) {
        return (feature.getModifiers () & Modifier.STATIC) > 0;
    }
    
    private int getAccessLevel(Feature f) {
        int mod = f.getModifiers();
        if ((mod & Modifier.PUBLIC) > 0)
            return 3;
        if ((mod & Modifier.PROTECTED) > 0)
            return 2;
        if ((mod & Modifier.PRIVATE) > 0)
            return 0;
        return 1;
    }
    
    private Method isOverridden(Method method, String name, List argTypes) {
        if (!CheckUtils.isVirtual(method))
            return null;

        ClassDefinition jc = method.getDeclaringClass();
        LinkedList subtypes = new LinkedList();
        addSubtypes(jc, subtypes);
        while (subtypes.size() > 0) {
            jc = (ClassDefinition) subtypes.removeFirst();
            if (jc instanceof UnresolvedClass) {
                continue;
            }
            Method m = jc.getMethod(name, argTypes, false);
            if ((m != null) && CheckUtils.isVirtual(m))
                return m;
            addSubtypes(jc, subtypes);
        }
        return null;
    }

    private Problem willBeOverridden(Method method, String name, List argTypes, Problem problem) {
        int accessLevel = getAccessLevel(method);
        if (accessLevel == 0)
            return null;
                
        boolean isStatic = isStatic(method);
        boolean isFinal = (method.getModifiers() & Modifier.FINAL) > 0;
        Method temp = null;
        ClassDefinition jc = method.getDeclaringClass();
        LinkedList subtypes = new LinkedList();
        addSubtypes(jc, subtypes);
        while (subtypes.size() > 0) {
            jc = (ClassDefinition) subtypes.removeFirst();
            if (jc instanceof UnresolvedClass) {
                continue;
            }
            Method m = jc.getMethod(name, argTypes, false);
            if (m != null) {
                if (temp == null)
                    temp = m;
                if (isFinal) {
                    String msg = new MessageFormat(getString("ERR_WillBeOverridden_final")).format(
                        new Object[] {
                            method.getName(),
                            getDefClassName(method.getDeclaringClass()),
                            m.getName(),
                            getDefClassName(m.getDeclaringClass())
                        }
                    );
                    return createProblem(problem, true, msg);
                }
                if (getAccessLevel(m) < accessLevel) {
                    String msg = new MessageFormat(getString("ERR_WillBeOverridden_access")).format(
                        new Object[] {
                            method.getName(),
                            getDefClassName(method.getDeclaringClass()),
                            m.getName(),
                            getDefClassName(m.getDeclaringClass())
                        }
                    );
                    return createProblem(problem, true, msg);
                }
                if (isStatic != isStatic(m)) {
                    String msg = new MessageFormat(getString("ERR_WillBeOverridden_static")).format(
                        new Object[] {
                            isStatic ? getString("LBL_static") : getString("LBL_instance"),
                            method.getName(),
                            getDefClassName(method.getDeclaringClass()),
                            isStatic(m) ? getString("LBL_static") : getString("LBL_instance"),
                            m.getName(),
                            getDefClassName(m.getDeclaringClass())
                        }
                    );
                    return createProblem(problem, true, msg);
                }
            } else {
                addSubtypes(jc, subtypes);
            }
        }
        if (temp != null) {
            String msg = new MessageFormat(getString("ERR_WillBeOverridden")).format(
                new Object[] {
                    method.getName(),
                    getDefClassName(method.getDeclaringClass()),
                    temp.getName(),
                    getDefClassName(temp.getDeclaringClass())
                }
            );
            return createProblem(problem, false, msg);
        } else {
            return problem;
        }
    }
    
    private Problem willOverride(Method method, String name, List argTypes, Problem problem) {
        int accessLevel = getAccessLevel(method);
        boolean isStatic = isStatic(method);        
        Method temp = null;
        ClassDefinition jc = method.getDeclaringClass ();
        LinkedList supertypes = new LinkedList ();

        supertypes.addAll (jc.getInterfaces());
        jc = jc.getSuperClass ();
        if (jc != null)
            supertypes.add (jc);
        while (supertypes.size () > 0) {
            jc = (ClassDefinition) supertypes.removeFirst ();
            if (jc instanceof UnresolvedClass) {
                continue;
            }
            Method m = jc.getMethod (name, argTypes, false);
            if (m != null) {
                if (temp == null)
                    temp = m;
                if ((m.getModifiers() & Modifier.FINAL) > 0) {
                    String msg = new MessageFormat(getString("ERR_WillOverride_final")).format(
                        new Object[] {
                            method.getName(),
                            getDefClassName(method.getDeclaringClass()),
                            m.getName(),
                            getDefClassName(m.getDeclaringClass())
                        }
                    );
                    return createProblem(problem, true, msg);
                }
                if (getAccessLevel(m) > accessLevel) {
                    String msg = new MessageFormat(getString("ERR_WillOverride_access")).format(
                        new Object[] {
                            method.getName(),
                            getDefClassName(method.getDeclaringClass()),
                            m.getName(),
                            getDefClassName(m.getDeclaringClass())
                        }
                    );
                    return createProblem(problem, true, msg);
                }
                if (isStatic != isStatic(m)) {
                    String msg = new MessageFormat(getString("ERR_WillOverride_static")).format(
                        new Object[] {
                            isStatic ? getString("LBL_static") : getString("LBL_instance"),
                            method.getName(),
                            getDefClassName(method.getDeclaringClass()),
                            isStatic(m) ? getString("LBL_static") : getString("LBL_instance"),
                            m.getName(),
                            getDefClassName(m.getDeclaringClass())
                        }
                    );
                    return createProblem(problem, true, msg);
                }
            } else {
                supertypes.addAll (jc.getInterfaces ());
                jc = jc.getSuperClass ();
                if (jc != null)
                    supertypes.add (jc);
            }
        } // while
        if (temp != null) {
            String msg = new MessageFormat(getString("ERR_WillOverride")).format(
                new Object[] {
                    method.getName(),
                    getDefClassName(method.getDeclaringClass()),
                    temp.getName(),
                    getDefClassName(temp.getDeclaringClass())
                }
            );
            return createProblem(problem, false, msg);
        } else {
            return problem;
        }
    }
    
    private Method overrides(Method method, String name, List argTypes, boolean findFinal) {
        Method res = null;
        if (!CheckUtils.isVirtual(method))
            return null;

        ClassDefinition jc = method.getDeclaringClass ();
        LinkedList supertypes = new LinkedList ();

        supertypes.addAll (jc.getInterfaces());
        jc = jc.getSuperClass ();
        if (jc != null)
            supertypes.add (jc);
        while (supertypes.size () > 0) {
            jc = (ClassDefinition) supertypes.removeFirst ();
            if (jc instanceof UnresolvedClass) {
                continue;
            }
            Method m = jc.getMethod (name, argTypes, false);
            if ((m != null) && CheckUtils.isVirtual(m)) {
                if ((m.getModifiers () & Modifier.FINAL) > 0) {
                    res = m;
                    break;
                } else if (res == null) {
                    res = m;
                    if (!findFinal)
                        break;
                }
            }
            supertypes.addAll (jc.getInterfaces ());
            jc = jc.getSuperClass ();
            if (jc != null)
                supertypes.add (jc);
        }        
        return res;
    }

    private Field hides (Field field, String name) {
        if (!CheckUtils.isVirtual(field))
            return null;

        ClassDefinition jc = field.getDeclaringClass ();
        jc = jc.getSuperClass ();
        while (jc != null) {
            Field f = jc.getField (name, false);
            if ((f != null) && CheckUtils.isVirtual(f))
                return f;
            jc = jc.getSuperClass ();
        }
        return null;
    }

    private String variableClashes(String newName, Feature scope) {
        if (varNames==null)
            varNames=CheckUtils.getAllVariableNames(scope);
        if (varNames.contains(newName)) {
                return new MessageFormat (getString ("ERR_LocVariableClash")).format (
                    new Object[] {newName}
                );
        }
        return null;
    }
    
    private String clashes(Feature feature, String newName) {
        ClassDefinition dc = feature.getDeclaringClass ();
        if (feature instanceof TypeParameter) {
            // TODO: any check?
        } else if (feature instanceof JavaClass) {
            if (dc != null) {
                String result = checkInnersForClash(newName, dc);
                if (result != null)
                    return result;
            } else {
                Element composite = (Element) feature.refImmediateComposite();
                if (composite instanceof Resource) {
                    Resource resource = (Resource)composite;
                    DataObject dobj = JavaMetamodel.getManager().getDataObject(resource);
                    FileObject primFile = dobj.getPrimaryFile();
                    FileObject folder = primFile.getParent();
                    FileObject[] children = folder.getChildren();
                    for (int x = 0; x < children.length; x++) {
                        if (children[x] != primFile && !children[x].isVirtual() && children[x].getName().equals(newName) && "java".equals(children[x].getExt())) { //NOI18N
                            return new MessageFormat(getString("ERR_ClassClash")).format(
                                    new Object[] {newName, resource.getPackageName()}
                            );
                        }
                    }
                }
            }
        } else if (feature instanceof Method) {
            List params = getParamTypes ((Method) feature);
            if (dc.getMethod(newName, params, false) != null) {
                return new MessageFormat (getString ("ERR_MethodClash")).format (
                    new Object[] {newName, getDefClassName(dc)}
                );
            } // if
        } else if (feature instanceof Field) {
            if (dc.getField(newName, false) != null) {
                return new MessageFormat (getString ("ERR_FieldClash")).format (
                    new Object[] {newName, getDefClassName(dc)}
                );
            } // if
        }
        return null;
    }

    private String checkInnersForClash(final String newName, final ClassDefinition dc) {
        Iterator iter = dc.getFeatures ().iterator ();
        while (iter.hasNext ()) {
            Object obj = iter.next();
            if (!(obj instanceof JavaClass))
                continue;
            JavaClass nestedClass = (JavaClass) obj;
            if (nestedClass.getSimpleName ().equals (newName)) {
                return new MessageFormat (getString ("ERR_InnerClassClash")).format (
                    new Object[] {newName, getDefClassName(dc)}
                );
            }
        } // while
        return null;
    }
    
    String getElementName(NamedElement elem) {
        if (elem instanceof JavaClass) {
            return ((JavaClass) elem).getSimpleName();
        } else {
            return elem.getName();
        }
    }
    
    String getDefClassName(ClassDefinition cd) {
        if (cd instanceof JavaClass) {
            return ((JavaClass) cd).getName();
        } else {
            return "";
        }
    }
    
    private List getParamTypes(Method method) {
        List types = new LinkedList ();
        Iterator iter = method.getParameters ().iterator ();
        while (iter.hasNext ())
            types.add (getRealType(((Parameter) iter.next ()).getType ()));
        return types;
    }
    
    private static Type getRealType(Type type) {
        if (type instanceof ParameterizedType) {
            return ((ParameterizedType) type).getDefinition();
        }
        return type;
    }
    
    /**
     * Tests, if the renamed object should cause resource rename. Checks
     * if the object is java class. Then it checks for resource. If the
     * resource exists and resource name is the same as the name of class
     * is the same as the name of resource, it returns true. In all other
     * cases it returns false.
     *
     * @return  true, if the renamed object should cause resource rename
     */
    static boolean isResourceClass(Resource res, RefObject refObject) { //todo (#pf): try to find out better name for this method.
        if (res == null || !(refObject instanceof JavaClass))
            return false;
        int classCount = 0;
        for (Iterator iter = res.getClassifiers().iterator(); iter.hasNext(); ) {
            if (iter.next() instanceof JavaClass) {
                classCount++;
            }
        }
        if (classCount == 1) {
            return true;
        }
        String relativeResName = res.getName();
        String javaClassName = ((JavaClass) refObject).getSimpleName();
        int begin = relativeResName.lastIndexOf('/') + 1;
        if (begin < 0) begin = 0;
        int end = relativeResName.lastIndexOf('.');
        if (javaClassName.equals(relativeResName.substring(begin, end)))
            return true;
        else
            return false;
    }

    private void addSubtypes(ClassDefinition cd, List list) {
        if (!(cd instanceof JavaClass)) {
            return;
        }
        JavaClass jc = (JavaClass) cd;
        Collection subtypes = null;
        if (jc.isInterface()) {            
            subtypes = jc.getImplementors();
        } else {
            subtypes = jc.getSubClasses();            
        }
        list.addAll(subtypes);        
    }
    
    private static final String getString(String key) {
        return NbBundle.getMessage(RenameRefactoring.class, key);
    }
    
    public void start(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStart(event.getOperationType(), event.getCount());
    }
    
    public void step(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStep();
    }
    
    public void stop(org.netbeans.modules.javacore.internalapi.ProgressEvent event) {
        fireProgressListenerStop();
    }

//    public void setClassPath() {
//        if (NbAbstractRefactoring.isElementAvail((Element) jmiObject) == null && (jmiObject instanceof Method)) {
//            Collection c = ((MethodImpl) jmiObject).getOverridenMethods();
//            if (!c.isEmpty()) {
//                setClassPath(c);
//                return;
//            }
//        }
//        setClassPath((Element) jmiObject);
//    }

    // RenameDOElement ..........................................................
    private class RenameDOElement extends SimpleRefactoringElementImpl implements ExternalChange {
        private final String text;
        private PositionBounds bounds = null;
        private String oldName = null;
        private Resource res = null;
        private RefObject refObject;
        
        public RenameDOElement(RefObject refObject) {
            this.refObject = refObject;
            Object o;
            if (refObject instanceof Resource) {
                o = refObject;
            } else {
                o = refObject.refImmediateComposite();
            }
            if (o instanceof Resource) res = (Resource) o;
            String bundleName = null;
            if (refObject instanceof Resource) {
                bundleName = "LBL_RenameClassDO";          //NOI18N
            } else if (refObject instanceof JavaEnum) {
                bundleName = "LBL_RenameEnum"; // NOI18N
            } else if (refObject instanceof AnnotationType) {
                bundleName = "LBL_RenameAnnotationType"; // NOI18N
            } else if (refObject instanceof TypeParameter) {
                bundleName = "LBL_RenameTypeParameter"; // NOI18N
            } else if (refObject instanceof JavaClass) {
                bundleName = isResourceClass() ? "LBL_RenameClassDO" : "LBL_RenameClass";          //NOI18N
            } else if (refObject instanceof Method) {
                bundleName = "LBL_RenameMethod";          //NOI18N
            } else if (refObject instanceof Field) {
                bundleName ="LBL_RenameField";          //NOI18N
            } else if (refObject instanceof Parameter) {
                bundleName = "LBL_RenameParameter"; // NOI18N
            } else if (refObject instanceof LocalVariable) {
                bundleName = "LBL_RenameLocVariable"; // NOI18N
            } else if (refObject instanceof Attribute) {
                bundleName = "LBL_RenameAttribute"; // NOI18N
            } else {
                assert false:"Invalid type "+refObject.getClass(); // NOI18N
            }
            text = MessageFormat.format(NbBundle.getMessage(RenameRefactoring.class, bundleName), new Object[] {newName});
        }
         
        private boolean isResourceClass() {
            return RenameRefactoringPlugin.isResourceClass(res, refObject);
        } 
        
        public String getDisplayText() {
            return text;
        }
        
        public Element getJavaElement() {
            return (Element) refObject;
        }
        
        public PositionBounds getPosition() {
            if (bounds == null) {
                if (!(refObject instanceof Resource)) {
                    bounds = JavaMetamodel.getManager().getElementPosition((Element)refObject);
                }
            }
            return bounds;
        }
        
        public String getText() {
            return getDisplayText();
        }
        
        public void performChange() {
            if (refObject instanceof Resource) {
                 JavaMetamodel.getManager().registerExtChange(this);
            } else {
                NamedElement obj = (NamedElement) refObject;
                if (obj instanceof JavaClass) {
                    oldName = ((JavaClass) obj).getSimpleName();
                    if (isResourceClass()) {
                        JavaMetamodel.getManager().registerExtChange(this);
                    }
                    ((JavaClass) obj).setSimpleName(newName);
                } else {
                    oldName = obj.getName();
                    obj.setName(newName);
                }
            }
        }
        
        private void doRename() {
            try {
                DataObject dobj = JavaMetamodel.getManager().getDataObject(res);
                oldName = dobj.getName();
                dobj.rename(newName);  
            } catch (DataObjectNotFoundException e) {
                throw (RuntimeException) new RuntimeException().initCause(e);
            } catch (IOException e) {
                throw (RuntimeException) new RuntimeException().initCause(e);
            }
        }
        
        public void performExternalChange () {
            doRename();
        }
        
        public void undoExternalChange() {
           if (oldName == null) return;
            String temp = newName;
            newName = oldName;
            oldName = temp; 
            doRename();
            newName = temp; 
        }

        public FileObject getParentFile() {
            return null;
        }
    } // RenameDOElement

    // RenamePackageElement ..........................................................
    private class RenamePackageElement extends RenameUsageElement {
        private String oldName = null;

        public RenamePackageElement(RefObject jmiObject, Element feature, String newName) {
                super(jmiObject, feature, newName);
        }
        
        public void performChange() {
            MultipartId mpi = (MultipartId) feature;
            oldName = mpi.getName();
            mpi.setName(newName);
        }
        
    } // RenamePackageElement
    
    // RenameDataFolder ..........................................................
    private class RenameDataFolder extends SimpleRefactoringElementImpl implements RefactoringElementImplementation, ExternalChange {
        private final String text;
        private PositionBounds bounds;
        private String oldName, newName;
        private DataFolder folder;
        
        public RenameDataFolder(DataFolder folder, String name) {
            newName = name; 
            this.folder = folder;
            text = MessageFormat.format(NbBundle.getMessage(RenameRefactoring.class, "LBL_RenameFolder"), new Object[] {folder.getName(), newName});
        }
        
        public String getDisplayText() {
            return text;
        }
        
        public Element getJavaElement() {
            return (Element) jmiObject;
        }
        
        public PositionBounds getPosition() {
            if (bounds == null) {
                bounds = JavaMetamodel.getManager().getElementPosition((Element)jmiObject);
            }
            return bounds;
        }
        
        
        public String getText() {
            return getDisplayText();
        }
        
        public void performChange() {
             JavaMetamodel.getManager().registerExtChange(this);
        }
        
        private void doRename() {
            oldName = folder.getName();
            try {
                folder.rename(newName);
            } catch (java.io.IOException e) {
                throw (RuntimeException) new RuntimeException().initCause(e);
            }
        }
        
        public void performExternalChange () {
            doRename();
        }
        
        public void undoExternalChange() {
            if (oldName == null) return;
            String temp = newName;
            newName = oldName;
            oldName = temp; 
            doRename();
            newName = temp; 
        }

        public FileObject getParentFile() {
            return null;
        }
        
    } // RenameDataFolder
}
