/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */

package org.netbeans.modules.refactoring.plugins;

import java.lang.reflect.Modifier;
import java.text.MessageFormat;
import java.util.*;
import org.netbeans.jmi.javamodel.*;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.javacore.internalapi.JavaModelUtil;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.ExtractMethodRefactoring;
import org.netbeans.modules.refactoring.api.ExtractMethodRefactoring.ParameterInfo;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.openide.filesystems.FileObject;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;
import org.openide.util.Utilities;

/**
 *
 * @author Tomas Hurka
 */
public class ExtractMethodRefactoringPlugin extends JavaRefactoringPlugin {
    private ExtractMethodRefactoring refactoring;
    private Statement firstStatement,lastStatement;
    private List selectedStatements;
    private Set writeVars,readVars,unusedDeclVars,declVars,newLocalVars,exceptions;
    private Variable returnedVariable;
    private Feature feature;
    private TypeReference returnType;
    private boolean retValDefinedInSelectedStmts;
    private boolean hasReturn;
    
    public ExtractMethodRefactoringPlugin(ExtractMethodRefactoring ref) {
        refactoring=ref;
        writeVars=new HashSet();
        readVars=new HashSet();
        unusedDeclVars=new HashSet();
        declVars=new HashSet();
        newLocalVars=new HashSet();
    }

    public Problem checkParameters() {
        Problem p=fastCheckParameters();
        if (p!=null)
            return p;
        List params=getParamTypes();
        ClassDefinition dc=getFeature().getDeclaringClass();
        String newName=refactoring.getName();
        if (dc.getMethod(newName, params, false) != null) {
            String errText=new MessageFormat(getString("ERR_MethodClash")).format (
                new Object[] {newName, getDefClassName(dc)}
            );
            return createProblem(null,true,errText);
        }
        return null;
    }

    public Problem fastCheckParameters() {
        Problem result;
        ParameterInfo[] pars;
        
        result=checkName(refactoring.getName());
        if (result != null)
            return result;
        pars=refactoring.getParamTable();
        for (int i=0;i<pars.length;i++) {
            result=checkName(pars[i].getName());
            if (result!=null)
                return result;
        }
        return null;
    }

    public Problem preCheck() {
        fireProgressListenerStart(AbstractRefactoring.PRE_CHECK, 8);
        try {
            Problem problem;
            
            if (refactoring.getStartOffset()<0 || refactoring.getEndOffset()<0) {
                return createProblem(null, true, getString("ERR_ExtractMethodWrongElement"));
            }
            fireProgressListenerStep();
            problem=checkStatements();
            if (problem!=null)
                return problem;
            fireProgressListenerStep();
            problem=checkJump();
            if (problem!=null)
                return problem;
            fireProgressListenerStep();
            computeAllVaribleAccess(selectedStatements);
            fireProgressListenerStep();
            problem=checkReturnValue();
            if (problem!=null)
                return problem;
            fireProgressListenerStep();
            problem=computeExceptions();
            if (problem!=null)
                return problem;
            fireProgressListenerStep();
            computeDuplicatedVars();
            fireProgressListenerStep();
            refactoring.setParamTable(getInputParameters());
            refactoring.setReturnType(getReturnType());
            refactoring.setStaticMod(Modifier.isStatic(getFeature().getModifiers()));
            fireProgressListenerStep();
            return null;
        } finally {
            fireProgressListenerStop();
        }
    }

    public Problem prepare(RefactoringElementsBag refactoringElements) {
        refactoringElements.add(refactoring,new ExtractMethodElement());
        return null;
    }
    
    private Problem checkStatements() {
        Resource rsc=refactoring.getResource();
        int start=refactoring.getStartOffset();
        int end=refactoring.getEndOffset();
        selectedStatements=JavaModelUtil.getSelectedStatements(rsc,start,end);
        
        if (selectedStatements==null) {
            return createProblem(null, true, getString("ERR_ExtractMethodWrongSelection"));                
        }
        return null;
    }

    private Problem checkJump() {
        Iterator sIt=selectedStatements.iterator();
        Statement jumpStatement=null;
        
        while(sIt.hasNext()) {
            Statement jumpStmt=findJump((Statement)sIt.next(), false);
            
            if (jumpStmt==null)
                continue;
            if (jumpStatement==null) {
                jumpStatement=jumpStmt;
                continue;
            }
            if (!jumpStatement.refClass().equals(jumpStmt.refClass())) {
                return createProblem(null, true, getString("ERR_ExtractMethodContainsFixBreakReturn"));  
            }
            jumpStatement=jumpStmt;
        }
        if (jumpStatement==null)
            return null;
        if (!jumpStatement.equals(getLastStatement())) {
            return createProblem(null, true, getString("ERR_ExtractMethodContainsBreakOrReturn"));
        }
        if (jumpStatement instanceof ReturnStatement) {
            if (((ReturnStatement)jumpStatement).getExpression()!=null) {
                hasReturn=true;
                return null;
            }
        }
        selectedStatements.remove(selectedStatements.size()-1);
        return null;
    }
        
    private Boolean readsVariableAfterSelection(Variable var) {
        Element varScope=(Element)var.refImmediateComposite();
        if (varScope instanceof LocalVarDeclaration)
            varScope=(Element)varScope.refImmediateComposite();
        Statement last=getLastStatement();
        int lastOffset=last.getEndOffset();
        List children=varScope.getChildren();
        VariableAnalyzer ca=new VariableAnalyzer(var, lastOffset+1, varScope.getEndOffset());
        
        return ca.readsVariableInElement(children);
    }
    
    private Statement findJump(Statement st, boolean ignoreBreakCont) {
        Iterator chIt;

        if (st instanceof ReturnStatement) {
            return st;
        }
        if (!ignoreBreakCont && (st instanceof BreakStatement || st instanceof ContinueStatement)) {
            return st;
        }
        if (st instanceof SwitchStatement || st instanceof WhileStatement || st instanceof DoStatement || st instanceof ForStatement || st instanceof ForEachStatement)
            ignoreBreakCont=true;
        chIt=st.getChildren().iterator();
        while(chIt.hasNext()) {
            Element el=(Element)chIt.next();
            
            if (el instanceof Statement && !(el instanceof JavaClass)) {
                Statement jumpStmt=findJump((Statement)el, ignoreBreakCont);
                
                if (jumpStmt!=null)
                    return jumpStmt;
            }
        }
        return null;
    }
    
    ParameterInfo[] getInputParameters() {
        List pars=new ArrayList();
        Iterator it=readVars.iterator();
        
        while(it.hasNext()) {
            Variable v=(Variable)it.next();
            VariableAnalyzer ca=new VariableAnalyzer(v);
            
            if (Boolean.FALSE.equals(ca.readsVariableInElement(selectedStatements))) {
                newLocalVars.add(v);
            } else {
                pars.add(new ParameterInfo(v));
            }
        }
        it=writeVars.iterator();
        while(it.hasNext()) {
            Variable v=(Variable)it.next();
            VariableAnalyzer ca=new VariableAnalyzer(v);
            
            if (readVars.contains(v))
                continue;
            if (isInsideSelectedStatements(v))
                continue;
            if (Boolean.FALSE.equals(ca.readsVariableInElement(selectedStatements))) {
                newLocalVars.add(v);                
            } else {
                pars.add(new ParameterInfo(v));
            }
        }
        return (ParameterInfo[])pars.toArray(new ParameterInfo[pars.size()]);
    }
    
    private Problem checkReturnValue() {
        Iterator it;
        int lastOffset=getLastStatement().getEndOffset();
        Set retVar=new HashSet();
        
        checkLoop(retVar);
        it=writeVars.iterator();
        while(it.hasNext()) {
            Variable var=(Variable)it.next();
            
            if (Boolean.TRUE.equals(readsVariableAfterSelection(var))) {
                retVar.add(var);
            }
        }
        if (retVar.size()>1) {
            StringBuffer conflictNames=new StringBuffer();

            it=retVar.iterator();
            while(it.hasNext()) {
                Variable v=(Variable)it.next();
                
                conflictNames.append(v.getName()).append(' ');
            }
            String msg = new MessageFormat(getString("ERR_ExtractMethodTooMuchRetValues")).format(new Object[] {conflictNames}); //NOI18N
            return createProblem(null, true, msg);
        }
        if (hasReturn && !retVar.isEmpty()) {
            Variable v=(Variable)retVar.iterator().next();
            String conflictNames=v.getName();
            String msg = new MessageFormat(getString("ERR_ExtractMethodRetValueAndReturn")).format(new Object[] {conflictNames}); //NOI18N
            return createProblem(null, true, msg);
        }
        if (!retVar.isEmpty()) {
            returnedVariable=(Variable)retVar.iterator().next();
            retValDefinedInSelectedStmts=isInsideSelectedStatements(returnedVariable);
        }
        return null;
    }
    
    private void checkLoop(Set retVar) {
        boolean hasLoop=false;
        Element el=getFirstStatement();
        
        while(!(el instanceof ClassMember)) {
            el=(Element)el.refImmediateComposite();
            if (el instanceof WhileStatement || el instanceof ForStatement || el instanceof DoStatement) {
                hasLoop=true;
                break;
            }
        }
        if (!hasLoop)
            return;
        Iterator it=writeVars.iterator();
        while(it.hasNext()) {
            Variable var=(Variable)it.next();
            
            if (readVars.contains(var)) {
                retVar.add(var);
            }
        }
    }
    
    private Problem computeExceptions() {
        exceptions=JavaModelUtil.getExceptionsFromStatements(selectedStatements);
        return null;
    }
    
    private void computeAllVaribleAccess(List elements) {
        Iterator elIt=elements.iterator();
        while(elIt.hasNext()) {
            Object el=elIt.next();
            
            if (el instanceof VariableAccess) {
                VariableAccess acc=(VariableAccess)el;
                Variable variable=(Variable)acc.getElement();
                
                if (!(variable instanceof Field)) {
                    if (isInsideSelectedStatements(variable)) {
                        if (!selectedStatements.contains(variable))
                            continue;
                    }
                    unusedDeclVars.remove(acc);
                    if (acc.isWrite()) {
                        writeVars.add(variable);
                    } 
                    if (acc.isRead()) {
                        readVars.add(variable);
                    }
                }
            } else if (el instanceof MultipartId) {
                continue;
            } else if (el instanceof LocalVarDeclaration) {
                if (selectedStatements.contains(el)) {
                    LocalVarDeclaration varDecl=(LocalVarDeclaration)el;
                    Iterator varIt=varDecl.getVariables().iterator();
                    
                    while(varIt.hasNext()) {
                        LocalVariable locVar=(LocalVariable)varIt.next();
                        
                        if (locVar.getInitialValue()!=null) {
                            writeVars.add(locVar);
                        } else {
                            unusedDeclVars.add(locVar);
                        }
                        declVars.add(locVar);
                    }
                }
            }
            computeAllVaribleAccess(((Element)el).getChildren());
        }
    }
    
    private boolean isInsideSelectedStatements(Element el) {
        int elOffset=el.getStartOffset();
        if (elOffset>=getFirstStatement().getStartOffset() && elOffset<=getLastStatement().getEndOffset()) {
            return true;
        }
        return false;
    }
    
    private Problem checkName(String name) {
        if (!Utilities.isJavaIdentifier(name)) {
            String msg = new MessageFormat(getString("ERR_InvalidIdentifier")).format(new Object[] {name}); //NOI18N
            return createProblem(null, true, msg);
        }
        return null;
    }

    private static String getString(String key) {
        return NbBundle.getMessage(ExtractMethodRefactoring.class, key);
    }

    String getDefClassName(ClassDefinition cd) {
        if (cd instanceof JavaClass) {
            return ((JavaClass) cd).getName();
        } else {
            return "";
        }
    }

    private Statement getFirstStatement() {
        return (Statement)selectedStatements.get(0);
    }

    private Statement getLastStatement() {
        return (Statement)selectedStatements.get(selectedStatements.size()-1);
    }
    
    private Feature getFeature() {
        if (feature==null) {
            feature=JavaModelUtil.getDeclaringFeature(getFirstStatement());
        }
        return feature;
    }

    private List getParamTypes() {
        ParameterInfo[] pars=refactoring.getParamTable();
        List types=new ArrayList(pars.length);
        int i;
        
        for (i=0;i<pars.length;i++) {
            types.add(pars[i].getVariable().getType());
        }
        return types;
    }


    private TypeReference getReturnType() {
        if (returnType==null) {
            returnType=createReturnType();
        }
        return returnType;
    }
    
    private TypeReference createReturnType() {
        TypeReference ref;
        int dims;
        Feature f=getFeature();
        JavaModelPackage pck=(JavaModelPackage)f.refImmediatePackage();

        if (hasReturn) {
            Method m=(Method)f;
            ref=m.getTypeName();
            dims=m.getDimCount();
        } else if (returnedVariable!=null) {
            ref=returnedVariable.getTypeName();
            dims=returnedVariable.getDimCount();
        } else {
            return pck.getMultipartId().createMultipartId("void",null,null); // NOI18N
        }
        ref=(TypeReference)ref.duplicate();
        if (dims==0) {
            return ref;
        }
        if (ref instanceof ArrayReference) {
            ArrayReference arr=(ArrayReference)ref;

            arr.setDimCount(arr.getDimCount()+dims);
            return arr;
        }
        return pck.getArrayReference().createArrayReference(null, (MultipartId)ref, dims);
    }

    private void computeDuplicatedVars() {
        Iterator locVarIt=declVars.iterator();
        
        while(locVarIt.hasNext()) {
            LocalVariable loc=(LocalVariable)locVarIt.next();
            
            if (loc.equals(returnedVariable)) {
                locVarIt.remove();
            }
            else if (unusedDeclVars.contains(loc)) {
                locVarIt.remove();
            }
            else if (!isAccessedAfterSelection(loc)) {
                locVarIt.remove();                
            }
        }
    }

    private boolean isAccessedAfterSelection(LocalVariable loc) {
        Iterator usagesIt=loc.getReferences().iterator();
        
        while(usagesIt.hasNext()) {
            Element usage=(Element)usagesIt.next();
            
            if (!isInsideSelectedStatements(usage))
                return true;
        }
        return false;
    }
        
    private class VariableAnalyzer {
        private Variable var;
        private int startOffset,endOffset;
        
        private VariableAnalyzer(Variable v) {
            this(v, 0, Integer.MAX_VALUE);
        }
        
        private VariableAnalyzer(Variable v,int start,int end) {
            var=v;
            startOffset=start;
            endOffset=end;
        }
        
        private Boolean readsVariableInElement(Collection els) {
            Iterator chIt=els.iterator();

            while(chIt.hasNext()) {
                Boolean ret=readsVariableInElement((Element)chIt.next());

                if (ret!=null)
                    return ret;
            }
            return null;
        }

        private boolean isInRange(int offset) {
            return offset>=startOffset && offset<=endOffset;
        }
        
        private Boolean readsVariableInElement(Element el) {
            if (el==null || (!isInRange(el.getStartOffset()) && !isInRange(el.getEndOffset())))
                return null;
            if (el instanceof VariableAccess) {
                VariableAccess acc=(VariableAccess)el;
                Variable variable=(Variable)acc.getElement();

                if (var.equals(variable)) {
                    return Boolean.valueOf(acc.isRead());
                }
                return null;
            } else if (el instanceof MultipartId) {
                return null;
            } else if (el instanceof Assignment) {
                Assignment assign=(Assignment)el;
                Boolean ret=readsVariableInElement(assign.getRightSide());

                if (ret==null)
                    ret=readsVariableInElement(assign.getLeftSide());
                return ret;
            } else if (el instanceof IfStatement) {
                IfStatement ifSt=(IfStatement)el;
                Boolean ret=readsVariableInElement(ifSt.getExpression());
                Boolean thenPart;
                Boolean elsePart;

                if (ret!=null)
                    return ret;
                thenPart=readsVariableInElement(ifSt.getThenPart());
                elsePart=readsVariableInElement(ifSt.getElsePart());
                if (Boolean.TRUE.equals(thenPart) || Boolean.TRUE.equals(elsePart))
                    return Boolean.TRUE;
                if (Boolean.FALSE.equals(thenPart) && Boolean.FALSE.equals(elsePart))
                    return Boolean.FALSE;
                return null;
            } else if (el instanceof WhileStatement) {
                WhileStatement wst=(WhileStatement)el;
                Boolean ret=readsVariableInElement(wst.getExpression());
                Boolean body;

                if (ret!=null)
                    return ret;
                body=readsVariableInElement(wst.getBody());
                if (Boolean.TRUE.equals(body))
                    return body;
                return null;
            } else if (el instanceof ForStatement) {
                ForStatement forSt=(ForStatement)el;
                Boolean ret=readsVariableInElement(forSt.getInit());
                List bodyList;
                Statement body;

                if (ret!=null)
                    return ret;
                ret=readsVariableInElement(forSt.getExpression());
                if (ret!=null)
                    return ret;
                bodyList=new ArrayList(2);
                body=forSt.getBody();
                if (body!=null)
                    bodyList.add(body);
                bodyList.addAll(forSt.getSteps());
                ret=readsVariableInElement(bodyList);
                if (Boolean.TRUE.equals(ret))
                    return ret;
                return null;            
            } else if (el instanceof TryStatement) {
                TryStatement trySt=(TryStatement)el;
                Boolean ret=readsVariableInElement(trySt.getBody());
                Iterator catchIt;

                if (ret!=null)
                    return ret;
                catchIt=trySt.getCatches().iterator();
                while(catchIt.hasNext()) {
                    if (Boolean.TRUE.equals(readsVariableInElement((Catch)catchIt.next()))) {
                        return Boolean.TRUE;
                    }
                }
                return readsVariableInElement(trySt.getFinalizer());
            } else if (el instanceof SwitchStatement) {
                SwitchStatement switchSt=(SwitchStatement)el;
                Boolean ret=readsVariableInElement(switchSt.getExpression());
                Iterator caseIt;

                if (ret!=null)
                    return ret;
                caseIt=switchSt.getCases().iterator();
                while(caseIt.hasNext()) {
                    if (Boolean.TRUE.equals(readsVariableInElement((Case)caseIt.next()))) {
                        return Boolean.TRUE;
                    }
                }
                return null;
            } else {
                return readsVariableInElement(el.getChildren());
            }
        }
        
    }
    
    private class ExtractMethodElement extends SimpleRefactoringElementImpl {
        private Resource rsc;
        private JavaModelPackage pck;
        
        private ExtractMethodElement() {
            rsc=getFirstStatement().getResource();
            pck=(JavaModelPackage)rsc.refImmediatePackage();
        }
        
        public String getDisplayText() {
            return getText();
        }

        public Element getJavaElement() {
            return getFeature();
        }

        public FileObject getParentFile() {
            return JavaModel.getFileObject(getResource()); 
        }

        public PositionBounds getPosition() {
            PositionBounds st=JavaMetamodel.getManager().getElementPosition(getFirstStatement());
            PositionBounds end=JavaMetamodel.getManager().getElementPosition(getLastStatement());
            return new PositionBounds(st.getBegin(), end.getEnd());
        }

        public String getText() {
            String sourceText=getResource().getSourceText();
            
            return CheckUtils.htmlize(sourceText.substring(getFirstStatement().getStartOffset(),getLastStatement().getEndOffset()));
        }

        public void performChange() {
            Feature f=getFeature();
            String name=refactoring.getName();
            int mods=refactoring.getModifier() | (refactoring.isStaticMod()?Modifier.STATIC:0);
            List duplVars=createDuplicatedVars();
            Statement newStmt=createReplacementStatement();
            List statements=createStatements(newStmt);
            insertDuplicatedVars(duplVars, newStmt);
            StatementBlock body=pck.getStatementBlock().createStatementBlock(statements);
            List pars=createFormalParameters();
            TypeReference retType=getReturnType();
            List exNames=createExceptionNames(f);
            Method m=pck.getMethod().createMethod(name, null, mods, null, null, body, null, null, pars, exNames, retType, 0);
            List features=f.getDeclaringClass().getFeatures();
            int index=features.indexOf(f);
            
            features.add(index+1,m);
        }
                
        private List createExceptionNames(Feature scope) {
            List excs=new ArrayList();
            Iterator exIt=exceptions.iterator();
            
            while(exIt.hasNext()) {
               excs.add(JavaModelUtil.resolveImportsForClass(scope,(JavaClass)exIt.next()));
            }
            return excs;
        }
        
        private List createFormalParameters() {
            ParameterInfo[] parTable=refactoring.getParamTable();
            int i;
            ParameterClass parCls=pck.getParameter();
            List pars=new ArrayList();
            
            for (i=0;i<parTable.length;i++) {
                ParameterInfo pInfo=parTable[i];
                Variable var=pInfo.getVariable();
                int dimcount = var.getDimCount();
                if (var instanceof Parameter) {
                    if (((Parameter) var).isVarArg()) {
                        dimcount++;
                    }
                }
                TypeReference parType=(TypeReference)var.getTypeName().duplicate();
                Parameter p=parCls.createParameter(pInfo.getName(), null, !writeVars.contains(var), parType, dimcount, false);
                pars.add(p);
            }
            return pars;
        }
        
        private Statement createReplacementStatement() {
            List pars=createParameters();
            String name=refactoring.getName();
            MethodInvocation inv=pck.getMethodInvocation().createMethodInvocation(name, pars, null, false);
            
            if (hasReturn) {
                return pck.getReturnStatement().createReturnStatement(inv);
            } else if (returnedVariable!=null){
                if (retValDefinedInSelectedStmts) {
                    return createVarDecl(returnedVariable, inv);
                } else {
                    Assignment assign=pck.getAssignment().createAssignment(createVarAccess(returnedVariable), OperatorEnum.ASSIGN ,inv);
                    
                    return pck.getExpressionStatement().createExpressionStatement(assign);
                }
            }
            return pck.getExpressionStatement().createExpressionStatement(inv);
        }
        
        private List createParameters() {
            ParameterInfo[] parTable=refactoring.getParamTable();
            VariableAccessClass parCls=pck.getVariableAccess();
            List pars=new ArrayList();
            int i;
            
            for (i=0;i<parTable.length;i++) {
                pars.add(parCls.createVariableAccess(parTable[i].getName(), null, false));
            }
            return pars;            
        }
        
        private VariableAccess createVarAccess(Variable var) {
            return pck.getVariableAccess().createVariableAccess(returnedVariable.getName(), null, false);
        }
        
        private LocalVarDeclaration createVarDecl(Variable var,InitialValue initVal) {
            LocalVariable locVar=pck.getLocalVariable().createLocalVariable(var.getName(), null, false, null, var.getDimCount(), initVal, null);
            TypeReference typeName=(TypeReference)var.getTypeName().duplicate();
            
            return pck.getLocalVarDeclaration().createLocalVarDeclaration(false, typeName, Collections.singletonList(locVar));
        }
        
        private List createStatements(Statement newStmt) {
            List stmts=new ArrayList();
            Element parent=(Element)getFirstStatement().refImmediateComposite();
            Iterator it=newLocalVars.iterator();            
            
            while(it.hasNext()) {
                stmts.add(createVarDecl((Variable)it.next(),null));
            }
            it=selectedStatements.iterator();
            while(it.hasNext()) {
                Statement st=(Statement)it.next();
                
                if (!unusedDeclVars.contains(st)) {
                    parent.replaceChild(st, newStmt);
                    stmts.add(st);
                    newStmt=null;
                }
            }
            if (returnedVariable!=null) {
                stmts.add(pck.getReturnStatement().createReturnStatement(createVarAccess(returnedVariable)));
            }
            return stmts;
        }
                
        private Resource getResource() {
            return rsc;
        }

        private List createDuplicatedVars() {
            LocalVarDeclarationClass declClass=pck.getLocalVarDeclaration();
            LocalVariableClass varClass=pck.getLocalVariable();
            List vars=new ArrayList();
            Iterator it=declVars.iterator();
            
            while(it.hasNext()) {
                LocalVariable loc=(LocalVariable)it.next();                
                LocalVariable lvar=varClass.createLocalVariable(loc.getName(),null,false,null,loc.getDimCount(),null,null);
                TypeReference typeName=(TypeReference)loc.getTypeName().duplicate();

                vars.add(declClass.createLocalVarDeclaration(false,typeName,Collections.singletonList(lvar)));
            }
            return vars;
        }

        private void insertDuplicatedVars(List duplVars, Statement newStmt) {
            if (!duplVars.isEmpty()) {
                Object parent=newStmt.refImmediateComposite();

                if (parent instanceof StatementBlock) {
                    StatementBlock pblock=(StatementBlock)parent;
                    List stmts=pblock.getStatements();
                    int offset=stmts.indexOf(newStmt);

                    stmts.addAll(offset,duplVars);
                }
            }
        }
    }
}
