/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 * 
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 * 
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring.api;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import org.netbeans.jmi.javamodel.Field;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.MultipartId;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.ParameterizedType;
import org.netbeans.modules.refactoring.CheckUtils;
import org.netbeans.modules.refactoring.classpath.Util;
import org.netbeans.modules.refactoring.Utilities;

/** Pull Up Refactoring implementation class.
 *
 * @author Martin Matula
 */
public final class PullUpRefactoring extends AbstractRefactoring {
    private static final MemberInfo[] EMPTY_MEMBERS = new MemberInfo[0];

    // source type
    private final JavaClass sourceType;
    // parameters of the refactoring
    private JavaClass targetType;
    private MemberInfo[] members;
    
    // cached supertypes of the source type
    private JavaClass[] supertypes;
    
    /** Creates a new instance of PullUpRefactoring 
     * @param sourceType Type the members of which should be pulled up.
     */
    public PullUpRefactoring(JavaClass sourceType) {
        // check if the sourceType is a ParameterizedType - if so, unwrap it
        if (sourceType instanceof ParameterizedType) {
            this.sourceType = ((ParameterizedType) sourceType).getDefinition();
        } else {
            this.sourceType = sourceType;
        }
    }
    
    /** Method called by the refactoring framework - gives the refactoring implementation
     * a chance to narrow the scope of the refactoring.
     */
    protected void setClassPath() {
        // leave the complete classpath
        Util.setDefaultClassPath();
    }

    /** Returns the type the members of which should be pulled up
     * by this refactoring.
     * @return Source of the members to be pulled up.
     */
    public JavaClass getSourceType() {
        return sourceType;
    }

    // --- PARAMETERS ----------------------------------------------------------
    
    /** Returns target supertype to pull members up to.
     * @return Target supertype or null if no target supertype is set.
     */
    public JavaClass getTargetType() {
        return targetType;
    }

    /** Sets target supertype to pull members up to.
     * @param targetType Target supertype.
     */
    public void setTargetType(JavaClass targetType) {
        this.targetType = targetType;
    }

    /** Returns descriptors of the members to pull up.
     * @return Member descriptors.
     */
    public MemberInfo[] getMembers() {
        // never return null
        return members == null ? EMPTY_MEMBERS : members;
    }

    /** Sets members (using their descriptors) to pull up.
     * @param members Descriptors of members to be pulled up.
     */
    public void setMembers(MemberInfo[] members) {
        this.members = members;
    }
    
    // --- HELPER METHODS ------------------------------------------------------
    
    /** Returns supertypes of the source type that the members could
     * be pull up to.
     * @return Supertypes available for pulling up members to.
     */
    public JavaClass[] collectSupertypes() {
        if (supertypes == null) {
            if (sourceType != null) {
                List list = new ArrayList();
                // collect all supertypes recursivelly using a helper method
                collectSupertypes(sourceType, list, new HashSet());
                supertypes = (JavaClass[]) list.toArray(new JavaClass[list.size()]);
            } else {
                supertypes = new JavaClass[0];
            }
        }
        return supertypes;
    }
    
    // helper method for collecting supertypes
    private static void collectSupertypes(JavaClass type, List result, Set visited) {
        JavaClass superClass = Utilities.getRealClass(type.getSuperClass());
        ArrayList supertypes = new ArrayList();
        
        // get superclass (if not visited already)
        if (superClass != null && visited.add(superClass)) {
            supertypes.add(superClass);
            // add it to the result set if its source is available
            if (Utilities.isFromSource(superClass) && !CheckUtils.isFromLibrary(superClass.getResource())) {
                result.add(superClass);
            }
        }
        
        // get all implemented super interfaces (if not visited already)
        for (Iterator it = type.getInterfaces().iterator(); it.hasNext();) {
            JavaClass ifc = Utilities.getRealClass((JavaClass) it.next());
            if (visited.add(ifc)) {
                supertypes.add(ifc);
                // add it to the result set if its source is available
                if (Utilities.isFromSource(ifc)) {
                    result.add(ifc);
                }
            }
        }
        
        // iterate through the collected direct supertypes
        // and collect their supertypes recursivelly
        // (this is done in a separate loop to preserve logical ordering
        // of the supertypes)
        for (Iterator it = supertypes.iterator(); it.hasNext();) {
            collectSupertypes((JavaClass) it.next(), result, visited);
        }
    }
    
   
    // --- HELPER CLASSES ------------------------------------------------------
    
    /** Class describing a member to be pulled up.
     */
    public static final class MemberInfo {
        public final NamedElement member;
        public final boolean makeAbstract;
        
        private MemberInfo(NamedElement member, boolean makeAbstract) {
            this.member = member;
            this.makeAbstract = makeAbstract;
        }

        /** Creates a new instance of MemberInfo describing a method.
         * @param method Method to be pulled up.
         * @param makeAbstract Indicates whether the method should be made abstract
         *              in the supertype.
         */
        public MemberInfo(Method method, boolean makeAbstract) {
            this((NamedElement) method, makeAbstract);
        }
        
        /** Creates a new instance of MemberInfo describing an inner class
         * to be pulled up.
         * @param field Inner class to be pulled up.
         */
        public MemberInfo(JavaClass innerClass) {
            this(innerClass, false);
        }
        
        /** Creates a new instance of MemberInfo describing a field 
         * to be pulled up.
         * @param field Field to be pulled up.
         */
        public MemberInfo(Field field) {
            this(field, false);
        }
        
        /** Creates a new instance of MemberInfo describing an interface name
         * from the implements clause that should be pulled up.
         * @param interfaceName Interface name to be pulled up.
         */
        public MemberInfo(MultipartId interfaceName) {
            this(interfaceName, false);
        }
    }
}
