/*
 * The contents of this file are subject to the terms of the Common Development
 * and Distribution License (the License). You may not use this file except in
 * compliance with the License.
 *
 * You can obtain a copy of the License at http://www.netbeans.org/cddl.html
 * or http://www.netbeans.org/cddl.txt.
 * 
 * When distributing Covered Code, include this CDDL Header Notice in each file
 * and include the License file at http://www.netbeans.org/cddl.txt.
 * If applicable, add the following below the CDDL Header, with the fields
 * enclosed by brackets [] replaced by your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.modules.refactoring;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.netbeans.jmi.javamodel.Element;
import org.netbeans.jmi.javamodel.LocalVarDeclaration;
import org.netbeans.jmi.javamodel.NamedElement;
import org.netbeans.jmi.javamodel.Resource;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.spi.SimpleRefactoringElementImpl;
import org.openide.ErrorManager;
import org.openide.filesystems.FileLock;
import org.openide.filesystems.FileObject;
import org.openide.loaders.DataObject;
import org.openide.loaders.DataObjectNotFoundException;
import org.openide.text.PositionBounds;
import org.openide.util.NbBundle;

/**
 * The RefactoringElement used by SafeDeleteRefactoringPlugin
 * @author Bharath Ravikumar, Jan Becicka
 */
public class SafeDeleteRefactoringElement extends SimpleRefactoringElementImpl {
    
    private final String text;
    
    private Element refactoredElement;
    
    private ExternalChange extChange;
    
    public SafeDeleteRefactoringElement(Element refactoredElementObj) {
        this.refactoredElement = refactoredElementObj;
        if (refactoredElement instanceof Resource) {
            extChange = new ResourceDeletionChange((Resource) refactoredElement);
            FileObject fileObject = JavaModel.getFileObject((Resource) refactoredElement);
            text = NbBundle.getMessage(SafeDeleteRefactoringElement.class, "TXT_SafeDel_Delete", fileObject.getNameExt());
        } else
            if (refactoredElement instanceof NamedElement) {
            text = NbBundle.getMessage(SafeDeleteRefactoringElement.class, "TXT_SafeDel_Delete", ((NamedElement) refactoredElement).getName());
            } else {
            String msg = NbBundle.getMessage(SafeDeleteRefactoringElement.class, "TXT_SafeDel_Delete");
            text = MessageFormat.format(msg, new Object[] {refactoredElement});
            }
    }
    
    public void performChange() {
        if (refactoredElement instanceof Resource) {
            //If a resource has been passed, the absence of any underlying child element has
            //already been ensured.It is hence safe to directly delete it.
            extChange.performExternalChange();
            JavaMetamodel.getManager().registerUndoElement(extChange);
            return;
        } else if (refactoredElement instanceof org.netbeans.jmi.javamodel.LocalVariable) {
            Object comp = refactoredElement.refImmediateComposite();
            if (comp instanceof LocalVarDeclaration) {
                List variables = ((LocalVarDeclaration) comp).getVariables();
                if (variables.size() == 1) {
                    refactoredElement = (Element) comp;
                }
            }
        }
        refactoredElement.refDelete();
        return;
    }
    
    public String getText() {
        return text;
    }
    
    public PositionBounds getPosition() {
        return JavaMetamodel.getManager().getElementPosition(refactoredElement);
    }
    
    public FileObject getParentFile() {
        return JavaMetamodel.getManager().getFileObject(refactoredElement.getResource());
    }
    
    public Element getJavaElement() {
        return refactoredElement;
    }
    
    public String getDisplayText() {
        return getText();
    }
    
    private static class ResourceDeletionChange implements ExternalChange {
        
        private DataObject primaryDataObj;
        
        private FileObject parentFileObj;
        
        private Resource refactoredResource;
        
        private Map fileObjMap = new HashMap();
        
        /**
         * This flag indicates whether the change is being executed the first time
         * or if the current invocation is a result of a Refactoring-Redo.
         * This is needed to store the correct version of the source text for
         * an Undo
         */
        private boolean initialExecution = true;
        
        private ResourceDeletionChange(Resource refactoredResource) {
            this.refactoredResource = refactoredResource;
            try {
                backupFileObjects(refactoredResource);
            } catch (DataObjectNotFoundException ex) {
                ErrorManager.getDefault().notify(ex);
            }
        }
        
        public void performExternalChange() {
            try {
                //Indicates whether this is the first invocation of the external change
                //For the first invocation, the data would've been backed up in the
                //constructor (by which time the containing classes would not have been
                //deleted.So, that's the right place to do the backup.)
                if(!initialExecution)
                    backupFileObjects(refactoredResource); //Record current state
                else
                    initialExecution = false;
                
                //Now do the deletion of data object
                primaryDataObj.delete();
                refactoredResource.refDelete();
            } catch (IOException exception) {
                ErrorManager.getDefault().notify(exception);
                return;
            }
        }
        
        public void undoExternalChange() {
            try {
                populateFileObjects();
            } catch (DataObjectNotFoundException notFndEx) {
                ErrorManager.getDefault().notify(notFndEx);
            } catch (IOException ioEx) {
                ErrorManager.getDefault().notify(ioEx);
            }
        }
        
        private void backupFileObjects(final Resource refactoredResource) throws DataObjectNotFoundException {
            FileObject javaFileObject = JavaModel.getFileObject((Resource) refactoredResource);
            //set the parent folder's file object
            parentFileObj = javaFileObject.getParent();
            primaryDataObj = DataObject.find(javaFileObject);
            Set fileObjSet = primaryDataObj.files();
            Iterator fileObjIterator = fileObjSet.iterator();
            while (fileObjIterator.hasNext()) {
                FileObject fileObj = (FileObject) fileObjIterator.next();
                String fileName = fileObj.getNameExt();
                try {
                    String fileContent = readFileContent(fileObj);
                    fileObjMap.put(fileName, fileContent);
                } catch (FileNotFoundException ex) {
                    ErrorManager.getDefault().notify(ex);
                } catch (IOException ex) {
                    ErrorManager.getDefault().notify(ex);
                }
            }
        }
        
        private String readFileContent(FileObject fileObject) throws FileNotFoundException, IOException {
            Resource resource = JavaModel.getResource(fileObject);
            if (resource != null)
                return resource.getSourceText();
            else {
                StringBuffer buffer = new StringBuffer();
                BufferedReader bis = new BufferedReader(new InputStreamReader(fileObject.getInputStream()));
                String sourceString = null;
                while( (sourceString = bis.readLine()) != null)
                    buffer.append(sourceString );
                bis.close();
                return buffer.toString();
            }
        }
        
        private void populateFileObjects() throws DataObjectNotFoundException, IOException {
            Iterator fileObjIte = fileObjMap.keySet().iterator();
            FileObject fObj = null;
            while (fileObjIte.hasNext()) {
                String fileName = fileObjIte.next().toString();
                //Generate a key to retrieve this file later.
                fObj = parentFileObj.createData(fileName);
                writeToFile(fObj, (String) fileObjMap.get(fileName));
            }
            primaryDataObj = DataObject.find(fObj);
            refactoredResource = JavaMetamodel.getManager().getResource(primaryDataObj.getPrimaryFile());
        }
        
        private void writeToFile(FileObject fileObject, String fileContent) throws IOException {
            FileLock lock = null;
            PrintWriter writer = null;
            
            try {
                lock = fileObject.lock();
                OutputStream fOS = fileObject.getOutputStream(lock);
                writer = new PrintWriter(fOS);
                writer.print(fileContent);
            } finally {
                if(writer != null) {
                    writer.close();
                }
                if(lock != null) {
                    lock.releaseLock();
                }
            }
        }
    }
}
